#!/usr/bin/env python3

# Copyright 2017 Patrick O. Perry.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import math
import re

PATTERN = re.compile(r"""^([0-9A-Fa-f]+)        # (first code)
                          (\.\.([0-9A-Fa-f]+))? # (.. last code)?
                          \s*
                          ;                     # ;
                          \s*
                          (\w+)                 # (property name)
                          \s*
                          ;                     # ;
                          \s*
                          (\w+)                 # (property value)
                          \s*
                          (\#.*)?$              # (# comment)?""", re.X)

UNICODE_MAX = 0x10FFFF

DERIVED_NORMALIZATION_PROPS = "data/ucd/DerivedNormalizationProps.txt"
try:
    infile = open(DERIVED_NORMALIZATION_PROPS)
except FileNotFoundError:
    infile = open("../" + DERIVED_NORMALIZATION_PROPS, "r")

code_props = ['Yes'] * (UNICODE_MAX + 1)

for line in infile:
    m = PATTERN.match(line)
    if m is None or m.group(4) != 'NFC_QC':
        continue
    begin = int(m.group(1), 16)
    if m.group(3) is None:
        end = begin + 1
    else:
        end = int(m.group(3), 16) + 1
    prop = m.group(5)
    if prop == 'M':
        prop = 'Maybe'
    elif prop == 'N':
        prop = 'No'
    elif prop == 'Y':
        prop = 'Yes'
    for code in range(begin, end):
        code_props[code] = prop
    #print(line, end = "")
    #print('[', '{:04X}'.format(begin), ',', '{:04X}'.format(end), '): ',
    #      prop, sep = '')
infile.close()


prop_names = set(code_props)

prop_vals = {}
prop_vals['No'] = 0
prop_vals['Yes'] = 1
prop_vals['Maybe'] = -1

def compute_tables(block_size):
    nblock = len(code_props) // block_size
    stage1 = [None] * nblock
    stage2 = []
    stage2_dict = {}
    for i in range(nblock):
        begin = i * block_size
        end = begin + block_size
        block = tuple(code_props[begin:end])
        if block in stage2_dict:
            j = stage2_dict[block]
        else:
            j = len(stage2)
            stage2_dict[block] = j
            stage2.append(block)
        stage1[i] = j
    return (stage1,stage2)

def stage1_item_size(nstage2):
    nbyte = math.ceil(math.log(nstage2, 2) / 8)
    size = 2**math.ceil(math.log(nbyte, 2))
    return size

page_size = 4096
block_size = 256

nbytes = {}

best_block_size = 1
smallest_size = len(code_props)

for i in range(1,17):
    block_size = 2**i
    stage1,stage2 = compute_tables(block_size)
    #
    nbyte1 = len(stage1) * stage1_item_size(len(stage2))
    nbyte2 = len(stage2) * block_size
    #
    nbyte1 = math.ceil(nbyte1 / page_size) * page_size
    nbyte2 = math.ceil(nbyte2 / page_size) * page_size
    nbyte = nbyte1 + nbyte2
    nbytes[block_size] = nbyte
    #
    if nbyte < smallest_size:
        smallest_size = nbyte
        best_block_size = block_size


block_size = best_block_size
stage1,stage2 = compute_tables(block_size)

type1_size = stage1_item_size(len(stage2))

if type1_size == 1:
    type1 = 'uint8_t'
elif type1_size == 2:
    type1 = 'uint16_t'
elif type1_size == 4:
    type1 = 'uint32_t'
else:
    type1 = 'uint64_t'

type2 = 'int8_t'



# Write normalizationprop.h to stdout

print("/* This file is automatically generated. DO NOT EDIT!")
print("   Instead, edit gen-normalization.py and re-run.  */")
print("")
print("/*")
print(" * Unicode NFC_QC property values.")
print(" *")
print(" * Defined in UAX #15 \"Unicode Normalization Forms\"")
print(" *")
print(" *     http://www.unicode.org/reports/tr15/")
print(" *")
print(" * Section 9, \"Detecting Normalization Forms.\"")
print(" *")
print(" *")
print(" * We use the two-stage lookup strategy described at")
print(" *")
print(" *     http://www.strchr.com/multi-stage_tables")
print(" *")
print(" */")
print("")
print("#ifndef NORMALIZATIONPROP_H")
print("#define NORMALIZATIONPROP_H")
print("")
print("#include <stdint.h>")
print("")
print("enum nfc_qc_prop {")
for i in range(len(prop_names)):
    prop = sorted(prop_names)[i]
    if i > 0:
        print(",")
    print("\tNFC_QC_" + prop.upper() + " = " + str(prop_vals[prop]), end="")
print("\n};")
print("")
print("static const " + type1 + " nfc_qc_stage1[] = {")
for i in range(len(stage1) - 1):
    if i % 16  == 0:
        print("/* U+{:04X} */".format(i * block_size), end="")
    print("{0: >3},".format(stage1[i]), end="")
    if i % 16 == 15:
        print("")
print("{0: >3}".format(stage1[len(stage1) - 1]))
print("};")
print("")
print("static const " + type2 + " nfc_qc_stage2[][" +
        str(block_size) + "] = {")
for i in range(len(stage2)):
    print("  /* block " + str(i) + " */")
    print("  {", end="")
    for j in range(block_size):
        print("{0: >3}".format(prop_vals[stage2[i][j]]), end="")
        if j + 1 == block_size:
            print("\n  }", end="")
        else:
            print(",", end="")
            if j % 16 == 15:
                print("\n   ", end="")
    if i + 1 != len(stage2):
        print(",\n")
    else:
        print("")
print("};")

print("")
print("static int nfc_qc(uint32_t code)")
print("{")
print("\tconst uint32_t block_size = " + str(block_size) + ";")
print("\t" + type1 + " i = nfc_qc_stage1[code / block_size];")
print("\treturn nfc_qc_stage2[i][code % block_size];")
print("}")
print("")
print("#endif /* NORMALIZATIONPROP_H */")
