% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/r2_bayes.R
\name{r2_bayes}
\alias{r2_bayes}
\alias{r2_posterior}
\alias{r2_posterior.brmsfit}
\alias{r2_posterior.stanreg}
\alias{r2_posterior.BFBayesFactor}
\title{Bayesian R2}
\usage{
r2_bayes(model, robust = TRUE, ci = 0.95, verbose = TRUE, ...)

r2_posterior(model, ...)

\method{r2_posterior}{brmsfit}(model, verbose = TRUE, ...)

\method{r2_posterior}{stanreg}(model, verbose = TRUE, ...)

\method{r2_posterior}{BFBayesFactor}(model, average = FALSE, prior_odds = NULL, verbose = TRUE, ...)
}
\arguments{
\item{model}{A Bayesian regression model (from \strong{brms},
\strong{rstanarm}, \strong{BayesFactor}, etc).}

\item{robust}{Logical, if \code{TRUE}, the median instead of mean is used to
calculate the central tendency of the variances.}

\item{ci}{Value or vector of probability of the CI (between 0 and 1) to be
estimated.}

\item{verbose}{Toggle off warnings.}

\item{...}{Arguments passed to \code{r2_posterior()}.}

\item{average}{Compute model-averaged index? See \code{\link[bayestestR:weighted_posteriors]{bayestestR::weighted_posteriors()}}.}

\item{prior_odds}{Optional vector of prior odds for the models compared to
the first model (or the denominator, for \code{BFBayesFactor} objects). For
\code{data.frame}s, this will be used as the basis of weighting.}
}
\value{
A list with the Bayesian R2 value. For mixed models, a list with the
Bayesian R2 value and the marginal Bayesian R2 value. The standard errors
and credible intervals for the R2 values are saved as attributes.
}
\description{
Compute R2 for Bayesian models. For mixed models (including a
random part), it additionally computes the R2 related to the fixed effects
only (marginal R2). While \code{r2_bayes()} returns a single R2 value,
\code{r2_posterior()} returns a posterior sample of Bayesian R2 values.
}
\details{
\code{r2_bayes()} returns an "unadjusted" R2 value. See
\code{\link[=r2_loo]{r2_loo()}} to calculate a LOO-adjusted R2, which comes
conceptually closer to an adjusted R2 measure.

For mixed models, the conditional and marginal R2 are returned. The marginal
R2 considers only the variance of the fixed effects, while the conditional
R2 takes both the fixed and random effects into account.

\code{r2_posterior()} is the actual workhorse for \code{r2_bayes()} and
returns a posterior sample of Bayesian R2 values.
}
\examples{
\dontshow{if (require("rstanarm") && require("rstantools") && require("BayesFactor") && require("brms")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(performance)
\donttest{
model <- suppressWarnings(rstanarm::stan_glm(
  mpg ~ wt + cyl,
  data = mtcars,
  chains = 1,
  iter = 500,
  refresh = 0,
  show_messages = FALSE
))
r2_bayes(model)

model <- suppressWarnings(rstanarm::stan_lmer(
  Petal.Length ~ Petal.Width + (1 | Species),
  data = iris,
  chains = 1,
  iter = 500,
  refresh = 0
))
r2_bayes(model)
}

BFM <- BayesFactor::generalTestBF(mpg ~ qsec + gear, data = mtcars, progress = FALSE)
FM <- BayesFactor::lmBF(mpg ~ qsec + gear, data = mtcars)

r2_bayes(FM)
r2_bayes(BFM[3])
r2_bayes(BFM, average = TRUE) # across all models

# with random effects:
mtcars$gear <- factor(mtcars$gear)
model <- BayesFactor::lmBF(
  mpg ~ hp + cyl + gear + gear:wt,
  mtcars,
  progress = FALSE,
  whichRandom = c("gear", "gear:wt")
)

r2_bayes(model)

\donttest{
model <- suppressWarnings(brms::brm(
  mpg ~ wt + cyl,
  data = mtcars,
  silent = 2,
  refresh = 0
))
r2_bayes(model)

model <- suppressWarnings(brms::brm(
  Petal.Length ~ Petal.Width + (1 | Species),
  data = iris,
  silent = 2,
  refresh = 0
))
r2_bayes(model)
}
\dontshow{\}) # examplesIf}
}
\references{
Gelman, A., Goodrich, B., Gabry, J., and Vehtari, A. (2018).
R-squared for Bayesian regression models. The American Statistician, 1–6.
\doi{10.1080/00031305.2018.1549100}
}
