/*	Data_Table

PIRL CVS ID: Data_Table.java,v 1.9 2012/04/16 06:08:56 castalia Exp

Copyright (C) 2003-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Database;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.table.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Vector;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.Set;
import java.util.Iterator;
import java.util.EventObject;
import java.lang.Float;

/**	<I>Data_Table</I> manages a JTable for Database_View.
<P>
	@author	Bradford Castalia, UA/PIRL
	@version 1.9
*/
public class Data_Table
	extends JPanel
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Database.Data_Table (1.9 2012/04/16 06:08:56)";

private JScrollPane				Table_View_Pane;

//	Data table elements.
private JTable					Table_View;
private Data_Table_Model		Table_Model;

//	Record numbers table elements.
private JTable					Record_Numbers;
private AbstractTableModel		Record_Numbers_Table_Model;
private TableColumnModel		Record_Numbers_Column_Model;
private JLabel					Record_Numbers_Label,
								Total_Records_Label;
private static final int		RECORD_NUMBERS_COLUMN_WIDTH = 70;

//	Colors.
private Color					Record_Numbers_Color = new Color
									((float)0.85, (float)1.00, (float)0.85),
								Uneditable_Cell_Color = new Color
									((float)1.00, (float)1.00, (float)1.00),
								Edited_Cell_Color = new Color
									((float)1.00, (float)1.00, (float)0.00),
								Editable_Cell_Color = new Color
									((float)1.00, (float)1.00, (float)0.90);
//	Work-around for bug in OS-X java.
private static final Color		BLACK = new Color
									((float)0.00, (float)0.00, (float)0.00);

private static final String		NL = Database.NL;


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_SETUP			= 1 << 0,
	DEBUG_UI_LAYOUT		= 1 << 1,
	DEBUG_LOCATIONS		= 1 << 2,
	DEBUG_EDITOR		= 1 << 3,
	DEBUG_RENDERER		= 1 << 4,
	DEBUG_COLUMN_MODEL	= 1 << 5,
	DEBUG_DATA			= 1 << 6,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
public Data_Table
	(
	Vector	data_values,
	Vector	field_values,
	boolean	editable,
	Vector	editable_fields
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> Data_Table");
//	Create the table's data model.
Table_Model = new Data_Table_Model (data_values, field_values);
//	Set the editable fields.
Table_Model.Editable (editable, editable_fields);

//	Build the table view.
Create_GUI ();
Set_Column_Sizes (Table_View);
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("<<< Data_Table");
}

public Data_Table
	(
	Vector	data_values,
	Vector	field_values,
	boolean	editable
	)
{this (data_values, field_values, editable, null);}

public Data_Table
	(
	Vector	data_values,
	Vector	field_values
	)
{this (data_values, field_values, true, null);}

public Data_Table
	(
	Vector	table,
	boolean	editable,
	Vector	editable_fields
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> Data_Table");
Vector
	field_names = null;
if (table != null && ! table.isEmpty ())
	field_names = (Vector)table.remove (0);
//	Create the table's data model.
Table_Model = new Data_Table_Model (table, field_names);

//	Set the editable fields.
Table_Model.Editable (editable, editable_fields);

//	Build the table view.
Create_GUI ();
Set_Column_Sizes (Table_View);
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("<<< Data_Table");
}

public Data_Table
	(
	Vector	table,
	boolean	editable
	)
{this (table, editable, null);}

public Data_Table
	(
	Vector	table
	)
{this (table, true, null);}

public Data_Table ()
{this (null, false, null);}

/*==============================================================================
	Accessors
*/
public JTable Table_View ()
{return Table_View;}

public Data_Table_Model Table_Model ()
{return Table_Model;}

/*------------------------------------------------------------------------------
	Data Table
*/
public void Data
	(
	Vector	data_values,
	Vector	field_names,
	boolean	editable,
	Vector	editable_fields
	)
{
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println (">>> Data_Table.Data");

//	Apply the data to the table's data model.
Table_Model.Data (data_values, field_names, editable, editable_fields);

//	Update the table pane.
Set_Column_Sizes (Table_View);
Show_Total_Records ();

if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println ("<<< Data_Table.Data");
}

public void Data
	(
	Vector	data_values,
	Vector	field_names,
	boolean	editable
	)
{Data (data_values, field_names, editable, null);}

public void Data
	(
	Vector	table,
	boolean	editable
	)
{
Vector
	field_names = null;
if (table != null && ! table.isEmpty ())
	field_names = (Vector)table.remove (0);
Data (table, field_names, editable, null);
}

public void Data
	(
	Vector	table
	)
{Data (table, true);}

public void Data ()
{Data (null, null, false, null);}

public void setDataVector
	(
	Vector	data_values,
	Vector	field_names
	)
{Data (data_values, field_names, true, null);}

/*------------------------------------------------------------------------------
	Cell Editing
*/
public void Editable
	(
	boolean	editable,
	int		field_number
	)
{Table_Model.Editable (editable, field_number); repaint ();}

public void Editable
	(
	boolean	editable,
	Vector	field_numbers
	)
{Table_Model.Editable (editable, field_numbers); repaint ();}

public void Editable
	(
	boolean	editable
	)
{Table_Model.Editable (editable); repaint ();}

public Vector Edited_Cells ()
{return Table_Model.Edited_Cells ();}

public void Clear_Edited_Cells ()
{Table_Model.Clear_Edited_Cells ();}

/*------------------------------------------------------------------------------
	Colors
*/
public Color Record_Numbers_Color ()
{return Record_Numbers_Color;}

public Color Record_Numbers_Color
	(
	Color	color
	)
{
Color previous_color = Record_Numbers_Color;
Record_Numbers_Color = color;
Record_Numbers_Label.setBackground (Record_Numbers_Color);
Record_Numbers.setBackground (Record_Numbers_Color);
Total_Records_Label.setBackground (Record_Numbers_Color);
return previous_color;
}

public Color Uneditable_Cell_Color ()
{return Uneditable_Cell_Color;}

public Color Uneditable_Cell_Color
	(
	Color	color
	)
{
Color previous_color = Uneditable_Cell_Color;
Uneditable_Cell_Color = color;
return previous_color;
}

public Color Edited_Cell_Color ()
{return Edited_Cell_Color;}

public Color Edited_Cell_Color
	(
	Color	color
	)
{
Color previous_color = Edited_Cell_Color;
Edited_Cell_Color = color;
return previous_color;
}

public Color Editable_Cell_Color ()
{return Editable_Cell_Color;}

public Color Editable_Cell_Color
	(
	Color	color
	)
{
Color previous_color = Editable_Cell_Color;
Editable_Cell_Color = color;
return previous_color;
}

/*==============================================================================
	Table GUI
*/
private void Create_GUI ()
{
if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println (">>> Data_Table.Create_GUI");
GridBagLayout grid	= new GridBagLayout ();
GridBagConstraints location = new GridBagConstraints ();
setLayout (grid);

/*------------------------------------------------------------------------------
	Record Numbers

	Table record numbering is provided by a JTable in a Viewport that
	is located in the row header area of the ScrollPane containing the
	main JTable for the table data. The JTable for row numbers uses a
	trivial AbstractTableModel that presents a single column of Integer
	values corresponding to the row number (+1 for row counts). These
	values are generated on demand by the <CODE>getValueAt</CODE>
	method. The number of rows is obtained from the <CODE>
	Table_Model.getRowCount</CODE> method. The main Table_Model JTable
	shares the same SelectionModel to keep selections in sync.
*/
if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println
		("    Data_Table.Create_GUI - Constructing Record_Numbers_Table_Model");
Record_Numbers_Table_Model = new AbstractTableModel ()
	{
	public String getColumnName (int column)
	{return "Record";}
	public int getRowCount ()
	{return Table_Model.getRowCount ();}
	public int getColumnCount ()
	{return 1;}
	public Object getValueAt (int row, int column)
	{return new Integer (row + 1);}
	public Class getColumnClass (int column)
	{return Integer.class;}
	};
if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println
		("    Data_Table.Create_GUI - Constructing Record_Numbers_Column_Model");
Record_Numbers_Column_Model = new DefaultTableColumnModel ()
	{
	public void addColumn (TableColumn column)
	{
	if ((DEBUG & DEBUG_COLUMN_MODEL) != 0)
		System.out.println
			("--- Data_Table.Record_Numbers_Column_Model.addColumn: "
			+ column.getModelIndex ());
	DefaultTableCellRenderer
		renderer = new DefaultTableCellRenderer ();
	renderer.setHorizontalAlignment (SwingConstants.RIGHT);
	column.setCellRenderer (renderer);
	column.setMaxWidth (RECORD_NUMBERS_COLUMN_WIDTH);
	column.setResizable (false);
	super.addColumn (column);
	}};

if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		("    Data_Table.Create_GUI: TableModelListener enabled.");
//	Coordinate the Record_Numbers_Table_Model with the Table_Model.
Table_Model.addTableModelListener (new TableModelListener ()
	{
	public void tableChanged (TableModelEvent event)
	{
	int
		operation = event.getType ();
	if ((DEBUG & DEBUG_DATA) != 0)
		{
		System.out.println ("--- Data_Table.TableModelListener event:");
		switch (operation)
			{
			case TableModelEvent.DELETE:
				System.out.print ("DELETE"); break;
			case TableModelEvent.INSERT:
				System.out.print ("INSERT"); break;
			case TableModelEvent.UPDATE:
				System.out.print ("UPDATE"); break;
			default:
				System.out.print ("Type " + operation);
			}
		System.out.println
			(": rows " + event.getFirstRow () + " to " + event.getLastRow ()
			+", column " + event.getColumn ());
		}
	if (operation == TableModelEvent.DELETE)
		Record_Numbers_Table_Model.fireTableRowsDeleted
			(event.getFirstRow (), event.getLastRow ());
	else if (operation == TableModelEvent.INSERT)
		Record_Numbers_Table_Model.fireTableRowsInserted
			(event.getFirstRow (), event.getLastRow ());
	if (operation != TableModelEvent.UPDATE)
		Total_Records_Label.setText
			(String.valueOf (Table_View.getRowCount ()));
	}});

if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println
		("    Data_Table.Create_GUI - Constructing Record_Numbers JTable");
Record_Numbers = new JTable
	(Record_Numbers_Table_Model, Record_Numbers_Column_Model);
if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println
		("    Data_Table.Create_GUI - createDefaultColumnsFromModel");
Record_Numbers.createDefaultColumnsFromModel ();
Record_Numbers.setMaximumSize (new Dimension
	(RECORD_NUMBERS_COLUMN_WIDTH, 10000));
Record_Numbers.setBackground (Record_Numbers_Color);
Record_Numbers.setColumnSelectionAllowed (false);
Record_Numbers.setCellSelectionEnabled (false);
Record_Numbers.setAutoResizeMode (JTable.AUTO_RESIZE_OFF);

/*------------------------------------------------------------------------------
	Data Table
*/
if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println
		("    Data_Table.Create_GUI - Constructing Table_View JTable");
Table_View = new JTable (Table_Model);
//Table_View.createDefaultColumnsFromModel ();
Table_View.setAutoResizeMode (JTable.AUTO_RESIZE_ALL_COLUMNS);

//	Override the default cell renderer and editor.
Table_View.setDefaultRenderer (Object.class, new Data_Cell_Renderer ());
Table_View.setDefaultEditor (Object.class, new Data_Cell_Editor ());

//	Share the SelectionModel to keep selections in sync.
Table_View.setSelectionModel (Record_Numbers.getSelectionModel ());
Table_View.setCellSelectionEnabled (true);

Table_View.getTableHeader ().setBorder (BorderFactory.createMatteBorder
	(0, 1, 0, 0, BLACK));

/*------------------------------------------------------------------------------
	Scroll pane
*/
Table_View_Pane = new JScrollPane (Table_View);
Table_View_Pane.setViewportBorder (BorderFactory.createMatteBorder
	(0, 1, 0, 0, BLACK));

//	Put the record numbers table in a viewport in the scroll pane row header.
JViewport viewport = new JViewport ();
viewport.setView (Record_Numbers);
viewport.setPreferredSize (Record_Numbers.getMaximumSize ());
Table_View_Pane.setRowHeader (viewport);

//	Put a record numbers table header in the scroll pane upper-left corner.
Record_Numbers_Label	= new JLabel ("Record", SwingConstants.CENTER);
Record_Numbers_Label.setBorder (BorderFactory.createMatteBorder
	(0, 0, 2, 0, BLACK));
Record_Numbers_Label.setMinimumSize (new Dimension
	(RECORD_NUMBERS_COLUMN_WIDTH + 3,
	Table_View.getTableHeader ().getHeight ()));
Record_Numbers_Label.setBackground (Record_Numbers_Color);
Record_Numbers_Label.setOpaque (true);
Table_View_Pane.setCorner (JScrollPane.UPPER_LEFT_CORNER, Record_Numbers_Label);

//	Put a total records label in the scroll pane lower-left corner.
Total_Records_Label		= new JLabel
	(String.valueOf (Table_View.getRowCount ()), SwingConstants.RIGHT);
Total_Records_Label.setBorder (BorderFactory.createMatteBorder
	(2, 0,0, 0, BLACK));
Total_Records_Label.setMinimumSize (new Dimension
	(RECORD_NUMBERS_COLUMN_WIDTH + 3, Record_Numbers.getRowHeight ()));
Total_Records_Label.setBackground (Record_Numbers_Color);
Total_Records_Label.setOpaque (true);
Table_View_Pane.setCorner (JScrollPane.LOWER_LEFT_CORNER, Total_Records_Label);

//	Keep the horizonal scroll bar aways visible.
Table_View_Pane.setHorizontalScrollBarPolicy
	(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);

//	Put the scroll pane into the enclosing pane.
location.insets		= new Insets (0, 10, 0, 10);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
add (Table_View_Pane, location);

if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println ("<<< Data_Table.Create_GUI");
}

/*	Sets the width of each table column to accommodate its field name,
	and turns off column auto resizing if the table is wider than its pane.
*/
private void Set_Column_Sizes
	(
	JTable	table
	)
{
if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println
		(">>> Data_Table.Set_Column_Sizes: "
		+ table.getColumnCount () + " columns");
TableCellRenderer
	header = Table_View.getTableHeader().getDefaultRenderer();
TableColumn
	column;
int
	column_number,
	column_count = Table_View.getColumnCount (),
	column_width,
	table_width = 0;

for (column_number = 0;
	 column_number < column_count;
	 column_number++)
	{
	column = Table_View.getColumnModel ().getColumn (column_number);
	column_width = header.getTableCellRendererComponent
		(null, column.getHeaderValue (), false, false, 0, 0)
		.getPreferredSize ().width + 8;
	column.setPreferredWidth (column_width);
	column.setMinWidth (column_width);
	table_width += column_width;
	if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
		System.out.println
			("    " + column_number + ": " + column_width);
	}
table_width += Record_Numbers.getWidth ();
if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println
		("    Table_View_Pane width = " + Table_View_Pane.getWidth () + NL
		+"              table_width = " + table_width);
Table_View.setAutoResizeMode
	((table_width > Table_View_Pane.getWidth ()) ?
		JTable.AUTO_RESIZE_OFF : JTable.AUTO_RESIZE_ALL_COLUMNS);
Table_View.doLayout ();
if ((DEBUG & DEBUG_UI_LAYOUT) != 0)
	System.out.println ("<<< Data_Table.Set_Column_Sizes");
}

private void Show_Total_Records ()
{
Total_Records_Label.setText (String.valueOf (Table_View.getRowCount ()));
Record_Numbers_Table_Model.fireTableDataChanged ();
}

/*==============================================================================
	Cell Renderer
*/
class Data_Cell_Renderer
	extends DefaultTableCellRenderer
{
public Component getTableCellRendererComponent
	(
	JTable	table,
	Object	value,
	boolean	is_selected,
	boolean	has_focus,
	int		row,
	int		column
	)
{
if ((DEBUG & DEBUG_RENDERER) != 0)
	System.out.println
		("--- Data_Table.Data_Cell_Renderer.getTableCellRendererComponent:" + NL
		+ "    " + row + ',' + column
		+ ": selected = " + is_selected
		+ ", has focus = " + has_focus
		+ " - " + value);
Component
	component = super.getTableCellRendererComponent
					(table, value, is_selected, has_focus, row, column);
if (value instanceof String)
	{
	try
		{
		Float.valueOf ((String)value);
		setHorizontalAlignment (SwingConstants.RIGHT);
		}
	catch (NumberFormatException exception)
		{setHorizontalAlignment (SwingConstants.LEFT);}
	}
else
	setHorizontalAlignment (SwingConstants.LEFT);
if (Table_Model.is_Cell_Edited (row, column))
	component.setBackground (Edited_Cell_Color);
else if (Table_Model.isCellEditable (row, column))
	component.setBackground (Editable_Cell_Color);
else
	component.setBackground (Uneditable_Cell_Color);
return component;
}
}	//	End of Data_Cell_Renderer class.

/*==============================================================================
	Cell Editor
*/
class Data_Cell_Editor
	extends DefaultCellEditor
{
private int			Edited_Row		= -1,
					Edited_Column	= -1;

Data_Cell_Editor ()
{
super (new JTextField ());
super.setClickCountToStart (2);
if ((DEBUG & DEBUG_EDITOR) != 0)
	System.out.println
		("--- Data_Table.Data_Cell_Editor:");
((JTextField)super.getComponent ()).addKeyListener (new KeyListener ()
	{
	public void keyPressed (KeyEvent key_event) {}

	public void keyTyped (KeyEvent key_event)
	{
	if ((DEBUG & DEBUG_EDITOR) != 0)
		System.out.println
			("--- Data_Table.Data_Cell_Editor: keyTyped:");
	if (Edited_Row < 0)
		{
		Edited_Row = Table_View.getEditingRow ();
		Edited_Column = Table_View.getEditingColumn ();
		if ((DEBUG & DEBUG_EDITOR) != 0)
			System.out.println
				("    Edited_Row " + Edited_Row
				+ ", Edited_Column " + Edited_Column);
		}
	}

	public void keyReleased (KeyEvent key_event) {}
	});
}

public Component getTableCellEditorComponent
	(
	JTable	table,
	Object	value,
	boolean	is_selected,
	int		row,
	int		column
	)
{
if ((DEBUG & DEBUG_EDITOR) != 0)
	System.out.println
		("--- Data_Table.Data_Cell_Editor.getTableCellEditorComponent:" + NL
		+ "    " + row + ',' + column
		+ ": selected = " + is_selected
		+ " - " + value);
Edited_Row = -1;
return super.getTableCellEditorComponent
			(table, value, is_selected, row, column);
}

public boolean stopCellEditing ()
{
if ((DEBUG & DEBUG_EDITOR) != 0)
	System.out.println
		("--- Data_Table.Data_Cell_Editor.stopCellEditing: "
		+ Table_View.getSelectedRow () + ","
		+ Table_View.getSelectedColumn ());
if (Edited_Row >= 0)
	{
	//Table_Model.Edited_Location (Edited_Row, Edited_Column);
	if ((DEBUG & DEBUG_EDITOR) != 0)
		Table_Model.Edited_Cells ();
	}
return super.stopCellEditing ();
}
}	//	End of Data_Cell_Editor class.

/*==============================================================================
	Testing stub
*/
public static void main (String[] arguments)
{
int
	rows    = 30,
	columns = 15;
if (arguments.length > 0)
	{
	try
		{
		rows = Integer.parseInt (arguments[0]);
		if (arguments.length > 1)
			columns = Integer.parseInt (arguments[1]);
		}
	catch (NumberFormatException exception) {rows = -1;}
	if (rows < 0 || columns < 0)
		{
		System.out.println (NL
			+"Usage: Data_Table [<rows> [<columns>]");
		System.exit (1);
		}
	}

//	Main panel.
JPanel panel = new JPanel ();
GridBagLayout grid	= new GridBagLayout ();
GridBagConstraints location = new GridBagConstraints ();
panel.setLayout (grid);

//	Select fields that can be edited.
Vector
	editable_fields = new Vector ();
editable_fields.add (new Integer (0));
editable_fields.add (new Integer (2));
editable_fields.add (new Integer (4));

//	The Data_Table.
final Data_Table
	Table = new Data_Table (table_data (rows, columns), true, editable_fields);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
panel.add (Table, location);

//	Report a listing of edited cells.
JButton
	Report = new JButton ("Report");
Report.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{
	Vector
		edited_cells = Table.Edited_Cells ();
	System.out.println (edited_cells.size () + " cells edited");
	Iterator
		cells = edited_cells.iterator ();
	while (cells.hasNext ())
		{
		Cell
			cell = (Cell)cells.next ();
		System.out.println
			("    " + cell.row + "," + cell.column + ": " + cell.new_value);
		}
	}});
location.gridwidth	= 1;
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
location.weighty	= 0.0;
panel.add (Report, location);

//	Reset all edited cells.
JButton
	Reset = new JButton ("Reset");
Reset.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Table.Clear_Edited_Cells ();}});
panel.add (Reset, location);

//	Turn editing off and on.
final JButton
	Editable = new JButton ("Edit None");
Editable.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{
	String
		label = Editable.getText ();
	if (label.equals ("Edit None"))
		{
		Table.Editable (false);
		Editable.setText ("Edit All");
		}
	else
		{
		Table.Editable (true);
		Editable.setText ("Edit None");
		}
	}});
panel.add (Editable, location);

//	Reload new rows and columns.
JLabel label = new JLabel ("  Rows:");
location.anchor		= GridBagConstraints.EAST;
panel.add (label, location);
final JTextField Rows = new JTextField (String.valueOf (rows), 5);
location.anchor		= GridBagConstraints.WEST;
panel.add (Rows, location);

label = new JLabel ("  Columns:");
location.anchor		= GridBagConstraints.EAST;
panel.add (label, location);
final JTextField Columns = new JTextField (String.valueOf (columns), 5);
location.anchor		= GridBagConstraints.WEST;
panel.add (Columns, location);

Rows.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{
	try
		{
		int
			rows    = Integer.parseInt (Rows.getText ()),
			columns = Integer.parseInt (Columns.getText ());
		Table.Data (table_data (rows, columns));
		}
	catch (NumberFormatException exception) {}
	}});
Columns.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{
	try
		{
		int
			rows    = Integer.parseInt (Rows.getText ()),
			columns = Integer.parseInt (Columns.getText ());
		Table.Data (table_data (rows, columns));
		}
	catch (NumberFormatException exception) {}
	}});

final JFrame
	frame = new JFrame ("Data_Table");
frame.addWindowListener (new WindowAdapter ()
	{public void windowClosing (WindowEvent e)
	{System.exit (0);}});
frame.setContentPane (panel);
frame.pack ();
frame.setVisible (true);
}

private static Vector table_data
	(
	int		rows,
	int		columns
	)
{
//	Assemble a table of data.
Vector
	table = new Vector (),
	record;
int
	row,
	column;
for (row = 0;
	 row < rows + 1;
	 row++)
	{
	record = new Vector ();
	for (column = 0;
		 column < columns;
		 column++)
		{
		if (row == 0)
			record.add ("Field " + column);
		else if (row == 1)
			record.add ("cell "
				+ String.valueOf (row - 1) + '.' + String.valueOf (column));
		else
			record.add
				(String.valueOf (row - 1) + '.' + String.valueOf (column));
		}
	table.add (record);
	}
return table;
}

}	//	End of Data_Table class.

/* *****************************************************************************
	Data Table Model
*/
class Data_Table_Model
	extends DefaultTableModel
{
public static final String
	ID = "PIRL.Database.Data_Table_Model (1.9 2012/04/16 06:08:56)";

private static Vector	EMPTY_FIELD = new Vector (1);
static 					{EMPTY_FIELD.add ("Empty");}
private static Vector	EMPTY_TABLE = new Vector (1);

//	The Set of column numbers (Integer) that can be edited.
private TreeSet			Editable_Fields = new TreeSet ();

/**	A map of data table cells that have been edited.
<P>
	The Edited_Cells_Map maps the number of each row (as an Integer)
	that contains an edited cell to a TreeMap of each column number (as
	an Integer) in the row that was edited mapped to its old (unedited)
	value. The use of TreeMaps ensures that the Edited_Cells_Map will
	be ordered by row and by column within rows.
*/
private TreeMap			Edited_Cells_Map = new TreeMap ();


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_DATA_VALUE	= 1 << 0,
	DEBUG_DATA_CHANGE	= 1 << 1,
	DEBUG_EDITABLE		= 1 << 2,
	DEBUG_METADATA		= 1 << 3,
	DEBUG_LOCATIONS		= 1 << 4,
	DEBUG_ALL			= -1,

	DEBUG				=	DEBUG_OFF;

/*==============================================================================
	Constructors
*/
public Data_Table_Model ()
{super (EMPTY_TABLE, EMPTY_FIELD);}

public Data_Table_Model
	(
	Vector	data_values,
	Vector	field_names
	)
{
super (((data_values == null || field_names == null || field_names.isEmpty ()) ?
			EMPTY_TABLE : data_values),
       ((data_values == null || field_names == null || field_names.isEmpty ()) ?
			EMPTY_FIELD : field_names));
}

/*==============================================================================
	AbstractTableModel interface
*/
public void setValueAt
	(
	Object	value,
	int		row,
	int		column
	)
	throws ArrayIndexOutOfBoundsException
{
if ((DEBUG & DEBUG_DATA_VALUE) != 0)
	System.out.println
		("--- Data_Table_Model.setValueAt: "
		+ row + "," + column + ", "
		+ value + '(' + getValueAt (row, column) + ')');
if (! getValueAt (row, column).equals (value))
	{
	Edited_Cell (row, column, (String)getValueAt (row, column));
	super.setValueAt (value, row, column);
	}
}

public boolean isCellEditable
	(
	int		row,
	int		column
	)
{
if ((DEBUG & DEBUG_DATA_VALUE) != 0)
	System.out.print
		("--- Data_Table_Model.isCellEditable (" + row + ", " + column + "): ");
if ((DEBUG & DEBUG_DATA_VALUE) != 0)
	System.out.println (Editable_Fields.contains (new Integer (column)));
return Editable_Fields.contains (new Integer (column));
}

/*==============================================================================
	Editing
*/
/*------------------------------------------------------------------------------
	Editable Cells
*/
private void Editable
	(
	boolean	editable,
	int		field_number,
	boolean	signal_change
	)
{
if ((DEBUG & DEBUG_EDITABLE) != 0)
	System.out.println
		("--- Data_Table_Model.Editable: " + editable + " - " + field_number);
if (editable)
	{
	if (field_number < 0)
		{
		int
			total_fields = getColumnCount ();
		field_number = 0;
		while (field_number < total_fields)
			Editable_Fields.add (new Integer (field_number++));
		}
	else
		Editable_Fields.add (new Integer (field_number));
	}
else
	{
	if (field_number < 0)
		Editable_Fields.clear ();
	else
		Editable_Fields.remove (new Integer (field_number));
	}
if (signal_change)
	fireTableDataChanged ();
}

public void Editable
	(
	boolean	editable,
	int		field_number
	)
{Editable (editable, field_number, true);}

private void Editable
	(
	boolean	editable,
	Vector	field_numbers,
	boolean	signal_change
	)
{
if (field_numbers == null)
	Editable (editable, -1, true);	//	Apply to all fields.
else
	{
	Iterator
		fields = field_numbers.iterator ();
	while (fields.hasNext ())
		Editable (editable, ((Integer)fields.next ()).intValue (),
			//	Signal data change on the last field number.
			(fields.hasNext () ? false : signal_change));
	}
}

public void Editable
	(
	boolean	editable,
	Vector	field_numbers
	)
{Editable (editable, field_numbers, true);}

public void Editable
	(
	boolean	editable
	)
{Editable (editable, -1, true);}

/*------------------------------------------------------------------------------
	Edited Cells
*/
/**	Gets a list of cells that have been edited.
<P>
	The contents of the Edited_Cells_Map is converted to a Vector of
	<CODE>{@link Cell Cell}</CODE> objects, one for each cell that was
	edited. The Cell objects will be ordered in the Vector by row and
	column within each row.
<P>
	@return	A Vector of Cell objects.
*/
public Vector Edited_Cells ()
{
Vector
	edited_cells = new Vector ();
if ((DEBUG & DEBUG_LOCATIONS) != 0)
	System.out.println ("--- Data_Table_Model.Edited_Cells:");
Set row_list = Edited_Cells_Map.keySet ();
Iterator rows = row_list.iterator ();
while (rows.hasNext ())
	{
	Integer row = (Integer)rows.next ();
	TreeMap column_map = (TreeMap)Edited_Cells_Map.get (row);
	Set column_list = column_map.keySet ();
	Iterator columns = column_list.iterator ();
	while (columns.hasNext ())
		{
		Integer column = (Integer)columns.next ();
		edited_cells.add (new Cell
			(row, column, (String)getValueAt
				(row.intValue (), column.intValue ()),
				(String)column_map.get (column)));
		if ((DEBUG & DEBUG_LOCATIONS) != 0)
			System.out.println ("    "
				+ ((Cell)edited_cells.lastElement ()).row + ","
				+ ((Cell)edited_cells.lastElement ()).column + ": "
				+ ((Cell)edited_cells.lastElement ()).new_value + "("
				+ ((Cell)edited_cells.lastElement ()).old_value + ")");
		}
	}
return edited_cells;
}

public void Clear_Edited_Cells ()
{
if (Edited_Cells_Map.size () != 0)
	{
	Edited_Cells_Map.clear ();
	fireTableDataChanged ();
	}
}

public boolean is_Cell_Edited
	(
	int		row,
	int		column
	)
{
TreeMap
	columns;
if ((DEBUG & DEBUG_LOCATIONS) != 0)
	System.out.println ("--- Data_Table_Model.is_Cell_Edited ("
		+ row + ", " + column + "): "
		+ ((columns = (TreeMap)Edited_Cells_Map.get (new Integer (row))) != null &&
		columns.containsKey (new Integer (column))));
return ((columns = (TreeMap)Edited_Cells_Map.get (new Integer (row))) != null &&
		columns.containsKey (new Integer (column)));
}

private void Edited_Cell
	(
	int		row,
	int		column,
	String	value
	)
{
if ((DEBUG & DEBUG_LOCATIONS) != 0)
	System.out.print
		("--- Data_Table_Model.Edited_Cell: "
		+ row + "," + column + ": " + value);
Integer
	row_number = new Integer (row),
	column_number = new Integer (column);
TreeMap
	column_numbers;
if ((column_numbers = (TreeMap)Edited_Cells_Map.get (row_number)) != null)
	{
	if (column_numbers.containsKey (column_number))
		{
		if ((DEBUG & DEBUG_LOCATIONS) != 0)
			System.out.println ("redundant");
		return;
		}
	}
else
	{
	column_numbers = new TreeMap ();
	Edited_Cells_Map.put (row_number, column_numbers);
	}
column_numbers.put (column_number, value);
if ((DEBUG & DEBUG_LOCATIONS) != 0)
	System.out.println ("added");
}

/*==============================================================================
	Data
*/
public void Data
	(
	Vector	data_values,
	Vector	field_names,
	boolean	editable,
	Vector	editable_fields
	)
{
if ((DEBUG & DEBUG_DATA_CHANGE) != 0)
	System.out.println (">>> Data_Table_Model.Data");

//	Apply the data values and field names.
if (data_values == null ||
	field_names == null ||
	field_names.isEmpty ())
	{
	data_values = EMPTY_TABLE;
	field_names = EMPTY_FIELD;
	}
if ((DEBUG & DEBUG_DATA_CHANGE) != 0)
	System.out.println
		(">>> Data_Table_Model.Data: "
		+ data_values.size () + " rows, " + field_names.size () + " columns");

//	Apply the editable fields (don't update the table, yet).
if (editable_fields == null)
	{
	int
		total_fields = field_names.size (),
		field_number = 0;
	while (field_number < total_fields)
		Editable (editable, field_number++, false);
	}
else
	Editable (editable, editable_fields, false);

//	Apply the new data.
super.setDataVector (data_values, field_names);
if ((DEBUG & DEBUG_DATA_CHANGE) != 0)
	System.out.println ("<<< Data_Table_Model.Data: "
		+ getColumnCount () + " columns");
}

}	//	End of Data_Table_Model class.

/* *****************************************************************************
	Edited Cells
*/
/**	Information about an edited cell.
<P>
	Objects of this class are returned, in a Vector, to report
	to the user information on the cells that have been edited.
<P>
	@see	Data_Table_Model#Edited_Cells()
*/
class Cell
{
int row;
int column;
String new_value;
String old_value;

public Cell
	(
	int		row,
	int		column,
	String	new_value,
	String	old_value
	)
{
this.row = row;
this.column = column;
this.new_value = new_value;
this.old_value = old_value;
}

public Cell
	(
	Integer	row,
	Integer	column,
	String	new_value,
	String	old_value
	)
{
this.row = row.intValue ();
this.column = column.intValue ();
this.new_value = new_value;
this.old_value = old_value;
}

private Cell ()
{}

}	//	End of Cell class.
