<?php
/**
 * The Kronolith_View_Year:: class provides an API for viewing years.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @author  Jan Schneider <jan@horde.org>
 * @package Kronolith
 */
class Kronolith_View_Year
{
    public $year;
    protected $_events = array();

    /**
     *
     * @param Horde_Date $date
     *
     * @return Kronolith_View_Year
     */
    public function __construct(Horde_Date $date)
    {
        $this->year = $date->year;
        $startDate = new Horde_Date(array('year' => $this->year,
                                          'month' => 1,
                                          'mday' => 1));
        $endDate = new Horde_Date(array('year' => $this->year,
                                        'month' => 12,
                                        'mday' => 31));

        try {
            $this->_events = Kronolith::listEvents($startDate, $endDate, $GLOBALS['calendar_manager']->get(Kronolith::DISPLAY_CALENDARS));
        } catch (Exception $e) {
            $GLOBALS['notification']->push($e, 'horde.error');
            $this->_events = array();
        }
        if (!is_array($this->_events)) {
            $this->_events = array();
        }
    }

    public function html()
    {
        global $prefs;

        $html = '<table id="kronolith-view-year" class="kronolith-minical"><tr>';
        for ($month = 1; $month <= 12; ++$month) {
            $html .= '<td>';

            // Heading for each month.
            $date = new Horde_Date(sprintf('%04d%02d01010101', $this->year, $month));
            $html .= '<table><thead><tr class="kronolith-minical-nav"><th colspan="7">'
                . Horde::url('month.php')->add('date', $date->dateString())->link()
                . $date->strftime('%B')
                . '</a></th></tr><tr><th class="kronolith-minical-empty">&nbsp;</th>';
            if (!$prefs->getValue('week_start_monday')) {
                $html .= '<th>' . _("Su"). '</th>';
            }
            $html .= '<th>' . _("Mo") . '</th>' .
                '<th>' . _("Tu") . '</th>' .
                '<th>' . _("We") . '</th>' .
                '<th>' . _("Th") . '</th>' .
                '<th>' . _("Fr") . '</th>' .
                '<th>' . _("Sa") . '</th>';
            if ($prefs->getValue('week_start_monday')) {
                $html .= '<th>' . _("Su") . '</th>';
            }
            $html .= '</tr></thead><tbody><tr><td class="kronolith-minical-week">';

            $startday = new Horde_Date(array('mday' => 1,
                                             'month' => $month,
                                             'year' => $this->year));
            $startday = $startday->dayOfWeek();

            $daysInView = Date_Calc::weeksInMonth($month, $this->year) * 7;
            if (!$prefs->getValue('week_start_monday')) {
                $startOfView = 1 - $startday;

                // We may need to adjust the number of days in the
                // view if we're starting weeks on Sunday.
                if ($startday == Horde_Date::DATE_SUNDAY) {
                    $daysInView -= 7;
                }
                $endday = new Horde_Date(array('mday' => Horde_Date_Utils::daysInMonth($month, $this->year),
                                               'month' => $month,
                                               'year' => $this->year));
                $endday = $endday->dayOfWeek();
                if ($endday == Horde_Date::DATE_SUNDAY) {
                    $daysInView += 7;
                }
            } else {
                if ($startday == Horde_Date::DATE_SUNDAY) {
                    $startOfView = -5;
                } else {
                    $startOfView = 2 - $startday;
                }
            }

            $currentCalendars = array(true);
            foreach ($currentCalendars as $id => $cal) {
                $cell = 0;
                for ($day = $startOfView; $day < $startOfView + $daysInView; ++$day) {
                    $date = new Kronolith_Day($month, $day, $this->year);
                    $date->hour = $prefs->getValue('twentyFour') ? 12 : 6;
                    $week = $date->weekOfYear();

                    if ($cell % 7 == 0) {
                        if ($cell != 0) {
                            $html .= "</tr>\n<tr><td class=\"kronolith-minical-week\">";
                        }
                        $html .= (int)$date->weekOfYear() . '</td>';
                    }
                    if ($date->month != $month) {
                        $style = 'kronolith-other-month';
                    } else {
                        $style = '';
                    }

                    /* Set up the link to the day view. */
                    $url = Horde::url('day.php', true)
                        ->add('date', $date->dateString());

                    if ($date->month == $month &&
                              !empty($this->_events[$date->dateString()])) {
                        /* There are events; create a cell with tooltip to list
                         * them. */
                        $day_events = '';
                        foreach ($this->_events[$date->dateString()] as $event) {
                            if ($event->status == Kronolith::STATUS_CONFIRMED) {
                                /* Set the background color to distinguish the
                                 * day */
                                $style = 'year-event';
                            }

                            if ($event->isAllDay()) {
                                $day_events .= _("All day");
                            } else {
                                $day_events .= $event->start->strftime($prefs->getValue('twentyFour') ? '%R' : '%I:%M%p') . '-' . $event->end->strftime($prefs->getValue('twentyFour') ? '%R' : '%I:%M%p');
                            }
                            $day_events .= ':'
                                . ($event->getLocation() ? ' (' . $event->getLocation() . ')' : '')
                                . ' ' . $event->getTitle() . "\n";
                        }
                        /* Bold the cell if there are events. */
                        $cellday = '<strong>' . Horde::linkTooltip($url, _("View Day"), '', '', '', $day_events) . $date->mday . '</a></strong>';
                    } else {
                        /* No events, plain link to the day. */
                        $cellday = Horde::linkTooltip($url, _("View Day")) . $date->mday . '</a>';
                    }
                    if ($date->isToday() && $date->month == $month) {
                        $style .= ' kronolith-today';
                    }

                    $html .= '<td align="center" class="' . $style . '" height="10" width="5%" valign="top">' .
                        $cellday . '</td>';
                    ++$cell;
                }
            }

            $html .= '</tr></tbody></table></td>';
            if ($month % 3 == 0 && $month != 12) {
                $html .= '</tr><tr>';
            }
        }

        echo $html . '</tr></table>';
    }

    public function link($offset = 0, $full = false)
    {
        return Horde::url('year.php', $full)
            ->add('date', ($this->year + $offset) . '0101');
    }

    public function getName()
    {
        return 'Year';
    }

}
