///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/viewport/input/ViewportInputHandler.h>
#include <core/viewport/input/XFormManager.h>
#include <core/viewport/input/SelectionMode.h>
#include <core/viewport/ViewportManager.h>
#include <core/data/DataSetManager.h>

namespace Core {

/// The singleton instance of this class.
ViewportInputManager* ViewportInputManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the viewport input manager.
******************************************************************************/
ViewportInputManager::ViewportInputManager()
{
	// Reset the viewport input manager when a new scene has been loaded.
	connect(&DATASET_MANAGER, SIGNAL(dataSetReset(DataSet*)), this, SLOT(reset()));
}

/******************************************************************************
* Returns the currently active ViewportInputHandler that
* handles the mouse events in viewports.
******************************************************************************/
ViewportInputHandler* ViewportInputManager::currentHandler() const
{
	if(inputHandlerStack.empty()) return NULL;
	return inputHandlerStack.back().get();
}

/******************************************************************************
* Pushes a handler onto the stack and makes it active.
******************************************************************************/
void ViewportInputManager::pushInputHandler(const ViewportInputHandler::SmartPtr& handler)
{
    CHECK_OBJECT_POINTER(handler);

	ViewportInputHandler::SmartPtr oldHandler = currentHandler();
	if(handler == oldHandler) return;

	if(oldHandler != NULL) {
		if(handler->handlerActivationType() == ViewportInputHandler::EXCLUSIVE) {
			// Remove all handlers from the stack
			inputHandlerStack.clear();
		}
		else if(handler->handlerActivationType() == ViewportInputHandler::NORMAL) {
			// Remove all non-exclusive handlers from the stack
			for(int i = inputHandlerStack.size(); i--; ) {
				CHECK_OBJECT_POINTER(inputHandlerStack[i]);
				if(inputHandlerStack[i]->handlerActivationType() != ViewportInputHandler::EXCLUSIVE) {
					inputHandlerStack.remove(i);
				}
			}
		}
		else if(handler->handlerActivationType() == ViewportInputHandler::TEMPORARY) {
			// Remove all temporary handlers from the stack.
			if(oldHandler->handlerActivationType() == ViewportInputHandler::TEMPORARY)
				inputHandlerStack.pop_back();
		}
		else { OVITO_ASSERT(false); }
	}

	// Put new handler on the stack.
	inputHandlerStack.push_back(handler);
	if(oldHandler != NULL) oldHandler->onDeactivated();
	handler->onActivated();
	inputModeChanged(oldHandler.get(), handler.get());

	// Redraw viewports if the old or the new handler uses overlays.
	if((oldHandler && oldHandler->hasOverlay()) || (handler && handler->hasOverlay()))
		VIEWPORT_MANAGER.updateViewports();
}

/******************************************************************************
* Removes a handler from the stack and deactivates it if
* it is currently active.
******************************************************************************/
void ViewportInputManager::removeInputHandler(ViewportInputHandler* handler)
{
	int index = inputHandlerStack.indexOf(handler);
	if(index < 0) return;
	CHECK_OBJECT_POINTER(handler);
	if(index == inputHandlerStack.size()-1) {
		ViewportInputHandler::SmartPtr oldHandler = handler;
		inputHandlerStack.remove(index);
		handler->onDeactivated();
		if(!inputHandlerStack.empty())
			currentHandler()->onActivated();
		inputModeChanged(handler, currentHandler());

		// Redraw viewports if the old or the new handler uses overlays.
		if((oldHandler && oldHandler->hasOverlay()) || (currentHandler() && currentHandler()->hasOverlay()))
			VIEWPORT_MANAGER.updateViewports();
	}
	else {
		// Redraw viewports if the removed handler used overlays.
		if(handler->hasOverlay())
			VIEWPORT_MANAGER.updateViewports();

		inputHandlerStack.remove(index);
	}
}

/******************************************************************************
* Resets the input mode stack to its initial state on application startup.
******************************************************************************/
void ViewportInputManager::reset()
{
	// Remove all input modes from the stack
	while(currentHandler() != NULL)
		removeInputHandler(currentHandler());

	// Activate selection mode by default
	pushInputHandler(XFORM_MANAGER.objectSelectionMode());
}

};
