///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file PluginClass.h
 * \brief Contains the definition of the Core::PluginClass class.
 */

#ifndef __OVITO_PLUGINCLASS_H
#define __OVITO_PLUGINCLASS_H

#include <core/Core.h>
#include "NativePluginClassDescriptor.h"

namespace Core {

class ObjectSaveStream;		// defined in ObjectSaveStream.h
class ObjectLoadStream;		// defined in ObjectLoadStream.h

#ifdef _DEBUG
	/// Checks whether a pointer to a PluginClass derived object is valid.
	#define CHECK_OBJECT_POINTER(object) { CHECK_POINTER(boost::get_pointer(object)) OVITO_ASSERT_MSG((object)->__isObjectAlive(), "CHECK_OBJECT_POINTER", "PluginClass derived object pointer is invalid. Object has already been deleted."); }
#else
	/// Do nothing for release builds.
	#define CHECK_OBJECT_POINTER(object)
#endif

/**
 * \brief Universal base class for all plug-in classes.
 *
 * Almost all classes in this application are derived from this base class.
 * The PluginClass provides several services:
 *
 * \li Every PluginClass derived class has a descriptor object associated with it that
 *     uniquely identifies it. This PluginClassDescriptor allows the developer to
 *     create instances of the class at runtime without using the \c new operator.
 * \li A reference counting mechanism.
 * \li Serialization and deserialization.
 *
 * \author Alexander Stukowski
 * \sa PluginClassDescriptor, Plugin, PluginManager
 */
class CORE_DLLEXPORT PluginClass : public QObject
{
public:

	/// \brief These are the flags that an be set for a PluginClass derived object.
	/// \sa setPluginClassFlag(), getPluginClassFlag()
	enum PluginClassFlag {

		/// \brief Prevents the object from being auto-deleted when all references have gone.
		/// \sa PluginClass::objectReferenceCount(), PluginClass::autoDeleteObject()
		FLAG_OBJECT_LOCKED = (1<<0),

		/// \brief Is set while the object is being loaded from an input stream (i.e. during the execution of PluginClass::loadFromStream()).
		/// \sa PluginClass::loadFromStream()
		FLAG_OBJ_BEING_LOADED = (1<<1),

		/// \brief All bits above this value are available for the use by plug-in developers in derived classes.
		FLAG_USER = (1<<16),
	};
	Q_DECLARE_FLAGS(PluginClassFlags, PluginClassFlag);

public:

	/// \brief The default constructor.
	/// \param isLoading Specifies whether this object is being created during a normal operation (\c isLoading==false)
	///                  or because it is being loaded from a data stream (\c isLoading==true).
	///
	/// If the constructor is called with \c isLoading==true then it should not initialize the data fields
	/// that have been stored in the data stream by the saveToStream() method. Shortly after the constructor returns
	/// the loadFromStream() method will be called by the system to load the data field values from the input stream.
	///
	/// \sa loadFromStream()
	PluginClass(bool isLoading = false) : referenceCount(0), flags(0) {
#ifdef _DEBUG
		magicDebugCode = 0x87ABCDEF;
#endif
	}

	/// \brief The virtual destructor.
	///
	/// The default implementation does nothing.
	virtual ~PluginClass() {
#ifdef _DEBUG
		magicDebugCode = 0xFEDCBA87;
#endif
	}

	/// \brief Deletes this object.
	///
	/// The default implementation of this method calls "delete this" to delete the object from memory.
	/// But there is one exception: If the PluginClassFlag::FLAG_OBJECT_LOCKED is set then the object
	/// is not deleted an this method does nothing.
	virtual void autoDeleteObject();

public:

	/// \brief Saves the class' contents to an output stream.
	/// \param stream The destination stream.
	///
	/// Derived classes can overwrite this virtual method to store their specific data
	/// in the output stream. The derived class \b must always call the base implementation of the saveToStream() method
	/// before it writes its own data to the stream.
	///
	/// The base implementation of PluginClass::saveToStream() method saves the
	/// PluginClassFlags of this object to the output stream.
	///
	/// \sa loadFromStream()
	virtual void saveToStream(ObjectSaveStream& stream);

	/// \brief Loads the class' contents from an input stream.
	/// \param stream The source stream.
	/// \throw Exception when a parsing error has occurred.
	///
	/// Derived classes can overwrite this virtual method to read their specific data
	/// from the input stream. The derived class \b must always call the loadFromStream() method
	/// of the base class before it reads its own data from the stream.
	///
	/// The base implementation of PluginClass::loadFromStream() method restores the
	/// PluginClassFlags for this object that had been serialized to the data stream.
	///
	/// \note The PluginClass object is not in a fully initialized state when the loadFromStream() method is called.
	///       It has just been created with \c isLoading set to \c true in the constructor.
	///       In particular the developer cannot assume that all other objects stored in the data stream and
	///       referenced by this object have already been restored at the time loadFromStream() is called.
	///       But for each PluginClass derived object stored in the data stream the loadFromStreamComplete() method
	///       will be called after all objects have been deserialized. So if the object has to do some post-deserialization
	///       tasks that rely on other objects referenced by this object then this should be done in an implementation of
	///       the loadFromStreamComplete() method.
	///
	/// \sa saveToStream(), loadFromStreamComplete()
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// This method is called once for this object after it has been
	/// loaded from the input stream and each other object in the stream
	/// has been loaded as well. This function can therefore safely access
	/// sub-objects stored in this class that have been loaded via ObjectLoadStream::loadObject().
	/// \sa loadFromStream()
	virtual void loadFromStreamComplete() {}

	/// \brief Sets the value of one or more flags.
	/// \param f Sepcifies the flags to set. Each bit that is set in \a f is assigned the given \a value (\c true or \c false).
	/// \param value The value that is assigned to the flags specified by \a f.
	/// \sa clearPluginClassFlag(), getPluginClassFlag()
	void setPluginClassFlag(PluginClassFlags f, bool value = true) { if(value) flags |= f; else flags &= ~f; }

	/// \brief Clears one or more flags.
	/// \param f Each flag bit that is set in \a f is cleared.
	/// \sa setPluginClassFlag(), getPluginClassFlag()
	void clearPluginClassFlag(PluginClassFlags f) { setPluginClassFlag(f, false); }

	/// \brief Tests whether one or more flags are set.
	/// \param f Specifies the flags to test.
	/// \return \a true if any of the bits set in \a f are set for this PluginClass object; \c false if none of them are.
	/// \sa setPluginClassFlag(), getPluginClassFlags()
	bool getPluginClassFlag(PluginClassFlags f) const { return (flags & f) != 0; }

	/// \brief Returns all flags that are set for this PluginClass object.
	/// \return All flags that are set to \c true.
	/// \sa getPluginClassFlag()
	PluginClassFlags getPluginClassFlags() const { return flags; }

	/// \brief Returns the current value of the object's reference counter.
	/// \return The reference count for this object.
	size_t objectReferenceCount() const { return referenceCount; }

#ifdef _DEBUG
	bool __isObjectAlive() const { return this->magicDebugCode == 0x87ABCDEF; }
#endif

private:

	/// The number of existing references to this object.
	size_t referenceCount;

	/// The flag bits.
    PluginClassFlags flags;

private:

	/// \brief Increments the reference count by one.
	/// \sa decrementReferenceCount()
	void incrementReferenceCount() {
		CHECK_OBJECT_POINTER(this);
		++referenceCount;
	}

	/// \brief Decrements the reference count by one.
	///
	/// If the refence count becomes zero then the object is auto-deleted.
	/// \sa autoDeleteObject(), incrementReferenceCount()
	void decrementReferenceCount() {
		CHECK_OBJECT_POINTER(this);
		OVITO_ASSERT_MSG(referenceCount > 0, "PluginClass::decrementReferenceCount()", "Negative reference count.");
		--referenceCount;
		if(referenceCount == 0)
			autoDeleteObject();
	}

#ifdef _DEBUG
	/// This field is filled with a special code value in the constructor to indicate that
	/// the object is alive and has not been deleted.
	/// The object destructor sets the field back to zero.
	quint32 magicDebugCode;
#endif

private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(PluginClass);

	friend void intrusive_ptr_add_ref(PluginClass* target);
	friend void intrusive_ptr_release(PluginClass* target);
};

Q_DECLARE_OPERATORS_FOR_FLAGS(PluginClass::PluginClassFlags)

/// This is used by the intrusive_ptr class from the Boost library.
inline void intrusive_ptr_add_ref(PluginClass* target) { target->incrementReferenceCount(); }

/// This is used by the intrusive_ptr class from the Boost library.
inline void intrusive_ptr_release(PluginClass* target) { target->decrementReferenceCount(); }

/// \brief Dynamic casting function for PluginClass derived object.
///
/// Returns the given object cast to type \c T if the object is of type \c T
/// (or of a subclass); otherwise returns \c NULL.
template<class T, class U>
inline T* dynamic_object_cast(U* obj) {
	return qobject_cast<T*>(obj);
}

/// \brief Dynamic casting function for PluginClass derived object.
///
/// Returns the given object cast to type \c T if the object is of type \c T
/// (or of a subclass); otherwise returns \c NULL.
template<class T, class U>
inline const T* dynamic_object_cast(const U* obj) {
	return qobject_cast<const T*>(obj);
}

/// \brief Dynamic casting function for smart pointer to PluginClass derived objects.
///
/// Returns the given object cast to type \c T if the object is of type \c T
/// (or of a subclass); otherwise returns \c NULL.
template<class T, class U>
inline intrusive_ptr<T> dynamic_object_cast(const intrusive_ptr<U>& obj) {
	return intrusive_ptr<T>(qobject_cast<T*>(obj.get()));
}

/// \brief Static casting function for PluginClass derived object.
///
/// Returns the given object cast to type \c T.
/// Performs a runtime check of the object type in debug build.
template<class T, class U>
inline T* static_object_cast(U* obj) {
	OVITO_ASSERT_MSG(!obj || obj->pluginClassDescriptor()->isKindOf(PLUGINCLASSINFO(T)), "static_object_cast", "Runtime type check failed. The source object is not an instance of the destination class.");
	return static_cast<T*>(obj);
}

/// \brief Static casting function for PluginClass derived object.
///
/// Returns the given object cast to type \c T.
/// Performs a runtime check of the object type in debug build.
template<class T, class U>
inline const T* static_object_cast(const U* obj) {
	OVITO_ASSERT_MSG(!obj || obj->pluginClassDescriptor()->isKindOf(PLUGINCLASSINFO(T)), "static_object_cast", "Runtime type check failed. The source object is not an instance of the destination class.");
	return static_cast<const T*>(obj);
}

/// \brief Static casting function for smart pointers to PluginClass derived objects.
///
/// Returns the given object cast to type \c T.
/// Performs a runtime check of the object type in debug build.
template<class T, class U>
inline intrusive_ptr<T> static_object_cast(const intrusive_ptr<U>& obj) {
	OVITO_ASSERT_MSG(!obj || obj->pluginClassDescriptor()->isKindOf(PLUGINCLASSINFO(T)), "static_object_cast", "Runtime type check failed. The source object is not an instance of the destination class.");
	return boost::static_pointer_cast<T>(obj);
}

};	// End of namespace Core

#include <core/data/ObjectLoadStream.h>
#include <core/data/ObjectSaveStream.h>

#endif // __OVITO_PLUGINCLASS_H
