///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __SHOW_PERIODIC_IMAGES_MODIFIER_H
#define __SHOW_PERIODIC_IMAGES_MODIFIER_H

#include <core/Core.h>
#include <atomviz/AtomViz.h>
#include <atomviz/modifier/AtomsObjectModifierBase.h>

namespace AtomViz {

/**
 * \brief This modifier duplicates all atoms multiple times and shifts them by
 *        one of the simulation cell vectors to visualize the periodic images.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT ShowPeriodicImagesModifier : public AtomsObjectModifierBase
{
public:

	/// \brief Constructs a new instance of this class.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	ShowPeriodicImagesModifier(bool isLoading = false);

	/// \brief Asks the modifier for its validity interval at the given time.
	///
	/// This method returns the maximum time interval during which
	/// the object's parameters stay constant and that includes the given
	/// point in time.
	virtual TimeInterval modifierValidity(TimeTicks time) { return TimeForever; }

protected:

	/// \brief Modifies the atoms object.
	///
	/// The time interval passed to the function is cut down by this method to the interval
	/// where the output object produced by this modifier is valid/constant.
	/// This method is part of the implementation of the abstract AtomsObjectModifierBase class.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

private:

	/// Controls whether the periodic images are shown in the X direction.
	PropertyField<bool> _showImageX;
	/// Controls whether the periodic images are shown in the Y direction.
	PropertyField<bool> _showImageY;
	/// Controls whether the periodic images are shown in the Z direction.
	PropertyField<bool> _showImageZ;

	/// Controls the number of periodic images shown in the X direction.
	PropertyField<int> _numImagesX;
	/// Controls the number of periodic images shown in the Y direction.
	PropertyField<int> _numImagesY;
	/// Controls the number of periodic images shown in the Z direction.
	PropertyField<int> _numImagesZ;

	/// Controls whether the size of the simulation box is adjusted to the extended system.
	PropertyField<bool> _adjustBoxSize;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ShowPeriodicImagesModifier)
	DECLARE_PROPERTY_FIELD(_showImageX)
	DECLARE_PROPERTY_FIELD(_showImageY)
	DECLARE_PROPERTY_FIELD(_showImageZ)
	DECLARE_PROPERTY_FIELD(_numImagesX)
	DECLARE_PROPERTY_FIELD(_numImagesY)
	DECLARE_PROPERTY_FIELD(_numImagesZ)
	DECLARE_PROPERTY_FIELD(_adjustBoxSize)
};

/**
 * \brief A properties editor for the ShowPeriodicImagesModifier class.
 *
 * This editor class creates and manages the user interface through which the
 * user can alter the modifier's parameters.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT ShowPeriodicImagesModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(ShowPeriodicImagesModifierEditor)
};

};	// End of namespace AtomViz

#endif // __SHOW_PERIODIC_IMAGES_MODIFIER_H
