PROGRAM create_restartfield
!
!****	create_restartfield
!
!	Purpose:
!	--------
!	This program creates an analytical field on a grid given as argument
!       
!***	History:
!       -------
!       Version   Programmer      Date        Description
!       -------   ----------      ----        -----------
!         1.0     Sophie Valcke   2007/12/06  Creation
!*----------------------------------------------------------------
!
!** ++ calling argument
!       1- source grid acronym (CHARACTER(len=4))
!       2- analytical field (1, 2, or 3, CHARACTER(len=1))
!             1) F = 2 - cos[Pi*acos(cos(lat)cos(lon)]
!             2) F = 2 + [(cos(lat))**2]*cos(2*lon)
!             3) F = 2 + [(sin(2*lon))**16]*cos(16lon)
!
!** ++ modules and includes
!
  IMPLICIT NONE
  INCLUDE 'netcdf.inc'
!
!** ++ declarations
!
  CHARACTER(len=4)       :: cl_grd
  CHARACTER(len=1)       :: cl_fld
  CHARACTER(len=8)       :: cl_nam
  CHARACTER(len=64)      :: cl_ficgrd, cl_ficmsk
  INTEGER                :: il_fileid, il_lonid, il_latid, il_mskid 
  INTEGER                :: il_ficid, il_fldid
  INTEGER                :: il_ndims, il_i, il_j, il_ij, il_type
  INTEGER, DIMENSION(:), ALLOCATABLE :: il_dimids, il_msk, il_i_dimid, il_j_dimid
  REAL*4,  DIMENSION(:), ALLOCATABLE :: rl_lon
  REAL*4,  DIMENSION(:), ALLOCATABLE :: rl_lat
  REAL*4,  DIMENSION(:), ALLOCATABLE           :: rl_fld
  DOUBLE PRECISION, PARAMETER                 :: two = 2.
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_lon
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_lat
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_fld
  DOUBLE PRECISION, PARAMETER    :: dp_pi=3.14159265359
  DOUBLE PRECISION, PARAMETER    :: dp_length= 1.2*dp_pi
  DOUBLE PRECISION, PARAMETER    :: dp_conv = dp_pi/180.
  LOGICAL                :: ll_dbl

!*----------------------------------------------------------------
!
! Get arguments giving source grid acronym and field type
! 
  CALL getarg(1,cl_ficgrd)
!  PRINT *, 'Source grid file name = ', cl_ficgrd
  CALL getarg(2,cl_ficmsk)
!  PRINT *, 'Source mask file name= ', cl_ficmsk
  CALL getarg(3,cl_grd)
!  PRINT *, 'Source grid acronym = ', cl_grd
  CALL getarg(4,cl_fld)
!  PRINT *, 'Analytical field number = ', cl_fld
!
! Open grids file and get grid longitudes and latitudes
!
  CALL hdlerr(NF_OPEN(cl_ficgrd, NF_NOWRITE, il_fileid))
!  
  cl_nam=cl_grd//".lon" 
!  WRITE(*,*) 'cl_nam ', cl_nam
  CALL hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_lonid))
!  WRITE(*,*) 'il_lonid', il_lonid
  cl_nam=cl_grd//".lat" 
!  WRITE(*,*) 'cl_nam ', cl_nam
  CALL hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_latid))
!  WRITE(*,*) 'il_latid', il_latid
!
  CALL hdlerr(NF_INQ_VARNDIMS(il_fileid, il_lonid, il_ndims))
!  WRITE(*,*) 'il_ndims =', il_ndims
  ALLOCATE (il_dimids(il_ndims))
  CALL hdlerr(NF_INQ_VARDIMID(il_fileid, il_lonid, il_dimids))
!  WRITE(*,*) 'il_dimids =', il_dimids
  CALL hdlerr(NF_INQ_VARTYPE(il_fileid, il_lonid, il_type))
  ll_dbl = .false.
  IF (il_type == NF_DOUBLE) ll_dbl = .TRUE.
!  WRITE(*,*) 'il_dimids =', il_dimids
  IF (il_ndims == 1) THEN
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(1), il_ij))
!      WRITE(*,*) 'il_ij= ', il_ij
  ELSE
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(1), il_i))
!      WRITE(*,*) 'il_i= ', il_i
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(2), il_j))
!      WRITE(*,*) 'il_j= ', il_j
      il_ij = il_i*il_j
!      WRITE(*,*) 'il_ij= ', il_ij
  ENDIF
  IF (ll_dbl) THEN
      ALLOCATE (dl_lon(il_ij))
      ALLOCATE (dl_lat(il_ij))
      CALL hdlerr(NF_GET_VAR_DOUBLE (il_fileid, il_lonid, dl_lon))
      CALL hdlerr(NF_GET_VAR_DOUBLE (il_fileid, il_latid, dl_lat))
  ELSE
      ALLOCATE (rl_lon(il_ij))
      ALLOCATE (rl_lat(il_ij))
      CALL hdlerr(NF_GET_VAR_REAL (il_fileid, il_lonid, rl_lon))
      CALL hdlerr(NF_GET_VAR_REAL (il_fileid, il_latid, rl_lat))
  ENDIF
  CALL hdlerr(NF_CLOSE(il_fileid))
!
! Open mask file and get mask file
!
  CALL hdlerr(NF_OPEN(cl_ficmsk, NF_NOWRITE, il_fileid))
!  
  cl_nam=cl_grd//".msk" 
!  write(*,*) 'cl_nam ', cl_nam
  call hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_mskid))
!  write(*,*) 'il_mskid', il_mskid
!
  ALLOCATE (il_msk(il_ij))
  CALL hdlerr(NF_GET_VAR_INT (il_fileid, il_mskid, il_msk))
  CALL hdlerr(NF_CLOSE(il_fileid))
!
! Create field and apply mask
!
  IF (ll_dbl) THEN
      dl_lat = dl_lat * dp_conv
      dl_lon = dl_lon * dp_conv
      ALLOCATE (dl_fld(il_ij))
      IF (cl_fld == '1') THEN
          dl_fld =  two - COS(dp_pi*(ACOS(COS(dl_lat)*COS(dl_lon))/dp_length))
      ELSE IF (cl_fld == '2') THEN
          dl_fld = two + COS(dl_lat)**2*COS(two*dl_lon)
      ELSE IF (cl_fld == '3') THEN
          dl_fld = two + SIN(two*dl_lat)**16*COS(16.*dl_lon)
      ELSE IF (cl_fld == '4') THEN
          dl_fld = 6.0
      ELSE
          WRITE(*,*) 'Bad analytical field number (2nd calling argument)'
          WRITE(*,*) 'should be either "1", "2" or "3"'
          STOP
      ENDIF
!
      WHERE (il_msk == 1)
          dl_fld = 0.00
      END WHERE
  ELSE
      rl_lat = rl_lat * dp_conv
      rl_lon = rl_lon * dp_conv     
      ALLOCATE (rl_fld(il_ij))
      IF (cl_fld == '1') THEN
          rl_fld =  two - COS(dp_pi*(ACOS(-COS(rl_lat)*COS(rl_lon))/dp_length))
      ELSE IF (cl_fld == '2') THEN
          rl_fld = two + COS(rl_lat)**2*COS(two*rl_lon)
      ELSE IF (cl_fld == '3') THEN
          rl_fld = two + SIN(two*rl_lat)**16*COS(16.*rl_lon)
      ELSE IF (cl_fld == '4') THEN
          rl_fld = 6.0
      ELSE
          WRITE(*,*) 'Bad analytical field number (2nd calling argument)'
          WRITE(*,*) 'should be either "1", "2" or "3"'
          STOP
      ENDIF
!
      WHERE (il_msk == 1)
          rl_fld = 0.00
      END WHERE
  ENDIF
!
! Create file and write the field
!
! Create file
  CALL hdlerr(NF_CREATE('fldin.nc', 0, il_ficid))
!
! Create dimensions
  IF (il_ndims == 1) THEN
      CALL hdlerr(NF_DEF_DIM(il_ficid, 'il_ij', il_ij, il_dimids(1)))
  ELSE
      CALL hdlerr(NF_DEF_DIM(il_ficid, 'il_i', il_i,il_dimids(1)))
      CALL hdlerr(NF_DEF_DIM(il_ficid, 'il_j', il_j,il_dimids(2)))
  ENDIF
!
! Create variables
  CALL hdlerr(NF_DEF_VAR (il_ficid, 'field_in', il_type, il_ndims, &
     il_dimids, il_fldid))
  cl_nam=cl_grd//".lon" 
  CALL hdlerr(NF_DEF_VAR (il_ficid, cl_nam, il_type, il_ndims, il_dimids, &
     il_lonid))
!  WRITE(*,*) 'il_lonid = ', il_lonid
  cl_nam=cl_grd//".lat" 
  CALL hdlerr(NF_DEF_VAR (il_ficid, cl_nam, il_type, il_ndims, il_dimids, &
     il_latid))
!  WRITE(*,*) 'il_latid = ', il_latid
!
! End of definition phase
  CALL hdlerr(NF_ENDDEF(il_ficid))
!
! Write the field
  IF (ll_dbl) THEN
      CALL hdlerr(NF_PUT_VAR_DOUBLE (il_ficid, il_lonid, dl_lon)) 
      CALL hdlerr(NF_PUT_VAR_DOUBLE (il_ficid, il_lonid, dl_lat)) 
      CALL hdlerr(NF_PUT_VAR_DOUBLE (il_ficid, il_fldid, dl_fld))
  ELSE
      CALL hdlerr(NF_PUT_VAR_REAL (il_ficid, il_lonid, rl_lon)) 
      CALL hdlerr(NF_PUT_VAR_REAL (il_ficid, il_lonid, rl_lat))
      CALL hdlerr(NF_PUT_VAR_REAL (il_ficid, il_fldid, rl_fld))
      WRITE(77,*) rl_fld
  ENDIF
!
! Close the file
  CALL hdlerr(NF_CLOSE(il_ficid))
!
END PROGRAM create_restartfield
!
!*----------------------------------------------------------------
!*----------------------------------------------------------------
!
SUBROUTINE hdlerr(istatus)

  INTEGER                 :: istatus
  INCLUDE 'netcdf.inc'

  IF (istatus .ne. NF_NOERR) THEN
      print *, NF_STRERROR(istatus)
      stop 'stopped'
  ENDIF

  RETURN

END SUBROUTINE hdlerr
!
!*----------------------------------------------------------------
!*----------------------------------------------------------------
!
