/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <string>
using std::string;


#include "LanguageHelper.h" 
#include "MagMLHelper.h" 
#include "Highlighter.h"



// -----------------------------------------------------------------------
// ------------------- Syntax highlighter code ---------------------------
//        this class is used by the MagMLHelper object, defined below
// -----------------------------------------------------------------------


class MagMLHighlighter : public Highlighter
{
public:
     MagMLHighlighter(QTextDocument *parent = 0);
    ~MagMLHighlighter(){}

     virtual void highlightBlock(const QString &text);

protected:

     QRegExp commentStartExpression;
     QRegExp commentEndExpression;

     QTextCharFormat tagNameFormat;
     QTextCharFormat attributeNameFormat;
     QTextCharFormat tagSymbolsFormat;
     QTextCharFormat quotationFormat;
     QTextCharFormat multiLineCommentFormat;
};


MagMLHighlighter::MagMLHighlighter(QTextDocument *parent)
     : Highlighter(parent)
{
    HighlightingRule rule;


    // Note that the order in which these highlighting rules are given
    // DOES matter! The rules seem to be applied in the order in which
    // they are given.

    // tag names
    tagNameFormat.setForeground(Qt::darkBlue);
    rule.pattern = QRegExp("(<\\w*)|(</\\w*>)|(>)");
    rule.format = tagNameFormat;
    highlightingRules.append(rule);


    // tag symbols
    rule.pattern = QRegExp("(<)|(</)|(>)");
    rule.format = tagSymbolsFormat;
    highlightingRules.append(rule);


    // attribute names
    attributeNameFormat.setForeground(Qt::red);
    rule.pattern = QRegExp("(\\w+\\s*)(?==)");  // (?==) means 'true if '=', but don't match it'
    rule.format = attributeNameFormat;
    highlightingRules.append(rule);


    // strings
    quotationFormat.setForeground(Qt::darkRed);
    rule.pattern = QRegExp("(\"[^\"]*\")|(\'[^\']*\')");
    rule.format = quotationFormat;
    highlightingRules.append(rule);


    // multi-line comments
    commentStartExpression = QRegExp("<!--");
    commentEndExpression = QRegExp("-->");
    multiLineCommentFormat.setFontItalic(true);
    multiLineCommentFormat.setForeground(Qt::darkGreen);

}



void MagMLHighlighter::highlightBlock(const QString &text)
{
    foreach (HighlightingRule rule, highlightingRules) 
    {
        QRegExp expression(rule.pattern);
        int index = text.indexOf(expression);
        while (index >= 0) 
        {
            int length = expression.matchedLength();
            setFormat(index, length, rule.format);
            index = text.indexOf(expression, index + length);
        }
    }


    // multi-line comments

    setCurrentBlockState(0);

    int startIndex = 0;
    if (previousBlockState() != 1)
        startIndex = text.indexOf(commentStartExpression);

    while (startIndex >= 0) 
    {
        int endIndex = text.indexOf(commentEndExpression, startIndex);
        int commentLength;
        if (endIndex == -1) 
        {
            setCurrentBlockState(1);
            commentLength = text.length() - startIndex;
        } 
        else 
        {
            commentLength = endIndex - startIndex
                            + commentEndExpression.matchedLength();
        }
        setFormat(startIndex, commentLength, multiLineCommentFormat);
        startIndex = text.indexOf(commentStartExpression,
                                  startIndex + commentLength);
    }


}





// -----------------------------------------------------------------------
// -------------------------- MagMLHelper code ---------------------------
// -----------------------------------------------------------------------


// ---------------------------------------------------------------------------
// MagMLHelper::MagMLHelper
// constructor
// ---------------------------------------------------------------------------

MagMLHelper::MagMLHelper()
{
    languageName_ = "MagML";
    className_    = "MAGML";
    serviceName_  = "MagML";
}


// ---------------------------------------------------------------------------
// MagMLHelper::~MagMLHelper
// destructor
// ---------------------------------------------------------------------------

MagMLHelper::~MagMLHelper()
{
}


// ---------------------------------------------------------------------------
// MagMLHelper::~createHighlighter
// creates a new syntax highlighter object
// ---------------------------------------------------------------------------

Highlighter *MagMLHelper::createHighlighter(QTextDocument *parent)
{
    return new MagMLHighlighter(parent);
}



// ---------------------------------------------------------------------------
// MagMLHelper::commentBlock
// puts the text into an XML comment block
// ---------------------------------------------------------------------------

QString MagMLHelper::commentBlock(const QString &text)
{
    QString result("<!--\n");

    result += text + "-->\n";

    return result;
}







