/*
 * Windows 950 codepage (Traditional Chinese) functions
 *
 * Copyright (c) 2008-2012, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This software is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <types.h>

#include "libuna_codepage_windows_950.h"
#include "libuna_inline.h"
#include "libuna_libcerror.h"
#include "libuna_types.h"

/* Extended ASCII to Unicode character lookup table for Windows 950 codepage
 * Unknown are filled with the Unicode replacement character 0xfffd
 */

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa140[ 63 ] = {
	0x3000, 0xff0c, 0x3001, 0x3002, 0xff0e, 0x2027, 0xff1b, 0xff1a,
	0xff1f, 0xff01, 0xfe30, 0x2026, 0x2025, 0xfe50, 0xfe51, 0xfe52,
	0x00b7, 0xfe54, 0xfe55, 0xfe56, 0xfe57, 0xff5c, 0x2013, 0xfe31,
	0x2014, 0xfe33, 0x2574, 0xfe34, 0xfe4f, 0xff08, 0xff09, 0xfe35,
	0xfe36, 0xff5b, 0xff5d, 0xfe37, 0xfe38, 0x3014, 0x3015, 0xfe39,
	0xfe3a, 0x3010, 0x3011, 0xfe3b, 0xfe3c, 0x300a, 0x300b, 0xfe3d,
	0xfe3e, 0x3008, 0x3009, 0xfe3f, 0xfe40, 0x300c, 0x300d, 0xfe41,
	0xfe42, 0x300e, 0x300f, 0xfe43, 0xfe44, 0xfe59, 0xfe5a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa1a1[ 94 ] = {
	        0xfe5b, 0xfe5c, 0xfe5d, 0xfe5e, 0x2018, 0x2019, 0x201c,
	0x201d, 0x301d, 0x301e, 0x2035, 0x2032, 0xff03, 0xff06, 0xff0a,
	0x203b, 0x00a7, 0x3003, 0x25cb, 0x25cf, 0x25b3, 0x25b2, 0x25ce,
	0x2606, 0x2605, 0x25c7, 0x25c6, 0x25a1, 0x25a0, 0x25bd, 0x25bc,
	0x32a3, 0x2105, 0x00af, 0xffe3, 0xff3f, 0x02cd, 0xfe49, 0xfe4a,
	0xfe4d, 0xfe4e, 0xfe4b, 0xfe4c, 0xfe5f, 0xfe60, 0xfe61, 0xff0b,
	0xff0d, 0x00d7, 0x00f7, 0x00b1, 0x221a, 0xff1c, 0xff1e, 0xff1d,
	0x2266, 0x2267, 0x2260, 0x221e, 0x2252, 0x2261, 0xfe62, 0xfe63,
	0xfe64, 0xfe65, 0xfe66, 0xff5e, 0x2229, 0x222a, 0x22a5, 0x2220,
	0x221f, 0x22bf, 0x33d2, 0x33d1, 0x222b, 0x222e, 0x2235, 0x2234,
	0x2640, 0x2642, 0x2295, 0x2299, 0x2191, 0x2193, 0x2190, 0x2192,
	0x2196, 0x2197, 0x2199, 0x2198, 0x2225, 0x2223, 0xff0f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa240[ 63 ] = {
	0xff3c, 0x2215, 0xfe68, 0xff04, 0xffe5, 0x3012, 0xffe0, 0xffe1,
	0xff05, 0xff20, 0x2103, 0x2109, 0xfe69, 0xfe6a, 0xfe6b, 0x33d5,
	0x339c, 0x339d, 0x339e, 0x33ce, 0x33a1, 0x338e, 0x338f, 0x33c4,
	0x00b0, 0x5159, 0x515b, 0x515e, 0x515d, 0x5161, 0x5163, 0x55e7,
	0x74e9, 0x7cce, 0x2581, 0x2582, 0x2583, 0x2584, 0x2585, 0x2586,
	0x2587, 0x2588, 0x258f, 0x258e, 0x258d, 0x258c, 0x258b, 0x258a,
	0x2589, 0x253c, 0x2534, 0x252c, 0x2524, 0x251c, 0x2594, 0x2500,
	0x2502, 0x2595, 0x250c, 0x2510, 0x2514, 0x2518, 0x256d
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa2a1[ 94 ] = {
	        0x256e, 0x2570, 0x256f, 0x2550, 0x255e, 0x256a, 0x2561,
	0x25e2, 0x25e3, 0x25e5, 0x25e4, 0x2571, 0x2572, 0x2573, 0xff10,
	0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17, 0xff18,
	0xff19, 0x2160, 0x2161, 0x2162, 0x2163, 0x2164, 0x2165, 0x2166,
	0x2167, 0x2168, 0x2169, 0x3021, 0x3022, 0x3023, 0x3024, 0x3025,
	0x3026, 0x3027, 0x3028, 0x3029, 0x5341, 0x5344, 0x5345, 0xff21,
	0xff22, 0xff23, 0xff24, 0xff25, 0xff26, 0xff27, 0xff28, 0xff29,
	0xff2a, 0xff2b, 0xff2c, 0xff2d, 0xff2e, 0xff2f, 0xff30, 0xff31,
	0xff32, 0xff33, 0xff34, 0xff35, 0xff36, 0xff37, 0xff38, 0xff39,
	0xff3a, 0xff41, 0xff42, 0xff43, 0xff44, 0xff45, 0xff46, 0xff47,
	0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c, 0xff4d, 0xff4e, 0xff4f,
	0xff50, 0xff51, 0xff52, 0xff53, 0xff54, 0xff55, 0xff56
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa340[ 63 ] = {
	0xff57, 0xff58, 0xff59, 0xff5a, 0x0391, 0x0392, 0x0393, 0x0394,
	0x0395, 0x0396, 0x0397, 0x0398, 0x0399, 0x039a, 0x039b, 0x039c,
	0x039d, 0x039e, 0x039f, 0x03a0, 0x03a1, 0x03a3, 0x03a4, 0x03a5,
	0x03a6, 0x03a7, 0x03a8, 0x03a9, 0x03b1, 0x03b2, 0x03b3, 0x03b4,
	0x03b5, 0x03b6, 0x03b7, 0x03b8, 0x03b9, 0x03ba, 0x03bb, 0x03bc,
	0x03bd, 0x03be, 0x03bf, 0x03c0, 0x03c1, 0x03c3, 0x03c4, 0x03c5,
	0x03c6, 0x03c7, 0x03c8, 0x03c9, 0x3105, 0x3106, 0x3107, 0x3108,
	0x3109, 0x310a, 0x310b, 0x310c, 0x310d, 0x310e, 0x310f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa3a1[ 31 ] = {
	        0x3110, 0x3111, 0x3112, 0x3113, 0x3114, 0x3115, 0x3116,
	0x3117, 0x3118, 0x3119, 0x311a, 0x311b, 0x311c, 0x311d, 0x311e,
	0x311f, 0x3120, 0x3121, 0x3122, 0x3123, 0x3124, 0x3125, 0x3126,
	0x3127, 0x3128, 0x3129, 0x02d9, 0x02c9, 0x02ca, 0x02c7, 0x02cb
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa440[ 63 ] = {
	0x4e00, 0x4e59, 0x4e01, 0x4e03, 0x4e43, 0x4e5d, 0x4e86, 0x4e8c,
	0x4eba, 0x513f, 0x5165, 0x516b, 0x51e0, 0x5200, 0x5201, 0x529b,
	0x5315, 0x5341, 0x535c, 0x53c8, 0x4e09, 0x4e0b, 0x4e08, 0x4e0a,
	0x4e2b, 0x4e38, 0x51e1, 0x4e45, 0x4e48, 0x4e5f, 0x4e5e, 0x4e8e,
	0x4ea1, 0x5140, 0x5203, 0x52fa, 0x5343, 0x53c9, 0x53e3, 0x571f,
	0x58eb, 0x5915, 0x5927, 0x5973, 0x5b50, 0x5b51, 0x5b53, 0x5bf8,
	0x5c0f, 0x5c22, 0x5c38, 0x5c71, 0x5ddd, 0x5de5, 0x5df1, 0x5df2,
	0x5df3, 0x5dfe, 0x5e72, 0x5efe, 0x5f0b, 0x5f13, 0x624d
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa4a1[ 94 ] = {
	        0x4e11, 0x4e10, 0x4e0d, 0x4e2d, 0x4e30, 0x4e39, 0x4e4b,
	0x5c39, 0x4e88, 0x4e91, 0x4e95, 0x4e92, 0x4e94, 0x4ea2, 0x4ec1,
	0x4ec0, 0x4ec3, 0x4ec6, 0x4ec7, 0x4ecd, 0x4eca, 0x4ecb, 0x4ec4,
	0x5143, 0x5141, 0x5167, 0x516d, 0x516e, 0x516c, 0x5197, 0x51f6,
	0x5206, 0x5207, 0x5208, 0x52fb, 0x52fe, 0x52ff, 0x5316, 0x5339,
	0x5348, 0x5347, 0x5345, 0x535e, 0x5384, 0x53cb, 0x53ca, 0x53cd,
	0x58ec, 0x5929, 0x592b, 0x592a, 0x592d, 0x5b54, 0x5c11, 0x5c24,
	0x5c3a, 0x5c6f, 0x5df4, 0x5e7b, 0x5eff, 0x5f14, 0x5f15, 0x5fc3,
	0x6208, 0x6236, 0x624b, 0x624e, 0x652f, 0x6587, 0x6597, 0x65a4,
	0x65b9, 0x65e5, 0x66f0, 0x6708, 0x6728, 0x6b20, 0x6b62, 0x6b79,
	0x6bcb, 0x6bd4, 0x6bdb, 0x6c0f, 0x6c34, 0x706b, 0x722a, 0x7236,
	0x723b, 0x7247, 0x7259, 0x725b, 0x72ac, 0x738b, 0x4e19
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa540[ 63 ] = {
	0x4e16, 0x4e15, 0x4e14, 0x4e18, 0x4e3b, 0x4e4d, 0x4e4f, 0x4e4e,
	0x4ee5, 0x4ed8, 0x4ed4, 0x4ed5, 0x4ed6, 0x4ed7, 0x4ee3, 0x4ee4,
	0x4ed9, 0x4ede, 0x5145, 0x5144, 0x5189, 0x518a, 0x51ac, 0x51f9,
	0x51fa, 0x51f8, 0x520a, 0x52a0, 0x529f, 0x5305, 0x5306, 0x5317,
	0x531d, 0x4edf, 0x534a, 0x5349, 0x5361, 0x5360, 0x536f, 0x536e,
	0x53bb, 0x53ef, 0x53e4, 0x53f3, 0x53ec, 0x53ee, 0x53e9, 0x53e8,
	0x53fc, 0x53f8, 0x53f5, 0x53eb, 0x53e6, 0x53ea, 0x53f2, 0x53f1,
	0x53f0, 0x53e5, 0x53ed, 0x53fb, 0x56db, 0x56da, 0x5916
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa5a1[ 94 ] = {
	        0x592e, 0x5931, 0x5974, 0x5976, 0x5b55, 0x5b83, 0x5c3c,
	0x5de8, 0x5de7, 0x5de6, 0x5e02, 0x5e03, 0x5e73, 0x5e7c, 0x5f01,
	0x5f18, 0x5f17, 0x5fc5, 0x620a, 0x6253, 0x6254, 0x6252, 0x6251,
	0x65a5, 0x65e6, 0x672e, 0x672c, 0x672a, 0x672b, 0x672d, 0x6b63,
	0x6bcd, 0x6c11, 0x6c10, 0x6c38, 0x6c41, 0x6c40, 0x6c3e, 0x72af,
	0x7384, 0x7389, 0x74dc, 0x74e6, 0x7518, 0x751f, 0x7528, 0x7529,
	0x7530, 0x7531, 0x7532, 0x7533, 0x758b, 0x767d, 0x76ae, 0x76bf,
	0x76ee, 0x77db, 0x77e2, 0x77f3, 0x793a, 0x79be, 0x7a74, 0x7acb,
	0x4e1e, 0x4e1f, 0x4e52, 0x4e53, 0x4e69, 0x4e99, 0x4ea4, 0x4ea6,
	0x4ea5, 0x4eff, 0x4f09, 0x4f19, 0x4f0a, 0x4f15, 0x4f0d, 0x4f10,
	0x4f11, 0x4f0f, 0x4ef2, 0x4ef6, 0x4efb, 0x4ef0, 0x4ef3, 0x4efd,
	0x4f01, 0x4f0b, 0x5149, 0x5147, 0x5146, 0x5148, 0x5168
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa640[ 63 ] = {
	0x5171, 0x518d, 0x51b0, 0x5217, 0x5211, 0x5212, 0x520e, 0x5216,
	0x52a3, 0x5308, 0x5321, 0x5320, 0x5370, 0x5371, 0x5409, 0x540f,
	0x540c, 0x540a, 0x5410, 0x5401, 0x540b, 0x5404, 0x5411, 0x540d,
	0x5408, 0x5403, 0x540e, 0x5406, 0x5412, 0x56e0, 0x56de, 0x56dd,
	0x5733, 0x5730, 0x5728, 0x572d, 0x572c, 0x572f, 0x5729, 0x5919,
	0x591a, 0x5937, 0x5938, 0x5984, 0x5978, 0x5983, 0x597d, 0x5979,
	0x5982, 0x5981, 0x5b57, 0x5b58, 0x5b87, 0x5b88, 0x5b85, 0x5b89,
	0x5bfa, 0x5c16, 0x5c79, 0x5dde, 0x5e06, 0x5e76, 0x5e74
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa6a1[ 94 ] = {
	        0x5f0f, 0x5f1b, 0x5fd9, 0x5fd6, 0x620e, 0x620c, 0x620d,
	0x6210, 0x6263, 0x625b, 0x6258, 0x6536, 0x65e9, 0x65e8, 0x65ec,
	0x65ed, 0x66f2, 0x66f3, 0x6709, 0x673d, 0x6734, 0x6731, 0x6735,
	0x6b21, 0x6b64, 0x6b7b, 0x6c16, 0x6c5d, 0x6c57, 0x6c59, 0x6c5f,
	0x6c60, 0x6c50, 0x6c55, 0x6c61, 0x6c5b, 0x6c4d, 0x6c4e, 0x7070,
	0x725f, 0x725d, 0x767e, 0x7af9, 0x7c73, 0x7cf8, 0x7f36, 0x7f8a,
	0x7fbd, 0x8001, 0x8003, 0x800c, 0x8012, 0x8033, 0x807f, 0x8089,
	0x808b, 0x808c, 0x81e3, 0x81ea, 0x81f3, 0x81fc, 0x820c, 0x821b,
	0x821f, 0x826e, 0x8272, 0x827e, 0x866b, 0x8840, 0x884c, 0x8863,
	0x897f, 0x9621, 0x4e32, 0x4ea8, 0x4f4d, 0x4f4f, 0x4f47, 0x4f57,
	0x4f5e, 0x4f34, 0x4f5b, 0x4f55, 0x4f30, 0x4f50, 0x4f51, 0x4f3d,
	0x4f3a, 0x4f38, 0x4f43, 0x4f54, 0x4f3c, 0x4f46, 0x4f63
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa740[ 63 ] = {
	0x4f5c, 0x4f60, 0x4f2f, 0x4f4e, 0x4f36, 0x4f59, 0x4f5d, 0x4f48,
	0x4f5a, 0x514c, 0x514b, 0x514d, 0x5175, 0x51b6, 0x51b7, 0x5225,
	0x5224, 0x5229, 0x522a, 0x5228, 0x52ab, 0x52a9, 0x52aa, 0x52ac,
	0x5323, 0x5373, 0x5375, 0x541d, 0x542d, 0x541e, 0x543e, 0x5426,
	0x544e, 0x5427, 0x5446, 0x5443, 0x5433, 0x5448, 0x5442, 0x541b,
	0x5429, 0x544a, 0x5439, 0x543b, 0x5438, 0x542e, 0x5435, 0x5436,
	0x5420, 0x543c, 0x5440, 0x5431, 0x542b, 0x541f, 0x542c, 0x56ea,
	0x56f0, 0x56e4, 0x56eb, 0x574a, 0x5751, 0x5740, 0x574d
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa7a1[ 94 ] = {
	        0x5747, 0x574e, 0x573e, 0x5750, 0x574f, 0x573b, 0x58ef,
	0x593e, 0x599d, 0x5992, 0x59a8, 0x599e, 0x59a3, 0x5999, 0x5996,
	0x598d, 0x59a4, 0x5993, 0x598a, 0x59a5, 0x5b5d, 0x5b5c, 0x5b5a,
	0x5b5b, 0x5b8c, 0x5b8b, 0x5b8f, 0x5c2c, 0x5c40, 0x5c41, 0x5c3f,
	0x5c3e, 0x5c90, 0x5c91, 0x5c94, 0x5c8c, 0x5deb, 0x5e0c, 0x5e8f,
	0x5e87, 0x5e8a, 0x5ef7, 0x5f04, 0x5f1f, 0x5f64, 0x5f62, 0x5f77,
	0x5f79, 0x5fd8, 0x5fcc, 0x5fd7, 0x5fcd, 0x5ff1, 0x5feb, 0x5ff8,
	0x5fea, 0x6212, 0x6211, 0x6284, 0x6297, 0x6296, 0x6280, 0x6276,
	0x6289, 0x626d, 0x628a, 0x627c, 0x627e, 0x6279, 0x6273, 0x6292,
	0x626f, 0x6298, 0x626e, 0x6295, 0x6293, 0x6291, 0x6286, 0x6539,
	0x653b, 0x6538, 0x65f1, 0x66f4, 0x675f, 0x674e, 0x674f, 0x6750,
	0x6751, 0x675c, 0x6756, 0x675e, 0x6749, 0x6746, 0x6760
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa840[ 63 ] = {
	0x6753, 0x6757, 0x6b65, 0x6bcf, 0x6c42, 0x6c5e, 0x6c99, 0x6c81,
	0x6c88, 0x6c89, 0x6c85, 0x6c9b, 0x6c6a, 0x6c7a, 0x6c90, 0x6c70,
	0x6c8c, 0x6c68, 0x6c96, 0x6c92, 0x6c7d, 0x6c83, 0x6c72, 0x6c7e,
	0x6c74, 0x6c86, 0x6c76, 0x6c8d, 0x6c94, 0x6c98, 0x6c82, 0x7076,
	0x707c, 0x707d, 0x7078, 0x7262, 0x7261, 0x7260, 0x72c4, 0x72c2,
	0x7396, 0x752c, 0x752b, 0x7537, 0x7538, 0x7682, 0x76ef, 0x77e3,
	0x79c1, 0x79c0, 0x79bf, 0x7a76, 0x7cfb, 0x7f55, 0x8096, 0x8093,
	0x809d, 0x8098, 0x809b, 0x809a, 0x80b2, 0x826f, 0x8292
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa8a1[ 94 ] = {
	        0x828b, 0x828d, 0x898b, 0x89d2, 0x8a00, 0x8c37, 0x8c46,
	0x8c55, 0x8c9d, 0x8d64, 0x8d70, 0x8db3, 0x8eab, 0x8eca, 0x8f9b,
	0x8fb0, 0x8fc2, 0x8fc6, 0x8fc5, 0x8fc4, 0x5de1, 0x9091, 0x90a2,
	0x90aa, 0x90a6, 0x90a3, 0x9149, 0x91c6, 0x91cc, 0x9632, 0x962e,
	0x9631, 0x962a, 0x962c, 0x4e26, 0x4e56, 0x4e73, 0x4e8b, 0x4e9b,
	0x4e9e, 0x4eab, 0x4eac, 0x4f6f, 0x4f9d, 0x4f8d, 0x4f73, 0x4f7f,
	0x4f6c, 0x4f9b, 0x4f8b, 0x4f86, 0x4f83, 0x4f70, 0x4f75, 0x4f88,
	0x4f69, 0x4f7b, 0x4f96, 0x4f7e, 0x4f8f, 0x4f91, 0x4f7a, 0x5154,
	0x5152, 0x5155, 0x5169, 0x5177, 0x5176, 0x5178, 0x51bd, 0x51fd,
	0x523b, 0x5238, 0x5237, 0x523a, 0x5230, 0x522e, 0x5236, 0x5241,
	0x52be, 0x52bb, 0x5352, 0x5354, 0x5353, 0x5351, 0x5366, 0x5377,
	0x5378, 0x5379, 0x53d6, 0x53d4, 0x53d7, 0x5473, 0x5475
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa940[ 63 ] = {
	0x5496, 0x5478, 0x5495, 0x5480, 0x547b, 0x5477, 0x5484, 0x5492,
	0x5486, 0x547c, 0x5490, 0x5471, 0x5476, 0x548c, 0x549a, 0x5462,
	0x5468, 0x548b, 0x547d, 0x548e, 0x56fa, 0x5783, 0x5777, 0x576a,
	0x5769, 0x5761, 0x5766, 0x5764, 0x577c, 0x591c, 0x5949, 0x5947,
	0x5948, 0x5944, 0x5954, 0x59be, 0x59bb, 0x59d4, 0x59b9, 0x59ae,
	0x59d1, 0x59c6, 0x59d0, 0x59cd, 0x59cb, 0x59d3, 0x59ca, 0x59af,
	0x59b3, 0x59d2, 0x59c5, 0x5b5f, 0x5b64, 0x5b63, 0x5b97, 0x5b9a,
	0x5b98, 0x5b9c, 0x5b99, 0x5b9b, 0x5c1a, 0x5c48, 0x5c45
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa9a1[ 94 ] = {
	        0x5c46, 0x5cb7, 0x5ca1, 0x5cb8, 0x5ca9, 0x5cab, 0x5cb1,
	0x5cb3, 0x5e18, 0x5e1a, 0x5e16, 0x5e15, 0x5e1b, 0x5e11, 0x5e78,
	0x5e9a, 0x5e97, 0x5e9c, 0x5e95, 0x5e96, 0x5ef6, 0x5f26, 0x5f27,
	0x5f29, 0x5f80, 0x5f81, 0x5f7f, 0x5f7c, 0x5fdd, 0x5fe0, 0x5ffd,
	0x5ff5, 0x5fff, 0x600f, 0x6014, 0x602f, 0x6035, 0x6016, 0x602a,
	0x6015, 0x6021, 0x6027, 0x6029, 0x602b, 0x601b, 0x6216, 0x6215,
	0x623f, 0x623e, 0x6240, 0x627f, 0x62c9, 0x62cc, 0x62c4, 0x62bf,
	0x62c2, 0x62b9, 0x62d2, 0x62db, 0x62ab, 0x62d3, 0x62d4, 0x62cb,
	0x62c8, 0x62a8, 0x62bd, 0x62bc, 0x62d0, 0x62d9, 0x62c7, 0x62cd,
	0x62b5, 0x62da, 0x62b1, 0x62d8, 0x62d6, 0x62d7, 0x62c6, 0x62ac,
	0x62ce, 0x653e, 0x65a7, 0x65bc, 0x65fa, 0x6614, 0x6613, 0x660c,
	0x6606, 0x6602, 0x660e, 0x6600, 0x660f, 0x6615, 0x660a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaa40[ 63 ] = {
	0x6607, 0x670d, 0x670b, 0x676d, 0x678b, 0x6795, 0x6771, 0x679c,
	0x6773, 0x6777, 0x6787, 0x679d, 0x6797, 0x676f, 0x6770, 0x677f,
	0x6789, 0x677e, 0x6790, 0x6775, 0x679a, 0x6793, 0x677c, 0x676a,
	0x6772, 0x6b23, 0x6b66, 0x6b67, 0x6b7f, 0x6c13, 0x6c1b, 0x6ce3,
	0x6ce8, 0x6cf3, 0x6cb1, 0x6ccc, 0x6ce5, 0x6cb3, 0x6cbd, 0x6cbe,
	0x6cbc, 0x6ce2, 0x6cab, 0x6cd5, 0x6cd3, 0x6cb8, 0x6cc4, 0x6cb9,
	0x6cc1, 0x6cae, 0x6cd7, 0x6cc5, 0x6cf1, 0x6cbf, 0x6cbb, 0x6ce1,
	0x6cdb, 0x6cca, 0x6cac, 0x6cef, 0x6cdc, 0x6cd6, 0x6ce0
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaaa1[ 94 ] = {
	        0x7095, 0x708e, 0x7092, 0x708a, 0x7099, 0x722c, 0x722d,
	0x7238, 0x7248, 0x7267, 0x7269, 0x72c0, 0x72ce, 0x72d9, 0x72d7,
	0x72d0, 0x73a9, 0x73a8, 0x739f, 0x73ab, 0x73a5, 0x753d, 0x759d,
	0x7599, 0x759a, 0x7684, 0x76c2, 0x76f2, 0x76f4, 0x77e5, 0x77fd,
	0x793e, 0x7940, 0x7941, 0x79c9, 0x79c8, 0x7a7a, 0x7a79, 0x7afa,
	0x7cfe, 0x7f54, 0x7f8c, 0x7f8b, 0x8005, 0x80ba, 0x80a5, 0x80a2,
	0x80b1, 0x80a1, 0x80ab, 0x80a9, 0x80b4, 0x80aa, 0x80af, 0x81e5,
	0x81fe, 0x820d, 0x82b3, 0x829d, 0x8299, 0x82ad, 0x82bd, 0x829f,
	0x82b9, 0x82b1, 0x82ac, 0x82a5, 0x82af, 0x82b8, 0x82a3, 0x82b0,
	0x82be, 0x82b7, 0x864e, 0x8671, 0x521d, 0x8868, 0x8ecb, 0x8fce,
	0x8fd4, 0x8fd1, 0x90b5, 0x90b8, 0x90b1, 0x90b6, 0x91c7, 0x91d1,
	0x9577, 0x9580, 0x961c, 0x9640, 0x963f, 0x963b, 0x9644
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xab40[ 63 ] = {
	0x9642, 0x96b9, 0x96e8, 0x9752, 0x975e, 0x4e9f, 0x4ead, 0x4eae,
	0x4fe1, 0x4fb5, 0x4faf, 0x4fbf, 0x4fe0, 0x4fd1, 0x4fcf, 0x4fdd,
	0x4fc3, 0x4fb6, 0x4fd8, 0x4fdf, 0x4fca, 0x4fd7, 0x4fae, 0x4fd0,
	0x4fc4, 0x4fc2, 0x4fda, 0x4fce, 0x4fde, 0x4fb7, 0x5157, 0x5192,
	0x5191, 0x51a0, 0x524e, 0x5243, 0x524a, 0x524d, 0x524c, 0x524b,
	0x5247, 0x52c7, 0x52c9, 0x52c3, 0x52c1, 0x530d, 0x5357, 0x537b,
	0x539a, 0x53db, 0x54ac, 0x54c0, 0x54a8, 0x54ce, 0x54c9, 0x54b8,
	0x54a6, 0x54b3, 0x54c7, 0x54c2, 0x54bd, 0x54aa, 0x54c1
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaba1[ 94 ] = {
	        0x54c4, 0x54c8, 0x54af, 0x54ab, 0x54b1, 0x54bb, 0x54a9,
	0x54a7, 0x54bf, 0x56ff, 0x5782, 0x578b, 0x57a0, 0x57a3, 0x57a2,
	0x57ce, 0x57ae, 0x5793, 0x5955, 0x5951, 0x594f, 0x594e, 0x5950,
	0x59dc, 0x59d8, 0x59ff, 0x59e3, 0x59e8, 0x5a03, 0x59e5, 0x59ea,
	0x59da, 0x59e6, 0x5a01, 0x59fb, 0x5b69, 0x5ba3, 0x5ba6, 0x5ba4,
	0x5ba2, 0x5ba5, 0x5c01, 0x5c4e, 0x5c4f, 0x5c4d, 0x5c4b, 0x5cd9,
	0x5cd2, 0x5df7, 0x5e1d, 0x5e25, 0x5e1f, 0x5e7d, 0x5ea0, 0x5ea6,
	0x5efa, 0x5f08, 0x5f2d, 0x5f65, 0x5f88, 0x5f85, 0x5f8a, 0x5f8b,
	0x5f87, 0x5f8c, 0x5f89, 0x6012, 0x601d, 0x6020, 0x6025, 0x600e,
	0x6028, 0x604d, 0x6070, 0x6068, 0x6062, 0x6046, 0x6043, 0x606c,
	0x606b, 0x606a, 0x6064, 0x6241, 0x62dc, 0x6316, 0x6309, 0x62fc,
	0x62ed, 0x6301, 0x62ee, 0x62fd, 0x6307, 0x62f1, 0x62f7
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xac40[ 63 ] = {
	0x62ef, 0x62ec, 0x62fe, 0x62f4, 0x6311, 0x6302, 0x653f, 0x6545,
	0x65ab, 0x65bd, 0x65e2, 0x6625, 0x662d, 0x6620, 0x6627, 0x662f,
	0x661f, 0x6628, 0x6631, 0x6624, 0x66f7, 0x67ff, 0x67d3, 0x67f1,
	0x67d4, 0x67d0, 0x67ec, 0x67b6, 0x67af, 0x67f5, 0x67e9, 0x67ef,
	0x67c4, 0x67d1, 0x67b4, 0x67da, 0x67e5, 0x67b8, 0x67cf, 0x67de,
	0x67f3, 0x67b0, 0x67d9, 0x67e2, 0x67dd, 0x67d2, 0x6b6a, 0x6b83,
	0x6b86, 0x6bb5, 0x6bd2, 0x6bd7, 0x6c1f, 0x6cc9, 0x6d0b, 0x6d32,
	0x6d2a, 0x6d41, 0x6d25, 0x6d0c, 0x6d31, 0x6d1e, 0x6d17
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaca1[ 94 ] = {
	        0x6d3b, 0x6d3d, 0x6d3e, 0x6d36, 0x6d1b, 0x6cf5, 0x6d39,
	0x6d27, 0x6d38, 0x6d29, 0x6d2e, 0x6d35, 0x6d0e, 0x6d2b, 0x70ab,
	0x70ba, 0x70b3, 0x70ac, 0x70af, 0x70ad, 0x70b8, 0x70ae, 0x70a4,
	0x7230, 0x7272, 0x726f, 0x7274, 0x72e9, 0x72e0, 0x72e1, 0x73b7,
	0x73ca, 0x73bb, 0x73b2, 0x73cd, 0x73c0, 0x73b3, 0x751a, 0x752d,
	0x754f, 0x754c, 0x754e, 0x754b, 0x75ab, 0x75a4, 0x75a5, 0x75a2,
	0x75a3, 0x7678, 0x7686, 0x7687, 0x7688, 0x76c8, 0x76c6, 0x76c3,
	0x76c5, 0x7701, 0x76f9, 0x76f8, 0x7709, 0x770b, 0x76fe, 0x76fc,
	0x7707, 0x77dc, 0x7802, 0x7814, 0x780c, 0x780d, 0x7946, 0x7949,
	0x7948, 0x7947, 0x79b9, 0x79ba, 0x79d1, 0x79d2, 0x79cb, 0x7a7f,
	0x7a81, 0x7aff, 0x7afd, 0x7c7d, 0x7d02, 0x7d05, 0x7d00, 0x7d09,
	0x7d07, 0x7d04, 0x7d06, 0x7f38, 0x7f8e, 0x7fbf, 0x8004
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xad40[ 63 ] = {
	0x8010, 0x800d, 0x8011, 0x8036, 0x80d6, 0x80e5, 0x80da, 0x80c3,
	0x80c4, 0x80cc, 0x80e1, 0x80db, 0x80ce, 0x80de, 0x80e4, 0x80dd,
	0x81f4, 0x8222, 0x82e7, 0x8303, 0x8305, 0x82e3, 0x82db, 0x82e6,
	0x8304, 0x82e5, 0x8302, 0x8309, 0x82d2, 0x82d7, 0x82f1, 0x8301,
	0x82dc, 0x82d4, 0x82d1, 0x82de, 0x82d3, 0x82df, 0x82ef, 0x8306,
	0x8650, 0x8679, 0x867b, 0x867a, 0x884d, 0x886b, 0x8981, 0x89d4,
	0x8a08, 0x8a02, 0x8a03, 0x8c9e, 0x8ca0, 0x8d74, 0x8d73, 0x8db4,
	0x8ecd, 0x8ecc, 0x8ff0, 0x8fe6, 0x8fe2, 0x8fea, 0x8fe5
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xada1[ 94 ] = {
	        0x8fed, 0x8feb, 0x8fe4, 0x8fe8, 0x90ca, 0x90ce, 0x90c1,
	0x90c3, 0x914b, 0x914a, 0x91cd, 0x9582, 0x9650, 0x964b, 0x964c,
	0x964d, 0x9762, 0x9769, 0x97cb, 0x97ed, 0x97f3, 0x9801, 0x98a8,
	0x98db, 0x98df, 0x9996, 0x9999, 0x4e58, 0x4eb3, 0x500c, 0x500d,
	0x5023, 0x4fef, 0x5026, 0x5025, 0x4ff8, 0x5029, 0x5016, 0x5006,
	0x503c, 0x501f, 0x501a, 0x5012, 0x5011, 0x4ffa, 0x5000, 0x5014,
	0x5028, 0x4ff1, 0x5021, 0x500b, 0x5019, 0x5018, 0x4ff3, 0x4fee,
	0x502d, 0x502a, 0x4ffe, 0x502b, 0x5009, 0x517c, 0x51a4, 0x51a5,
	0x51a2, 0x51cd, 0x51cc, 0x51c6, 0x51cb, 0x5256, 0x525c, 0x5254,
	0x525b, 0x525d, 0x532a, 0x537f, 0x539f, 0x539d, 0x53df, 0x54e8,
	0x5510, 0x5501, 0x5537, 0x54fc, 0x54e5, 0x54f2, 0x5506, 0x54fa,
	0x5514, 0x54e9, 0x54ed, 0x54e1, 0x5509, 0x54ee, 0x54ea
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xae40[ 63 ] = {
	0x54e6, 0x5527, 0x5507, 0x54fd, 0x550f, 0x5703, 0x5704, 0x57c2,
	0x57d4, 0x57cb, 0x57c3, 0x5809, 0x590f, 0x5957, 0x5958, 0x595a,
	0x5a11, 0x5a18, 0x5a1c, 0x5a1f, 0x5a1b, 0x5a13, 0x59ec, 0x5a20,
	0x5a23, 0x5a29, 0x5a25, 0x5a0c, 0x5a09, 0x5b6b, 0x5c58, 0x5bb0,
	0x5bb3, 0x5bb6, 0x5bb4, 0x5bae, 0x5bb5, 0x5bb9, 0x5bb8, 0x5c04,
	0x5c51, 0x5c55, 0x5c50, 0x5ced, 0x5cfd, 0x5cfb, 0x5cea, 0x5ce8,
	0x5cf0, 0x5cf6, 0x5d01, 0x5cf4, 0x5dee, 0x5e2d, 0x5e2b, 0x5eab,
	0x5ead, 0x5ea7, 0x5f31, 0x5f92, 0x5f91, 0x5f90, 0x6059
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaea1[ 94 ] = {
	        0x6063, 0x6065, 0x6050, 0x6055, 0x606d, 0x6069, 0x606f,
	0x6084, 0x609f, 0x609a, 0x608d, 0x6094, 0x608c, 0x6085, 0x6096,
	0x6247, 0x62f3, 0x6308, 0x62ff, 0x634e, 0x633e, 0x632f, 0x6355,
	0x6342, 0x6346, 0x634f, 0x6349, 0x633a, 0x6350, 0x633d, 0x632a,
	0x632b, 0x6328, 0x634d, 0x634c, 0x6548, 0x6549, 0x6599, 0x65c1,
	0x65c5, 0x6642, 0x6649, 0x664f, 0x6643, 0x6652, 0x664c, 0x6645,
	0x6641, 0x66f8, 0x6714, 0x6715, 0x6717, 0x6821, 0x6838, 0x6848,
	0x6846, 0x6853, 0x6839, 0x6842, 0x6854, 0x6829, 0x68b3, 0x6817,
	0x684c, 0x6851, 0x683d, 0x67f4, 0x6850, 0x6840, 0x683c, 0x6843,
	0x682a, 0x6845, 0x6813, 0x6818, 0x6841, 0x6b8a, 0x6b89, 0x6bb7,
	0x6c23, 0x6c27, 0x6c28, 0x6c26, 0x6c24, 0x6cf0, 0x6d6a, 0x6d95,
	0x6d88, 0x6d87, 0x6d66, 0x6d78, 0x6d77, 0x6d59, 0x6d93
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaf40[ 63 ] = {
	0x6d6c, 0x6d89, 0x6d6e, 0x6d5a, 0x6d74, 0x6d69, 0x6d8c, 0x6d8a,
	0x6d79, 0x6d85, 0x6d65, 0x6d94, 0x70ca, 0x70d8, 0x70e4, 0x70d9,
	0x70c8, 0x70cf, 0x7239, 0x7279, 0x72fc, 0x72f9, 0x72fd, 0x72f8,
	0x72f7, 0x7386, 0x73ed, 0x7409, 0x73ee, 0x73e0, 0x73ea, 0x73de,
	0x7554, 0x755d, 0x755c, 0x755a, 0x7559, 0x75be, 0x75c5, 0x75c7,
	0x75b2, 0x75b3, 0x75bd, 0x75bc, 0x75b9, 0x75c2, 0x75b8, 0x768b,
	0x76b0, 0x76ca, 0x76cd, 0x76ce, 0x7729, 0x771f, 0x7720, 0x7728,
	0x77e9, 0x7830, 0x7827, 0x7838, 0x781d, 0x7834, 0x7837
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xafa1[ 94 ] = {
	        0x7825, 0x782d, 0x7820, 0x781f, 0x7832, 0x7955, 0x7950,
	0x7960, 0x795f, 0x7956, 0x795e, 0x795d, 0x7957, 0x795a, 0x79e4,
	0x79e3, 0x79e7, 0x79df, 0x79e6, 0x79e9, 0x79d8, 0x7a84, 0x7a88,
	0x7ad9, 0x7b06, 0x7b11, 0x7c89, 0x7d21, 0x7d17, 0x7d0b, 0x7d0a,
	0x7d20, 0x7d22, 0x7d14, 0x7d10, 0x7d15, 0x7d1a, 0x7d1c, 0x7d0d,
	0x7d19, 0x7d1b, 0x7f3a, 0x7f5f, 0x7f94, 0x7fc5, 0x7fc1, 0x8006,
	0x8018, 0x8015, 0x8019, 0x8017, 0x803d, 0x803f, 0x80f1, 0x8102,
	0x80f0, 0x8105, 0x80ed, 0x80f4, 0x8106, 0x80f8, 0x80f3, 0x8108,
	0x80fd, 0x810a, 0x80fc, 0x80ef, 0x81ed, 0x81ec, 0x8200, 0x8210,
	0x822a, 0x822b, 0x8228, 0x822c, 0x82bb, 0x832b, 0x8352, 0x8354,
	0x834a, 0x8338, 0x8350, 0x8349, 0x8335, 0x8334, 0x834f, 0x8332,
	0x8339, 0x8336, 0x8317, 0x8340, 0x8331, 0x8328, 0x8343
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb040[ 63 ] = {
	0x8654, 0x868a, 0x86aa, 0x8693, 0x86a4, 0x86a9, 0x868c, 0x86a3,
	0x869c, 0x8870, 0x8877, 0x8881, 0x8882, 0x887d, 0x8879, 0x8a18,
	0x8a10, 0x8a0e, 0x8a0c, 0x8a15, 0x8a0a, 0x8a17, 0x8a13, 0x8a16,
	0x8a0f, 0x8a11, 0x8c48, 0x8c7a, 0x8c79, 0x8ca1, 0x8ca2, 0x8d77,
	0x8eac, 0x8ed2, 0x8ed4, 0x8ecf, 0x8fb1, 0x9001, 0x9006, 0x8ff7,
	0x9000, 0x8ffa, 0x8ff4, 0x9003, 0x8ffd, 0x9005, 0x8ff8, 0x9095,
	0x90e1, 0x90dd, 0x90e2, 0x9152, 0x914d, 0x914c, 0x91d8, 0x91dd,
	0x91d7, 0x91dc, 0x91d9, 0x9583, 0x9662, 0x9663, 0x9661
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb0a1[ 94 ] = {
	        0x965b, 0x965d, 0x9664, 0x9658, 0x965e, 0x96bb, 0x98e2,
	0x99ac, 0x9aa8, 0x9ad8, 0x9b25, 0x9b32, 0x9b3c, 0x4e7e, 0x507a,
	0x507d, 0x505c, 0x5047, 0x5043, 0x504c, 0x505a, 0x5049, 0x5065,
	0x5076, 0x504e, 0x5055, 0x5075, 0x5074, 0x5077, 0x504f, 0x500f,
	0x506f, 0x506d, 0x515c, 0x5195, 0x51f0, 0x526a, 0x526f, 0x52d2,
	0x52d9, 0x52d8, 0x52d5, 0x5310, 0x530f, 0x5319, 0x533f, 0x5340,
	0x533e, 0x53c3, 0x66fc, 0x5546, 0x556a, 0x5566, 0x5544, 0x555e,
	0x5561, 0x5543, 0x554a, 0x5531, 0x5556, 0x554f, 0x5555, 0x552f,
	0x5564, 0x5538, 0x552e, 0x555c, 0x552c, 0x5563, 0x5533, 0x5541,
	0x5557, 0x5708, 0x570b, 0x5709, 0x57df, 0x5805, 0x580a, 0x5806,
	0x57e0, 0x57e4, 0x57fa, 0x5802, 0x5835, 0x57f7, 0x57f9, 0x5920,
	0x5962, 0x5a36, 0x5a41, 0x5a49, 0x5a66, 0x5a6a, 0x5a40
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb140[ 63 ] = {
	0x5a3c, 0x5a62, 0x5a5a, 0x5a46, 0x5a4a, 0x5b70, 0x5bc7, 0x5bc5,
	0x5bc4, 0x5bc2, 0x5bbf, 0x5bc6, 0x5c09, 0x5c08, 0x5c07, 0x5c60,
	0x5c5c, 0x5c5d, 0x5d07, 0x5d06, 0x5d0e, 0x5d1b, 0x5d16, 0x5d22,
	0x5d11, 0x5d29, 0x5d14, 0x5d19, 0x5d24, 0x5d27, 0x5d17, 0x5de2,
	0x5e38, 0x5e36, 0x5e33, 0x5e37, 0x5eb7, 0x5eb8, 0x5eb6, 0x5eb5,
	0x5ebe, 0x5f35, 0x5f37, 0x5f57, 0x5f6c, 0x5f69, 0x5f6b, 0x5f97,
	0x5f99, 0x5f9e, 0x5f98, 0x5fa1, 0x5fa0, 0x5f9c, 0x607f, 0x60a3,
	0x6089, 0x60a0, 0x60a8, 0x60cb, 0x60b4, 0x60e6, 0x60bd
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb1a1[ 94 ] = {
	        0x60c5, 0x60bb, 0x60b5, 0x60dc, 0x60bc, 0x60d8, 0x60d5,
	0x60c6, 0x60df, 0x60b8, 0x60da, 0x60c7, 0x621a, 0x621b, 0x6248,
	0x63a0, 0x63a7, 0x6372, 0x6396, 0x63a2, 0x63a5, 0x6377, 0x6367,
	0x6398, 0x63aa, 0x6371, 0x63a9, 0x6389, 0x6383, 0x639b, 0x636b,
	0x63a8, 0x6384, 0x6388, 0x6399, 0x63a1, 0x63ac, 0x6392, 0x638f,
	0x6380, 0x637b, 0x6369, 0x6368, 0x637a, 0x655d, 0x6556, 0x6551,
	0x6559, 0x6557, 0x555f, 0x654f, 0x6558, 0x6555, 0x6554, 0x659c,
	0x659b, 0x65ac, 0x65cf, 0x65cb, 0x65cc, 0x65ce, 0x665d, 0x665a,
	0x6664, 0x6668, 0x6666, 0x665e, 0x66f9, 0x52d7, 0x671b, 0x6881,
	0x68af, 0x68a2, 0x6893, 0x68b5, 0x687f, 0x6876, 0x68b1, 0x68a7,
	0x6897, 0x68b0, 0x6883, 0x68c4, 0x68ad, 0x6886, 0x6885, 0x6894,
	0x689d, 0x68a8, 0x689f, 0x68a1, 0x6882, 0x6b32, 0x6bba
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb240[ 63 ] = {
	0x6beb, 0x6bec, 0x6c2b, 0x6d8e, 0x6dbc, 0x6df3, 0x6dd9, 0x6db2,
	0x6de1, 0x6dcc, 0x6de4, 0x6dfb, 0x6dfa, 0x6e05, 0x6dc7, 0x6dcb,
	0x6daf, 0x6dd1, 0x6dae, 0x6dde, 0x6df9, 0x6db8, 0x6df7, 0x6df5,
	0x6dc5, 0x6dd2, 0x6e1a, 0x6db5, 0x6dda, 0x6deb, 0x6dd8, 0x6dea,
	0x6df1, 0x6dee, 0x6de8, 0x6dc6, 0x6dc4, 0x6daa, 0x6dec, 0x6dbf,
	0x6de6, 0x70f9, 0x7109, 0x710a, 0x70fd, 0x70ef, 0x723d, 0x727d,
	0x7281, 0x731c, 0x731b, 0x7316, 0x7313, 0x7319, 0x7387, 0x7405,
	0x740a, 0x7403, 0x7406, 0x73fe, 0x740d, 0x74e0, 0x74f6
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb2a1[ 94 ] = {
	        0x74f7, 0x751c, 0x7522, 0x7565, 0x7566, 0x7562, 0x7570,
	0x758f, 0x75d4, 0x75d5, 0x75b5, 0x75ca, 0x75cd, 0x768e, 0x76d4,
	0x76d2, 0x76db, 0x7737, 0x773e, 0x773c, 0x7736, 0x7738, 0x773a,
	0x786b, 0x7843, 0x784e, 0x7965, 0x7968, 0x796d, 0x79fb, 0x7a92,
	0x7a95, 0x7b20, 0x7b28, 0x7b1b, 0x7b2c, 0x7b26, 0x7b19, 0x7b1e,
	0x7b2e, 0x7c92, 0x7c97, 0x7c95, 0x7d46, 0x7d43, 0x7d71, 0x7d2e,
	0x7d39, 0x7d3c, 0x7d40, 0x7d30, 0x7d33, 0x7d44, 0x7d2f, 0x7d42,
	0x7d32, 0x7d31, 0x7f3d, 0x7f9e, 0x7f9a, 0x7fcc, 0x7fce, 0x7fd2,
	0x801c, 0x804a, 0x8046, 0x812f, 0x8116, 0x8123, 0x812b, 0x8129,
	0x8130, 0x8124, 0x8202, 0x8235, 0x8237, 0x8236, 0x8239, 0x838e,
	0x839e, 0x8398, 0x8378, 0x83a2, 0x8396, 0x83bd, 0x83ab, 0x8392,
	0x838a, 0x8393, 0x8389, 0x83a0, 0x8377, 0x837b, 0x837c
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb340[ 63 ] = {
	0x8386, 0x83a7, 0x8655, 0x5f6a, 0x86c7, 0x86c0, 0x86b6, 0x86c4,
	0x86b5, 0x86c6, 0x86cb, 0x86b1, 0x86af, 0x86c9, 0x8853, 0x889e,
	0x8888, 0x88ab, 0x8892, 0x8896, 0x888d, 0x888b, 0x8993, 0x898f,
	0x8a2a, 0x8a1d, 0x8a23, 0x8a25, 0x8a31, 0x8a2d, 0x8a1f, 0x8a1b,
	0x8a22, 0x8c49, 0x8c5a, 0x8ca9, 0x8cac, 0x8cab, 0x8ca8, 0x8caa,
	0x8ca7, 0x8d67, 0x8d66, 0x8dbe, 0x8dba, 0x8edb, 0x8edf, 0x9019,
	0x900d, 0x901a, 0x9017, 0x9023, 0x901f, 0x901d, 0x9010, 0x9015,
	0x901e, 0x9020, 0x900f, 0x9022, 0x9016, 0x901b, 0x9014
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb3a1[ 94 ] = {
	        0x90e8, 0x90ed, 0x90fd, 0x9157, 0x91ce, 0x91f5, 0x91e6,
	0x91e3, 0x91e7, 0x91ed, 0x91e9, 0x9589, 0x966a, 0x9675, 0x9673,
	0x9678, 0x9670, 0x9674, 0x9676, 0x9677, 0x966c, 0x96c0, 0x96ea,
	0x96e9, 0x7ae0, 0x7adf, 0x9802, 0x9803, 0x9b5a, 0x9ce5, 0x9e75,
	0x9e7f, 0x9ea5, 0x9ebb, 0x50a2, 0x508d, 0x5085, 0x5099, 0x5091,
	0x5080, 0x5096, 0x5098, 0x509a, 0x6700, 0x51f1, 0x5272, 0x5274,
	0x5275, 0x5269, 0x52de, 0x52dd, 0x52db, 0x535a, 0x53a5, 0x557b,
	0x5580, 0x55a7, 0x557c, 0x558a, 0x559d, 0x5598, 0x5582, 0x559c,
	0x55aa, 0x5594, 0x5587, 0x558b, 0x5583, 0x55b3, 0x55ae, 0x559f,
	0x553e, 0x55b2, 0x559a, 0x55bb, 0x55ac, 0x55b1, 0x557e, 0x5589,
	0x55ab, 0x5599, 0x570d, 0x582f, 0x582a, 0x5834, 0x5824, 0x5830,
	0x5831, 0x5821, 0x581d, 0x5820, 0x58f9, 0x58fa, 0x5960
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb440[ 63 ] = {
	0x5a77, 0x5a9a, 0x5a7f, 0x5a92, 0x5a9b, 0x5aa7, 0x5b73, 0x5b71,
	0x5bd2, 0x5bcc, 0x5bd3, 0x5bd0, 0x5c0a, 0x5c0b, 0x5c31, 0x5d4c,
	0x5d50, 0x5d34, 0x5d47, 0x5dfd, 0x5e45, 0x5e3d, 0x5e40, 0x5e43,
	0x5e7e, 0x5eca, 0x5ec1, 0x5ec2, 0x5ec4, 0x5f3c, 0x5f6d, 0x5fa9,
	0x5faa, 0x5fa8, 0x60d1, 0x60e1, 0x60b2, 0x60b6, 0x60e0, 0x611c,
	0x6123, 0x60fa, 0x6115, 0x60f0, 0x60fb, 0x60f4, 0x6168, 0x60f1,
	0x610e, 0x60f6, 0x6109, 0x6100, 0x6112, 0x621f, 0x6249, 0x63a3,
	0x638c, 0x63cf, 0x63c0, 0x63e9, 0x63c9, 0x63c6, 0x63cd
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb4a1[ 94 ] = {
	        0x63d2, 0x63e3, 0x63d0, 0x63e1, 0x63d6, 0x63ed, 0x63ee,
	0x6376, 0x63f4, 0x63ea, 0x63db, 0x6452, 0x63da, 0x63f9, 0x655e,
	0x6566, 0x6562, 0x6563, 0x6591, 0x6590, 0x65af, 0x666e, 0x6670,
	0x6674, 0x6676, 0x666f, 0x6691, 0x667a, 0x667e, 0x6677, 0x66fe,
	0x66ff, 0x671f, 0x671d, 0x68fa, 0x68d5, 0x68e0, 0x68d8, 0x68d7,
	0x6905, 0x68df, 0x68f5, 0x68ee, 0x68e7, 0x68f9, 0x68d2, 0x68f2,
	0x68e3, 0x68cb, 0x68cd, 0x690d, 0x6912, 0x690e, 0x68c9, 0x68da,
	0x696e, 0x68fb, 0x6b3e, 0x6b3a, 0x6b3d, 0x6b98, 0x6b96, 0x6bbc,
	0x6bef, 0x6c2e, 0x6c2f, 0x6c2c, 0x6e2f, 0x6e38, 0x6e54, 0x6e21,
	0x6e32, 0x6e67, 0x6e4a, 0x6e20, 0x6e25, 0x6e23, 0x6e1b, 0x6e5b,
	0x6e58, 0x6e24, 0x6e56, 0x6e6e, 0x6e2d, 0x6e26, 0x6e6f, 0x6e34,
	0x6e4d, 0x6e3a, 0x6e2c, 0x6e43, 0x6e1d, 0x6e3e, 0x6ecb
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb540[ 63 ] = {
	0x6e89, 0x6e19, 0x6e4e, 0x6e63, 0x6e44, 0x6e72, 0x6e69, 0x6e5f,
	0x7119, 0x711a, 0x7126, 0x7130, 0x7121, 0x7136, 0x716e, 0x711c,
	0x724c, 0x7284, 0x7280, 0x7336, 0x7325, 0x7334, 0x7329, 0x743a,
	0x742a, 0x7433, 0x7422, 0x7425, 0x7435, 0x7436, 0x7434, 0x742f,
	0x741b, 0x7426, 0x7428, 0x7525, 0x7526, 0x756b, 0x756a, 0x75e2,
	0x75db, 0x75e3, 0x75d9, 0x75d8, 0x75de, 0x75e0, 0x767b, 0x767c,
	0x7696, 0x7693, 0x76b4, 0x76dc, 0x774f, 0x77ed, 0x785d, 0x786c,
	0x786f, 0x7a0d, 0x7a08, 0x7a0b, 0x7a05, 0x7a00, 0x7a98
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb5a1[ 94 ] = {
	        0x7a97, 0x7a96, 0x7ae5, 0x7ae3, 0x7b49, 0x7b56, 0x7b46,
	0x7b50, 0x7b52, 0x7b54, 0x7b4d, 0x7b4b, 0x7b4f, 0x7b51, 0x7c9f,
	0x7ca5, 0x7d5e, 0x7d50, 0x7d68, 0x7d55, 0x7d2b, 0x7d6e, 0x7d72,
	0x7d61, 0x7d66, 0x7d62, 0x7d70, 0x7d73, 0x5584, 0x7fd4, 0x7fd5,
	0x800b, 0x8052, 0x8085, 0x8155, 0x8154, 0x814b, 0x8151, 0x814e,
	0x8139, 0x8146, 0x813e, 0x814c, 0x8153, 0x8174, 0x8212, 0x821c,
	0x83e9, 0x8403, 0x83f8, 0x840d, 0x83e0, 0x83c5, 0x840b, 0x83c1,
	0x83ef, 0x83f1, 0x83f4, 0x8457, 0x840a, 0x83f0, 0x840c, 0x83cc,
	0x83fd, 0x83f2, 0x83ca, 0x8438, 0x840e, 0x8404, 0x83dc, 0x8407,
	0x83d4, 0x83df, 0x865b, 0x86df, 0x86d9, 0x86ed, 0x86d4, 0x86db,
	0x86e4, 0x86d0, 0x86de, 0x8857, 0x88c1, 0x88c2, 0x88b1, 0x8983,
	0x8996, 0x8a3b, 0x8a60, 0x8a55, 0x8a5e, 0x8a3c, 0x8a41
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb640[ 63 ] = {
	0x8a54, 0x8a5b, 0x8a50, 0x8a46, 0x8a34, 0x8a3a, 0x8a36, 0x8a56,
	0x8c61, 0x8c82, 0x8caf, 0x8cbc, 0x8cb3, 0x8cbd, 0x8cc1, 0x8cbb,
	0x8cc0, 0x8cb4, 0x8cb7, 0x8cb6, 0x8cbf, 0x8cb8, 0x8d8a, 0x8d85,
	0x8d81, 0x8dce, 0x8ddd, 0x8dcb, 0x8dda, 0x8dd1, 0x8dcc, 0x8ddb,
	0x8dc6, 0x8efb, 0x8ef8, 0x8efc, 0x8f9c, 0x902e, 0x9035, 0x9031,
	0x9038, 0x9032, 0x9036, 0x9102, 0x90f5, 0x9109, 0x90fe, 0x9163,
	0x9165, 0x91cf, 0x9214, 0x9215, 0x9223, 0x9209, 0x921e, 0x920d,
	0x9210, 0x9207, 0x9211, 0x9594, 0x958f, 0x958b, 0x9591
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb6a1[ 94 ] = {
	        0x9593, 0x9592, 0x958e, 0x968a, 0x968e, 0x968b, 0x967d,
	0x9685, 0x9686, 0x968d, 0x9672, 0x9684, 0x96c1, 0x96c5, 0x96c4,
	0x96c6, 0x96c7, 0x96ef, 0x96f2, 0x97cc, 0x9805, 0x9806, 0x9808,
	0x98e7, 0x98ea, 0x98ef, 0x98e9, 0x98f2, 0x98ed, 0x99ae, 0x99ad,
	0x9ec3, 0x9ecd, 0x9ed1, 0x4e82, 0x50ad, 0x50b5, 0x50b2, 0x50b3,
	0x50c5, 0x50be, 0x50ac, 0x50b7, 0x50bb, 0x50af, 0x50c7, 0x527f,
	0x5277, 0x527d, 0x52df, 0x52e6, 0x52e4, 0x52e2, 0x52e3, 0x532f,
	0x55df, 0x55e8, 0x55d3, 0x55e6, 0x55ce, 0x55dc, 0x55c7, 0x55d1,
	0x55e3, 0x55e4, 0x55ef, 0x55da, 0x55e1, 0x55c5, 0x55c6, 0x55e5,
	0x55c9, 0x5712, 0x5713, 0x585e, 0x5851, 0x5858, 0x5857, 0x585a,
	0x5854, 0x586b, 0x584c, 0x586d, 0x584a, 0x5862, 0x5852, 0x584b,
	0x5967, 0x5ac1, 0x5ac9, 0x5acc, 0x5abe, 0x5abd, 0x5abc
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb740[ 63 ] = {
	0x5ab3, 0x5ac2, 0x5ab2, 0x5d69, 0x5d6f, 0x5e4c, 0x5e79, 0x5ec9,
	0x5ec8, 0x5f12, 0x5f59, 0x5fac, 0x5fae, 0x611a, 0x610f, 0x6148,
	0x611f, 0x60f3, 0x611b, 0x60f9, 0x6101, 0x6108, 0x614e, 0x614c,
	0x6144, 0x614d, 0x613e, 0x6134, 0x6127, 0x610d, 0x6106, 0x6137,
	0x6221, 0x6222, 0x6413, 0x643e, 0x641e, 0x642a, 0x642d, 0x643d,
	0x642c, 0x640f, 0x641c, 0x6414, 0x640d, 0x6436, 0x6416, 0x6417,
	0x6406, 0x656c, 0x659f, 0x65b0, 0x6697, 0x6689, 0x6687, 0x6688,
	0x6696, 0x6684, 0x6698, 0x668d, 0x6703, 0x6994, 0x696d
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb7a1[ 94 ] = {
	        0x695a, 0x6977, 0x6960, 0x6954, 0x6975, 0x6930, 0x6982,
	0x694a, 0x6968, 0x696b, 0x695e, 0x6953, 0x6979, 0x6986, 0x695d,
	0x6963, 0x695b, 0x6b47, 0x6b72, 0x6bc0, 0x6bbf, 0x6bd3, 0x6bfd,
	0x6ea2, 0x6eaf, 0x6ed3, 0x6eb6, 0x6ec2, 0x6e90, 0x6e9d, 0x6ec7,
	0x6ec5, 0x6ea5, 0x6e98, 0x6ebc, 0x6eba, 0x6eab, 0x6ed1, 0x6e96,
	0x6e9c, 0x6ec4, 0x6ed4, 0x6eaa, 0x6ea7, 0x6eb4, 0x714e, 0x7159,
	0x7169, 0x7164, 0x7149, 0x7167, 0x715c, 0x716c, 0x7166, 0x714c,
	0x7165, 0x715e, 0x7146, 0x7168, 0x7156, 0x723a, 0x7252, 0x7337,
	0x7345, 0x733f, 0x733e, 0x746f, 0x745a, 0x7455, 0x745f, 0x745e,
	0x7441, 0x743f, 0x7459, 0x745b, 0x745c, 0x7576, 0x7578, 0x7600,
	0x75f0, 0x7601, 0x75f2, 0x75f1, 0x75fa, 0x75ff, 0x75f4, 0x75f3,
	0x76de, 0x76df, 0x775b, 0x776b, 0x7766, 0x775e, 0x7763
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb840[ 63 ] = {
	0x7779, 0x776a, 0x776c, 0x775c, 0x7765, 0x7768, 0x7762, 0x77ee,
	0x788e, 0x78b0, 0x7897, 0x7898, 0x788c, 0x7889, 0x787c, 0x7891,
	0x7893, 0x787f, 0x797a, 0x797f, 0x7981, 0x842c, 0x79bd, 0x7a1c,
	0x7a1a, 0x7a20, 0x7a14, 0x7a1f, 0x7a1e, 0x7a9f, 0x7aa0, 0x7b77,
	0x7bc0, 0x7b60, 0x7b6e, 0x7b67, 0x7cb1, 0x7cb3, 0x7cb5, 0x7d93,
	0x7d79, 0x7d91, 0x7d81, 0x7d8f, 0x7d5b, 0x7f6e, 0x7f69, 0x7f6a,
	0x7f72, 0x7fa9, 0x7fa8, 0x7fa4, 0x8056, 0x8058, 0x8086, 0x8084,
	0x8171, 0x8170, 0x8178, 0x8165, 0x816e, 0x8173, 0x816b
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb8a1[ 94 ] = {
	        0x8179, 0x817a, 0x8166, 0x8205, 0x8247, 0x8482, 0x8477,
	0x843d, 0x8431, 0x8475, 0x8466, 0x846b, 0x8449, 0x846c, 0x845b,
	0x843c, 0x8435, 0x8461, 0x8463, 0x8469, 0x846d, 0x8446, 0x865e,
	0x865c, 0x865f, 0x86f9, 0x8713, 0x8708, 0x8707, 0x8700, 0x86fe,
	0x86fb, 0x8702, 0x8703, 0x8706, 0x870a, 0x8859, 0x88df, 0x88d4,
	0x88d9, 0x88dc, 0x88d8, 0x88dd, 0x88e1, 0x88ca, 0x88d5, 0x88d2,
	0x899c, 0x89e3, 0x8a6b, 0x8a72, 0x8a73, 0x8a66, 0x8a69, 0x8a70,
	0x8a87, 0x8a7c, 0x8a63, 0x8aa0, 0x8a71, 0x8a85, 0x8a6d, 0x8a62,
	0x8a6e, 0x8a6c, 0x8a79, 0x8a7b, 0x8a3e, 0x8a68, 0x8c62, 0x8c8a,
	0x8c89, 0x8cca, 0x8cc7, 0x8cc8, 0x8cc4, 0x8cb2, 0x8cc3, 0x8cc2,
	0x8cc5, 0x8de1, 0x8ddf, 0x8de8, 0x8def, 0x8df3, 0x8dfa, 0x8dea,
	0x8de4, 0x8de6, 0x8eb2, 0x8f03, 0x8f09, 0x8efe, 0x8f0a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb940[ 63 ] = {
	0x8f9f, 0x8fb2, 0x904b, 0x904a, 0x9053, 0x9042, 0x9054, 0x903c,
	0x9055, 0x9050, 0x9047, 0x904f, 0x904e, 0x904d, 0x9051, 0x903e,
	0x9041, 0x9112, 0x9117, 0x916c, 0x916a, 0x9169, 0x91c9, 0x9237,
	0x9257, 0x9238, 0x923d, 0x9240, 0x923e, 0x925b, 0x924b, 0x9264,
	0x9251, 0x9234, 0x9249, 0x924d, 0x9245, 0x9239, 0x923f, 0x925a,
	0x9598, 0x9698, 0x9694, 0x9695, 0x96cd, 0x96cb, 0x96c9, 0x96ca,
	0x96f7, 0x96fb, 0x96f9, 0x96f6, 0x9756, 0x9774, 0x9776, 0x9810,
	0x9811, 0x9813, 0x980a, 0x9812, 0x980c, 0x98fc, 0x98f4
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb9a1[ 94 ] = {
	        0x98fd, 0x98fe, 0x99b3, 0x99b1, 0x99b4, 0x9ae1, 0x9ce9,
	0x9e82, 0x9f0e, 0x9f13, 0x9f20, 0x50e7, 0x50ee, 0x50e5, 0x50d6,
	0x50ed, 0x50da, 0x50d5, 0x50cf, 0x50d1, 0x50f1, 0x50ce, 0x50e9,
	0x5162, 0x51f3, 0x5283, 0x5282, 0x5331, 0x53ad, 0x55fe, 0x5600,
	0x561b, 0x5617, 0x55fd, 0x5614, 0x5606, 0x5609, 0x560d, 0x560e,
	0x55f7, 0x5616, 0x561f, 0x5608, 0x5610, 0x55f6, 0x5718, 0x5716,
	0x5875, 0x587e, 0x5883, 0x5893, 0x588a, 0x5879, 0x5885, 0x587d,
	0x58fd, 0x5925, 0x5922, 0x5924, 0x596a, 0x5969, 0x5ae1, 0x5ae6,
	0x5ae9, 0x5ad7, 0x5ad6, 0x5ad8, 0x5ae3, 0x5b75, 0x5bde, 0x5be7,
	0x5be1, 0x5be5, 0x5be6, 0x5be8, 0x5be2, 0x5be4, 0x5bdf, 0x5c0d,
	0x5c62, 0x5d84, 0x5d87, 0x5e5b, 0x5e63, 0x5e55, 0x5e57, 0x5e54,
	0x5ed3, 0x5ed6, 0x5f0a, 0x5f46, 0x5f70, 0x5fb9, 0x6147
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xba40[ 63 ] = {
	0x613f, 0x614b, 0x6177, 0x6162, 0x6163, 0x615f, 0x615a, 0x6158,
	0x6175, 0x622a, 0x6487, 0x6458, 0x6454, 0x64a4, 0x6478, 0x645f,
	0x647a, 0x6451, 0x6467, 0x6434, 0x646d, 0x647b, 0x6572, 0x65a1,
	0x65d7, 0x65d6, 0x66a2, 0x66a8, 0x669d, 0x699c, 0x69a8, 0x6995,
	0x69c1, 0x69ae, 0x69d3, 0x69cb, 0x699b, 0x69b7, 0x69bb, 0x69ab,
	0x69b4, 0x69d0, 0x69cd, 0x69ad, 0x69cc, 0x69a6, 0x69c3, 0x69a3,
	0x6b49, 0x6b4c, 0x6c33, 0x6f33, 0x6f14, 0x6efe, 0x6f13, 0x6ef4,
	0x6f29, 0x6f3e, 0x6f20, 0x6f2c, 0x6f0f, 0x6f02, 0x6f22
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbaa1[ 94 ] = {
	        0x6eff, 0x6eef, 0x6f06, 0x6f31, 0x6f38, 0x6f32, 0x6f23,
	0x6f15, 0x6f2b, 0x6f2f, 0x6f88, 0x6f2a, 0x6eec, 0x6f01, 0x6ef2,
	0x6ecc, 0x6ef7, 0x7194, 0x7199, 0x717d, 0x718a, 0x7184, 0x7192,
	0x723e, 0x7292, 0x7296, 0x7344, 0x7350, 0x7464, 0x7463, 0x746a,
	0x7470, 0x746d, 0x7504, 0x7591, 0x7627, 0x760d, 0x760b, 0x7609,
	0x7613, 0x76e1, 0x76e3, 0x7784, 0x777d, 0x777f, 0x7761, 0x78c1,
	0x789f, 0x78a7, 0x78b3, 0x78a9, 0x78a3, 0x798e, 0x798f, 0x798d,
	0x7a2e, 0x7a31, 0x7aaa, 0x7aa9, 0x7aed, 0x7aef, 0x7ba1, 0x7b95,
	0x7b8b, 0x7b75, 0x7b97, 0x7b9d, 0x7b94, 0x7b8f, 0x7bb8, 0x7b87,
	0x7b84, 0x7cb9, 0x7cbd, 0x7cbe, 0x7dbb, 0x7db0, 0x7d9c, 0x7dbd,
	0x7dbe, 0x7da0, 0x7dca, 0x7db4, 0x7db2, 0x7db1, 0x7dba, 0x7da2,
	0x7dbf, 0x7db5, 0x7db8, 0x7dad, 0x7dd2, 0x7dc7, 0x7dac
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbb40[ 63 ] = {
	0x7f70, 0x7fe0, 0x7fe1, 0x7fdf, 0x805e, 0x805a, 0x8087, 0x8150,
	0x8180, 0x818f, 0x8188, 0x818a, 0x817f, 0x8182, 0x81e7, 0x81fa,
	0x8207, 0x8214, 0x821e, 0x824b, 0x84c9, 0x84bf, 0x84c6, 0x84c4,
	0x8499, 0x849e, 0x84b2, 0x849c, 0x84cb, 0x84b8, 0x84c0, 0x84d3,
	0x8490, 0x84bc, 0x84d1, 0x84ca, 0x873f, 0x871c, 0x873b, 0x8722,
	0x8725, 0x8734, 0x8718, 0x8755, 0x8737, 0x8729, 0x88f3, 0x8902,
	0x88f4, 0x88f9, 0x88f8, 0x88fd, 0x88e8, 0x891a, 0x88ef, 0x8aa6,
	0x8a8c, 0x8a9e, 0x8aa3, 0x8a8d, 0x8aa1, 0x8a93, 0x8aa4
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbba1[ 94 ] = {
	        0x8aaa, 0x8aa5, 0x8aa8, 0x8a98, 0x8a91, 0x8a9a, 0x8aa7,
	0x8c6a, 0x8c8d, 0x8c8c, 0x8cd3, 0x8cd1, 0x8cd2, 0x8d6b, 0x8d99,
	0x8d95, 0x8dfc, 0x8f14, 0x8f12, 0x8f15, 0x8f13, 0x8fa3, 0x9060,
	0x9058, 0x905c, 0x9063, 0x9059, 0x905e, 0x9062, 0x905d, 0x905b,
	0x9119, 0x9118, 0x911e, 0x9175, 0x9178, 0x9177, 0x9174, 0x9278,
	0x9280, 0x9285, 0x9298, 0x9296, 0x927b, 0x9293, 0x929c, 0x92a8,
	0x927c, 0x9291, 0x95a1, 0x95a8, 0x95a9, 0x95a3, 0x95a5, 0x95a4,
	0x9699, 0x969c, 0x969b, 0x96cc, 0x96d2, 0x9700, 0x977c, 0x9785,
	0x97f6, 0x9817, 0x9818, 0x98af, 0x98b1, 0x9903, 0x9905, 0x990c,
	0x9909, 0x99c1, 0x9aaf, 0x9ab0, 0x9ae6, 0x9b41, 0x9b42, 0x9cf4,
	0x9cf6, 0x9cf3, 0x9ebc, 0x9f3b, 0x9f4a, 0x5104, 0x5100, 0x50fb,
	0x50f5, 0x50f9, 0x5102, 0x5108, 0x5109, 0x5105, 0x51dc
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbc40[ 63 ] = {
	0x5287, 0x5288, 0x5289, 0x528d, 0x528a, 0x52f0, 0x53b2, 0x562e,
	0x563b, 0x5639, 0x5632, 0x563f, 0x5634, 0x5629, 0x5653, 0x564e,
	0x5657, 0x5674, 0x5636, 0x562f, 0x5630, 0x5880, 0x589f, 0x589e,
	0x58b3, 0x589c, 0x58ae, 0x58a9, 0x58a6, 0x596d, 0x5b09, 0x5afb,
	0x5b0b, 0x5af5, 0x5b0c, 0x5b08, 0x5bee, 0x5bec, 0x5be9, 0x5beb,
	0x5c64, 0x5c65, 0x5d9d, 0x5d94, 0x5e62, 0x5e5f, 0x5e61, 0x5ee2,
	0x5eda, 0x5edf, 0x5edd, 0x5ee3, 0x5ee0, 0x5f48, 0x5f71, 0x5fb7,
	0x5fb5, 0x6176, 0x6167, 0x616e, 0x615d, 0x6155, 0x6182
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbca1[ 94 ] = {
	        0x617c, 0x6170, 0x616b, 0x617e, 0x61a7, 0x6190, 0x61ab,
	0x618e, 0x61ac, 0x619a, 0x61a4, 0x6194, 0x61ae, 0x622e, 0x6469,
	0x646f, 0x6479, 0x649e, 0x64b2, 0x6488, 0x6490, 0x64b0, 0x64a5,
	0x6493, 0x6495, 0x64a9, 0x6492, 0x64ae, 0x64ad, 0x64ab, 0x649a,
	0x64ac, 0x6499, 0x64a2, 0x64b3, 0x6575, 0x6577, 0x6578, 0x66ae,
	0x66ab, 0x66b4, 0x66b1, 0x6a23, 0x6a1f, 0x69e8, 0x6a01, 0x6a1e,
	0x6a19, 0x69fd, 0x6a21, 0x6a13, 0x6a0a, 0x69f3, 0x6a02, 0x6a05,
	0x69ed, 0x6a11, 0x6b50, 0x6b4e, 0x6ba4, 0x6bc5, 0x6bc6, 0x6f3f,
	0x6f7c, 0x6f84, 0x6f51, 0x6f66, 0x6f54, 0x6f86, 0x6f6d, 0x6f5b,
	0x6f78, 0x6f6e, 0x6f8e, 0x6f7a, 0x6f70, 0x6f64, 0x6f97, 0x6f58,
	0x6ed5, 0x6f6f, 0x6f60, 0x6f5f, 0x719f, 0x71ac, 0x71b1, 0x71a8,
	0x7256, 0x729b, 0x734e, 0x7357, 0x7469, 0x748b, 0x7483
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbd40[ 63 ] = {
	0x747e, 0x7480, 0x757f, 0x7620, 0x7629, 0x761f, 0x7624, 0x7626,
	0x7621, 0x7622, 0x769a, 0x76ba, 0x76e4, 0x778e, 0x7787, 0x778c,
	0x7791, 0x778b, 0x78cb, 0x78c5, 0x78ba, 0x78ca, 0x78be, 0x78d5,
	0x78bc, 0x78d0, 0x7a3f, 0x7a3c, 0x7a40, 0x7a3d, 0x7a37, 0x7a3b,
	0x7aaf, 0x7aae, 0x7bad, 0x7bb1, 0x7bc4, 0x7bb4, 0x7bc6, 0x7bc7,
	0x7bc1, 0x7ba0, 0x7bcc, 0x7cca, 0x7de0, 0x7df4, 0x7def, 0x7dfb,
	0x7dd8, 0x7dec, 0x7ddd, 0x7de8, 0x7de3, 0x7dda, 0x7dde, 0x7de9,
	0x7d9e, 0x7dd9, 0x7df2, 0x7df9, 0x7f75, 0x7f77, 0x7faf
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbda1[ 94 ] = {
	        0x7fe9, 0x8026, 0x819b, 0x819c, 0x819d, 0x81a0, 0x819a,
	0x8198, 0x8517, 0x853d, 0x851a, 0x84ee, 0x852c, 0x852d, 0x8513,
	0x8511, 0x8523, 0x8521, 0x8514, 0x84ec, 0x8525, 0x84ff, 0x8506,
	0x8782, 0x8774, 0x8776, 0x8760, 0x8766, 0x8778, 0x8768, 0x8759,
	0x8757, 0x874c, 0x8753, 0x885b, 0x885d, 0x8910, 0x8907, 0x8912,
	0x8913, 0x8915, 0x890a, 0x8abc, 0x8ad2, 0x8ac7, 0x8ac4, 0x8a95,
	0x8acb, 0x8af8, 0x8ab2, 0x8ac9, 0x8ac2, 0x8abf, 0x8ab0, 0x8ad6,
	0x8acd, 0x8ab6, 0x8ab9, 0x8adb, 0x8c4c, 0x8c4e, 0x8c6c, 0x8ce0,
	0x8cde, 0x8ce6, 0x8ce4, 0x8cec, 0x8ced, 0x8ce2, 0x8ce3, 0x8cdc,
	0x8cea, 0x8ce1, 0x8d6d, 0x8d9f, 0x8da3, 0x8e2b, 0x8e10, 0x8e1d,
	0x8e22, 0x8e0f, 0x8e29, 0x8e1f, 0x8e21, 0x8e1e, 0x8eba, 0x8f1d,
	0x8f1b, 0x8f1f, 0x8f29, 0x8f26, 0x8f2a, 0x8f1c, 0x8f1e
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbe40[ 63 ] = {
	0x8f25, 0x9069, 0x906e, 0x9068, 0x906d, 0x9077, 0x9130, 0x912d,
	0x9127, 0x9131, 0x9187, 0x9189, 0x918b, 0x9183, 0x92c5, 0x92bb,
	0x92b7, 0x92ea, 0x92ac, 0x92e4, 0x92c1, 0x92b3, 0x92bc, 0x92d2,
	0x92c7, 0x92f0, 0x92b2, 0x95ad, 0x95b1, 0x9704, 0x9706, 0x9707,
	0x9709, 0x9760, 0x978d, 0x978b, 0x978f, 0x9821, 0x982b, 0x981c,
	0x98b3, 0x990a, 0x9913, 0x9912, 0x9918, 0x99dd, 0x99d0, 0x99df,
	0x99db, 0x99d1, 0x99d5, 0x99d2, 0x99d9, 0x9ab7, 0x9aee, 0x9aef,
	0x9b27, 0x9b45, 0x9b44, 0x9b77, 0x9b6f, 0x9d06, 0x9d09
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbea1[ 94 ] = {
	        0x9d03, 0x9ea9, 0x9ebe, 0x9ece, 0x58a8, 0x9f52, 0x5112,
	0x5118, 0x5114, 0x5110, 0x5115, 0x5180, 0x51aa, 0x51dd, 0x5291,
	0x5293, 0x52f3, 0x5659, 0x566b, 0x5679, 0x5669, 0x5664, 0x5678,
	0x566a, 0x5668, 0x5665, 0x5671, 0x566f, 0x566c, 0x5662, 0x5676,
	0x58c1, 0x58be, 0x58c7, 0x58c5, 0x596e, 0x5b1d, 0x5b34, 0x5b78,
	0x5bf0, 0x5c0e, 0x5f4a, 0x61b2, 0x6191, 0x61a9, 0x618a, 0x61cd,
	0x61b6, 0x61be, 0x61ca, 0x61c8, 0x6230, 0x64c5, 0x64c1, 0x64cb,
	0x64bb, 0x64bc, 0x64da, 0x64c4, 0x64c7, 0x64c2, 0x64cd, 0x64bf,
	0x64d2, 0x64d4, 0x64be, 0x6574, 0x66c6, 0x66c9, 0x66b9, 0x66c4,
	0x66c7, 0x66b8, 0x6a3d, 0x6a38, 0x6a3a, 0x6a59, 0x6a6b, 0x6a58,
	0x6a39, 0x6a44, 0x6a62, 0x6a61, 0x6a4b, 0x6a47, 0x6a35, 0x6a5f,
	0x6a48, 0x6b59, 0x6b77, 0x6c05, 0x6fc2, 0x6fb1, 0x6fa1
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbf40[ 63 ] = {
	0x6fc3, 0x6fa4, 0x6fc1, 0x6fa7, 0x6fb3, 0x6fc0, 0x6fb9, 0x6fb6,
	0x6fa6, 0x6fa0, 0x6fb4, 0x71be, 0x71c9, 0x71d0, 0x71d2, 0x71c8,
	0x71d5, 0x71b9, 0x71ce, 0x71d9, 0x71dc, 0x71c3, 0x71c4, 0x7368,
	0x749c, 0x74a3, 0x7498, 0x749f, 0x749e, 0x74e2, 0x750c, 0x750d,
	0x7634, 0x7638, 0x763a, 0x76e7, 0x76e5, 0x77a0, 0x779e, 0x779f,
	0x77a5, 0x78e8, 0x78da, 0x78ec, 0x78e7, 0x79a6, 0x7a4d, 0x7a4e,
	0x7a46, 0x7a4c, 0x7a4b, 0x7aba, 0x7bd9, 0x7c11, 0x7bc9, 0x7be4,
	0x7bdb, 0x7be1, 0x7be9, 0x7be6, 0x7cd5, 0x7cd6, 0x7e0a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbfa1[ 94 ] = {
	        0x7e11, 0x7e08, 0x7e1b, 0x7e23, 0x7e1e, 0x7e1d, 0x7e09,
	0x7e10, 0x7f79, 0x7fb2, 0x7ff0, 0x7ff1, 0x7fee, 0x8028, 0x81b3,
	0x81a9, 0x81a8, 0x81fb, 0x8208, 0x8258, 0x8259, 0x854a, 0x8559,
	0x8548, 0x8568, 0x8569, 0x8543, 0x8549, 0x856d, 0x856a, 0x855e,
	0x8783, 0x879f, 0x879e, 0x87a2, 0x878d, 0x8861, 0x892a, 0x8932,
	0x8925, 0x892b, 0x8921, 0x89aa, 0x89a6, 0x8ae6, 0x8afa, 0x8aeb,
	0x8af1, 0x8b00, 0x8adc, 0x8ae7, 0x8aee, 0x8afe, 0x8b01, 0x8b02,
	0x8af7, 0x8aed, 0x8af3, 0x8af6, 0x8afc, 0x8c6b, 0x8c6d, 0x8c93,
	0x8cf4, 0x8e44, 0x8e31, 0x8e34, 0x8e42, 0x8e39, 0x8e35, 0x8f3b,
	0x8f2f, 0x8f38, 0x8f33, 0x8fa8, 0x8fa6, 0x9075, 0x9074, 0x9078,
	0x9072, 0x907c, 0x907a, 0x9134, 0x9192, 0x9320, 0x9336, 0x92f8,
	0x9333, 0x932f, 0x9322, 0x92fc, 0x932b, 0x9304, 0x931a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc040[ 63 ] = {
	0x9310, 0x9326, 0x9321, 0x9315, 0x932e, 0x9319, 0x95bb, 0x96a7,
	0x96a8, 0x96aa, 0x96d5, 0x970e, 0x9711, 0x9716, 0x970d, 0x9713,
	0x970f, 0x975b, 0x975c, 0x9766, 0x9798, 0x9830, 0x9838, 0x983b,
	0x9837, 0x982d, 0x9839, 0x9824, 0x9910, 0x9928, 0x991e, 0x991b,
	0x9921, 0x991a, 0x99ed, 0x99e2, 0x99f1, 0x9ab8, 0x9abc, 0x9afb,
	0x9aed, 0x9b28, 0x9b91, 0x9d15, 0x9d23, 0x9d26, 0x9d28, 0x9d12,
	0x9d1b, 0x9ed8, 0x9ed4, 0x9f8d, 0x9f9c, 0x512a, 0x511f, 0x5121,
	0x5132, 0x52f5, 0x568e, 0x5680, 0x5690, 0x5685, 0x5687
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc0a1[ 94 ] = {
	        0x568f, 0x58d5, 0x58d3, 0x58d1, 0x58ce, 0x5b30, 0x5b2a,
	0x5b24, 0x5b7a, 0x5c37, 0x5c68, 0x5dbc, 0x5dba, 0x5dbd, 0x5db8,
	0x5e6b, 0x5f4c, 0x5fbd, 0x61c9, 0x61c2, 0x61c7, 0x61e6, 0x61cb,
	0x6232, 0x6234, 0x64ce, 0x64ca, 0x64d8, 0x64e0, 0x64f0, 0x64e6,
	0x64ec, 0x64f1, 0x64e2, 0x64ed, 0x6582, 0x6583, 0x66d9, 0x66d6,
	0x6a80, 0x6a94, 0x6a84, 0x6aa2, 0x6a9c, 0x6adb, 0x6aa3, 0x6a7e,
	0x6a97, 0x6a90, 0x6aa0, 0x6b5c, 0x6bae, 0x6bda, 0x6c08, 0x6fd8,
	0x6ff1, 0x6fdf, 0x6fe0, 0x6fdb, 0x6fe4, 0x6feb, 0x6fef, 0x6f80,
	0x6fec, 0x6fe1, 0x6fe9, 0x6fd5, 0x6fee, 0x6ff0, 0x71e7, 0x71df,
	0x71ee, 0x71e6, 0x71e5, 0x71ed, 0x71ec, 0x71f4, 0x71e0, 0x7235,
	0x7246, 0x7370, 0x7372, 0x74a9, 0x74b0, 0x74a6, 0x74a8, 0x7646,
	0x7642, 0x764c, 0x76ea, 0x77b3, 0x77aa, 0x77b0, 0x77ac
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc140[ 63 ] = {
	0x77a7, 0x77ad, 0x77ef, 0x78f7, 0x78fa, 0x78f4, 0x78ef, 0x7901,
	0x79a7, 0x79aa, 0x7a57, 0x7abf, 0x7c07, 0x7c0d, 0x7bfe, 0x7bf7,
	0x7c0c, 0x7be0, 0x7ce0, 0x7cdc, 0x7cde, 0x7ce2, 0x7cdf, 0x7cd9,
	0x7cdd, 0x7e2e, 0x7e3e, 0x7e46, 0x7e37, 0x7e32, 0x7e43, 0x7e2b,
	0x7e3d, 0x7e31, 0x7e45, 0x7e41, 0x7e34, 0x7e39, 0x7e48, 0x7e35,
	0x7e3f, 0x7e2f, 0x7f44, 0x7ff3, 0x7ffc, 0x8071, 0x8072, 0x8070,
	0x806f, 0x8073, 0x81c6, 0x81c3, 0x81ba, 0x81c2, 0x81c0, 0x81bf,
	0x81bd, 0x81c9, 0x81be, 0x81e8, 0x8209, 0x8271, 0x85aa
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc1a1[ 94 ] = {
	        0x8584, 0x857e, 0x859c, 0x8591, 0x8594, 0x85af, 0x859b,
	0x8587, 0x85a8, 0x858a, 0x8667, 0x87c0, 0x87d1, 0x87b3, 0x87d2,
	0x87c6, 0x87ab, 0x87bb, 0x87ba, 0x87c8, 0x87cb, 0x893b, 0x8936,
	0x8944, 0x8938, 0x893d, 0x89ac, 0x8b0e, 0x8b17, 0x8b19, 0x8b1b,
	0x8b0a, 0x8b20, 0x8b1d, 0x8b04, 0x8b10, 0x8c41, 0x8c3f, 0x8c73,
	0x8cfa, 0x8cfd, 0x8cfc, 0x8cf8, 0x8cfb, 0x8da8, 0x8e49, 0x8e4b,
	0x8e48, 0x8e4a, 0x8f44, 0x8f3e, 0x8f42, 0x8f45, 0x8f3f, 0x907f,
	0x907d, 0x9084, 0x9081, 0x9082, 0x9080, 0x9139, 0x91a3, 0x919e,
	0x919c, 0x934d, 0x9382, 0x9328, 0x9375, 0x934a, 0x9365, 0x934b,
	0x9318, 0x937e, 0x936c, 0x935b, 0x9370, 0x935a, 0x9354, 0x95ca,
	0x95cb, 0x95cc, 0x95c8, 0x95c6, 0x96b1, 0x96b8, 0x96d6, 0x971c,
	0x971e, 0x97a0, 0x97d3, 0x9846, 0x98b6, 0x9935, 0x9a01
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc240[ 63 ] = {
	0x99ff, 0x9bae, 0x9bab, 0x9baa, 0x9bad, 0x9d3b, 0x9d3f, 0x9e8b,
	0x9ecf, 0x9ede, 0x9edc, 0x9edd, 0x9edb, 0x9f3e, 0x9f4b, 0x53e2,
	0x5695, 0x56ae, 0x58d9, 0x58d8, 0x5b38, 0x5f5d, 0x61e3, 0x6233,
	0x64f4, 0x64f2, 0x64fe, 0x6506, 0x64fa, 0x64fb, 0x64f7, 0x65b7,
	0x66dc, 0x6726, 0x6ab3, 0x6aac, 0x6ac3, 0x6abb, 0x6ab8, 0x6ac2,
	0x6aae, 0x6aaf, 0x6b5f, 0x6b78, 0x6baf, 0x7009, 0x700b, 0x6ffe,
	0x7006, 0x6ffa, 0x7011, 0x700f, 0x71fb, 0x71fc, 0x71fe, 0x71f8,
	0x7377, 0x7375, 0x74a7, 0x74bf, 0x7515, 0x7656, 0x7658
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc2a1[ 94 ] = {
	        0x7652, 0x77bd, 0x77bf, 0x77bb, 0x77bc, 0x790e, 0x79ae,
	0x7a61, 0x7a62, 0x7a60, 0x7ac4, 0x7ac5, 0x7c2b, 0x7c27, 0x7c2a,
	0x7c1e, 0x7c23, 0x7c21, 0x7ce7, 0x7e54, 0x7e55, 0x7e5e, 0x7e5a,
	0x7e61, 0x7e52, 0x7e59, 0x7f48, 0x7ff9, 0x7ffb, 0x8077, 0x8076,
	0x81cd, 0x81cf, 0x820a, 0x85cf, 0x85a9, 0x85cd, 0x85d0, 0x85c9,
	0x85b0, 0x85ba, 0x85b9, 0x85a6, 0x87ef, 0x87ec, 0x87f2, 0x87e0,
	0x8986, 0x89b2, 0x89f4, 0x8b28, 0x8b39, 0x8b2c, 0x8b2b, 0x8c50,
	0x8d05, 0x8e59, 0x8e63, 0x8e66, 0x8e64, 0x8e5f, 0x8e55, 0x8ec0,
	0x8f49, 0x8f4d, 0x9087, 0x9083, 0x9088, 0x91ab, 0x91ac, 0x91d0,
	0x9394, 0x938a, 0x9396, 0x93a2, 0x93b3, 0x93ae, 0x93ac, 0x93b0,
	0x9398, 0x939a, 0x9397, 0x95d4, 0x95d6, 0x95d0, 0x95d5, 0x96e2,
	0x96dc, 0x96d9, 0x96db, 0x96de, 0x9724, 0x97a3, 0x97a6
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc340[ 63 ] = {
	0x97ad, 0x97f9, 0x984d, 0x984f, 0x984c, 0x984e, 0x9853, 0x98ba,
	0x993e, 0x993f, 0x993d, 0x992e, 0x99a5, 0x9a0e, 0x9ac1, 0x9b03,
	0x9b06, 0x9b4f, 0x9b4e, 0x9b4d, 0x9bca, 0x9bc9, 0x9bfd, 0x9bc8,
	0x9bc0, 0x9d51, 0x9d5d, 0x9d60, 0x9ee0, 0x9f15, 0x9f2c, 0x5133,
	0x56a5, 0x58de, 0x58df, 0x58e2, 0x5bf5, 0x9f90, 0x5eec, 0x61f2,
	0x61f7, 0x61f6, 0x61f5, 0x6500, 0x650f, 0x66e0, 0x66dd, 0x6ae5,
	0x6add, 0x6ada, 0x6ad3, 0x701b, 0x701f, 0x7028, 0x701a, 0x701d,
	0x7015, 0x7018, 0x7206, 0x720d, 0x7258, 0x72a2, 0x7378
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc3a1[ 94 ] = {
	        0x737a, 0x74bd, 0x74ca, 0x74e3, 0x7587, 0x7586, 0x765f,
	0x7661, 0x77c7, 0x7919, 0x79b1, 0x7a6b, 0x7a69, 0x7c3e, 0x7c3f,
	0x7c38, 0x7c3d, 0x7c37, 0x7c40, 0x7e6b, 0x7e6d, 0x7e79, 0x7e69,
	0x7e6a, 0x7f85, 0x7e73, 0x7fb6, 0x7fb9, 0x7fb8, 0x81d8, 0x85e9,
	0x85dd, 0x85ea, 0x85d5, 0x85e4, 0x85e5, 0x85f7, 0x87fb, 0x8805,
	0x880d, 0x87f9, 0x87fe, 0x8960, 0x895f, 0x8956, 0x895e, 0x8b41,
	0x8b5c, 0x8b58, 0x8b49, 0x8b5a, 0x8b4e, 0x8b4f, 0x8b46, 0x8b59,
	0x8d08, 0x8d0a, 0x8e7c, 0x8e72, 0x8e87, 0x8e76, 0x8e6c, 0x8e7a,
	0x8e74, 0x8f54, 0x8f4e, 0x8fad, 0x908a, 0x908b, 0x91b1, 0x91ae,
	0x93e1, 0x93d1, 0x93df, 0x93c3, 0x93c8, 0x93dc, 0x93dd, 0x93d6,
	0x93e2, 0x93cd, 0x93d8, 0x93e4, 0x93d7, 0x93e8, 0x95dc, 0x96b4,
	0x96e3, 0x972a, 0x9727, 0x9761, 0x97dc, 0x97fb, 0x985e
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc440[ 63 ] = {
	0x9858, 0x985b, 0x98bc, 0x9945, 0x9949, 0x9a16, 0x9a19, 0x9b0d,
	0x9be8, 0x9be7, 0x9bd6, 0x9bdb, 0x9d89, 0x9d61, 0x9d72, 0x9d6a,
	0x9d6c, 0x9e92, 0x9e97, 0x9e93, 0x9eb4, 0x52f8, 0x56a8, 0x56b7,
	0x56b6, 0x56b4, 0x56bc, 0x58e4, 0x5b40, 0x5b43, 0x5b7d, 0x5bf6,
	0x5dc9, 0x61f8, 0x61fa, 0x6518, 0x6514, 0x6519, 0x66e6, 0x6727,
	0x6aec, 0x703e, 0x7030, 0x7032, 0x7210, 0x737b, 0x74cf, 0x7662,
	0x7665, 0x7926, 0x792a, 0x792c, 0x792b, 0x7ac7, 0x7af6, 0x7c4c,
	0x7c43, 0x7c4d, 0x7cef, 0x7cf0, 0x8fae, 0x7e7d, 0x7e7c
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc4a1[ 94 ] = {
	        0x7e82, 0x7f4c, 0x8000, 0x81da, 0x8266, 0x85fb, 0x85f9,
	0x8611, 0x85fa, 0x8606, 0x860b, 0x8607, 0x860a, 0x8814, 0x8815,
	0x8964, 0x89ba, 0x89f8, 0x8b70, 0x8b6c, 0x8b66, 0x8b6f, 0x8b5f,
	0x8b6b, 0x8d0f, 0x8d0d, 0x8e89, 0x8e81, 0x8e85, 0x8e82, 0x91b4,
	0x91cb, 0x9418, 0x9403, 0x93fd, 0x95e1, 0x9730, 0x98c4, 0x9952,
	0x9951, 0x99a8, 0x9a2b, 0x9a30, 0x9a37, 0x9a35, 0x9c13, 0x9c0d,
	0x9e79, 0x9eb5, 0x9ee8, 0x9f2f, 0x9f5f, 0x9f63, 0x9f61, 0x5137,
	0x5138, 0x56c1, 0x56c0, 0x56c2, 0x5914, 0x5c6c, 0x5dcd, 0x61fc,
	0x61fe, 0x651d, 0x651c, 0x6595, 0x66e9, 0x6afb, 0x6b04, 0x6afa,
	0x6bb2, 0x704c, 0x721b, 0x72a7, 0x74d6, 0x74d4, 0x7669, 0x77d3,
	0x7c50, 0x7e8f, 0x7e8c, 0x7fbc, 0x8617, 0x862d, 0x861a, 0x8823,
	0x8822, 0x8821, 0x881f, 0x896a, 0x896c, 0x89bd, 0x8b74
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc540[ 63 ] = {
	0x8b77, 0x8b7d, 0x8d13, 0x8e8a, 0x8e8d, 0x8e8b, 0x8f5f, 0x8faf,
	0x91ba, 0x942e, 0x9433, 0x9435, 0x943a, 0x9438, 0x9432, 0x942b,
	0x95e2, 0x9738, 0x9739, 0x9732, 0x97ff, 0x9867, 0x9865, 0x9957,
	0x9a45, 0x9a43, 0x9a40, 0x9a3e, 0x9acf, 0x9b54, 0x9b51, 0x9c2d,
	0x9c25, 0x9daf, 0x9db4, 0x9dc2, 0x9db8, 0x9e9d, 0x9eef, 0x9f19,
	0x9f5c, 0x9f66, 0x9f67, 0x513c, 0x513b, 0x56c8, 0x56ca, 0x56c9,
	0x5b7f, 0x5dd4, 0x5dd2, 0x5f4e, 0x61ff, 0x6524, 0x6b0a, 0x6b61,
	0x7051, 0x7058, 0x7380, 0x74e4, 0x758a, 0x766e, 0x766c
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc5a1[ 94 ] = {
	        0x79b3, 0x7c60, 0x7c5f, 0x807e, 0x807d, 0x81df, 0x8972,
	0x896f, 0x89fc, 0x8b80, 0x8d16, 0x8d17, 0x8e91, 0x8e93, 0x8f61,
	0x9148, 0x9444, 0x9451, 0x9452, 0x973d, 0x973e, 0x97c3, 0x97c1,
	0x986b, 0x9955, 0x9a55, 0x9a4d, 0x9ad2, 0x9b1a, 0x9c49, 0x9c31,
	0x9c3e, 0x9c3b, 0x9dd3, 0x9dd7, 0x9f34, 0x9f6c, 0x9f6a, 0x9f94,
	0x56cc, 0x5dd6, 0x6200, 0x6523, 0x652b, 0x652a, 0x66ec, 0x6b10,
	0x74da, 0x7aca, 0x7c64, 0x7c63, 0x7c65, 0x7e93, 0x7e96, 0x7e94,
	0x81e2, 0x8638, 0x863f, 0x8831, 0x8b8a, 0x9090, 0x908f, 0x9463,
	0x9460, 0x9464, 0x9768, 0x986f, 0x995c, 0x9a5a, 0x9a5b, 0x9a57,
	0x9ad3, 0x9ad4, 0x9ad1, 0x9c54, 0x9c57, 0x9c56, 0x9de5, 0x9e9f,
	0x9ef4, 0x56d1, 0x58e9, 0x652c, 0x705e, 0x7671, 0x7672, 0x77d7,
	0x7f50, 0x7f88, 0x8836, 0x8839, 0x8862, 0x8b93, 0x8b92
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc640[ 63 ] = {
	0x8b96, 0x8277, 0x8d1b, 0x91c0, 0x946a, 0x9742, 0x9748, 0x9744,
	0x97c6, 0x9870, 0x9a5f, 0x9b22, 0x9b58, 0x9c5f, 0x9df9, 0x9dfa,
	0x9e7c, 0x9e7d, 0x9f07, 0x9f77, 0x9f72, 0x5ef3, 0x6b16, 0x7063,
	0x7c6c, 0x7c6e, 0x883b, 0x89c0, 0x8ea1, 0x91c1, 0x9472, 0x9470,
	0x9871, 0x995e, 0x9ad6, 0x9b23, 0x9ecc, 0x7064, 0x77da, 0x8b9a,
	0x9477, 0x97c9, 0x9a62, 0x9a65, 0x7e9c, 0x8b9c, 0x8eaa, 0x91c5,
	0x947d, 0x947e, 0x947c, 0x9c77, 0x9c78, 0x9ef7, 0x8c54, 0x947f,
	0x9e1a, 0x7228, 0x9a6a, 0x9b31, 0x9e1b, 0x9e1e, 0x7c72
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc940[ 63 ] = {
	0x4e42, 0x4e5c, 0x51f5, 0x531a, 0x5382, 0x4e07, 0x4e0c, 0x4e47,
	0x4e8d, 0x56d7, 0xfa0c, 0x5c6e, 0x5f73, 0x4e0f, 0x5187, 0x4e0e,
	0x4e2e, 0x4e93, 0x4ec2, 0x4ec9, 0x4ec8, 0x5198, 0x52fc, 0x536c,
	0x53b9, 0x5720, 0x5903, 0x592c, 0x5c10, 0x5dff, 0x65e1, 0x6bb3,
	0x6bcc, 0x6c14, 0x723f, 0x4e31, 0x4e3c, 0x4ee8, 0x4edc, 0x4ee9,
	0x4ee1, 0x4edd, 0x4eda, 0x520c, 0x531c, 0x534c, 0x5722, 0x5723,
	0x5917, 0x592f, 0x5b81, 0x5b84, 0x5c12, 0x5c3b, 0x5c74, 0x5c73,
	0x5e04, 0x5e80, 0x5e82, 0x5fc9, 0x6209, 0x6250, 0x6c15
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc9a1[ 94 ] = {
	        0x6c36, 0x6c43, 0x6c3f, 0x6c3b, 0x72ae, 0x72b0, 0x738a,
	0x79b8, 0x808a, 0x961e, 0x4f0e, 0x4f18, 0x4f2c, 0x4ef5, 0x4f14,
	0x4ef1, 0x4f00, 0x4ef7, 0x4f08, 0x4f1d, 0x4f02, 0x4f05, 0x4f22,
	0x4f13, 0x4f04, 0x4ef4, 0x4f12, 0x51b1, 0x5213, 0x5209, 0x5210,
	0x52a6, 0x5322, 0x531f, 0x534d, 0x538a, 0x5407, 0x56e1, 0x56df,
	0x572e, 0x572a, 0x5734, 0x593c, 0x5980, 0x597c, 0x5985, 0x597b,
	0x597e, 0x5977, 0x597f, 0x5b56, 0x5c15, 0x5c25, 0x5c7c, 0x5c7a,
	0x5c7b, 0x5c7e, 0x5ddf, 0x5e75, 0x5e84, 0x5f02, 0x5f1a, 0x5f74,
	0x5fd5, 0x5fd4, 0x5fcf, 0x625c, 0x625e, 0x6264, 0x6261, 0x6266,
	0x6262, 0x6259, 0x6260, 0x625a, 0x6265, 0x65ef, 0x65ee, 0x673e,
	0x6739, 0x6738, 0x673b, 0x673a, 0x673f, 0x673c, 0x6733, 0x6c18,
	0x6c46, 0x6c52, 0x6c5c, 0x6c4f, 0x6c4a, 0x6c54, 0x6c4b
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xca40[ 63 ] = {
	0x6c4c, 0x7071, 0x725e, 0x72b4, 0x72b5, 0x738e, 0x752a, 0x767f,
	0x7a75, 0x7f51, 0x8278, 0x827c, 0x8280, 0x827d, 0x827f, 0x864d,
	0x897e, 0x9099, 0x9097, 0x9098, 0x909b, 0x9094, 0x9622, 0x9624,
	0x9620, 0x9623, 0x4f56, 0x4f3b, 0x4f62, 0x4f49, 0x4f53, 0x4f64,
	0x4f3e, 0x4f67, 0x4f52, 0x4f5f, 0x4f41, 0x4f58, 0x4f2d, 0x4f33,
	0x4f3f, 0x4f61, 0x518f, 0x51b9, 0x521c, 0x521e, 0x5221, 0x52ad,
	0x52ae, 0x5309, 0x5363, 0x5372, 0x538e, 0x538f, 0x5430, 0x5437,
	0x542a, 0x5454, 0x5445, 0x5419, 0x541c, 0x5425, 0x5418
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcaa1[ 94 ] = {
	        0x543d, 0x544f, 0x5441, 0x5428, 0x5424, 0x5447, 0x56ee,
	0x56e7, 0x56e5, 0x5741, 0x5745, 0x574c, 0x5749, 0x574b, 0x5752,
	0x5906, 0x5940, 0x59a6, 0x5998, 0x59a0, 0x5997, 0x598e, 0x59a2,
	0x5990, 0x598f, 0x59a7, 0x59a1, 0x5b8e, 0x5b92, 0x5c28, 0x5c2a,
	0x5c8d, 0x5c8f, 0x5c88, 0x5c8b, 0x5c89, 0x5c92, 0x5c8a, 0x5c86,
	0x5c93, 0x5c95, 0x5de0, 0x5e0a, 0x5e0e, 0x5e8b, 0x5e89, 0x5e8c,
	0x5e88, 0x5e8d, 0x5f05, 0x5f1d, 0x5f78, 0x5f76, 0x5fd2, 0x5fd1,
	0x5fd0, 0x5fed, 0x5fe8, 0x5fee, 0x5ff3, 0x5fe1, 0x5fe4, 0x5fe3,
	0x5ffa, 0x5fef, 0x5ff7, 0x5ffb, 0x6000, 0x5ff4, 0x623a, 0x6283,
	0x628c, 0x628e, 0x628f, 0x6294, 0x6287, 0x6271, 0x627b, 0x627a,
	0x6270, 0x6281, 0x6288, 0x6277, 0x627d, 0x6272, 0x6274, 0x6537,
	0x65f0, 0x65f4, 0x65f3, 0x65f2, 0x65f5, 0x6745, 0x6747
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcb40[ 63 ] = {
	0x6759, 0x6755, 0x674c, 0x6748, 0x675d, 0x674d, 0x675a, 0x674b,
	0x6bd0, 0x6c19, 0x6c1a, 0x6c78, 0x6c67, 0x6c6b, 0x6c84, 0x6c8b,
	0x6c8f, 0x6c71, 0x6c6f, 0x6c69, 0x6c9a, 0x6c6d, 0x6c87, 0x6c95,
	0x6c9c, 0x6c66, 0x6c73, 0x6c65, 0x6c7b, 0x6c8e, 0x7074, 0x707a,
	0x7263, 0x72bf, 0x72bd, 0x72c3, 0x72c6, 0x72c1, 0x72ba, 0x72c5,
	0x7395, 0x7397, 0x7393, 0x7394, 0x7392, 0x753a, 0x7539, 0x7594,
	0x7595, 0x7681, 0x793d, 0x8034, 0x8095, 0x8099, 0x8090, 0x8092,
	0x809c, 0x8290, 0x828f, 0x8285, 0x828e, 0x8291, 0x8293
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcba1[ 94 ] = {
	        0x828a, 0x8283, 0x8284, 0x8c78, 0x8fc9, 0x8fbf, 0x909f,
	0x90a1, 0x90a5, 0x909e, 0x90a7, 0x90a0, 0x9630, 0x9628, 0x962f,
	0x962d, 0x4e33, 0x4f98, 0x4f7c, 0x4f85, 0x4f7d, 0x4f80, 0x4f87,
	0x4f76, 0x4f74, 0x4f89, 0x4f84, 0x4f77, 0x4f4c, 0x4f97, 0x4f6a,
	0x4f9a, 0x4f79, 0x4f81, 0x4f78, 0x4f90, 0x4f9c, 0x4f94, 0x4f9e,
	0x4f92, 0x4f82, 0x4f95, 0x4f6b, 0x4f6e, 0x519e, 0x51bc, 0x51be,
	0x5235, 0x5232, 0x5233, 0x5246, 0x5231, 0x52bc, 0x530a, 0x530b,
	0x533c, 0x5392, 0x5394, 0x5487, 0x547f, 0x5481, 0x5491, 0x5482,
	0x5488, 0x546b, 0x547a, 0x547e, 0x5465, 0x546c, 0x5474, 0x5466,
	0x548d, 0x546f, 0x5461, 0x5460, 0x5498, 0x5463, 0x5467, 0x5464,
	0x56f7, 0x56f9, 0x576f, 0x5772, 0x576d, 0x576b, 0x5771, 0x5770,
	0x5776, 0x5780, 0x5775, 0x577b, 0x5773, 0x5774, 0x5762
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcc40[ 63 ] = {
	0x5768, 0x577d, 0x590c, 0x5945, 0x59b5, 0x59ba, 0x59cf, 0x59ce,
	0x59b2, 0x59cc, 0x59c1, 0x59b6, 0x59bc, 0x59c3, 0x59d6, 0x59b1,
	0x59bd, 0x59c0, 0x59c8, 0x59b4, 0x59c7, 0x5b62, 0x5b65, 0x5b93,
	0x5b95, 0x5c44, 0x5c47, 0x5cae, 0x5ca4, 0x5ca0, 0x5cb5, 0x5caf,
	0x5ca8, 0x5cac, 0x5c9f, 0x5ca3, 0x5cad, 0x5ca2, 0x5caa, 0x5ca7,
	0x5c9d, 0x5ca5, 0x5cb6, 0x5cb0, 0x5ca6, 0x5e17, 0x5e14, 0x5e19,
	0x5f28, 0x5f22, 0x5f23, 0x5f24, 0x5f54, 0x5f82, 0x5f7e, 0x5f7d,
	0x5fde, 0x5fe5, 0x602d, 0x6026, 0x6019, 0x6032, 0x600b
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcca1[ 94 ] = {
	        0x6034, 0x600a, 0x6017, 0x6033, 0x601a, 0x601e, 0x602c,
	0x6022, 0x600d, 0x6010, 0x602e, 0x6013, 0x6011, 0x600c, 0x6009,
	0x601c, 0x6214, 0x623d, 0x62ad, 0x62b4, 0x62d1, 0x62be, 0x62aa,
	0x62b6, 0x62ca, 0x62ae, 0x62b3, 0x62af, 0x62bb, 0x62a9, 0x62b0,
	0x62b8, 0x653d, 0x65a8, 0x65bb, 0x6609, 0x65fc, 0x6604, 0x6612,
	0x6608, 0x65fb, 0x6603, 0x660b, 0x660d, 0x6605, 0x65fd, 0x6611,
	0x6610, 0x66f6, 0x670a, 0x6785, 0x676c, 0x678e, 0x6792, 0x6776,
	0x677b, 0x6798, 0x6786, 0x6784, 0x6774, 0x678d, 0x678c, 0x677a,
	0x679f, 0x6791, 0x6799, 0x6783, 0x677d, 0x6781, 0x6778, 0x6779,
	0x6794, 0x6b25, 0x6b80, 0x6b7e, 0x6bde, 0x6c1d, 0x6c93, 0x6cec,
	0x6ceb, 0x6cee, 0x6cd9, 0x6cb6, 0x6cd4, 0x6cad, 0x6ce7, 0x6cb7,
	0x6cd0, 0x6cc2, 0x6cba, 0x6cc3, 0x6cc6, 0x6ced, 0x6cf2
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcd40[ 63 ] = {
	0x6cd2, 0x6cdd, 0x6cb4, 0x6c8a, 0x6c9d, 0x6c80, 0x6cde, 0x6cc0,
	0x6d30, 0x6ccd, 0x6cc7, 0x6cb0, 0x6cf9, 0x6ccf, 0x6ce9, 0x6cd1,
	0x7094, 0x7098, 0x7085, 0x7093, 0x7086, 0x7084, 0x7091, 0x7096,
	0x7082, 0x709a, 0x7083, 0x726a, 0x72d6, 0x72cb, 0x72d8, 0x72c9,
	0x72dc, 0x72d2, 0x72d4, 0x72da, 0x72cc, 0x72d1, 0x73a4, 0x73a1,
	0x73ad, 0x73a6, 0x73a2, 0x73a0, 0x73ac, 0x739d, 0x74dd, 0x74e8,
	0x753f, 0x7540, 0x753e, 0x758c, 0x7598, 0x76af, 0x76f3, 0x76f1,
	0x76f0, 0x76f5, 0x77f8, 0x77fc, 0x77f9, 0x77fb, 0x77fa
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcda1[ 94 ] = {
	        0x77f7, 0x7942, 0x793f, 0x79c5, 0x7a78, 0x7a7b, 0x7afb,
	0x7c75, 0x7cfd, 0x8035, 0x808f, 0x80ae, 0x80a3, 0x80b8, 0x80b5,
	0x80ad, 0x8220, 0x82a0, 0x82c0, 0x82ab, 0x829a, 0x8298, 0x829b,
	0x82b5, 0x82a7, 0x82ae, 0x82bc, 0x829e, 0x82ba, 0x82b4, 0x82a8,
	0x82a1, 0x82a9, 0x82c2, 0x82a4, 0x82c3, 0x82b6, 0x82a2, 0x8670,
	0x866f, 0x866d, 0x866e, 0x8c56, 0x8fd2, 0x8fcb, 0x8fd3, 0x8fcd,
	0x8fd6, 0x8fd5, 0x8fd7, 0x90b2, 0x90b4, 0x90af, 0x90b3, 0x90b0,
	0x9639, 0x963d, 0x963c, 0x963a, 0x9643, 0x4fcd, 0x4fc5, 0x4fd3,
	0x4fb2, 0x4fc9, 0x4fcb, 0x4fc1, 0x4fd4, 0x4fdc, 0x4fd9, 0x4fbb,
	0x4fb3, 0x4fdb, 0x4fc7, 0x4fd6, 0x4fba, 0x4fc0, 0x4fb9, 0x4fec,
	0x5244, 0x5249, 0x52c0, 0x52c2, 0x533d, 0x537c, 0x5397, 0x5396,
	0x5399, 0x5398, 0x54ba, 0x54a1, 0x54ad, 0x54a5, 0x54cf
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xce40[ 63 ] = {
	0x54c3, 0x830d, 0x54b7, 0x54ae, 0x54d6, 0x54b6, 0x54c5, 0x54c6,
	0x54a0, 0x5470, 0x54bc, 0x54a2, 0x54be, 0x5472, 0x54de, 0x54b0,
	0x57b5, 0x579e, 0x579f, 0x57a4, 0x578c, 0x5797, 0x579d, 0x579b,
	0x5794, 0x5798, 0x578f, 0x5799, 0x57a5, 0x579a, 0x5795, 0x58f4,
	0x590d, 0x5953, 0x59e1, 0x59de, 0x59ee, 0x5a00, 0x59f1, 0x59dd,
	0x59fa, 0x59fd, 0x59fc, 0x59f6, 0x59e4, 0x59f2, 0x59f7, 0x59db,
	0x59e9, 0x59f3, 0x59f5, 0x59e0, 0x59fe, 0x59f4, 0x59ed, 0x5ba8,
	0x5c4c, 0x5cd0, 0x5cd8, 0x5ccc, 0x5cd7, 0x5ccb, 0x5cdb
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcea1[ 94 ] = {
	        0x5cde, 0x5cda, 0x5cc9, 0x5cc7, 0x5cca, 0x5cd6, 0x5cd3,
	0x5cd4, 0x5ccf, 0x5cc8, 0x5cc6, 0x5cce, 0x5cdf, 0x5cf8, 0x5df9,
	0x5e21, 0x5e22, 0x5e23, 0x5e20, 0x5e24, 0x5eb0, 0x5ea4, 0x5ea2,
	0x5e9b, 0x5ea3, 0x5ea5, 0x5f07, 0x5f2e, 0x5f56, 0x5f86, 0x6037,
	0x6039, 0x6054, 0x6072, 0x605e, 0x6045, 0x6053, 0x6047, 0x6049,
	0x605b, 0x604c, 0x6040, 0x6042, 0x605f, 0x6024, 0x6044, 0x6058,
	0x6066, 0x606e, 0x6242, 0x6243, 0x62cf, 0x630d, 0x630b, 0x62f5,
	0x630e, 0x6303, 0x62eb, 0x62f9, 0x630f, 0x630c, 0x62f8, 0x62f6,
	0x6300, 0x6313, 0x6314, 0x62fa, 0x6315, 0x62fb, 0x62f0, 0x6541,
	0x6543, 0x65aa, 0x65bf, 0x6636, 0x6621, 0x6632, 0x6635, 0x661c,
	0x6626, 0x6622, 0x6633, 0x662b, 0x663a, 0x661d, 0x6634, 0x6639,
	0x662e, 0x670f, 0x6710, 0x67c1, 0x67f2, 0x67c8, 0x67ba
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcf40[ 63 ] = {
	0x67dc, 0x67bb, 0x67f8, 0x67d8, 0x67c0, 0x67b7, 0x67c5, 0x67eb,
	0x67e4, 0x67df, 0x67b5, 0x67cd, 0x67b3, 0x67f7, 0x67f6, 0x67ee,
	0x67e3, 0x67c2, 0x67b9, 0x67ce, 0x67e7, 0x67f0, 0x67b2, 0x67fc,
	0x67c6, 0x67ed, 0x67cc, 0x67ae, 0x67e6, 0x67db, 0x67fa, 0x67c9,
	0x67ca, 0x67c3, 0x67ea, 0x67cb, 0x6b28, 0x6b82, 0x6b84, 0x6bb6,
	0x6bd6, 0x6bd8, 0x6be0, 0x6c20, 0x6c21, 0x6d28, 0x6d34, 0x6d2d,
	0x6d1f, 0x6d3c, 0x6d3f, 0x6d12, 0x6d0a, 0x6cda, 0x6d33, 0x6d04,
	0x6d19, 0x6d3a, 0x6d1a, 0x6d11, 0x6d00, 0x6d1d, 0x6d42
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcfa1[ 94 ] = {
	        0x6d01, 0x6d18, 0x6d37, 0x6d03, 0x6d0f, 0x6d40, 0x6d07,
	0x6d20, 0x6d2c, 0x6d08, 0x6d22, 0x6d09, 0x6d10, 0x70b7, 0x709f,
	0x70be, 0x70b1, 0x70b0, 0x70a1, 0x70b4, 0x70b5, 0x70a9, 0x7241,
	0x7249, 0x724a, 0x726c, 0x7270, 0x7273, 0x726e, 0x72ca, 0x72e4,
	0x72e8, 0x72eb, 0x72df, 0x72ea, 0x72e6, 0x72e3, 0x7385, 0x73cc,
	0x73c2, 0x73c8, 0x73c5, 0x73b9, 0x73b6, 0x73b5, 0x73b4, 0x73eb,
	0x73bf, 0x73c7, 0x73be, 0x73c3, 0x73c6, 0x73b8, 0x73cb, 0x74ec,
	0x74ee, 0x752e, 0x7547, 0x7548, 0x75a7, 0x75aa, 0x7679, 0x76c4,
	0x7708, 0x7703, 0x7704, 0x7705, 0x770a, 0x76f7, 0x76fb, 0x76fa,
	0x77e7, 0x77e8, 0x7806, 0x7811, 0x7812, 0x7805, 0x7810, 0x780f,
	0x780e, 0x7809, 0x7803, 0x7813, 0x794a, 0x794c, 0x794b, 0x7945,
	0x7944, 0x79d5, 0x79cd, 0x79cf, 0x79d6, 0x79ce, 0x7a80
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd040[ 63 ] = {
	0x7a7e, 0x7ad1, 0x7b00, 0x7b01, 0x7c7a, 0x7c78, 0x7c79, 0x7c7f,
	0x7c80, 0x7c81, 0x7d03, 0x7d08, 0x7d01, 0x7f58, 0x7f91, 0x7f8d,
	0x7fbe, 0x8007, 0x800e, 0x800f, 0x8014, 0x8037, 0x80d8, 0x80c7,
	0x80e0, 0x80d1, 0x80c8, 0x80c2, 0x80d0, 0x80c5, 0x80e3, 0x80d9,
	0x80dc, 0x80ca, 0x80d5, 0x80c9, 0x80cf, 0x80d7, 0x80e6, 0x80cd,
	0x81ff, 0x8221, 0x8294, 0x82d9, 0x82fe, 0x82f9, 0x8307, 0x82e8,
	0x8300, 0x82d5, 0x833a, 0x82eb, 0x82d6, 0x82f4, 0x82ec, 0x82e1,
	0x82f2, 0x82f5, 0x830c, 0x82fb, 0x82f6, 0x82f0, 0x82ea
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd0a1[ 94 ] = {
	        0x82e4, 0x82e0, 0x82fa, 0x82f3, 0x82ed, 0x8677, 0x8674,
	0x867c, 0x8673, 0x8841, 0x884e, 0x8867, 0x886a, 0x8869, 0x89d3,
	0x8a04, 0x8a07, 0x8d72, 0x8fe3, 0x8fe1, 0x8fee, 0x8fe0, 0x90f1,
	0x90bd, 0x90bf, 0x90d5, 0x90c5, 0x90be, 0x90c7, 0x90cb, 0x90c8,
	0x91d4, 0x91d3, 0x9654, 0x964f, 0x9651, 0x9653, 0x964a, 0x964e,
	0x501e, 0x5005, 0x5007, 0x5013, 0x5022, 0x5030, 0x501b, 0x4ff5,
	0x4ff4, 0x5033, 0x5037, 0x502c, 0x4ff6, 0x4ff7, 0x5017, 0x501c,
	0x5020, 0x5027, 0x5035, 0x502f, 0x5031, 0x500e, 0x515a, 0x5194,
	0x5193, 0x51ca, 0x51c4, 0x51c5, 0x51c8, 0x51ce, 0x5261, 0x525a,
	0x5252, 0x525e, 0x525f, 0x5255, 0x5262, 0x52cd, 0x530e, 0x539e,
	0x5526, 0x54e2, 0x5517, 0x5512, 0x54e7, 0x54f3, 0x54e4, 0x551a,
	0x54ff, 0x5504, 0x5508, 0x54eb, 0x5511, 0x5505, 0x54f1
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd140[ 63 ] = {
	0x550a, 0x54fb, 0x54f7, 0x54f8, 0x54e0, 0x550e, 0x5503, 0x550b,
	0x5701, 0x5702, 0x57cc, 0x5832, 0x57d5, 0x57d2, 0x57ba, 0x57c6,
	0x57bd, 0x57bc, 0x57b8, 0x57b6, 0x57bf, 0x57c7, 0x57d0, 0x57b9,
	0x57c1, 0x590e, 0x594a, 0x5a19, 0x5a16, 0x5a2d, 0x5a2e, 0x5a15,
	0x5a0f, 0x5a17, 0x5a0a, 0x5a1e, 0x5a33, 0x5b6c, 0x5ba7, 0x5bad,
	0x5bac, 0x5c03, 0x5c56, 0x5c54, 0x5cec, 0x5cff, 0x5cee, 0x5cf1,
	0x5cf7, 0x5d00, 0x5cf9, 0x5e29, 0x5e28, 0x5ea8, 0x5eae, 0x5eaa,
	0x5eac, 0x5f33, 0x5f30, 0x5f67, 0x605d, 0x605a, 0x6067
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd1a1[ 94 ] = {
	        0x6041, 0x60a2, 0x6088, 0x6080, 0x6092, 0x6081, 0x609d,
	0x6083, 0x6095, 0x609b, 0x6097, 0x6087, 0x609c, 0x608e, 0x6219,
	0x6246, 0x62f2, 0x6310, 0x6356, 0x632c, 0x6344, 0x6345, 0x6336,
	0x6343, 0x63e4, 0x6339, 0x634b, 0x634a, 0x633c, 0x6329, 0x6341,
	0x6334, 0x6358, 0x6354, 0x6359, 0x632d, 0x6347, 0x6333, 0x635a,
	0x6351, 0x6338, 0x6357, 0x6340, 0x6348, 0x654a, 0x6546, 0x65c6,
	0x65c3, 0x65c4, 0x65c2, 0x664a, 0x665f, 0x6647, 0x6651, 0x6712,
	0x6713, 0x681f, 0x681a, 0x6849, 0x6832, 0x6833, 0x683b, 0x684b,
	0x684f, 0x6816, 0x6831, 0x681c, 0x6835, 0x682b, 0x682d, 0x682f,
	0x684e, 0x6844, 0x6834, 0x681d, 0x6812, 0x6814, 0x6826, 0x6828,
	0x682e, 0x684d, 0x683a, 0x6825, 0x6820, 0x6b2c, 0x6b2f, 0x6b2d,
	0x6b31, 0x6b34, 0x6b6d, 0x8082, 0x6b88, 0x6be6, 0x6be4
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd240[ 63 ] = {
	0x6be8, 0x6be3, 0x6be2, 0x6be7, 0x6c25, 0x6d7a, 0x6d63, 0x6d64,
	0x6d76, 0x6d0d, 0x6d61, 0x6d92, 0x6d58, 0x6d62, 0x6d6d, 0x6d6f,
	0x6d91, 0x6d8d, 0x6def, 0x6d7f, 0x6d86, 0x6d5e, 0x6d67, 0x6d60,
	0x6d97, 0x6d70, 0x6d7c, 0x6d5f, 0x6d82, 0x6d98, 0x6d2f, 0x6d68,
	0x6d8b, 0x6d7e, 0x6d80, 0x6d84, 0x6d16, 0x6d83, 0x6d7b, 0x6d7d,
	0x6d75, 0x6d90, 0x70dc, 0x70d3, 0x70d1, 0x70dd, 0x70cb, 0x7f39,
	0x70e2, 0x70d7, 0x70d2, 0x70de, 0x70e0, 0x70d4, 0x70cd, 0x70c5,
	0x70c6, 0x70c7, 0x70da, 0x70ce, 0x70e1, 0x7242, 0x7278
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd2a1[ 94 ] = {
	        0x7277, 0x7276, 0x7300, 0x72fa, 0x72f4, 0x72fe, 0x72f6,
	0x72f3, 0x72fb, 0x7301, 0x73d3, 0x73d9, 0x73e5, 0x73d6, 0x73bc,
	0x73e7, 0x73e3, 0x73e9, 0x73dc, 0x73d2, 0x73db, 0x73d4, 0x73dd,
	0x73da, 0x73d7, 0x73d8, 0x73e8, 0x74de, 0x74df, 0x74f4, 0x74f5,
	0x7521, 0x755b, 0x755f, 0x75b0, 0x75c1, 0x75bb, 0x75c4, 0x75c0,
	0x75bf, 0x75b6, 0x75ba, 0x768a, 0x76c9, 0x771d, 0x771b, 0x7710,
	0x7713, 0x7712, 0x7723, 0x7711, 0x7715, 0x7719, 0x771a, 0x7722,
	0x7727, 0x7823, 0x782c, 0x7822, 0x7835, 0x782f, 0x7828, 0x782e,
	0x782b, 0x7821, 0x7829, 0x7833, 0x782a, 0x7831, 0x7954, 0x795b,
	0x794f, 0x795c, 0x7953, 0x7952, 0x7951, 0x79eb, 0x79ec, 0x79e0,
	0x79ee, 0x79ed, 0x79ea, 0x79dc, 0x79de, 0x79dd, 0x7a86, 0x7a89,
	0x7a85, 0x7a8b, 0x7a8c, 0x7a8a, 0x7a87, 0x7ad8, 0x7b10
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd340[ 63 ] = {
	0x7b04, 0x7b13, 0x7b05, 0x7b0f, 0x7b08, 0x7b0a, 0x7b0e, 0x7b09,
	0x7b12, 0x7c84, 0x7c91, 0x7c8a, 0x7c8c, 0x7c88, 0x7c8d, 0x7c85,
	0x7d1e, 0x7d1d, 0x7d11, 0x7d0e, 0x7d18, 0x7d16, 0x7d13, 0x7d1f,
	0x7d12, 0x7d0f, 0x7d0c, 0x7f5c, 0x7f61, 0x7f5e, 0x7f60, 0x7f5d,
	0x7f5b, 0x7f96, 0x7f92, 0x7fc3, 0x7fc2, 0x7fc0, 0x8016, 0x803e,
	0x8039, 0x80fa, 0x80f2, 0x80f9, 0x80f5, 0x8101, 0x80fb, 0x8100,
	0x8201, 0x822f, 0x8225, 0x8333, 0x832d, 0x8344, 0x8319, 0x8351,
	0x8325, 0x8356, 0x833f, 0x8341, 0x8326, 0x831c, 0x8322
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd3a1[ 94 ] = {
	        0x8342, 0x834e, 0x831b, 0x832a, 0x8308, 0x833c, 0x834d,
	0x8316, 0x8324, 0x8320, 0x8337, 0x832f, 0x8329, 0x8347, 0x8345,
	0x834c, 0x8353, 0x831e, 0x832c, 0x834b, 0x8327, 0x8348, 0x8653,
	0x8652, 0x86a2, 0x86a8, 0x8696, 0x868d, 0x8691, 0x869e, 0x8687,
	0x8697, 0x8686, 0x868b, 0x869a, 0x8685, 0x86a5, 0x8699, 0x86a1,
	0x86a7, 0x8695, 0x8698, 0x868e, 0x869d, 0x8690, 0x8694, 0x8843,
	0x8844, 0x886d, 0x8875, 0x8876, 0x8872, 0x8880, 0x8871, 0x887f,
	0x886f, 0x8883, 0x887e, 0x8874, 0x887c, 0x8a12, 0x8c47, 0x8c57,
	0x8c7b, 0x8ca4, 0x8ca3, 0x8d76, 0x8d78, 0x8db5, 0x8db7, 0x8db6,
	0x8ed1, 0x8ed3, 0x8ffe, 0x8ff5, 0x9002, 0x8fff, 0x8ffb, 0x9004,
	0x8ffc, 0x8ff6, 0x90d6, 0x90e0, 0x90d9, 0x90da, 0x90e3, 0x90df,
	0x90e5, 0x90d8, 0x90db, 0x90d7, 0x90dc, 0x90e4, 0x9150
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd440[ 63 ] = {
	0x914e, 0x914f, 0x91d5, 0x91e2, 0x91da, 0x965c, 0x965f, 0x96bc,
	0x98e3, 0x9adf, 0x9b2f, 0x4e7f, 0x5070, 0x506a, 0x5061, 0x505e,
	0x5060, 0x5053, 0x504b, 0x505d, 0x5072, 0x5048, 0x504d, 0x5041,
	0x505b, 0x504a, 0x5062, 0x5015, 0x5045, 0x505f, 0x5069, 0x506b,
	0x5063, 0x5064, 0x5046, 0x5040, 0x506e, 0x5073, 0x5057, 0x5051,
	0x51d0, 0x526b, 0x526d, 0x526c, 0x526e, 0x52d6, 0x52d3, 0x532d,
	0x539c, 0x5575, 0x5576, 0x553c, 0x554d, 0x5550, 0x5534, 0x552a,
	0x5551, 0x5562, 0x5536, 0x5535, 0x5530, 0x5552, 0x5545
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd4a1[ 94 ] = {
	        0x550c, 0x5532, 0x5565, 0x554e, 0x5539, 0x5548, 0x552d,
	0x553b, 0x5540, 0x554b, 0x570a, 0x5707, 0x57fb, 0x5814, 0x57e2,
	0x57f6, 0x57dc, 0x57f4, 0x5800, 0x57ed, 0x57fd, 0x5808, 0x57f8,
	0x580b, 0x57f3, 0x57cf, 0x5807, 0x57ee, 0x57e3, 0x57f2, 0x57e5,
	0x57ec, 0x57e1, 0x580e, 0x57fc, 0x5810, 0x57e7, 0x5801, 0x580c,
	0x57f1, 0x57e9, 0x57f0, 0x580d, 0x5804, 0x595c, 0x5a60, 0x5a58,
	0x5a55, 0x5a67, 0x5a5e, 0x5a38, 0x5a35, 0x5a6d, 0x5a50, 0x5a5f,
	0x5a65, 0x5a6c, 0x5a53, 0x5a64, 0x5a57, 0x5a43, 0x5a5d, 0x5a52,
	0x5a44, 0x5a5b, 0x5a48, 0x5a8e, 0x5a3e, 0x5a4d, 0x5a39, 0x5a4c,
	0x5a70, 0x5a69, 0x5a47, 0x5a51, 0x5a56, 0x5a42, 0x5a5c, 0x5b72,
	0x5b6e, 0x5bc1, 0x5bc0, 0x5c59, 0x5d1e, 0x5d0b, 0x5d1d, 0x5d1a,
	0x5d20, 0x5d0c, 0x5d28, 0x5d0d, 0x5d26, 0x5d25, 0x5d0f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd540[ 63 ] = {
	0x5d30, 0x5d12, 0x5d23, 0x5d1f, 0x5d2e, 0x5e3e, 0x5e34, 0x5eb1,
	0x5eb4, 0x5eb9, 0x5eb2, 0x5eb3, 0x5f36, 0x5f38, 0x5f9b, 0x5f96,
	0x5f9f, 0x608a, 0x6090, 0x6086, 0x60be, 0x60b0, 0x60ba, 0x60d3,
	0x60d4, 0x60cf, 0x60e4, 0x60d9, 0x60dd, 0x60c8, 0x60b1, 0x60db,
	0x60b7, 0x60ca, 0x60bf, 0x60c3, 0x60cd, 0x60c0, 0x6332, 0x6365,
	0x638a, 0x6382, 0x637d, 0x63bd, 0x639e, 0x63ad, 0x639d, 0x6397,
	0x63ab, 0x638e, 0x636f, 0x6387, 0x6390, 0x636e, 0x63af, 0x6375,
	0x639c, 0x636d, 0x63ae, 0x637c, 0x63a4, 0x633b, 0x639f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd5a1[ 94 ] = {
	        0x6378, 0x6385, 0x6381, 0x6391, 0x638d, 0x6370, 0x6553,
	0x65cd, 0x6665, 0x6661, 0x665b, 0x6659, 0x665c, 0x6662, 0x6718,
	0x6879, 0x6887, 0x6890, 0x689c, 0x686d, 0x686e, 0x68ae, 0x68ab,
	0x6956, 0x686f, 0x68a3, 0x68ac, 0x68a9, 0x6875, 0x6874, 0x68b2,
	0x688f, 0x6877, 0x6892, 0x687c, 0x686b, 0x6872, 0x68aa, 0x6880,
	0x6871, 0x687e, 0x689b, 0x6896, 0x688b, 0x68a0, 0x6889, 0x68a4,
	0x6878, 0x687b, 0x6891, 0x688c, 0x688a, 0x687d, 0x6b36, 0x6b33,
	0x6b37, 0x6b38, 0x6b91, 0x6b8f, 0x6b8d, 0x6b8e, 0x6b8c, 0x6c2a,
	0x6dc0, 0x6dab, 0x6db4, 0x6db3, 0x6e74, 0x6dac, 0x6de9, 0x6de2,
	0x6db7, 0x6df6, 0x6dd4, 0x6e00, 0x6dc8, 0x6de0, 0x6ddf, 0x6dd6,
	0x6dbe, 0x6de5, 0x6ddc, 0x6ddd, 0x6ddb, 0x6df4, 0x6dca, 0x6dbd,
	0x6ded, 0x6df0, 0x6dba, 0x6dd5, 0x6dc2, 0x6dcf, 0x6dc9
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd640[ 63 ] = {
	0x6dd0, 0x6df2, 0x6dd3, 0x6dfd, 0x6dd7, 0x6dcd, 0x6de3, 0x6dbb,
	0x70fa, 0x710d, 0x70f7, 0x7117, 0x70f4, 0x710c, 0x70f0, 0x7104,
	0x70f3, 0x7110, 0x70fc, 0x70ff, 0x7106, 0x7113, 0x7100, 0x70f8,
	0x70f6, 0x710b, 0x7102, 0x710e, 0x727e, 0x727b, 0x727c, 0x727f,
	0x731d, 0x7317, 0x7307, 0x7311, 0x7318, 0x730a, 0x7308, 0x72ff,
	0x730f, 0x731e, 0x7388, 0x73f6, 0x73f8, 0x73f5, 0x7404, 0x7401,
	0x73fd, 0x7407, 0x7400, 0x73fa, 0x73fc, 0x73ff, 0x740c, 0x740b,
	0x73f4, 0x7408, 0x7564, 0x7563, 0x75ce, 0x75d2, 0x75cf
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd6a1[ 94 ] = {
	        0x75cb, 0x75cc, 0x75d1, 0x75d0, 0x768f, 0x7689, 0x76d3,
	0x7739, 0x772f, 0x772d, 0x7731, 0x7732, 0x7734, 0x7733, 0x773d,
	0x7725, 0x773b, 0x7735, 0x7848, 0x7852, 0x7849, 0x784d, 0x784a,
	0x784c, 0x7826, 0x7845, 0x7850, 0x7964, 0x7967, 0x7969, 0x796a,
	0x7963, 0x796b, 0x7961, 0x79bb, 0x79fa, 0x79f8, 0x79f6, 0x79f7,
	0x7a8f, 0x7a94, 0x7a90, 0x7b35, 0x7b47, 0x7b34, 0x7b25, 0x7b30,
	0x7b22, 0x7b24, 0x7b33, 0x7b18, 0x7b2a, 0x7b1d, 0x7b31, 0x7b2b,
	0x7b2d, 0x7b2f, 0x7b32, 0x7b38, 0x7b1a, 0x7b23, 0x7c94, 0x7c98,
	0x7c96, 0x7ca3, 0x7d35, 0x7d3d, 0x7d38, 0x7d36, 0x7d3a, 0x7d45,
	0x7d2c, 0x7d29, 0x7d41, 0x7d47, 0x7d3e, 0x7d3f, 0x7d4a, 0x7d3b,
	0x7d28, 0x7f63, 0x7f95, 0x7f9c, 0x7f9d, 0x7f9b, 0x7fca, 0x7fcb,
	0x7fcd, 0x7fd0, 0x7fd1, 0x7fc7, 0x7fcf, 0x7fc9, 0x801f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd740[ 63 ] = {
	0x801e, 0x801b, 0x8047, 0x8043, 0x8048, 0x8118, 0x8125, 0x8119,
	0x811b, 0x812d, 0x811f, 0x812c, 0x811e, 0x8121, 0x8115, 0x8127,
	0x811d, 0x8122, 0x8211, 0x8238, 0x8233, 0x823a, 0x8234, 0x8232,
	0x8274, 0x8390, 0x83a3, 0x83a8, 0x838d, 0x837a, 0x8373, 0x83a4,
	0x8374, 0x838f, 0x8381, 0x8395, 0x8399, 0x8375, 0x8394, 0x83a9,
	0x837d, 0x8383, 0x838c, 0x839d, 0x839b, 0x83aa, 0x838b, 0x837e,
	0x83a5, 0x83af, 0x8388, 0x8397, 0x83b0, 0x837f, 0x83a6, 0x8387,
	0x83ae, 0x8376, 0x839a, 0x8659, 0x8656, 0x86bf, 0x86b7
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd7a1[ 94 ] = {
	        0x86c2, 0x86c1, 0x86c5, 0x86ba, 0x86b0, 0x86c8, 0x86b9,
	0x86b3, 0x86b8, 0x86cc, 0x86b4, 0x86bb, 0x86bc, 0x86c3, 0x86bd,
	0x86be, 0x8852, 0x8889, 0x8895, 0x88a8, 0x88a2, 0x88aa, 0x889a,
	0x8891, 0x88a1, 0x889f, 0x8898, 0x88a7, 0x8899, 0x889b, 0x8897,
	0x88a4, 0x88ac, 0x888c, 0x8893, 0x888e, 0x8982, 0x89d6, 0x89d9,
	0x89d5, 0x8a30, 0x8a27, 0x8a2c, 0x8a1e, 0x8c39, 0x8c3b, 0x8c5c,
	0x8c5d, 0x8c7d, 0x8ca5, 0x8d7d, 0x8d7b, 0x8d79, 0x8dbc, 0x8dc2,
	0x8db9, 0x8dbf, 0x8dc1, 0x8ed8, 0x8ede, 0x8edd, 0x8edc, 0x8ed7,
	0x8ee0, 0x8ee1, 0x9024, 0x900b, 0x9011, 0x901c, 0x900c, 0x9021,
	0x90ef, 0x90ea, 0x90f0, 0x90f4, 0x90f2, 0x90f3, 0x90d4, 0x90eb,
	0x90ec, 0x90e9, 0x9156, 0x9158, 0x915a, 0x9153, 0x9155, 0x91ec,
	0x91f4, 0x91f1, 0x91f3, 0x91f8, 0x91e4, 0x91f9, 0x91ea
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd840[ 63 ] = {
	0x91eb, 0x91f7, 0x91e8, 0x91ee, 0x957a, 0x9586, 0x9588, 0x967c,
	0x966d, 0x966b, 0x9671, 0x966f, 0x96bf, 0x976a, 0x9804, 0x98e5,
	0x9997, 0x509b, 0x5095, 0x5094, 0x509e, 0x508b, 0x50a3, 0x5083,
	0x508c, 0x508e, 0x509d, 0x5068, 0x509c, 0x5092, 0x5082, 0x5087,
	0x515f, 0x51d4, 0x5312, 0x5311, 0x53a4, 0x53a7, 0x5591, 0x55a8,
	0x55a5, 0x55ad, 0x5577, 0x5645, 0x55a2, 0x5593, 0x5588, 0x558f,
	0x55b5, 0x5581, 0x55a3, 0x5592, 0x55a4, 0x557d, 0x558c, 0x55a6,
	0x557f, 0x5595, 0x55a1, 0x558e, 0x570c, 0x5829, 0x5837
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd8a1[ 94 ] = {
	        0x5819, 0x581e, 0x5827, 0x5823, 0x5828, 0x57f5, 0x5848,
	0x5825, 0x581c, 0x581b, 0x5833, 0x583f, 0x5836, 0x582e, 0x5839,
	0x5838, 0x582d, 0x582c, 0x583b, 0x5961, 0x5aaf, 0x5a94, 0x5a9f,
	0x5a7a, 0x5aa2, 0x5a9e, 0x5a78, 0x5aa6, 0x5a7c, 0x5aa5, 0x5aac,
	0x5a95, 0x5aae, 0x5a37, 0x5a84, 0x5a8a, 0x5a97, 0x5a83, 0x5a8b,
	0x5aa9, 0x5a7b, 0x5a7d, 0x5a8c, 0x5a9c, 0x5a8f, 0x5a93, 0x5a9d,
	0x5bea, 0x5bcd, 0x5bcb, 0x5bd4, 0x5bd1, 0x5bca, 0x5bce, 0x5c0c,
	0x5c30, 0x5d37, 0x5d43, 0x5d6b, 0x5d41, 0x5d4b, 0x5d3f, 0x5d35,
	0x5d51, 0x5d4e, 0x5d55, 0x5d33, 0x5d3a, 0x5d52, 0x5d3d, 0x5d31,
	0x5d59, 0x5d42, 0x5d39, 0x5d49, 0x5d38, 0x5d3c, 0x5d32, 0x5d36,
	0x5d40, 0x5d45, 0x5e44, 0x5e41, 0x5f58, 0x5fa6, 0x5fa5, 0x5fab,
	0x60c9, 0x60b9, 0x60cc, 0x60e2, 0x60ce, 0x60c4, 0x6114
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd940[ 63 ] = {
	0x60f2, 0x610a, 0x6116, 0x6105, 0x60f5, 0x6113, 0x60f8, 0x60fc,
	0x60fe, 0x60c1, 0x6103, 0x6118, 0x611d, 0x6110, 0x60ff, 0x6104,
	0x610b, 0x624a, 0x6394, 0x63b1, 0x63b0, 0x63ce, 0x63e5, 0x63e8,
	0x63ef, 0x63c3, 0x649d, 0x63f3, 0x63ca, 0x63e0, 0x63f6, 0x63d5,
	0x63f2, 0x63f5, 0x6461, 0x63df, 0x63be, 0x63dd, 0x63dc, 0x63c4,
	0x63d8, 0x63d3, 0x63c2, 0x63c7, 0x63cc, 0x63cb, 0x63c8, 0x63f0,
	0x63d7, 0x63d9, 0x6532, 0x6567, 0x656a, 0x6564, 0x655c, 0x6568,
	0x6565, 0x658c, 0x659d, 0x659e, 0x65ae, 0x65d0, 0x65d2
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd9a1[ 94 ] = {
	        0x667c, 0x666c, 0x667b, 0x6680, 0x6671, 0x6679, 0x666a,
	0x6672, 0x6701, 0x690c, 0x68d3, 0x6904, 0x68dc, 0x692a, 0x68ec,
	0x68ea, 0x68f1, 0x690f, 0x68d6, 0x68f7, 0x68eb, 0x68e4, 0x68f6,
	0x6913, 0x6910, 0x68f3, 0x68e1, 0x6907, 0x68cc, 0x6908, 0x6970,
	0x68b4, 0x6911, 0x68ef, 0x68c6, 0x6914, 0x68f8, 0x68d0, 0x68fd,
	0x68fc, 0x68e8, 0x690b, 0x690a, 0x6917, 0x68ce, 0x68c8, 0x68dd,
	0x68de, 0x68e6, 0x68f4, 0x68d1, 0x6906, 0x68d4, 0x68e9, 0x6915,
	0x6925, 0x68c7, 0x6b39, 0x6b3b, 0x6b3f, 0x6b3c, 0x6b94, 0x6b97,
	0x6b99, 0x6b95, 0x6bbd, 0x6bf0, 0x6bf2, 0x6bf3, 0x6c30, 0x6dfc,
	0x6e46, 0x6e47, 0x6e1f, 0x6e49, 0x6e88, 0x6e3c, 0x6e3d, 0x6e45,
	0x6e62, 0x6e2b, 0x6e3f, 0x6e41, 0x6e5d, 0x6e73, 0x6e1c, 0x6e33,
	0x6e4b, 0x6e40, 0x6e51, 0x6e3b, 0x6e03, 0x6e2e, 0x6e5e
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xda40[ 63 ] = {
	0x6e68, 0x6e5c, 0x6e61, 0x6e31, 0x6e28, 0x6e60, 0x6e71, 0x6e6b,
	0x6e39, 0x6e22, 0x6e30, 0x6e53, 0x6e65, 0x6e27, 0x6e78, 0x6e64,
	0x6e77, 0x6e55, 0x6e79, 0x6e52, 0x6e66, 0x6e35, 0x6e36, 0x6e5a,
	0x7120, 0x711e, 0x712f, 0x70fb, 0x712e, 0x7131, 0x7123, 0x7125,
	0x7122, 0x7132, 0x711f, 0x7128, 0x713a, 0x711b, 0x724b, 0x725a,
	0x7288, 0x7289, 0x7286, 0x7285, 0x728b, 0x7312, 0x730b, 0x7330,
	0x7322, 0x7331, 0x7333, 0x7327, 0x7332, 0x732d, 0x7326, 0x7323,
	0x7335, 0x730c, 0x742e, 0x742c, 0x7430, 0x742b, 0x7416
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdaa1[ 94 ] = {
	        0x741a, 0x7421, 0x742d, 0x7431, 0x7424, 0x7423, 0x741d,
	0x7429, 0x7420, 0x7432, 0x74fb, 0x752f, 0x756f, 0x756c, 0x75e7,
	0x75da, 0x75e1, 0x75e6, 0x75dd, 0x75df, 0x75e4, 0x75d7, 0x7695,
	0x7692, 0x76da, 0x7746, 0x7747, 0x7744, 0x774d, 0x7745, 0x774a,
	0x774e, 0x774b, 0x774c, 0x77de, 0x77ec, 0x7860, 0x7864, 0x7865,
	0x785c, 0x786d, 0x7871, 0x786a, 0x786e, 0x7870, 0x7869, 0x7868,
	0x785e, 0x7862, 0x7974, 0x7973, 0x7972, 0x7970, 0x7a02, 0x7a0a,
	0x7a03, 0x7a0c, 0x7a04, 0x7a99, 0x7ae6, 0x7ae4, 0x7b4a, 0x7b3b,
	0x7b44, 0x7b48, 0x7b4c, 0x7b4e, 0x7b40, 0x7b58, 0x7b45, 0x7ca2,
	0x7c9e, 0x7ca8, 0x7ca1, 0x7d58, 0x7d6f, 0x7d63, 0x7d53, 0x7d56,
	0x7d67, 0x7d6a, 0x7d4f, 0x7d6d, 0x7d5c, 0x7d6b, 0x7d52, 0x7d54,
	0x7d69, 0x7d51, 0x7d5f, 0x7d4e, 0x7f3e, 0x7f3f, 0x7f65
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdb40[ 63 ] = {
	0x7f66, 0x7fa2, 0x7fa0, 0x7fa1, 0x7fd7, 0x8051, 0x804f, 0x8050,
	0x80fe, 0x80d4, 0x8143, 0x814a, 0x8152, 0x814f, 0x8147, 0x813d,
	0x814d, 0x813a, 0x81e6, 0x81ee, 0x81f7, 0x81f8, 0x81f9, 0x8204,
	0x823c, 0x823d, 0x823f, 0x8275, 0x833b, 0x83cf, 0x83f9, 0x8423,
	0x83c0, 0x83e8, 0x8412, 0x83e7, 0x83e4, 0x83fc, 0x83f6, 0x8410,
	0x83c6, 0x83c8, 0x83eb, 0x83e3, 0x83bf, 0x8401, 0x83dd, 0x83e5,
	0x83d8, 0x83ff, 0x83e1, 0x83cb, 0x83ce, 0x83d6, 0x83f5, 0x83c9,
	0x8409, 0x840f, 0x83de, 0x8411, 0x8406, 0x83c2, 0x83f3
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdba1[ 94 ] = {
	        0x83d5, 0x83fa, 0x83c7, 0x83d1, 0x83ea, 0x8413, 0x83c3,
	0x83ec, 0x83ee, 0x83c4, 0x83fb, 0x83d7, 0x83e2, 0x841b, 0x83db,
	0x83fe, 0x86d8, 0x86e2, 0x86e6, 0x86d3, 0x86e3, 0x86da, 0x86ea,
	0x86dd, 0x86eb, 0x86dc, 0x86ec, 0x86e9, 0x86d7, 0x86e8, 0x86d1,
	0x8848, 0x8856, 0x8855, 0x88ba, 0x88d7, 0x88b9, 0x88b8, 0x88c0,
	0x88be, 0x88b6, 0x88bc, 0x88b7, 0x88bd, 0x88b2, 0x8901, 0x88c9,
	0x8995, 0x8998, 0x8997, 0x89dd, 0x89da, 0x89db, 0x8a4e, 0x8a4d,
	0x8a39, 0x8a59, 0x8a40, 0x8a57, 0x8a58, 0x8a44, 0x8a45, 0x8a52,
	0x8a48, 0x8a51, 0x8a4a, 0x8a4c, 0x8a4f, 0x8c5f, 0x8c81, 0x8c80,
	0x8cba, 0x8cbe, 0x8cb0, 0x8cb9, 0x8cb5, 0x8d84, 0x8d80, 0x8d89,
	0x8dd8, 0x8dd3, 0x8dcd, 0x8dc7, 0x8dd6, 0x8ddc, 0x8dcf, 0x8dd5,
	0x8dd9, 0x8dc8, 0x8dd7, 0x8dc5, 0x8eef, 0x8ef7, 0x8efa
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdc40[ 63 ] = {
	0x8ef9, 0x8ee6, 0x8eee, 0x8ee5, 0x8ef5, 0x8ee7, 0x8ee8, 0x8ef6,
	0x8eeb, 0x8ef1, 0x8eec, 0x8ef4, 0x8ee9, 0x902d, 0x9034, 0x902f,
	0x9106, 0x912c, 0x9104, 0x90ff, 0x90fc, 0x9108, 0x90f9, 0x90fb,
	0x9101, 0x9100, 0x9107, 0x9105, 0x9103, 0x9161, 0x9164, 0x915f,
	0x9162, 0x9160, 0x9201, 0x920a, 0x9225, 0x9203, 0x921a, 0x9226,
	0x920f, 0x920c, 0x9200, 0x9212, 0x91ff, 0x91fd, 0x9206, 0x9204,
	0x9227, 0x9202, 0x921c, 0x9224, 0x9219, 0x9217, 0x9205, 0x9216,
	0x957b, 0x958d, 0x958c, 0x9590, 0x9687, 0x967e, 0x9688
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdca1[ 94 ] = {
	        0x9689, 0x9683, 0x9680, 0x96c2, 0x96c8, 0x96c3, 0x96f1,
	0x96f0, 0x976c, 0x9770, 0x976e, 0x9807, 0x98a9, 0x98eb, 0x9ce6,
	0x9ef9, 0x4e83, 0x4e84, 0x4eb6, 0x50bd, 0x50bf, 0x50c6, 0x50ae,
	0x50c4, 0x50ca, 0x50b4, 0x50c8, 0x50c2, 0x50b0, 0x50c1, 0x50ba,
	0x50b1, 0x50cb, 0x50c9, 0x50b6, 0x50b8, 0x51d7, 0x527a, 0x5278,
	0x527b, 0x527c, 0x55c3, 0x55db, 0x55cc, 0x55d0, 0x55cb, 0x55ca,
	0x55dd, 0x55c0, 0x55d4, 0x55c4, 0x55e9, 0x55bf, 0x55d2, 0x558d,
	0x55cf, 0x55d5, 0x55e2, 0x55d6, 0x55c8, 0x55f2, 0x55cd, 0x55d9,
	0x55c2, 0x5714, 0x5853, 0x5868, 0x5864, 0x584f, 0x584d, 0x5849,
	0x586f, 0x5855, 0x584e, 0x585d, 0x5859, 0x5865, 0x585b, 0x583d,
	0x5863, 0x5871, 0x58fc, 0x5ac7, 0x5ac4, 0x5acb, 0x5aba, 0x5ab8,
	0x5ab1, 0x5ab5, 0x5ab0, 0x5abf, 0x5ac8, 0x5abb, 0x5ac6
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdd40[ 63 ] = {
	0x5ab7, 0x5ac0, 0x5aca, 0x5ab4, 0x5ab6, 0x5acd, 0x5ab9, 0x5a90,
	0x5bd6, 0x5bd8, 0x5bd9, 0x5c1f, 0x5c33, 0x5d71, 0x5d63, 0x5d4a,
	0x5d65, 0x5d72, 0x5d6c, 0x5d5e, 0x5d68, 0x5d67, 0x5d62, 0x5df0,
	0x5e4f, 0x5e4e, 0x5e4a, 0x5e4d, 0x5e4b, 0x5ec5, 0x5ecc, 0x5ec6,
	0x5ecb, 0x5ec7, 0x5f40, 0x5faf, 0x5fad, 0x60f7, 0x6149, 0x614a,
	0x612b, 0x6145, 0x6136, 0x6132, 0x612e, 0x6146, 0x612f, 0x614f,
	0x6129, 0x6140, 0x6220, 0x9168, 0x6223, 0x6225, 0x6224, 0x63c5,
	0x63f1, 0x63eb, 0x6410, 0x6412, 0x6409, 0x6420, 0x6424
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdda1[ 94 ] = {
	        0x6433, 0x6443, 0x641f, 0x6415, 0x6418, 0x6439, 0x6437,
	0x6422, 0x6423, 0x640c, 0x6426, 0x6430, 0x6428, 0x6441, 0x6435,
	0x642f, 0x640a, 0x641a, 0x6440, 0x6425, 0x6427, 0x640b, 0x63e7,
	0x641b, 0x642e, 0x6421, 0x640e, 0x656f, 0x6592, 0x65d3, 0x6686,
	0x668c, 0x6695, 0x6690, 0x668b, 0x668a, 0x6699, 0x6694, 0x6678,
	0x6720, 0x6966, 0x695f, 0x6938, 0x694e, 0x6962, 0x6971, 0x693f,
	0x6945, 0x696a, 0x6939, 0x6942, 0x6957, 0x6959, 0x697a, 0x6948,
	0x6949, 0x6935, 0x696c, 0x6933, 0x693d, 0x6965, 0x68f0, 0x6978,
	0x6934, 0x6969, 0x6940, 0x696f, 0x6944, 0x6976, 0x6958, 0x6941,
	0x6974, 0x694c, 0x693b, 0x694b, 0x6937, 0x695c, 0x694f, 0x6951,
	0x6932, 0x6952, 0x692f, 0x697b, 0x693c, 0x6b46, 0x6b45, 0x6b43,
	0x6b42, 0x6b48, 0x6b41, 0x6b9b, 0xfa0d, 0x6bfb, 0x6bfc
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xde40[ 63 ] = {
	0x6bf9, 0x6bf7, 0x6bf8, 0x6e9b, 0x6ed6, 0x6ec8, 0x6e8f, 0x6ec0,
	0x6e9f, 0x6e93, 0x6e94, 0x6ea0, 0x6eb1, 0x6eb9, 0x6ec6, 0x6ed2,
	0x6ebd, 0x6ec1, 0x6e9e, 0x6ec9, 0x6eb7, 0x6eb0, 0x6ecd, 0x6ea6,
	0x6ecf, 0x6eb2, 0x6ebe, 0x6ec3, 0x6edc, 0x6ed8, 0x6e99, 0x6e92,
	0x6e8e, 0x6e8d, 0x6ea4, 0x6ea1, 0x6ebf, 0x6eb3, 0x6ed0, 0x6eca,
	0x6e97, 0x6eae, 0x6ea3, 0x7147, 0x7154, 0x7152, 0x7163, 0x7160,
	0x7141, 0x715d, 0x7162, 0x7172, 0x7178, 0x716a, 0x7161, 0x7142,
	0x7158, 0x7143, 0x714b, 0x7170, 0x715f, 0x7150, 0x7153
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdea1[ 94 ] = {
	        0x7144, 0x714d, 0x715a, 0x724f, 0x728d, 0x728c, 0x7291,
	0x7290, 0x728e, 0x733c, 0x7342, 0x733b, 0x733a, 0x7340, 0x734a,
	0x7349, 0x7444, 0x744a, 0x744b, 0x7452, 0x7451, 0x7457, 0x7440,
	0x744f, 0x7450, 0x744e, 0x7442, 0x7446, 0x744d, 0x7454, 0x74e1,
	0x74ff, 0x74fe, 0x74fd, 0x751d, 0x7579, 0x7577, 0x6983, 0x75ef,
	0x760f, 0x7603, 0x75f7, 0x75fe, 0x75fc, 0x75f9, 0x75f8, 0x7610,
	0x75fb, 0x75f6, 0x75ed, 0x75f5, 0x75fd, 0x7699, 0x76b5, 0x76dd,
	0x7755, 0x775f, 0x7760, 0x7752, 0x7756, 0x775a, 0x7769, 0x7767,
	0x7754, 0x7759, 0x776d, 0x77e0, 0x7887, 0x789a, 0x7894, 0x788f,
	0x7884, 0x7895, 0x7885, 0x7886, 0x78a1, 0x7883, 0x7879, 0x7899,
	0x7880, 0x7896, 0x787b, 0x797c, 0x7982, 0x797d, 0x7979, 0x7a11,
	0x7a18, 0x7a19, 0x7a12, 0x7a17, 0x7a15, 0x7a22, 0x7a13
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdf40[ 63 ] = {
	0x7a1b, 0x7a10, 0x7aa3, 0x7aa2, 0x7a9e, 0x7aeb, 0x7b66, 0x7b64,
	0x7b6d, 0x7b74, 0x7b69, 0x7b72, 0x7b65, 0x7b73, 0x7b71, 0x7b70,
	0x7b61, 0x7b78, 0x7b76, 0x7b63, 0x7cb2, 0x7cb4, 0x7caf, 0x7d88,
	0x7d86, 0x7d80, 0x7d8d, 0x7d7f, 0x7d85, 0x7d7a, 0x7d8e, 0x7d7b,
	0x7d83, 0x7d7c, 0x7d8c, 0x7d94, 0x7d84, 0x7d7d, 0x7d92, 0x7f6d,
	0x7f6b, 0x7f67, 0x7f68, 0x7f6c, 0x7fa6, 0x7fa5, 0x7fa7, 0x7fdb,
	0x7fdc, 0x8021, 0x8164, 0x8160, 0x8177, 0x815c, 0x8169, 0x815b,
	0x8162, 0x8172, 0x6721, 0x815e, 0x8176, 0x8167, 0x816f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdfa1[ 94 ] = {
	        0x8144, 0x8161, 0x821d, 0x8249, 0x8244, 0x8240, 0x8242,
	0x8245, 0x84f1, 0x843f, 0x8456, 0x8476, 0x8479, 0x848f, 0x848d,
	0x8465, 0x8451, 0x8440, 0x8486, 0x8467, 0x8430, 0x844d, 0x847d,
	0x845a, 0x8459, 0x8474, 0x8473, 0x845d, 0x8507, 0x845e, 0x8437,
	0x843a, 0x8434, 0x847a, 0x8443, 0x8478, 0x8432, 0x8445, 0x8429,
	0x83d9, 0x844b, 0x842f, 0x8442, 0x842d, 0x845f, 0x8470, 0x8439,
	0x844e, 0x844c, 0x8452, 0x846f, 0x84c5, 0x848e, 0x843b, 0x8447,
	0x8436, 0x8433, 0x8468, 0x847e, 0x8444, 0x842b, 0x8460, 0x8454,
	0x846e, 0x8450, 0x870b, 0x8704, 0x86f7, 0x870c, 0x86fa, 0x86d6,
	0x86f5, 0x874d, 0x86f8, 0x870e, 0x8709, 0x8701, 0x86f6, 0x870d,
	0x8705, 0x88d6, 0x88cb, 0x88cd, 0x88ce, 0x88de, 0x88db, 0x88da,
	0x88cc, 0x88d0, 0x8985, 0x899b, 0x89df, 0x89e5, 0x89e4
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe040[ 63 ] = {
	0x89e1, 0x89e0, 0x89e2, 0x89dc, 0x89e6, 0x8a76, 0x8a86, 0x8a7f,
	0x8a61, 0x8a3f, 0x8a77, 0x8a82, 0x8a84, 0x8a75, 0x8a83, 0x8a81,
	0x8a74, 0x8a7a, 0x8c3c, 0x8c4b, 0x8c4a, 0x8c65, 0x8c64, 0x8c66,
	0x8c86, 0x8c84, 0x8c85, 0x8ccc, 0x8d68, 0x8d69, 0x8d91, 0x8d8c,
	0x8d8e, 0x8d8f, 0x8d8d, 0x8d93, 0x8d94, 0x8d90, 0x8d92, 0x8df0,
	0x8de0, 0x8dec, 0x8df1, 0x8dee, 0x8dd0, 0x8de9, 0x8de3, 0x8de2,
	0x8de7, 0x8df2, 0x8deb, 0x8df4, 0x8f06, 0x8eff, 0x8f01, 0x8f00,
	0x8f05, 0x8f07, 0x8f08, 0x8f02, 0x8f0b, 0x9052, 0x903f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe0a1[ 94 ] = {
	        0x9044, 0x9049, 0x903d, 0x9110, 0x910d, 0x910f, 0x9111,
	0x9116, 0x9114, 0x910b, 0x910e, 0x916e, 0x916f, 0x9248, 0x9252,
	0x9230, 0x923a, 0x9266, 0x9233, 0x9265, 0x925e, 0x9283, 0x922e,
	0x924a, 0x9246, 0x926d, 0x926c, 0x924f, 0x9260, 0x9267, 0x926f,
	0x9236, 0x9261, 0x9270, 0x9231, 0x9254, 0x9263, 0x9250, 0x9272,
	0x924e, 0x9253, 0x924c, 0x9256, 0x9232, 0x959f, 0x959c, 0x959e,
	0x959b, 0x9692, 0x9693, 0x9691, 0x9697, 0x96ce, 0x96fa, 0x96fd,
	0x96f8, 0x96f5, 0x9773, 0x9777, 0x9778, 0x9772, 0x980f, 0x980d,
	0x980e, 0x98ac, 0x98f6, 0x98f9, 0x99af, 0x99b2, 0x99b0, 0x99b5,
	0x9aad, 0x9aab, 0x9b5b, 0x9cea, 0x9ced, 0x9ce7, 0x9e80, 0x9efd,
	0x50e6, 0x50d4, 0x50d7, 0x50e8, 0x50f3, 0x50db, 0x50ea, 0x50dd,
	0x50e4, 0x50d3, 0x50ec, 0x50f0, 0x50ef, 0x50e3, 0x50e0
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe140[ 63 ] = {
	0x51d8, 0x5280, 0x5281, 0x52e9, 0x52eb, 0x5330, 0x53ac, 0x5627,
	0x5615, 0x560c, 0x5612, 0x55fc, 0x560f, 0x561c, 0x5601, 0x5613,
	0x5602, 0x55fa, 0x561d, 0x5604, 0x55ff, 0x55f9, 0x5889, 0x587c,
	0x5890, 0x5898, 0x5886, 0x5881, 0x587f, 0x5874, 0x588b, 0x587a,
	0x5887, 0x5891, 0x588e, 0x5876, 0x5882, 0x5888, 0x587b, 0x5894,
	0x588f, 0x58fe, 0x596b, 0x5adc, 0x5aee, 0x5ae5, 0x5ad5, 0x5aea,
	0x5ada, 0x5aed, 0x5aeb, 0x5af3, 0x5ae2, 0x5ae0, 0x5adb, 0x5aec,
	0x5ade, 0x5add, 0x5ad9, 0x5ae8, 0x5adf, 0x5b77, 0x5be0
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe1a1[ 94 ] = {
	        0x5be3, 0x5c63, 0x5d82, 0x5d80, 0x5d7d, 0x5d86, 0x5d7a,
	0x5d81, 0x5d77, 0x5d8a, 0x5d89, 0x5d88, 0x5d7e, 0x5d7c, 0x5d8d,
	0x5d79, 0x5d7f, 0x5e58, 0x5e59, 0x5e53, 0x5ed8, 0x5ed1, 0x5ed7,
	0x5ece, 0x5edc, 0x5ed5, 0x5ed9, 0x5ed2, 0x5ed4, 0x5f44, 0x5f43,
	0x5f6f, 0x5fb6, 0x612c, 0x6128, 0x6141, 0x615e, 0x6171, 0x6173,
	0x6152, 0x6153, 0x6172, 0x616c, 0x6180, 0x6174, 0x6154, 0x617a,
	0x615b, 0x6165, 0x613b, 0x616a, 0x6161, 0x6156, 0x6229, 0x6227,
	0x622b, 0x642b, 0x644d, 0x645b, 0x645d, 0x6474, 0x6476, 0x6472,
	0x6473, 0x647d, 0x6475, 0x6466, 0x64a6, 0x644e, 0x6482, 0x645e,
	0x645c, 0x644b, 0x6453, 0x6460, 0x6450, 0x647f, 0x643f, 0x646c,
	0x646b, 0x6459, 0x6465, 0x6477, 0x6573, 0x65a0, 0x66a1, 0x66a0,
	0x669f, 0x6705, 0x6704, 0x6722, 0x69b1, 0x69b6, 0x69c9
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe240[ 63 ] = {
	0x69a0, 0x69ce, 0x6996, 0x69b0, 0x69ac, 0x69bc, 0x6991, 0x6999,
	0x698e, 0x69a7, 0x698d, 0x69a9, 0x69be, 0x69af, 0x69bf, 0x69c4,
	0x69bd, 0x69a4, 0x69d4, 0x69b9, 0x69ca, 0x699a, 0x69cf, 0x69b3,
	0x6993, 0x69aa, 0x69a1, 0x699e, 0x69d9, 0x6997, 0x6990, 0x69c2,
	0x69b5, 0x69a5, 0x69c6, 0x6b4a, 0x6b4d, 0x6b4b, 0x6b9e, 0x6b9f,
	0x6ba0, 0x6bc3, 0x6bc4, 0x6bfe, 0x6ece, 0x6ef5, 0x6ef1, 0x6f03,
	0x6f25, 0x6ef8, 0x6f37, 0x6efb, 0x6f2e, 0x6f09, 0x6f4e, 0x6f19,
	0x6f1a, 0x6f27, 0x6f18, 0x6f3b, 0x6f12, 0x6eed, 0x6f0a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe2a1[ 94 ] = {
	        0x6f36, 0x6f73, 0x6ef9, 0x6eee, 0x6f2d, 0x6f40, 0x6f30,
	0x6f3c, 0x6f35, 0x6eeb, 0x6f07, 0x6f0e, 0x6f43, 0x6f05, 0x6efd,
	0x6ef6, 0x6f39, 0x6f1c, 0x6efc, 0x6f3a, 0x6f1f, 0x6f0d, 0x6f1e,
	0x6f08, 0x6f21, 0x7187, 0x7190, 0x7189, 0x7180, 0x7185, 0x7182,
	0x718f, 0x717b, 0x7186, 0x7181, 0x7197, 0x7244, 0x7253, 0x7297,
	0x7295, 0x7293, 0x7343, 0x734d, 0x7351, 0x734c, 0x7462, 0x7473,
	0x7471, 0x7475, 0x7472, 0x7467, 0x746e, 0x7500, 0x7502, 0x7503,
	0x757d, 0x7590, 0x7616, 0x7608, 0x760c, 0x7615, 0x7611, 0x760a,
	0x7614, 0x76b8, 0x7781, 0x777c, 0x7785, 0x7782, 0x776e, 0x7780,
	0x776f, 0x777e, 0x7783, 0x78b2, 0x78aa, 0x78b4, 0x78ad, 0x78a8,
	0x787e, 0x78ab, 0x789e, 0x78a5, 0x78a0, 0x78ac, 0x78a2, 0x78a4,
	0x7998, 0x798a, 0x798b, 0x7996, 0x7995, 0x7994, 0x7993
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe340[ 63 ] = {
	0x7997, 0x7988, 0x7992, 0x7990, 0x7a2b, 0x7a4a, 0x7a30, 0x7a2f,
	0x7a28, 0x7a26, 0x7aa8, 0x7aab, 0x7aac, 0x7aee, 0x7b88, 0x7b9c,
	0x7b8a, 0x7b91, 0x7b90, 0x7b96, 0x7b8d, 0x7b8c, 0x7b9b, 0x7b8e,
	0x7b85, 0x7b98, 0x5284, 0x7b99, 0x7ba4, 0x7b82, 0x7cbb, 0x7cbf,
	0x7cbc, 0x7cba, 0x7da7, 0x7db7, 0x7dc2, 0x7da3, 0x7daa, 0x7dc1,
	0x7dc0, 0x7dc5, 0x7d9d, 0x7dce, 0x7dc4, 0x7dc6, 0x7dcb, 0x7dcc,
	0x7daf, 0x7db9, 0x7d96, 0x7dbc, 0x7d9f, 0x7da6, 0x7dae, 0x7da9,
	0x7da1, 0x7dc9, 0x7f73, 0x7fe2, 0x7fe3, 0x7fe5, 0x7fde
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe3a1[ 94 ] = {
	        0x8024, 0x805d, 0x805c, 0x8189, 0x8186, 0x8183, 0x8187,
	0x818d, 0x818c, 0x818b, 0x8215, 0x8497, 0x84a4, 0x84a1, 0x849f,
	0x84ba, 0x84ce, 0x84c2, 0x84ac, 0x84ae, 0x84ab, 0x84b9, 0x84b4,
	0x84c1, 0x84cd, 0x84aa, 0x849a, 0x84b1, 0x84d0, 0x849d, 0x84a7,
	0x84bb, 0x84a2, 0x8494, 0x84c7, 0x84cc, 0x849b, 0x84a9, 0x84af,
	0x84a8, 0x84d6, 0x8498, 0x84b6, 0x84cf, 0x84a0, 0x84d7, 0x84d4,
	0x84d2, 0x84db, 0x84b0, 0x8491, 0x8661, 0x8733, 0x8723, 0x8728,
	0x876b, 0x8740, 0x872e, 0x871e, 0x8721, 0x8719, 0x871b, 0x8743,
	0x872c, 0x8741, 0x873e, 0x8746, 0x8720, 0x8732, 0x872a, 0x872d,
	0x873c, 0x8712, 0x873a, 0x8731, 0x8735, 0x8742, 0x8726, 0x8727,
	0x8738, 0x8724, 0x871a, 0x8730, 0x8711, 0x88f7, 0x88e7, 0x88f1,
	0x88f2, 0x88fa, 0x88fe, 0x88ee, 0x88fc, 0x88f6, 0x88fb
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe440[ 63 ] = {
	0x88f0, 0x88ec, 0x88eb, 0x899d, 0x89a1, 0x899f, 0x899e, 0x89e9,
	0x89eb, 0x89e8, 0x8aab, 0x8a99, 0x8a8b, 0x8a92, 0x8a8f, 0x8a96,
	0x8c3d, 0x8c68, 0x8c69, 0x8cd5, 0x8ccf, 0x8cd7, 0x8d96, 0x8e09,
	0x8e02, 0x8dff, 0x8e0d, 0x8dfd, 0x8e0a, 0x8e03, 0x8e07, 0x8e06,
	0x8e05, 0x8dfe, 0x8e00, 0x8e04, 0x8f10, 0x8f11, 0x8f0e, 0x8f0d,
	0x9123, 0x911c, 0x9120, 0x9122, 0x911f, 0x911d, 0x911a, 0x9124,
	0x9121, 0x911b, 0x917a, 0x9172, 0x9179, 0x9173, 0x92a5, 0x92a4,
	0x9276, 0x929b, 0x927a, 0x92a0, 0x9294, 0x92aa, 0x928d
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe4a1[ 94 ] = {
	        0x92a6, 0x929a, 0x92ab, 0x9279, 0x9297, 0x927f, 0x92a3,
	0x92ee, 0x928e, 0x9282, 0x9295, 0x92a2, 0x927d, 0x9288, 0x92a1,
	0x928a, 0x9286, 0x928c, 0x9299, 0x92a7, 0x927e, 0x9287, 0x92a9,
	0x929d, 0x928b, 0x922d, 0x969e, 0x96a1, 0x96ff, 0x9758, 0x977d,
	0x977a, 0x977e, 0x9783, 0x9780, 0x9782, 0x977b, 0x9784, 0x9781,
	0x977f, 0x97ce, 0x97cd, 0x9816, 0x98ad, 0x98ae, 0x9902, 0x9900,
	0x9907, 0x999d, 0x999c, 0x99c3, 0x99b9, 0x99bb, 0x99ba, 0x99c2,
	0x99bd, 0x99c7, 0x9ab1, 0x9ae3, 0x9ae7, 0x9b3e, 0x9b3f, 0x9b60,
	0x9b61, 0x9b5f, 0x9cf1, 0x9cf2, 0x9cf5, 0x9ea7, 0x50ff, 0x5103,
	0x5130, 0x50f8, 0x5106, 0x5107, 0x50f6, 0x50fe, 0x510b, 0x510c,
	0x50fd, 0x510a, 0x528b, 0x528c, 0x52f1, 0x52ef, 0x5648, 0x5642,
	0x564c, 0x5635, 0x5641, 0x564a, 0x5649, 0x5646, 0x5658
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe540[ 63 ] = {
	0x565a, 0x5640, 0x5633, 0x563d, 0x562c, 0x563e, 0x5638, 0x562a,
	0x563a, 0x571a, 0x58ab, 0x589d, 0x58b1, 0x58a0, 0x58a3, 0x58af,
	0x58ac, 0x58a5, 0x58a1, 0x58ff, 0x5aff, 0x5af4, 0x5afd, 0x5af7,
	0x5af6, 0x5b03, 0x5af8, 0x5b02, 0x5af9, 0x5b01, 0x5b07, 0x5b05,
	0x5b0f, 0x5c67, 0x5d99, 0x5d97, 0x5d9f, 0x5d92, 0x5da2, 0x5d93,
	0x5d95, 0x5da0, 0x5d9c, 0x5da1, 0x5d9a, 0x5d9e, 0x5e69, 0x5e5d,
	0x5e60, 0x5e5c, 0x7df3, 0x5edb, 0x5ede, 0x5ee1, 0x5f49, 0x5fb2,
	0x618b, 0x6183, 0x6179, 0x61b1, 0x61b0, 0x61a2, 0x6189
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe5a1[ 94 ] = {
	        0x619b, 0x6193, 0x61af, 0x61ad, 0x619f, 0x6192, 0x61aa,
	0x61a1, 0x618d, 0x6166, 0x61b3, 0x622d, 0x646e, 0x6470, 0x6496,
	0x64a0, 0x6485, 0x6497, 0x649c, 0x648f, 0x648b, 0x648a, 0x648c,
	0x64a3, 0x649f, 0x6468, 0x64b1, 0x6498, 0x6576, 0x657a, 0x6579,
	0x657b, 0x65b2, 0x65b3, 0x66b5, 0x66b0, 0x66a9, 0x66b2, 0x66b7,
	0x66aa, 0x66af, 0x6a00, 0x6a06, 0x6a17, 0x69e5, 0x69f8, 0x6a15,
	0x69f1, 0x69e4, 0x6a20, 0x69ff, 0x69ec, 0x69e2, 0x6a1b, 0x6a1d,
	0x69fe, 0x6a27, 0x69f2, 0x69ee, 0x6a14, 0x69f7, 0x69e7, 0x6a40,
	0x6a08, 0x69e6, 0x69fb, 0x6a0d, 0x69fc, 0x69eb, 0x6a09, 0x6a04,
	0x6a18, 0x6a25, 0x6a0f, 0x69f6, 0x6a26, 0x6a07, 0x69f4, 0x6a16,
	0x6b51, 0x6ba5, 0x6ba3, 0x6ba2, 0x6ba6, 0x6c01, 0x6c00, 0x6bff,
	0x6c02, 0x6f41, 0x6f26, 0x6f7e, 0x6f87, 0x6fc6, 0x6f92
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe640[ 63 ] = {
	0x6f8d, 0x6f89, 0x6f8c, 0x6f62, 0x6f4f, 0x6f85, 0x6f5a, 0x6f96,
	0x6f76, 0x6f6c, 0x6f82, 0x6f55, 0x6f72, 0x6f52, 0x6f50, 0x6f57,
	0x6f94, 0x6f93, 0x6f5d, 0x6f00, 0x6f61, 0x6f6b, 0x6f7d, 0x6f67,
	0x6f90, 0x6f53, 0x6f8b, 0x6f69, 0x6f7f, 0x6f95, 0x6f63, 0x6f77,
	0x6f6a, 0x6f7b, 0x71b2, 0x71af, 0x719b, 0x71b0, 0x71a0, 0x719a,
	0x71a9, 0x71b5, 0x719d, 0x71a5, 0x719e, 0x71a4, 0x71a1, 0x71aa,
	0x719c, 0x71a7, 0x71b3, 0x7298, 0x729a, 0x7358, 0x7352, 0x735e,
	0x735f, 0x7360, 0x735d, 0x735b, 0x7361, 0x735a, 0x7359
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe6a1[ 94 ] = {
	        0x7362, 0x7487, 0x7489, 0x748a, 0x7486, 0x7481, 0x747d,
	0x7485, 0x7488, 0x747c, 0x7479, 0x7508, 0x7507, 0x757e, 0x7625,
	0x761e, 0x7619, 0x761d, 0x761c, 0x7623, 0x761a, 0x7628, 0x761b,
	0x769c, 0x769d, 0x769e, 0x769b, 0x778d, 0x778f, 0x7789, 0x7788,
	0x78cd, 0x78bb, 0x78cf, 0x78cc, 0x78d1, 0x78ce, 0x78d4, 0x78c8,
	0x78c3, 0x78c4, 0x78c9, 0x799a, 0x79a1, 0x79a0, 0x799c, 0x79a2,
	0x799b, 0x6b76, 0x7a39, 0x7ab2, 0x7ab4, 0x7ab3, 0x7bb7, 0x7bcb,
	0x7bbe, 0x7bac, 0x7bce, 0x7baf, 0x7bb9, 0x7bca, 0x7bb5, 0x7cc5,
	0x7cc8, 0x7ccc, 0x7ccb, 0x7df7, 0x7ddb, 0x7dea, 0x7de7, 0x7dd7,
	0x7de1, 0x7e03, 0x7dfa, 0x7de6, 0x7df6, 0x7df1, 0x7df0, 0x7dee,
	0x7ddf, 0x7f76, 0x7fac, 0x7fb0, 0x7fad, 0x7fed, 0x7feb, 0x7fea,
	0x7fec, 0x7fe6, 0x7fe8, 0x8064, 0x8067, 0x81a3, 0x819f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe740[ 63 ] = {
	0x819e, 0x8195, 0x81a2, 0x8199, 0x8197, 0x8216, 0x824f, 0x8253,
	0x8252, 0x8250, 0x824e, 0x8251, 0x8524, 0x853b, 0x850f, 0x8500,
	0x8529, 0x850e, 0x8509, 0x850d, 0x851f, 0x850a, 0x8527, 0x851c,
	0x84fb, 0x852b, 0x84fa, 0x8508, 0x850c, 0x84f4, 0x852a, 0x84f2,
	0x8515, 0x84f7, 0x84eb, 0x84f3, 0x84fc, 0x8512, 0x84ea, 0x84e9,
	0x8516, 0x84fe, 0x8528, 0x851d, 0x852e, 0x8502, 0x84fd, 0x851e,
	0x84f6, 0x8531, 0x8526, 0x84e7, 0x84e8, 0x84f0, 0x84ef, 0x84f9,
	0x8518, 0x8520, 0x8530, 0x850b, 0x8519, 0x852f, 0x8662
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe7a1[ 94 ] = {
	        0x8756, 0x8763, 0x8764, 0x8777, 0x87e1, 0x8773, 0x8758,
	0x8754, 0x875b, 0x8752, 0x8761, 0x875a, 0x8751, 0x875e, 0x876d,
	0x876a, 0x8750, 0x874e, 0x875f, 0x875d, 0x876f, 0x876c, 0x877a,
	0x876e, 0x875c, 0x8765, 0x874f, 0x877b, 0x8775, 0x8762, 0x8767,
	0x8769, 0x885a, 0x8905, 0x890c, 0x8914, 0x890b, 0x8917, 0x8918,
	0x8919, 0x8906, 0x8916, 0x8911, 0x890e, 0x8909, 0x89a2, 0x89a4,
	0x89a3, 0x89ed, 0x89f0, 0x89ec, 0x8acf, 0x8ac6, 0x8ab8, 0x8ad3,
	0x8ad1, 0x8ad4, 0x8ad5, 0x8abb, 0x8ad7, 0x8abe, 0x8ac0, 0x8ac5,
	0x8ad8, 0x8ac3, 0x8aba, 0x8abd, 0x8ad9, 0x8c3e, 0x8c4d, 0x8c8f,
	0x8ce5, 0x8cdf, 0x8cd9, 0x8ce8, 0x8cda, 0x8cdd, 0x8ce7, 0x8da0,
	0x8d9c, 0x8da1, 0x8d9b, 0x8e20, 0x8e23, 0x8e25, 0x8e24, 0x8e2e,
	0x8e15, 0x8e1b, 0x8e16, 0x8e11, 0x8e19, 0x8e26, 0x8e27
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe840[ 63 ] = {
	0x8e14, 0x8e12, 0x8e18, 0x8e13, 0x8e1c, 0x8e17, 0x8e1a, 0x8f2c,
	0x8f24, 0x8f18, 0x8f1a, 0x8f20, 0x8f23, 0x8f16, 0x8f17, 0x9073,
	0x9070, 0x906f, 0x9067, 0x906b, 0x912f, 0x912b, 0x9129, 0x912a,
	0x9132, 0x9126, 0x912e, 0x9185, 0x9186, 0x918a, 0x9181, 0x9182,
	0x9184, 0x9180, 0x92d0, 0x92c3, 0x92c4, 0x92c0, 0x92d9, 0x92b6,
	0x92cf, 0x92f1, 0x92df, 0x92d8, 0x92e9, 0x92d7, 0x92dd, 0x92cc,
	0x92ef, 0x92c2, 0x92e8, 0x92ca, 0x92c8, 0x92ce, 0x92e6, 0x92cd,
	0x92d5, 0x92c9, 0x92e0, 0x92de, 0x92e7, 0x92d1, 0x92d3
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe8a1[ 94 ] = {
	        0x92b5, 0x92e1, 0x92c6, 0x92b4, 0x957c, 0x95ac, 0x95ab,
	0x95ae, 0x95b0, 0x96a4, 0x96a2, 0x96d3, 0x9705, 0x9708, 0x9702,
	0x975a, 0x978a, 0x978e, 0x9788, 0x97d0, 0x97cf, 0x981e, 0x981d,
	0x9826, 0x9829, 0x9828, 0x9820, 0x981b, 0x9827, 0x98b2, 0x9908,
	0x98fa, 0x9911, 0x9914, 0x9916, 0x9917, 0x9915, 0x99dc, 0x99cd,
	0x99cf, 0x99d3, 0x99d4, 0x99ce, 0x99c9, 0x99d6, 0x99d8, 0x99cb,
	0x99d7, 0x99cc, 0x9ab3, 0x9aec, 0x9aeb, 0x9af3, 0x9af2, 0x9af1,
	0x9b46, 0x9b43, 0x9b67, 0x9b74, 0x9b71, 0x9b66, 0x9b76, 0x9b75,
	0x9b70, 0x9b68, 0x9b64, 0x9b6c, 0x9cfc, 0x9cfa, 0x9cfd, 0x9cff,
	0x9cf7, 0x9d07, 0x9d00, 0x9cf9, 0x9cfb, 0x9d08, 0x9d05, 0x9d04,
	0x9e83, 0x9ed3, 0x9f0f, 0x9f10, 0x511c, 0x5113, 0x5117, 0x511a,
	0x5111, 0x51de, 0x5334, 0x53e1, 0x5670, 0x5660, 0x566e
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe940[ 63 ] = {
	0x5673, 0x5666, 0x5663, 0x566d, 0x5672, 0x565e, 0x5677, 0x571c,
	0x571b, 0x58c8, 0x58bd, 0x58c9, 0x58bf, 0x58ba, 0x58c2, 0x58bc,
	0x58c6, 0x5b17, 0x5b19, 0x5b1b, 0x5b21, 0x5b14, 0x5b13, 0x5b10,
	0x5b16, 0x5b28, 0x5b1a, 0x5b20, 0x5b1e, 0x5bef, 0x5dac, 0x5db1,
	0x5da9, 0x5da7, 0x5db5, 0x5db0, 0x5dae, 0x5daa, 0x5da8, 0x5db2,
	0x5dad, 0x5daf, 0x5db4, 0x5e67, 0x5e68, 0x5e66, 0x5e6f, 0x5ee9,
	0x5ee7, 0x5ee6, 0x5ee8, 0x5ee5, 0x5f4b, 0x5fbc, 0x619d, 0x61a8,
	0x6196, 0x61c5, 0x61b4, 0x61c6, 0x61c1, 0x61cc, 0x61ba
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe9a1[ 94 ] = {
	        0x61bf, 0x61b8, 0x618c, 0x64d7, 0x64d6, 0x64d0, 0x64cf,
	0x64c9, 0x64bd, 0x6489, 0x64c3, 0x64db, 0x64f3, 0x64d9, 0x6533,
	0x657f, 0x657c, 0x65a2, 0x66c8, 0x66be, 0x66c0, 0x66ca, 0x66cb,
	0x66cf, 0x66bd, 0x66bb, 0x66ba, 0x66cc, 0x6723, 0x6a34, 0x6a66,
	0x6a49, 0x6a67, 0x6a32, 0x6a68, 0x6a3e, 0x6a5d, 0x6a6d, 0x6a76,
	0x6a5b, 0x6a51, 0x6a28, 0x6a5a, 0x6a3b, 0x6a3f, 0x6a41, 0x6a6a,
	0x6a64, 0x6a50, 0x6a4f, 0x6a54, 0x6a6f, 0x6a69, 0x6a60, 0x6a3c,
	0x6a5e, 0x6a56, 0x6a55, 0x6a4d, 0x6a4e, 0x6a46, 0x6b55, 0x6b54,
	0x6b56, 0x6ba7, 0x6baa, 0x6bab, 0x6bc8, 0x6bc7, 0x6c04, 0x6c03,
	0x6c06, 0x6fad, 0x6fcb, 0x6fa3, 0x6fc7, 0x6fbc, 0x6fce, 0x6fc8,
	0x6f5e, 0x6fc4, 0x6fbd, 0x6f9e, 0x6fca, 0x6fa8, 0x7004, 0x6fa5,
	0x6fae, 0x6fba, 0x6fac, 0x6faa, 0x6fcf, 0x6fbf, 0x6fb8
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xea40[ 63 ] = {
	0x6fa2, 0x6fc9, 0x6fab, 0x6fcd, 0x6faf, 0x6fb2, 0x6fb0, 0x71c5,
	0x71c2, 0x71bf, 0x71b8, 0x71d6, 0x71c0, 0x71c1, 0x71cb, 0x71d4,
	0x71ca, 0x71c7, 0x71cf, 0x71bd, 0x71d8, 0x71bc, 0x71c6, 0x71da,
	0x71db, 0x729d, 0x729e, 0x7369, 0x7366, 0x7367, 0x736c, 0x7365,
	0x736b, 0x736a, 0x747f, 0x749a, 0x74a0, 0x7494, 0x7492, 0x7495,
	0x74a1, 0x750b, 0x7580, 0x762f, 0x762d, 0x7631, 0x763d, 0x7633,
	0x763c, 0x7635, 0x7632, 0x7630, 0x76bb, 0x76e6, 0x779a, 0x779d,
	0x77a1, 0x779c, 0x779b, 0x77a2, 0x77a3, 0x7795, 0x7799
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeaa1[ 94 ] = {
	        0x7797, 0x78dd, 0x78e9, 0x78e5, 0x78ea, 0x78de, 0x78e3,
	0x78db, 0x78e1, 0x78e2, 0x78ed, 0x78df, 0x78e0, 0x79a4, 0x7a44,
	0x7a48, 0x7a47, 0x7ab6, 0x7ab8, 0x7ab5, 0x7ab1, 0x7ab7, 0x7bde,
	0x7be3, 0x7be7, 0x7bdd, 0x7bd5, 0x7be5, 0x7bda, 0x7be8, 0x7bf9,
	0x7bd4, 0x7bea, 0x7be2, 0x7bdc, 0x7beb, 0x7bd8, 0x7bdf, 0x7cd2,
	0x7cd4, 0x7cd7, 0x7cd0, 0x7cd1, 0x7e12, 0x7e21, 0x7e17, 0x7e0c,
	0x7e1f, 0x7e20, 0x7e13, 0x7e0e, 0x7e1c, 0x7e15, 0x7e1a, 0x7e22,
	0x7e0b, 0x7e0f, 0x7e16, 0x7e0d, 0x7e14, 0x7e25, 0x7e24, 0x7f43,
	0x7f7b, 0x7f7c, 0x7f7a, 0x7fb1, 0x7fef, 0x802a, 0x8029, 0x806c,
	0x81b1, 0x81a6, 0x81ae, 0x81b9, 0x81b5, 0x81ab, 0x81b0, 0x81ac,
	0x81b4, 0x81b2, 0x81b7, 0x81a7, 0x81f2, 0x8255, 0x8256, 0x8257,
	0x8556, 0x8545, 0x856b, 0x854d, 0x8553, 0x8561, 0x8558
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeb40[ 63 ] = {
	0x8540, 0x8546, 0x8564, 0x8541, 0x8562, 0x8544, 0x8551, 0x8547,
	0x8563, 0x853e, 0x855b, 0x8571, 0x854e, 0x856e, 0x8575, 0x8555,
	0x8567, 0x8560, 0x858c, 0x8566, 0x855d, 0x8554, 0x8565, 0x856c,
	0x8663, 0x8665, 0x8664, 0x879b, 0x878f, 0x8797, 0x8793, 0x8792,
	0x8788, 0x8781, 0x8796, 0x8798, 0x8779, 0x8787, 0x87a3, 0x8785,
	0x8790, 0x8791, 0x879d, 0x8784, 0x8794, 0x879c, 0x879a, 0x8789,
	0x891e, 0x8926, 0x8930, 0x892d, 0x892e, 0x8927, 0x8931, 0x8922,
	0x8929, 0x8923, 0x892f, 0x892c, 0x891f, 0x89f1, 0x8ae0
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeba1[ 94 ] = {
	        0x8ae2, 0x8af2, 0x8af4, 0x8af5, 0x8add, 0x8b14, 0x8ae4,
	0x8adf, 0x8af0, 0x8ac8, 0x8ade, 0x8ae1, 0x8ae8, 0x8aff, 0x8aef,
	0x8afb, 0x8c91, 0x8c92, 0x8c90, 0x8cf5, 0x8cee, 0x8cf1, 0x8cf0,
	0x8cf3, 0x8d6c, 0x8d6e, 0x8da5, 0x8da7, 0x8e33, 0x8e3e, 0x8e38,
	0x8e40, 0x8e45, 0x8e36, 0x8e3c, 0x8e3d, 0x8e41, 0x8e30, 0x8e3f,
	0x8ebd, 0x8f36, 0x8f2e, 0x8f35, 0x8f32, 0x8f39, 0x8f37, 0x8f34,
	0x9076, 0x9079, 0x907b, 0x9086, 0x90fa, 0x9133, 0x9135, 0x9136,
	0x9193, 0x9190, 0x9191, 0x918d, 0x918f, 0x9327, 0x931e, 0x9308,
	0x931f, 0x9306, 0x930f, 0x937a, 0x9338, 0x933c, 0x931b, 0x9323,
	0x9312, 0x9301, 0x9346, 0x932d, 0x930e, 0x930d, 0x92cb, 0x931d,
	0x92fa, 0x9325, 0x9313, 0x92f9, 0x92f7, 0x9334, 0x9302, 0x9324,
	0x92ff, 0x9329, 0x9339, 0x9335, 0x932a, 0x9314, 0x930c
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xec40[ 63 ] = {
	0x930b, 0x92fe, 0x9309, 0x9300, 0x92fb, 0x9316, 0x95bc, 0x95cd,
	0x95be, 0x95b9, 0x95ba, 0x95b6, 0x95bf, 0x95b5, 0x95bd, 0x96a9,
	0x96d4, 0x970b, 0x9712, 0x9710, 0x9799, 0x9797, 0x9794, 0x97f0,
	0x97f8, 0x9835, 0x982f, 0x9832, 0x9924, 0x991f, 0x9927, 0x9929,
	0x999e, 0x99ee, 0x99ec, 0x99e5, 0x99e4, 0x99f0, 0x99e3, 0x99ea,
	0x99e9, 0x99e7, 0x9ab9, 0x9abf, 0x9ab4, 0x9abb, 0x9af6, 0x9afa,
	0x9af9, 0x9af7, 0x9b33, 0x9b80, 0x9b85, 0x9b87, 0x9b7c, 0x9b7e,
	0x9b7b, 0x9b82, 0x9b93, 0x9b92, 0x9b90, 0x9b7a, 0x9b95
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeca1[ 94 ] = {
	        0x9b7d, 0x9b88, 0x9d25, 0x9d17, 0x9d20, 0x9d1e, 0x9d14,
	0x9d29, 0x9d1d, 0x9d18, 0x9d22, 0x9d10, 0x9d19, 0x9d1f, 0x9e88,
	0x9e86, 0x9e87, 0x9eae, 0x9ead, 0x9ed5, 0x9ed6, 0x9efa, 0x9f12,
	0x9f3d, 0x5126, 0x5125, 0x5122, 0x5124, 0x5120, 0x5129, 0x52f4,
	0x5693, 0x568c, 0x568d, 0x5686, 0x5684, 0x5683, 0x567e, 0x5682,
	0x567f, 0x5681, 0x58d6, 0x58d4, 0x58cf, 0x58d2, 0x5b2d, 0x5b25,
	0x5b32, 0x5b23, 0x5b2c, 0x5b27, 0x5b26, 0x5b2f, 0x5b2e, 0x5b7b,
	0x5bf1, 0x5bf2, 0x5db7, 0x5e6c, 0x5e6a, 0x5fbe, 0x5fbb, 0x61c3,
	0x61b5, 0x61bc, 0x61e7, 0x61e0, 0x61e5, 0x61e4, 0x61e8, 0x61de,
	0x64ef, 0x64e9, 0x64e3, 0x64eb, 0x64e4, 0x64e8, 0x6581, 0x6580,
	0x65b6, 0x65da, 0x66d2, 0x6a8d, 0x6a96, 0x6a81, 0x6aa5, 0x6a89,
	0x6a9f, 0x6a9b, 0x6aa1, 0x6a9e, 0x6a87, 0x6a93, 0x6a8e
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xed40[ 63 ] = {
	0x6a95, 0x6a83, 0x6aa8, 0x6aa4, 0x6a91, 0x6a7f, 0x6aa6, 0x6a9a,
	0x6a85, 0x6a8c, 0x6a92, 0x6b5b, 0x6bad, 0x6c09, 0x6fcc, 0x6fa9,
	0x6ff4, 0x6fd4, 0x6fe3, 0x6fdc, 0x6fed, 0x6fe7, 0x6fe6, 0x6fde,
	0x6ff2, 0x6fdd, 0x6fe2, 0x6fe8, 0x71e1, 0x71f1, 0x71e8, 0x71f2,
	0x71e4, 0x71f0, 0x71e2, 0x7373, 0x736e, 0x736f, 0x7497, 0x74b2,
	0x74ab, 0x7490, 0x74aa, 0x74ad, 0x74b1, 0x74a5, 0x74af, 0x7510,
	0x7511, 0x7512, 0x750f, 0x7584, 0x7643, 0x7648, 0x7649, 0x7647,
	0x76a4, 0x76e9, 0x77b5, 0x77ab, 0x77b2, 0x77b7, 0x77b6
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeda1[ 94 ] = {
	        0x77b4, 0x77b1, 0x77a8, 0x77f0, 0x78f3, 0x78fd, 0x7902,
	0x78fb, 0x78fc, 0x78f2, 0x7905, 0x78f9, 0x78fe, 0x7904, 0x79ab,
	0x79a8, 0x7a5c, 0x7a5b, 0x7a56, 0x7a58, 0x7a54, 0x7a5a, 0x7abe,
	0x7ac0, 0x7ac1, 0x7c05, 0x7c0f, 0x7bf2, 0x7c00, 0x7bff, 0x7bfb,
	0x7c0e, 0x7bf4, 0x7c0b, 0x7bf3, 0x7c02, 0x7c09, 0x7c03, 0x7c01,
	0x7bf8, 0x7bfd, 0x7c06, 0x7bf0, 0x7bf1, 0x7c10, 0x7c0a, 0x7ce8,
	0x7e2d, 0x7e3c, 0x7e42, 0x7e33, 0x9848, 0x7e38, 0x7e2a, 0x7e49,
	0x7e40, 0x7e47, 0x7e29, 0x7e4c, 0x7e30, 0x7e3b, 0x7e36, 0x7e44,
	0x7e3a, 0x7f45, 0x7f7f, 0x7f7e, 0x7f7d, 0x7ff4, 0x7ff2, 0x802c,
	0x81bb, 0x81c4, 0x81cc, 0x81ca, 0x81c5, 0x81c7, 0x81bc, 0x81e9,
	0x825b, 0x825a, 0x825c, 0x8583, 0x8580, 0x858f, 0x85a7, 0x8595,
	0x85a0, 0x858b, 0x85a3, 0x857b, 0x85a4, 0x859a, 0x859e
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xee40[ 63 ] = {
	0x8577, 0x857c, 0x8589, 0x85a1, 0x857a, 0x8578, 0x8557, 0x858e,
	0x8596, 0x8586, 0x858d, 0x8599, 0x859d, 0x8581, 0x85a2, 0x8582,
	0x8588, 0x8585, 0x8579, 0x8576, 0x8598, 0x8590, 0x859f, 0x8668,
	0x87be, 0x87aa, 0x87ad, 0x87c5, 0x87b0, 0x87ac, 0x87b9, 0x87b5,
	0x87bc, 0x87ae, 0x87c9, 0x87c3, 0x87c2, 0x87cc, 0x87b7, 0x87af,
	0x87c4, 0x87ca, 0x87b4, 0x87b6, 0x87bf, 0x87b8, 0x87bd, 0x87de,
	0x87b2, 0x8935, 0x8933, 0x893c, 0x893e, 0x8941, 0x8952, 0x8937,
	0x8942, 0x89ad, 0x89af, 0x89ae, 0x89f2, 0x89f3, 0x8b1e
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeea1[ 94 ] = {
	        0x8b18, 0x8b16, 0x8b11, 0x8b05, 0x8b0b, 0x8b22, 0x8b0f,
	0x8b12, 0x8b15, 0x8b07, 0x8b0d, 0x8b08, 0x8b06, 0x8b1c, 0x8b13,
	0x8b1a, 0x8c4f, 0x8c70, 0x8c72, 0x8c71, 0x8c6f, 0x8c95, 0x8c94,
	0x8cf9, 0x8d6f, 0x8e4e, 0x8e4d, 0x8e53, 0x8e50, 0x8e4c, 0x8e47,
	0x8f43, 0x8f40, 0x9085, 0x907e, 0x9138, 0x919a, 0x91a2, 0x919b,
	0x9199, 0x919f, 0x91a1, 0x919d, 0x91a0, 0x93a1, 0x9383, 0x93af,
	0x9364, 0x9356, 0x9347, 0x937c, 0x9358, 0x935c, 0x9376, 0x9349,
	0x9350, 0x9351, 0x9360, 0x936d, 0x938f, 0x934c, 0x936a, 0x9379,
	0x9357, 0x9355, 0x9352, 0x934f, 0x9371, 0x9377, 0x937b, 0x9361,
	0x935e, 0x9363, 0x9367, 0x9380, 0x934e, 0x9359, 0x95c7, 0x95c0,
	0x95c9, 0x95c3, 0x95c5, 0x95b7, 0x96ae, 0x96b0, 0x96ac, 0x9720,
	0x971f, 0x9718, 0x971d, 0x9719, 0x979a, 0x97a1, 0x979c
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xef40[ 63 ] = {
	0x979e, 0x979d, 0x97d5, 0x97d4, 0x97f1, 0x9841, 0x9844, 0x984a,
	0x9849, 0x9845, 0x9843, 0x9925, 0x992b, 0x992c, 0x992a, 0x9933,
	0x9932, 0x992f, 0x992d, 0x9931, 0x9930, 0x9998, 0x99a3, 0x99a1,
	0x9a02, 0x99fa, 0x99f4, 0x99f7, 0x99f9, 0x99f8, 0x99f6, 0x99fb,
	0x99fd, 0x99fe, 0x99fc, 0x9a03, 0x9abe, 0x9afe, 0x9afd, 0x9b01,
	0x9afc, 0x9b48, 0x9b9a, 0x9ba8, 0x9b9e, 0x9b9b, 0x9ba6, 0x9ba1,
	0x9ba5, 0x9ba4, 0x9b86, 0x9ba2, 0x9ba0, 0x9baf, 0x9d33, 0x9d41,
	0x9d67, 0x9d36, 0x9d2e, 0x9d2f, 0x9d31, 0x9d38, 0x9d30
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xefa1[ 94 ] = {
	        0x9d45, 0x9d42, 0x9d43, 0x9d3e, 0x9d37, 0x9d40, 0x9d3d,
	0x7ff5, 0x9d2d, 0x9e8a, 0x9e89, 0x9e8d, 0x9eb0, 0x9ec8, 0x9eda,
	0x9efb, 0x9eff, 0x9f24, 0x9f23, 0x9f22, 0x9f54, 0x9fa0, 0x5131,
	0x512d, 0x512e, 0x5698, 0x569c, 0x5697, 0x569a, 0x569d, 0x5699,
	0x5970, 0x5b3c, 0x5c69, 0x5c6a, 0x5dc0, 0x5e6d, 0x5e6e, 0x61d8,
	0x61df, 0x61ed, 0x61ee, 0x61f1, 0x61ea, 0x61f0, 0x61eb, 0x61d6,
	0x61e9, 0x64ff, 0x6504, 0x64fd, 0x64f8, 0x6501, 0x6503, 0x64fc,
	0x6594, 0x65db, 0x66da, 0x66db, 0x66d8, 0x6ac5, 0x6ab9, 0x6abd,
	0x6ae1, 0x6ac6, 0x6aba, 0x6ab6, 0x6ab7, 0x6ac7, 0x6ab4, 0x6aad,
	0x6b5e, 0x6bc9, 0x6c0b, 0x7007, 0x700c, 0x700d, 0x7001, 0x7005,
	0x7014, 0x700e, 0x6fff, 0x7000, 0x6ffb, 0x7026, 0x6ffc, 0x6ff7,
	0x700a, 0x7201, 0x71ff, 0x71f9, 0x7203, 0x71fd, 0x7376
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf040[ 63 ] = {
	0x74b8, 0x74c0, 0x74b5, 0x74c1, 0x74be, 0x74b6, 0x74bb, 0x74c2,
	0x7514, 0x7513, 0x765c, 0x7664, 0x7659, 0x7650, 0x7653, 0x7657,
	0x765a, 0x76a6, 0x76bd, 0x76ec, 0x77c2, 0x77ba, 0x78ff, 0x790c,
	0x7913, 0x7914, 0x7909, 0x7910, 0x7912, 0x7911, 0x79ad, 0x79ac,
	0x7a5f, 0x7c1c, 0x7c29, 0x7c19, 0x7c20, 0x7c1f, 0x7c2d, 0x7c1d,
	0x7c26, 0x7c28, 0x7c22, 0x7c25, 0x7c30, 0x7e5c, 0x7e50, 0x7e56,
	0x7e63, 0x7e58, 0x7e62, 0x7e5f, 0x7e51, 0x7e60, 0x7e57, 0x7e53,
	0x7fb5, 0x7fb3, 0x7ff7, 0x7ff8, 0x8075, 0x81d1, 0x81d2
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf0a1[ 94 ] = {
	        0x81d0, 0x825f, 0x825e, 0x85b4, 0x85c6, 0x85c0, 0x85c3,
	0x85c2, 0x85b3, 0x85b5, 0x85bd, 0x85c7, 0x85c4, 0x85bf, 0x85cb,
	0x85ce, 0x85c8, 0x85c5, 0x85b1, 0x85b6, 0x85d2, 0x8624, 0x85b8,
	0x85b7, 0x85be, 0x8669, 0x87e7, 0x87e6, 0x87e2, 0x87db, 0x87eb,
	0x87ea, 0x87e5, 0x87df, 0x87f3, 0x87e4, 0x87d4, 0x87dc, 0x87d3,
	0x87ed, 0x87d8, 0x87e3, 0x87a4, 0x87d7, 0x87d9, 0x8801, 0x87f4,
	0x87e8, 0x87dd, 0x8953, 0x894b, 0x894f, 0x894c, 0x8946, 0x8950,
	0x8951, 0x8949, 0x8b2a, 0x8b27, 0x8b23, 0x8b33, 0x8b30, 0x8b35,
	0x8b47, 0x8b2f, 0x8b3c, 0x8b3e, 0x8b31, 0x8b25, 0x8b37, 0x8b26,
	0x8b36, 0x8b2e, 0x8b24, 0x8b3b, 0x8b3d, 0x8b3a, 0x8c42, 0x8c75,
	0x8c99, 0x8c98, 0x8c97, 0x8cfe, 0x8d04, 0x8d02, 0x8d00, 0x8e5c,
	0x8e62, 0x8e60, 0x8e57, 0x8e56, 0x8e5e, 0x8e65, 0x8e67
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf140[ 63 ] = {
	0x8e5b, 0x8e5a, 0x8e61, 0x8e5d, 0x8e69, 0x8e54, 0x8f46, 0x8f47,
	0x8f48, 0x8f4b, 0x9128, 0x913a, 0x913b, 0x913e, 0x91a8, 0x91a5,
	0x91a7, 0x91af, 0x91aa, 0x93b5, 0x938c, 0x9392, 0x93b7, 0x939b,
	0x939d, 0x9389, 0x93a7, 0x938e, 0x93aa, 0x939e, 0x93a6, 0x9395,
	0x9388, 0x9399, 0x939f, 0x938d, 0x93b1, 0x9391, 0x93b2, 0x93a4,
	0x93a8, 0x93b4, 0x93a3, 0x93a5, 0x95d2, 0x95d3, 0x95d1, 0x96b3,
	0x96d7, 0x96da, 0x5dc2, 0x96df, 0x96d8, 0x96dd, 0x9723, 0x9722,
	0x9725, 0x97ac, 0x97ae, 0x97a8, 0x97ab, 0x97a4, 0x97aa
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf1a1[ 94 ] = {
	        0x97a2, 0x97a5, 0x97d7, 0x97d9, 0x97d6, 0x97d8, 0x97fa,
	0x9850, 0x9851, 0x9852, 0x98b8, 0x9941, 0x993c, 0x993a, 0x9a0f,
	0x9a0b, 0x9a09, 0x9a0d, 0x9a04, 0x9a11, 0x9a0a, 0x9a05, 0x9a07,
	0x9a06, 0x9ac0, 0x9adc, 0x9b08, 0x9b04, 0x9b05, 0x9b29, 0x9b35,
	0x9b4a, 0x9b4c, 0x9b4b, 0x9bc7, 0x9bc6, 0x9bc3, 0x9bbf, 0x9bc1,
	0x9bb5, 0x9bb8, 0x9bd3, 0x9bb6, 0x9bc4, 0x9bb9, 0x9bbd, 0x9d5c,
	0x9d53, 0x9d4f, 0x9d4a, 0x9d5b, 0x9d4b, 0x9d59, 0x9d56, 0x9d4c,
	0x9d57, 0x9d52, 0x9d54, 0x9d5f, 0x9d58, 0x9d5a, 0x9e8e, 0x9e8c,
	0x9edf, 0x9f01, 0x9f00, 0x9f16, 0x9f25, 0x9f2b, 0x9f2a, 0x9f29,
	0x9f28, 0x9f4c, 0x9f55, 0x5134, 0x5135, 0x5296, 0x52f7, 0x53b4,
	0x56ab, 0x56ad, 0x56a6, 0x56a7, 0x56aa, 0x56ac, 0x58da, 0x58dd,
	0x58db, 0x5912, 0x5b3d, 0x5b3e, 0x5b3f, 0x5dc3, 0x5e70
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf240[ 63 ] = {
	0x5fbf, 0x61fb, 0x6507, 0x6510, 0x650d, 0x6509, 0x650c, 0x650e,
	0x6584, 0x65de, 0x65dd, 0x66de, 0x6ae7, 0x6ae0, 0x6acc, 0x6ad1,
	0x6ad9, 0x6acb, 0x6adf, 0x6adc, 0x6ad0, 0x6aeb, 0x6acf, 0x6acd,
	0x6ade, 0x6b60, 0x6bb0, 0x6c0c, 0x7019, 0x7027, 0x7020, 0x7016,
	0x702b, 0x7021, 0x7022, 0x7023, 0x7029, 0x7017, 0x7024, 0x701c,
	0x702a, 0x720c, 0x720a, 0x7207, 0x7202, 0x7205, 0x72a5, 0x72a6,
	0x72a4, 0x72a3, 0x72a1, 0x74cb, 0x74c5, 0x74b7, 0x74c3, 0x7516,
	0x7660, 0x77c9, 0x77ca, 0x77c4, 0x77f1, 0x791d, 0x791b
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf2a1[ 94 ] = {
	        0x7921, 0x791c, 0x7917, 0x791e, 0x79b0, 0x7a67, 0x7a68,
	0x7c33, 0x7c3c, 0x7c39, 0x7c2c, 0x7c3b, 0x7cec, 0x7cea, 0x7e76,
	0x7e75, 0x7e78, 0x7e70, 0x7e77, 0x7e6f, 0x7e7a, 0x7e72, 0x7e74,
	0x7e68, 0x7f4b, 0x7f4a, 0x7f83, 0x7f86, 0x7fb7, 0x7ffd, 0x7ffe,
	0x8078, 0x81d7, 0x81d5, 0x8264, 0x8261, 0x8263, 0x85eb, 0x85f1,
	0x85ed, 0x85d9, 0x85e1, 0x85e8, 0x85da, 0x85d7, 0x85ec, 0x85f2,
	0x85f8, 0x85d8, 0x85df, 0x85e3, 0x85dc, 0x85d1, 0x85f0, 0x85e6,
	0x85ef, 0x85de, 0x85e2, 0x8800, 0x87fa, 0x8803, 0x87f6, 0x87f7,
	0x8809, 0x880c, 0x880b, 0x8806, 0x87fc, 0x8808, 0x87ff, 0x880a,
	0x8802, 0x8962, 0x895a, 0x895b, 0x8957, 0x8961, 0x895c, 0x8958,
	0x895d, 0x8959, 0x8988, 0x89b7, 0x89b6, 0x89f6, 0x8b50, 0x8b48,
	0x8b4a, 0x8b40, 0x8b53, 0x8b56, 0x8b54, 0x8b4b, 0x8b55
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf340[ 63 ] = {
	0x8b51, 0x8b42, 0x8b52, 0x8b57, 0x8c43, 0x8c77, 0x8c76, 0x8c9a,
	0x8d06, 0x8d07, 0x8d09, 0x8dac, 0x8daa, 0x8dad, 0x8dab, 0x8e6d,
	0x8e78, 0x8e73, 0x8e6a, 0x8e6f, 0x8e7b, 0x8ec2, 0x8f52, 0x8f51,
	0x8f4f, 0x8f50, 0x8f53, 0x8fb4, 0x9140, 0x913f, 0x91b0, 0x91ad,
	0x93de, 0x93c7, 0x93cf, 0x93c2, 0x93da, 0x93d0, 0x93f9, 0x93ec,
	0x93cc, 0x93d9, 0x93a9, 0x93e6, 0x93ca, 0x93d4, 0x93ee, 0x93e3,
	0x93d5, 0x93c4, 0x93ce, 0x93c0, 0x93d2, 0x93e7, 0x957d, 0x95da,
	0x95db, 0x96e1, 0x9729, 0x972b, 0x972c, 0x9728, 0x9726
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf3a1[ 94 ] = {
	        0x97b3, 0x97b7, 0x97b6, 0x97dd, 0x97de, 0x97df, 0x985c,
	0x9859, 0x985d, 0x9857, 0x98bf, 0x98bd, 0x98bb, 0x98be, 0x9948,
	0x9947, 0x9943, 0x99a6, 0x99a7, 0x9a1a, 0x9a15, 0x9a25, 0x9a1d,
	0x9a24, 0x9a1b, 0x9a22, 0x9a20, 0x9a27, 0x9a23, 0x9a1e, 0x9a1c,
	0x9a14, 0x9ac2, 0x9b0b, 0x9b0a, 0x9b0e, 0x9b0c, 0x9b37, 0x9bea,
	0x9beb, 0x9be0, 0x9bde, 0x9be4, 0x9be6, 0x9be2, 0x9bf0, 0x9bd4,
	0x9bd7, 0x9bec, 0x9bdc, 0x9bd9, 0x9be5, 0x9bd5, 0x9be1, 0x9bda,
	0x9d77, 0x9d81, 0x9d8a, 0x9d84, 0x9d88, 0x9d71, 0x9d80, 0x9d78,
	0x9d86, 0x9d8b, 0x9d8c, 0x9d7d, 0x9d6b, 0x9d74, 0x9d75, 0x9d70,
	0x9d69, 0x9d85, 0x9d73, 0x9d7b, 0x9d82, 0x9d6f, 0x9d79, 0x9d7f,
	0x9d87, 0x9d68, 0x9e94, 0x9e91, 0x9ec0, 0x9efc, 0x9f2d, 0x9f40,
	0x9f41, 0x9f4d, 0x9f56, 0x9f57, 0x9f58, 0x5337, 0x56b2
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf440[ 63 ] = {
	0x56b5, 0x56b3, 0x58e3, 0x5b45, 0x5dc6, 0x5dc7, 0x5eee, 0x5eef,
	0x5fc0, 0x5fc1, 0x61f9, 0x6517, 0x6516, 0x6515, 0x6513, 0x65df,
	0x66e8, 0x66e3, 0x66e4, 0x6af3, 0x6af0, 0x6aea, 0x6ae8, 0x6af9,
	0x6af1, 0x6aee, 0x6aef, 0x703c, 0x7035, 0x702f, 0x7037, 0x7034,
	0x7031, 0x7042, 0x7038, 0x703f, 0x703a, 0x7039, 0x7040, 0x703b,
	0x7033, 0x7041, 0x7213, 0x7214, 0x72a8, 0x737d, 0x737c, 0x74ba,
	0x76ab, 0x76aa, 0x76be, 0x76ed, 0x77cc, 0x77ce, 0x77cf, 0x77cd,
	0x77f2, 0x7925, 0x7923, 0x7927, 0x7928, 0x7924, 0x7929
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf4a1[ 94 ] = {
	        0x79b2, 0x7a6e, 0x7a6c, 0x7a6d, 0x7af7, 0x7c49, 0x7c48,
	0x7c4a, 0x7c47, 0x7c45, 0x7cee, 0x7e7b, 0x7e7e, 0x7e81, 0x7e80,
	0x7fba, 0x7fff, 0x8079, 0x81db, 0x81d9, 0x820b, 0x8268, 0x8269,
	0x8622, 0x85ff, 0x8601, 0x85fe, 0x861b, 0x8600, 0x85f6, 0x8604,
	0x8609, 0x8605, 0x860c, 0x85fd, 0x8819, 0x8810, 0x8811, 0x8817,
	0x8813, 0x8816, 0x8963, 0x8966, 0x89b9, 0x89f7, 0x8b60, 0x8b6a,
	0x8b5d, 0x8b68, 0x8b63, 0x8b65, 0x8b67, 0x8b6d, 0x8dae, 0x8e86,
	0x8e88, 0x8e84, 0x8f59, 0x8f56, 0x8f57, 0x8f55, 0x8f58, 0x8f5a,
	0x908d, 0x9143, 0x9141, 0x91b7, 0x91b5, 0x91b2, 0x91b3, 0x940b,
	0x9413, 0x93fb, 0x9420, 0x940f, 0x9414, 0x93fe, 0x9415, 0x9410,
	0x9428, 0x9419, 0x940d, 0x93f5, 0x9400, 0x93f7, 0x9407, 0x940e,
	0x9416, 0x9412, 0x93fa, 0x9409, 0x93f8, 0x940a, 0x93ff
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf540[ 63 ] = {
	0x93fc, 0x940c, 0x93f6, 0x9411, 0x9406, 0x95de, 0x95e0, 0x95df,
	0x972e, 0x972f, 0x97b9, 0x97bb, 0x97fd, 0x97fe, 0x9860, 0x9862,
	0x9863, 0x985f, 0x98c1, 0x98c2, 0x9950, 0x994e, 0x9959, 0x994c,
	0x994b, 0x9953, 0x9a32, 0x9a34, 0x9a31, 0x9a2c, 0x9a2a, 0x9a36,
	0x9a29, 0x9a2e, 0x9a38, 0x9a2d, 0x9ac7, 0x9aca, 0x9ac6, 0x9b10,
	0x9b12, 0x9b11, 0x9c0b, 0x9c08, 0x9bf7, 0x9c05, 0x9c12, 0x9bf8,
	0x9c40, 0x9c07, 0x9c0e, 0x9c06, 0x9c17, 0x9c14, 0x9c09, 0x9d9f,
	0x9d99, 0x9da4, 0x9d9d, 0x9d92, 0x9d98, 0x9d90, 0x9d9b
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf5a1[ 94 ] = {
	        0x9da0, 0x9d94, 0x9d9c, 0x9daa, 0x9d97, 0x9da1, 0x9d9a,
	0x9da2, 0x9da8, 0x9d9e, 0x9da3, 0x9dbf, 0x9da9, 0x9d96, 0x9da6,
	0x9da7, 0x9e99, 0x9e9b, 0x9e9a, 0x9ee5, 0x9ee4, 0x9ee7, 0x9ee6,
	0x9f30, 0x9f2e, 0x9f5b, 0x9f60, 0x9f5e, 0x9f5d, 0x9f59, 0x9f91,
	0x513a, 0x5139, 0x5298, 0x5297, 0x56c3, 0x56bd, 0x56be, 0x5b48,
	0x5b47, 0x5dcb, 0x5dcf, 0x5ef1, 0x61fd, 0x651b, 0x6b02, 0x6afc,
	0x6b03, 0x6af8, 0x6b00, 0x7043, 0x7044, 0x704a, 0x7048, 0x7049,
	0x7045, 0x7046, 0x721d, 0x721a, 0x7219, 0x737e, 0x7517, 0x766a,
	0x77d0, 0x792d, 0x7931, 0x792f, 0x7c54, 0x7c53, 0x7cf2, 0x7e8a,
	0x7e87, 0x7e88, 0x7e8b, 0x7e86, 0x7e8d, 0x7f4d, 0x7fbb, 0x8030,
	0x81dd, 0x8618, 0x862a, 0x8626, 0x861f, 0x8623, 0x861c, 0x8619,
	0x8627, 0x862e, 0x8621, 0x8620, 0x8629, 0x861e, 0x8625
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf640[ 63 ] = {
	0x8829, 0x881d, 0x881b, 0x8820, 0x8824, 0x881c, 0x882b, 0x884a,
	0x896d, 0x8969, 0x896e, 0x896b, 0x89fa, 0x8b79, 0x8b78, 0x8b45,
	0x8b7a, 0x8b7b, 0x8d10, 0x8d14, 0x8daf, 0x8e8e, 0x8e8c, 0x8f5e,
	0x8f5b, 0x8f5d, 0x9146, 0x9144, 0x9145, 0x91b9, 0x943f, 0x943b,
	0x9436, 0x9429, 0x943d, 0x943c, 0x9430, 0x9439, 0x942a, 0x9437,
	0x942c, 0x9440, 0x9431, 0x95e5, 0x95e4, 0x95e3, 0x9735, 0x973a,
	0x97bf, 0x97e1, 0x9864, 0x98c9, 0x98c6, 0x98c0, 0x9958, 0x9956,
	0x9a39, 0x9a3d, 0x9a46, 0x9a44, 0x9a42, 0x9a41, 0x9a3a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf6a1[ 94 ] = {
	        0x9a3f, 0x9acd, 0x9b15, 0x9b17, 0x9b18, 0x9b16, 0x9b3a,
	0x9b52, 0x9c2b, 0x9c1d, 0x9c1c, 0x9c2c, 0x9c23, 0x9c28, 0x9c29,
	0x9c24, 0x9c21, 0x9db7, 0x9db6, 0x9dbc, 0x9dc1, 0x9dc7, 0x9dca,
	0x9dcf, 0x9dbe, 0x9dc5, 0x9dc3, 0x9dbb, 0x9db5, 0x9dce, 0x9db9,
	0x9dba, 0x9dac, 0x9dc8, 0x9db1, 0x9dad, 0x9dcc, 0x9db3, 0x9dcd,
	0x9db2, 0x9e7a, 0x9e9c, 0x9eeb, 0x9eee, 0x9eed, 0x9f1b, 0x9f18,
	0x9f1a, 0x9f31, 0x9f4e, 0x9f65, 0x9f64, 0x9f92, 0x4eb9, 0x56c6,
	0x56c5, 0x56cb, 0x5971, 0x5b4b, 0x5b4c, 0x5dd5, 0x5dd1, 0x5ef2,
	0x6521, 0x6520, 0x6526, 0x6522, 0x6b0b, 0x6b08, 0x6b09, 0x6c0d,
	0x7055, 0x7056, 0x7057, 0x7052, 0x721e, 0x721f, 0x72a9, 0x737f,
	0x74d8, 0x74d5, 0x74d9, 0x74d7, 0x766d, 0x76ad, 0x7935, 0x79b4,
	0x7a70, 0x7a71, 0x7c57, 0x7c5c, 0x7c59, 0x7c5b, 0x7c5a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf740[ 63 ] = {
	0x7cf4, 0x7cf1, 0x7e91, 0x7f4f, 0x7f87, 0x81de, 0x826b, 0x8634,
	0x8635, 0x8633, 0x862c, 0x8632, 0x8636, 0x882c, 0x8828, 0x8826,
	0x882a, 0x8825, 0x8971, 0x89bf, 0x89be, 0x89fb, 0x8b7e, 0x8b84,
	0x8b82, 0x8b86, 0x8b85, 0x8b7f, 0x8d15, 0x8e95, 0x8e94, 0x8e9a,
	0x8e92, 0x8e90, 0x8e96, 0x8e97, 0x8f60, 0x8f62, 0x9147, 0x944c,
	0x9450, 0x944a, 0x944b, 0x944f, 0x9447, 0x9445, 0x9448, 0x9449,
	0x9446, 0x973f, 0x97e3, 0x986a, 0x9869, 0x98cb, 0x9954, 0x995b,
	0x9a4e, 0x9a53, 0x9a54, 0x9a4c, 0x9a4f, 0x9a48, 0x9a4a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf7a1[ 94 ] = {
	        0x9a49, 0x9a52, 0x9a50, 0x9ad0, 0x9b19, 0x9b2b, 0x9b3b,
	0x9b56, 0x9b55, 0x9c46, 0x9c48, 0x9c3f, 0x9c44, 0x9c39, 0x9c33,
	0x9c41, 0x9c3c, 0x9c37, 0x9c34, 0x9c32, 0x9c3d, 0x9c36, 0x9ddb,
	0x9dd2, 0x9dde, 0x9dda, 0x9dcb, 0x9dd0, 0x9ddc, 0x9dd1, 0x9ddf,
	0x9de9, 0x9dd9, 0x9dd8, 0x9dd6, 0x9df5, 0x9dd5, 0x9ddd, 0x9eb6,
	0x9ef0, 0x9f35, 0x9f33, 0x9f32, 0x9f42, 0x9f6b, 0x9f95, 0x9fa2,
	0x513d, 0x5299, 0x58e8, 0x58e7, 0x5972, 0x5b4d, 0x5dd8, 0x882f,
	0x5f4f, 0x6201, 0x6203, 0x6204, 0x6529, 0x6525, 0x6596, 0x66eb,
	0x6b11, 0x6b12, 0x6b0f, 0x6bca, 0x705b, 0x705a, 0x7222, 0x7382,
	0x7381, 0x7383, 0x7670, 0x77d4, 0x7c67, 0x7c66, 0x7e95, 0x826c,
	0x863a, 0x8640, 0x8639, 0x863c, 0x8631, 0x863b, 0x863e, 0x8830,
	0x8832, 0x882e, 0x8833, 0x8976, 0x8974, 0x8973, 0x89fe
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf840[ 63 ] = {
	0x8b8c, 0x8b8e, 0x8b8b, 0x8b88, 0x8c45, 0x8d19, 0x8e98, 0x8f64,
	0x8f63, 0x91bc, 0x9462, 0x9455, 0x945d, 0x9457, 0x945e, 0x97c4,
	0x97c5, 0x9800, 0x9a56, 0x9a59, 0x9b1e, 0x9b1f, 0x9b20, 0x9c52,
	0x9c58, 0x9c50, 0x9c4a, 0x9c4d, 0x9c4b, 0x9c55, 0x9c59, 0x9c4c,
	0x9c4e, 0x9dfb, 0x9df7, 0x9def, 0x9de3, 0x9deb, 0x9df8, 0x9de4,
	0x9df6, 0x9de1, 0x9dee, 0x9de6, 0x9df2, 0x9df0, 0x9de2, 0x9dec,
	0x9df4, 0x9df3, 0x9de8, 0x9ded, 0x9ec2, 0x9ed0, 0x9ef2, 0x9ef3,
	0x9f06, 0x9f1c, 0x9f38, 0x9f37, 0x9f36, 0x9f43, 0x9f4f
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf8a1[ 94 ] = {
	        0x9f71, 0x9f70, 0x9f6e, 0x9f6f, 0x56d3, 0x56cd, 0x5b4e,
	0x5c6d, 0x652d, 0x66ed, 0x66ee, 0x6b13, 0x705f, 0x7061, 0x705d,
	0x7060, 0x7223, 0x74db, 0x74e5, 0x77d5, 0x7938, 0x79b7, 0x79b6,
	0x7c6a, 0x7e97, 0x7f89, 0x826d, 0x8643, 0x8838, 0x8837, 0x8835,
	0x884b, 0x8b94, 0x8b95, 0x8e9e, 0x8e9f, 0x8ea0, 0x8e9d, 0x91be,
	0x91bd, 0x91c2, 0x946b, 0x9468, 0x9469, 0x96e5, 0x9746, 0x9743,
	0x9747, 0x97c7, 0x97e5, 0x9a5e, 0x9ad5, 0x9b59, 0x9c63, 0x9c67,
	0x9c66, 0x9c62, 0x9c5e, 0x9c60, 0x9e02, 0x9dfe, 0x9e07, 0x9e03,
	0x9e06, 0x9e05, 0x9e00, 0x9e01, 0x9e09, 0x9dff, 0x9dfd, 0x9e04,
	0x9ea0, 0x9f1e, 0x9f46, 0x9f74, 0x9f75, 0x9f76, 0x56d4, 0x652e,
	0x65b8, 0x6b18, 0x6b19, 0x6b17, 0x6b1a, 0x7062, 0x7226, 0x72aa,
	0x77d8, 0x77d9, 0x7939, 0x7c69, 0x7c6b, 0x7cf6, 0x7e9a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf940[ 63 ] = {
	0x7e98, 0x7e9b, 0x7e99, 0x81e0, 0x81e1, 0x8646, 0x8647, 0x8648,
	0x8979, 0x897a, 0x897c, 0x897b, 0x89ff, 0x8b98, 0x8b99, 0x8ea5,
	0x8ea4, 0x8ea3, 0x946e, 0x946d, 0x946f, 0x9471, 0x9473, 0x9749,
	0x9872, 0x995f, 0x9c68, 0x9c6e, 0x9c6d, 0x9e0b, 0x9e0d, 0x9e10,
	0x9e0f, 0x9e12, 0x9e11, 0x9ea1, 0x9ef5, 0x9f09, 0x9f47, 0x9f78,
	0x9f7b, 0x9f7a, 0x9f79, 0x571e, 0x7066, 0x7c6f, 0x883c, 0x8db2,
	0x8ea6, 0x91c3, 0x9474, 0x9478, 0x9476, 0x9475, 0x9a60, 0x9c74,
	0x9c73, 0x9c71, 0x9c75, 0x9e14, 0x9e13, 0x9ef6, 0x9f0a
};

const uint16_t libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf9a1[ 94 ] = {
	        0x9fa4, 0x7068, 0x7065, 0x7cf7, 0x866a, 0x883e, 0x883d,
	0x883f, 0x8b9e, 0x8c9c, 0x8ea9, 0x8ec9, 0x974b, 0x9873, 0x9874,
	0x98cc, 0x9961, 0x99ab, 0x9a64, 0x9a66, 0x9a67, 0x9b24, 0x9e15,
	0x9e17, 0x9f48, 0x6207, 0x6b1e, 0x7227, 0x864c, 0x8ea8, 0x9482,
	0x9480, 0x9481, 0x9a69, 0x9a68, 0x9b2e, 0x9e19, 0x7229, 0x864b,
	0x8b9f, 0x9483, 0x9c79, 0x9eb7, 0x7675, 0x9a6b, 0x9c7a, 0x9e1d,
	0x7069, 0x706a, 0x9ea4, 0x9f7e, 0x9f49, 0x9f98, 0x7881, 0x92b9,
	0x88cf, 0x58bb, 0x6052, 0x7ca7, 0x5afa, 0x2554, 0x2566, 0x2557,
	0x2560, 0x256c, 0x2563, 0x255a, 0x2569, 0x255d, 0x2552, 0x2564,
	0x2555, 0x255e, 0x256a, 0x2561, 0x2558, 0x2567, 0x255b, 0x2553,
	0x2565, 0x2556, 0x255f, 0x256b, 0x2562, 0x2559, 0x2568, 0x255c,
	0x2551, 0x2550, 0x256d, 0x256e, 0x2570, 0x256f, 0x2593, 
};

/* Unicode to ASCII character lookup table for Windows 950 codepage
 * Unknown are filled with the ASCII replacement character 0x1a
 */

const uint16_t libuna_codepage_windows_950_unicode_to_byte_stream_base_0x0080[ 128 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb1a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc2a1,
	0x58a2, 0xd3a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x50a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd1a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd2a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_950_unicode_to_byte_stream_base_0x02c0[ 320 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbea3,
	0x001a, 0xbca3, 0xbda3, 0xbfa3, 0x001a, 0xc5a1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xbba3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x44a3, 0x45a3, 0x46a3, 0x47a3, 0x48a3, 0x49a3, 0x4aa3,
	0x4ba3, 0x4ca3, 0x4da3, 0x4ea3, 0x4fa3, 0x50a3, 0x51a3, 0x52a3,
	0x53a3, 0x54a3, 0x001a, 0x55a3, 0x56a3, 0x57a3, 0x58a3, 0x59a3,
	0x5aa3, 0x5ba3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5ca3, 0x5da3, 0x5ea3, 0x5fa3, 0x60a3, 0x61a3, 0x62a3,
	0x63a3, 0x64a3, 0x65a3, 0x66a3, 0x67a3, 0x68a3, 0x69a3, 0x6aa3,
	0x6ba3, 0x6ca3, 0x001a, 0x6da3, 0x6ea3, 0x6fa3, 0x70a3, 0x71a3,
	0x72a3, 0x73a3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_950_unicode_to_byte_stream_base_0x2000[ 768 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x56a1, 0x58a1, 0x001a, 0x001a, 0x001a,
	0xa5a1, 0xa6a1, 0x001a, 0x001a, 0xa7a1, 0xa8a1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ca1, 0x4ba1, 0x45a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xaca1, 0x001a, 0x001a, 0xaba1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb0a1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe1a3, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4aa2, 0x001a, 0xc1a1, 0x001a, 0x001a,
	0x001a, 0x4ba2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb9a2, 0xbaa2, 0xbba2, 0xbca2, 0xbda2, 0xbea2, 0xbfa2, 0xc0a2,
	0xc1a2, 0xc2a2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf6a1, 0xf4a1, 0xf7a1, 0xf5a1, 0x001a, 0x001a, 0xf8a1, 0xf9a1,
	0xfba1, 0xfaa1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x41a2, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd4a1, 0x001a, 0x001a, 0x001a, 0xdba1, 0xe8a1,
	0xe7a1, 0x001a, 0x001a, 0xfda1, 0x001a, 0xfca1, 0x001a, 0x001a,
	0x001a, 0xe4a1, 0xe5a1, 0xeca1, 0x001a, 0x001a, 0xeda1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xefa1, 0xeea1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xdca1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xdaa1, 0xdda1, 0x001a, 0x001a, 0x001a, 0x001a, 0xd8a1, 0xd9a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf2a1, 0x001a, 0x001a,
	0x001a, 0xf3a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe6a1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe9a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_950_unicode_to_byte_stream_base_0x2500[ 384 ] = {
	0x77a2, 0x001a, 0x78a2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x7aa2, 0x001a, 0x001a, 0x001a,
	0x7ba2, 0x001a, 0x001a, 0x001a, 0x7ca2, 0x001a, 0x001a, 0x001a,
	0x7da2, 0x001a, 0x001a, 0x001a, 0x75a2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x74a2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x73a2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x72a2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x71a2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa4a2, 0xf8f9, 0xe6f9, 0xeff9, 0xddf9, 0xe8f9, 0xf1f9, 0xdff9,
	0xecf9, 0xf5f9, 0xe3f9, 0xeef9, 0xf7f9, 0xe5f9, 0xa5a2, 0xf2f9,
	0xe0f9, 0xa7a2, 0xf4f9, 0xe2f9, 0xe7f9, 0xf0f9, 0xdef9, 0xedf9,
	0xf6f9, 0xe4f9, 0xa6a2, 0xf3f9, 0xe1f9, 0x7ea2, 0xa1a2, 0xa3a2,
	0xa2a2, 0xaca2, 0xada2, 0xaea2, 0x5aa1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x62a2, 0x63a2, 0x64a2, 0x65a2, 0x66a2, 0x67a2, 0x68a2,
	0x69a2, 0x70a2, 0x6fa2, 0x6ea2, 0x6da2, 0x6ca2, 0x6ba2, 0x6aa2,
	0x001a, 0x001a, 0x001a, 0xfef9, 0x76a2, 0x79a2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbda1, 0xbca1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb6a1, 0xb5a1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbfa1, 0xbea1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbba1, 0xbaa1,
	0x001a, 0x001a, 0x001a, 0xb3a1, 0x001a, 0x001a, 0xb7a1, 0xb4a1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa8a2, 0xa9a2, 0xaba2, 0xaaa2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb9a1, 0xb8a1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf0a1, 0x001a, 0xf1a1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_950_unicode_to_byte_stream_base_0x3000[ 1024 ] = {
	0x40a1, 0x42a1, 0x43a1, 0xb2a1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x71a1, 0x72a1, 0x6da1, 0x6ea1, 0x75a1, 0x76a1, 0x79a1, 0x7aa1,
	0x69a1, 0x6aa1, 0x45a2, 0x001a, 0x65a1, 0x66a1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa9a1, 0xaaa1, 0x001a,
	0x001a, 0xc3a2, 0xc4a2, 0xc5a2, 0xc6a2, 0xc7a2, 0xc8a2, 0xc9a2,
	0xcaa2, 0xcba2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x74a3, 0x75a3, 0x76a3,
	0x77a3, 0x78a3, 0x79a3, 0x7aa3, 0x7ba3, 0x7ca3, 0x7da3, 0x7ea3,
	0xa1a3, 0xa2a3, 0xa3a3, 0xa4a3, 0xa5a3, 0xa6a3, 0xa7a3, 0xa8a3,
	0xa9a3, 0xaaa3, 0xaba3, 0xaca3, 0xada3, 0xaea3, 0xafa3, 0xb0a3,
	0xb1a3, 0xb2a3, 0xb3a3, 0xb4a3, 0xb5a3, 0xb6a3, 0xb7a3, 0xb8a3,
	0xb9a3, 0xbaa3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc0a1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x55a2, 0x56a2,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x50a2, 0x51a2, 0x52a2, 0x001a,
	0x001a, 0x54a2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x57a2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x53a2, 0x001a,
	0x001a, 0xeba1, 0xeaa1, 0x001a, 0x001a, 0x4fa2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
};

const uint16_t libuna_codepage_windows_950_unicode_to_byte_stream_base_0x4e00[ 20928 ] = {
	0x40a4, 0x42a4, 0x001a, 0x43a4, 0x001a, 0x001a, 0x001a, 0x45c9,
	0x56a4, 0x54a4, 0x57a4, 0x55a4, 0x46c9, 0xa3a4, 0x4fc9, 0x4dc9,
	0xa2a4, 0xa1a4, 0x001a, 0x001a, 0x42a5, 0x41a5, 0x40a5, 0x001a,
	0x43a5, 0xfea4, 0x001a, 0x001a, 0x001a, 0x001a, 0xe0a5, 0xe1a5,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc3a8, 0x001a,
	0x001a, 0x001a, 0x001a, 0x58a4, 0x001a, 0xa4a4, 0x50c9, 0x001a,
	0xa5a4, 0x63c9, 0xeaa6, 0xb1cb, 0x001a, 0x001a, 0x001a, 0x001a,
	0x59a4, 0xa6a4, 0x001a, 0x44a5, 0x64c9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x40c9, 0x44a4, 0x001a, 0x5ba4, 0x001a, 0x47c9,
	0x5ca4, 0x001a, 0x001a, 0xa7a4, 0x001a, 0x45a5, 0x47a5, 0x46a5,
	0x001a, 0x001a, 0xe2a5, 0xe3a5, 0x001a, 0x001a, 0xc4a8, 0x001a,
	0xbcad, 0x41a4, 0x001a, 0x001a, 0x41c9, 0x45a4, 0x5ea4, 0x5da4,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xe4a5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc5a8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xaeb0, 0x4bd4,
	0x001a, 0x001a, 0xc3b6, 0xb1dc, 0xb2dc, 0x001a, 0x46a4, 0x001a,
	0xa9a4, 0x001a, 0x001a, 0xc6a8, 0x47a4, 0x48c9, 0x5fa4, 0x001a,
	0x001a, 0xaaa4, 0xaca4, 0x51c9, 0xada4, 0xaba4, 0x001a, 0x001a,
	0x001a, 0xe5a5, 0x001a, 0xc7a8, 0x001a, 0x001a, 0xc8a8, 0x45ab,
	0x001a, 0x60a4, 0xaea4, 0x001a, 0xe6a5, 0xe8a5, 0xe7a5, 0x001a,
	0xeba6, 0x001a, 0x001a, 0xc9a8, 0xcaa8, 0x46ab, 0x47ab, 0x001a,
	0x001a, 0x001a, 0x001a, 0xbdad, 0x001a, 0x001a, 0xb3dc, 0x001a,
	0x001a, 0xd6f6, 0x48a4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb0a4, 0xafa4, 0x52c9, 0xb1a4, 0xb7a4, 0x001a, 0xb2a4, 0xb3a4,
	0x54c9, 0x53c9, 0xb5a4, 0xb6a4, 0x001a, 0xb4a4, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4aa5, 0x4ba5, 0x4ca5, 0x4da5,
	0x49a5, 0x50a5, 0x6ac9, 0x001a, 0x66c9, 0x69c9, 0x51a5, 0x61a5,
	0x001a, 0x68c9, 0x001a, 0x4ea5, 0x4fa5, 0x48a5, 0x001a, 0x001a,
	0x65c9, 0x67c9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf5a5, 0xb0c9, 0xf2a5, 0xf6a5, 0xbac9, 0xaec9, 0xf3a5, 0xb2c9,
	0x001a, 0x001a, 0x001a, 0xf4a5, 0x001a, 0xf7a5, 0x001a, 0xe9a5,
	0xb1c9, 0xf8a5, 0xb5c9, 0x001a, 0xb9c9, 0xb6c9, 0x001a, 0x001a,
	0xb3c9, 0xeaa5, 0xeca5, 0xf9a5, 0x001a, 0xeea5, 0xabc9, 0xf1a5,
	0xefa5, 0xf0a5, 0xbbc9, 0xb8c9, 0xafc9, 0xeda5, 0x001a, 0x001a,
	0xacc9, 0xeba5, 0x001a, 0x001a, 0x001a, 0xb4c9, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb7c9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xadc9, 0x66ca, 0x001a, 0x42a7,
	0xf4a6, 0x001a, 0x001a, 0x67ca, 0xf1a6, 0x001a, 0x44a7, 0x001a,
	0xf9a6, 0x001a, 0xf8a6, 0x5bca, 0xfca6, 0xf7a6, 0x60ca, 0x68ca,
	0x001a, 0x64ca, 0x001a, 0xfaa6, 0x001a, 0x001a, 0xfda6, 0xeea6,
	0x47a7, 0x5dca, 0x001a, 0x001a, 0xbdcb, 0xeca6, 0x43a7, 0xeda6,
	0xf5a6, 0xf6a6, 0x62ca, 0x5eca, 0xfba6, 0xf3a6, 0x5aca, 0xefa6,
	0x65ca, 0x45a7, 0x48a7, 0xf2a6, 0x40a7, 0x46a7, 0xf0a6, 0x63ca,
	0x41a7, 0x69ca, 0x5cca, 0xfea6, 0x5fca, 0x001a, 0x001a, 0x61ca,
	0x001a, 0xd8a8, 0xbfcb, 0xcbcb, 0xd0a8, 0x001a, 0xcccb, 0xcba8,
	0xd5a8, 0x001a, 0x001a, 0xcea8, 0xb9cb, 0xd6a8, 0xb8cb, 0xbccb,
	0xc3cb, 0xc1cb, 0xdea8, 0xd9a8, 0xb3cb, 0xb5cb, 0xdba8, 0xcfa8,
	0xb6cb, 0xc2cb, 0xc9cb, 0xd4a8, 0xbbcb, 0xb4cb, 0xd3a8, 0xb7cb,
	0xd7a8, 0xbacb, 0x001a, 0xd2a8, 0x001a, 0xcda8, 0x001a, 0xdca8,
	0xc4cb, 0xdda8, 0xc8cb, 0x001a, 0xc6cb, 0xcacb, 0xdaa8, 0xbecb,
	0xb2cb, 0x001a, 0xc0cb, 0xd1a8, 0xc5cb, 0xcca8, 0xc7cb, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x56ab, 0x4aab,
	0x001a, 0x001a, 0xe0cd, 0xe8cd, 0x001a, 0x49ab, 0x51ab, 0x5dab,
	0x001a, 0xeecd, 0xeccd, 0xe7cd, 0x001a, 0x001a, 0x001a, 0x4bab,
	0xedcd, 0xe3cd, 0x59ab, 0x50ab, 0x58ab, 0xdecd, 0x001a, 0xeacd,
	0x001a, 0xe1cd, 0x54ab, 0xe2cd, 0x001a, 0xddcd, 0x5bab, 0x4eab,
	0x57ab, 0x4dab, 0x001a, 0xdfcd, 0xe4cd, 0x001a, 0xebcd, 0x55ab,
	0x52ab, 0xe6cd, 0x5aab, 0xe9cd, 0xe5cd, 0x4fab, 0x5cab, 0x53ab,
	0x4cab, 0x48ab, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xefcd, 0x001a, 0xd7ad, 0xc1ad,
	0x001a, 0xd1ad, 0x001a, 0xd6ad, 0xd0d0, 0xcfd0, 0xd4d0, 0xd5d0,
	0xc4ad, 0x001a, 0xcdad, 0x001a, 0x001a, 0x001a, 0xdaad, 0x001a,
	0xcead, 0x001a, 0x001a, 0x001a, 0x001a, 0xc9d0, 0xc7ad, 0xcad0,
	0x001a, 0xdcad, 0x001a, 0xd3ad, 0xbead, 0xbfad, 0xddd0, 0xbfb0,
	0x001a, 0xccad, 0xcbad, 0xcbd0, 0xcfad, 0x5bd4, 0xc6ad, 0xd6d0,
	0xd5ad, 0xd4ad, 0xcaad, 0xced0, 0xd7d0, 0x001a, 0xc8d0, 0xc9ad,
	0xd8d0, 0xd2ad, 0xccd0, 0xc0ad, 0x001a, 0xc3ad, 0xc2ad, 0xd9d0,
	0xd0ad, 0xc5ad, 0xd9ad, 0xdbad, 0xd3d0, 0xd8ad, 0x001a, 0xdbd0,
	0xcdd0, 0xdcd0, 0x001a, 0xd1d0, 0x001a, 0xdad0, 0x001a, 0xd2d0,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc8ad, 0x001a, 0x001a, 0x001a,
	0x63d4, 0x57d4, 0x001a, 0xb3b0, 0x001a, 0x5cd4, 0x62d4, 0xb2b0,
	0x55d4, 0xb6b0, 0x59d4, 0x52d4, 0xb4b0, 0x56d4, 0xb9b0, 0xbeb0,
	0x001a, 0x67d4, 0x001a, 0x51d4, 0x001a, 0xbab0, 0x001a, 0x66d4,
	0x001a, 0x001a, 0xb5b0, 0x58d4, 0xb1b0, 0x53d4, 0x4fd4, 0x5dd4,
	0x50d4, 0x4ed4, 0x5ad4, 0x60d4, 0x61d4, 0xb7b0, 0x001a, 0x001a,
	0x5bd8, 0x5ed4, 0x4dd4, 0x5fd4, 0x001a, 0xc1b0, 0x64d4, 0xc0b0,
	0x4cd4, 0x001a, 0x54d4, 0x65d4, 0xbcb0, 0xbbb0, 0xb8b0, 0xbdb0,
	0x001a, 0x001a, 0xafb0, 0x001a, 0x001a, 0xb0b0, 0x001a, 0x001a,
	0xc8b3, 0x001a, 0x5ed8, 0x57d8, 0x001a, 0xc5b3, 0x001a, 0x5fd8,
	0x001a, 0x001a, 0x001a, 0x55d8, 0x58d8, 0xc4b3, 0x59d8, 0x001a,
	0x001a, 0xc7b3, 0x5dd8, 0x001a, 0x53d8, 0x52d8, 0xc9b3, 0x001a,
	0xcab3, 0xc6b3, 0xcbb3, 0x51d8, 0x5cd8, 0x5ad8, 0x54d8, 0x001a,
	0x001a, 0x001a, 0xc3b3, 0x56d8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xcab6, 0xc4b6, 0xb7dc, 0xcdb6,
	0xbddc, 0xc0dc, 0xc6b6, 0xc7b6, 0xbadc, 0xc5b6, 0xc3dc, 0xcbb6,
	0xc4dc, 0x001a, 0xbfdc, 0xccb6, 0x001a, 0xb4dc, 0xc9b6, 0xb5dc,
	0x001a, 0xbedc, 0xbcdc, 0x001a, 0xb8dc, 0xc8b6, 0xb6dc, 0xceb6,
	0xbbdc, 0xc2dc, 0xb9dc, 0xc1dc, 0x001a, 0x001a, 0xb6b9, 0xb3b9,
	0x001a, 0xb4b9, 0x001a, 0xf9e0, 0xf1e0, 0xb2b9, 0xafb9, 0xf2e0,
	0x001a, 0x001a, 0xb1b9, 0xf5e0, 0x001a, 0xf7e0, 0x001a, 0x001a,
	0xfee0, 0x001a, 0x001a, 0xfde0, 0xf8e0, 0xaeb9, 0xf0e0, 0xacb9,
	0xf3e0, 0xb7b9, 0xf6e0, 0x001a, 0xfae0, 0xb0b9, 0xadb9, 0xfce0,
	0xfbe0, 0xb5b9, 0x001a, 0xf4e0, 0x001a, 0xf8bb, 0xece4, 0x001a,
	0xe9e4, 0xf9bb, 0x001a, 0xf7bb, 0x001a, 0xf0e4, 0xede4, 0xe6e4,
	0xf6bb, 0x001a, 0xfabb, 0xe7e4, 0xf5bb, 0xfdbb, 0xeae4, 0xebe4,
	0xfbbb, 0xfcbb, 0xf1e4, 0xeee4, 0xefe4, 0x001a, 0x001a, 0x001a,
	0xaabe, 0xf8e8, 0xa7be, 0xf5e8, 0xa9be, 0xabbe, 0x001a, 0xf6e8,
	0xa8be, 0x001a, 0xf7e8, 0x001a, 0xf4e8, 0x001a, 0x001a, 0x76c0,
	0xbdec, 0x77c0, 0xbbec, 0x001a, 0xbcec, 0xbaec, 0xb9ec, 0x001a,
	0x001a, 0xbeec, 0x75c0, 0x001a, 0x001a, 0xb8ef, 0xb9ef, 0x001a,
	0xe8e4, 0xb7ef, 0x78c0, 0x5fc3, 0xebf1, 0xecf1, 0x001a, 0xd7c4,
	0xd8c4, 0xc1f5, 0xc0f5, 0x6cc5, 0x6bc5, 0xd0f7, 0x001a, 0x49a4,
	0x61a4, 0xb9a4, 0x001a, 0xb8a4, 0x53a5, 0x52a5, 0xfca5, 0xfba5,
	0xfda5, 0xfaa5, 0x001a, 0x4aa7, 0x49a7, 0x4ba7, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe0a8, 0x001a, 0xdfa8, 0xe1a8, 0x001a, 0x5eab,
	0x001a, 0x59a2, 0xded0, 0x5aa2, 0xc2b0, 0x5ca2, 0x5ba2, 0x60d8,
	0x001a, 0x5da2, 0xb8b9, 0x5ea2, 0x001a, 0x4aa4, 0x001a, 0xbaa4,
	0xfea5, 0xe2a8, 0x001a, 0x4ba4, 0xbda4, 0xbba4, 0xbca4, 0x001a,
	0x001a, 0x40a6, 0x001a, 0x001a, 0x001a, 0x4ca7, 0xe4a8, 0xe3a8,
	0xe5a8, 0x001a, 0x001a, 0x001a, 0xddad, 0x001a, 0x001a, 0x001a,
	0xacbe, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ec9,
	0x001a, 0x54a5, 0x55a5, 0x001a, 0x001a, 0x41a6, 0x001a, 0x6aca,
	0x001a, 0x60ab, 0x5fab, 0xe0d0, 0xdfd0, 0xc3b0, 0x001a, 0xbea4,
	0x55c9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcdcb, 0x001a,
	0x61ab, 0x001a, 0xe0ad, 0x001a, 0xdead, 0xdfad, 0x001a, 0x001a,
	0x001a, 0x001a, 0xadbe, 0x001a, 0x56a5, 0x001a, 0x001a, 0x001a,
	0x42a6, 0xbcc9, 0x001a, 0x001a, 0x001a, 0x001a, 0x4da7, 0x4ea7,
	0x001a, 0x6bca, 0x001a, 0x001a, 0xcecb, 0xe6a8, 0xcfcb, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe2d0, 0xe3d0, 0xe3ad, 0x001a,
	0xe4d0, 0x001a, 0xe1d0, 0xe4ad, 0xe2ad, 0xe1ad, 0xe5d0, 0x001a,
	0x68d4, 0x001a, 0x001a, 0x001a, 0x61d8, 0x001a, 0x001a, 0xc5dc,
	0x40e1, 0x001a, 0x001a, 0x001a, 0xfebb, 0xaebe, 0xf9e8, 0x001a,
	0x4ca4, 0x5aa4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc4b0, 0xcdb3, 0x001a, 0xb9b9, 0x001a, 0x42c9, 0xbfa4, 0x001a,
	0x59a5, 0x57a5, 0x58a5, 0x001a, 0x001a, 0xe7a8, 0x001a, 0x001a,
	0x4da4, 0x4ea4, 0x001a, 0x62a4, 0x001a, 0x001a, 0xc0a4, 0xc1a4,
	0xc2a4, 0xbec9, 0x5aa5, 0x001a, 0x6bc9, 0x001a, 0x46a6, 0x001a,
	0xbfc9, 0x44a6, 0x45a6, 0xbdc9, 0x001a, 0x001a, 0x47a6, 0x43a6,
	0x001a, 0x001a, 0x001a, 0x001a, 0x6cca, 0xecaa, 0x6dca, 0x001a,
	0x001a, 0x6eca, 0x001a, 0x001a, 0x50a7, 0x4fa7, 0x001a, 0x001a,
	0x53a7, 0x51a7, 0x52a7, 0x001a, 0x001a, 0x001a, 0xeda8, 0x001a,
	0xeca8, 0xd4cb, 0xd1cb, 0xd2cb, 0x001a, 0xd0cb, 0xeea8, 0xeaa8,
	0xe9a8, 0x001a, 0xeba8, 0xe8a8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xefa8, 0x001a, 0x63ab, 0xf0cd, 0x001a, 0xd3cb, 0x68ab,
	0x001a, 0xf1cd, 0x64ab, 0x67ab, 0x66ab, 0x65ab, 0x62ab, 0x001a,
	0x001a, 0x001a, 0xe8d0, 0x001a, 0xe7ad, 0xebd0, 0xe5ad, 0x001a,
	0x001a, 0x001a, 0xe7d0, 0xe8ad, 0xe6ad, 0xe9ad, 0xe9d0, 0xead0,
	0x001a, 0xe6d0, 0xecd0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd1b3, 0xc5b0, 0x69d4, 0x6bd4, 0x6ad4, 0x6cd4, 0xc6b0,
	0x001a, 0x001a, 0xceb3, 0x001a, 0xcfb3, 0xd0b3, 0x001a, 0xd0b6,
	0xc7dc, 0x001a, 0xc6dc, 0xc8dc, 0xc9dc, 0xd1b6, 0x001a, 0xcfb6,
	0x41e1, 0x42e1, 0xbbb9, 0xbab9, 0x5ae3, 0x001a, 0x001a, 0x40bc,
	0x41bc, 0x42bc, 0x44bc, 0xf2e4, 0xf3e4, 0x43bc, 0x001a, 0x001a,
	0x001a, 0xafbe, 0x001a, 0xb0be, 0x001a, 0x001a, 0xedf1, 0xc3f5,
	0xc2f5, 0xd1f7, 0x001a, 0x4fa4, 0x001a, 0x001a, 0x001a, 0x5ca5,
	0x5ba5, 0x001a, 0x001a, 0x48a6, 0x001a, 0x001a, 0xc0c9, 0x001a,
	0x001a, 0x55a7, 0x56a7, 0x54a7, 0x57a7, 0x6fca, 0x70ca, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xf1a8, 0xd5cb, 0x001a, 0xf0a8, 0x001a,
	0xf2cd, 0x6cab, 0xf3cd, 0x6bab, 0x001a, 0x001a, 0x001a, 0x69ab,
	0x001a, 0x6aab, 0x001a, 0x001a, 0x001a, 0xedd0, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc7b0, 0x6ed4, 0x001a, 0xcab0, 0x6dd4, 0xe5b1,
	0xc9b0, 0xc8b0, 0x001a, 0xd4b3, 0x001a, 0xd3b3, 0xd2b3, 0xd2b6,
	0x001a, 0x001a, 0xd5b6, 0xd6b6, 0xd4b6, 0x001a, 0xd3b6, 0x001a,
	0x001a, 0x43e1, 0x001a, 0x44e1, 0x001a, 0x001a, 0x001a, 0xf5e4,
	0x45bc, 0xf4e4, 0x001a, 0xb1be, 0xbfec, 0x79c0, 0x001a, 0xeef1,
	0x55c4, 0x001a, 0x63a4, 0xc3a4, 0x56c9, 0x001a, 0xc4a4, 0xc5a4,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5da5, 0x5ea5, 0x001a,
	0x49a6, 0x71ca, 0xd6cb, 0xd7cb, 0x001a, 0x6dab, 0xeed0, 0xccb0,
	0xcbb0, 0x63d8, 0x62d8, 0x001a, 0x001a, 0x50a4, 0xc6a4, 0x5fa5,
	0x001a, 0xcdb0, 0x43c9, 0x001a, 0x6cc9, 0x60a5, 0x001a, 0xc2c9,
	0x4ba6, 0x4aa6, 0xc1c9, 0x58a7, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xeaad, 0x001a, 0x001a, 0x6fd4, 0x001a, 0xd7b6,
	0x45e1, 0xbcb9, 0x001a, 0x001a, 0xfae8, 0x001a, 0x001a, 0xfdf3,
	0x001a, 0xc7a4, 0x001a, 0x001a, 0xd8cb, 0xf4cd, 0xd0b0, 0xceb0,
	0xcfb0, 0xcca2, 0x001a, 0x64a4, 0xcda2, 0xcea2, 0x001a, 0xc9a4,
	0xc8a4, 0x63a5, 0x62a5, 0x001a, 0x6dc9, 0xc3c9, 0x001a, 0x001a,
	0x001a, 0xf5a8, 0xf2a8, 0xf4a8, 0xf3a8, 0x001a, 0x001a, 0x6eab,
	0x001a, 0x001a, 0xd5b3, 0x001a, 0x52a4, 0x001a, 0xcba4, 0x001a,
	0x65a5, 0x64a5, 0x001a, 0x72ca, 0x001a, 0x001a, 0xf6a8, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x57c9, 0x001a, 0x67a5, 0x66a5,
	0x4ca6, 0x4da6, 0x73ca, 0x59a7, 0x001a, 0x5aa7, 0x001a, 0xf7a8,
	0xf8a8, 0xf9a8, 0x001a, 0x6fab, 0xf5cd, 0x001a, 0x001a, 0xebad,
	0x001a, 0x001a, 0x44c9, 0x001a, 0xcca4, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc4c9, 0x001a, 0x001a, 0x001a, 0x74ca, 0x75ca,
	0x001a, 0x001a, 0xd9cb, 0x001a, 0xdacb, 0x001a, 0xf7cd, 0xf6cd,
	0xf9cd, 0xf8cd, 0x70ab, 0x001a, 0x70d4, 0xedad, 0xefd0, 0xecad,
	0x001a, 0x001a, 0x001a, 0x001a, 0x64d8, 0xd6b3, 0x001a, 0x65d8,
	0x001a, 0x001a, 0x001a, 0x001a, 0x46e1, 0xbdb9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x46bc, 0x001a, 0xeff1, 0x001a, 0x001a, 0x001a,
	0x001a, 0x58c9, 0x001a, 0x68a5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd1b0, 0x001a, 0x001a, 0x001a, 0x001a,
	0x53a4, 0x65a4, 0xcea4, 0xcda4, 0x001a, 0xcfa4, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xfba8, 0x001a, 0xfaa8, 0xfca8,
	0x001a, 0x001a, 0x001a, 0x71ab, 0x001a, 0x001a, 0x001a, 0xeead,
	0x001a, 0xfbe8, 0x4fc2, 0x66a4, 0x6aa5, 0x79a5, 0x74a5, 0x001a,
	0x6fa5, 0x6ea5, 0x75a5, 0x73a5, 0x6ca5, 0x7aa5, 0x6da5, 0x69a5,
	0x78a5, 0x77a5, 0x76a5, 0x6ba5, 0x001a, 0x72a5, 0x001a, 0x001a,
	0x71a5, 0x001a, 0x001a, 0x7ba5, 0x70a5, 0x001a, 0x001a, 0x001a,
	0x001a, 0x53a6, 0x001a, 0x59a6, 0x55a6, 0x001a, 0x5ba6, 0xc5c9,
	0x58a6, 0x4ea6, 0x51a6, 0x54a6, 0x50a6, 0x57a6, 0x5aa6, 0x4fa6,
	0x52a6, 0x56a6, 0x5ca6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7eca, 0x7bca, 0x001a, 0x67a7, 0x7cca, 0x5ba7, 0x5da7, 0x75a7,
	0x70a7, 0x001a, 0x001a, 0x001a, 0xa5ca, 0x7dca, 0x5fa7, 0x61a7,
	0xa4ca, 0x68a7, 0x78ca, 0x74a7, 0x76a7, 0x5ca7, 0x6da7, 0x001a,
	0x76ca, 0x73a7, 0x001a, 0x64a7, 0x001a, 0x6ea7, 0x6fa7, 0x77ca,
	0x6ca7, 0x6aa7, 0x001a, 0x6ba7, 0x71a7, 0xa1ca, 0x5ea7, 0x001a,
	0x72a7, 0xa3ca, 0x66a7, 0x63a7, 0x001a, 0x7aca, 0x62a7, 0xa6ca,
	0x65a7, 0x001a, 0x69a7, 0x001a, 0x001a, 0x001a, 0x60a7, 0xa2ca,
	0x001a, 0x001a, 0x001a, 0x001a, 0x79ca, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xebcb, 0xeacb, 0x4fa9, 0xedcb, 0xefcb, 0xe4cb, 0xe7cb, 0xeecb,
	0x50a9, 0x001a, 0x001a, 0xe1cb, 0xe5cb, 0x001a, 0x001a, 0xe9cb,
	0x49ce, 0x4ba9, 0x4dce, 0xfda8, 0xe6cb, 0xfea8, 0x4ca9, 0x45a9,
	0x41a9, 0x001a, 0xe2cb, 0x44a9, 0x49a9, 0x52a9, 0xe3cb, 0xdccb,
	0x43a9, 0xddcb, 0xdfcb, 0x001a, 0x46a9, 0x001a, 0x48a9, 0xdbcb,
	0xe0cb, 0x001a, 0x001a, 0x51a9, 0x4da9, 0xe8cb, 0x53a9, 0x001a,
	0x4aa9, 0xdecb, 0x47a9, 0x001a, 0x001a, 0x42a9, 0x40a9, 0x001a,
	0xeccb, 0x001a, 0x4ea9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x48ce, 0xfbcd, 0x4bce, 0x001a, 0x001a, 0xfdcd, 0x78ab, 0xa8ab,
	0x74ab, 0xa7ab, 0x7dab, 0xa4ab, 0x72ab, 0xfccd, 0x43ce, 0xa3ab,
	0x4fce, 0xa5ab, 0x001a, 0x79ab, 0x001a, 0x001a, 0x45ce, 0x42ce,
	0x77ab, 0x001a, 0xfacd, 0xa6ab, 0x4ace, 0x7cab, 0x4cce, 0xa9ab,
	0x73ab, 0x7eab, 0x7bab, 0x40ce, 0xa1ab, 0x46ce, 0x47ce, 0x7aab,
	0xa2ab, 0x76ab, 0x001a, 0x001a, 0x001a, 0x001a, 0x75ab, 0xfecd,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x44ce, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ece, 0x001a,
	0x44d1, 0xfbad, 0xf1d0, 0x001a, 0xf6d0, 0xf4ad, 0x40ae, 0xf4d0,
	0xefad, 0xf9ad, 0xfead, 0xfbd0, 0x001a, 0xfaad, 0xfdad, 0x001a,
	0x001a, 0xfed0, 0xf5ad, 0xf5d0, 0x001a, 0x001a, 0x001a, 0x42d1,
	0x43d1, 0x001a, 0xf7ad, 0x41d1, 0xf3ad, 0x43ae, 0x001a, 0xf8d0,
	0x001a, 0xf1ad, 0x001a, 0x46d1, 0xf9d0, 0xfdd0, 0xf6ad, 0x42ae,
	0xfad0, 0xfcad, 0x40d1, 0x47d1, 0xa1d4, 0x001a, 0x45d1, 0x44ae,
	0xf0ad, 0xfcd0, 0xf3d0, 0x001a, 0xf8ad, 0x001a, 0x001a, 0xf2d0,
	0x001a, 0x001a, 0xf7d0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf0d0, 0x41ae,
	0x001a, 0x001a, 0x77d4, 0x001a, 0xe4b0, 0xa7d4, 0xe2b0, 0xdfb0,
	0x7cd4, 0xdbb0, 0xa2d4, 0xe6b0, 0x76d4, 0x7bd4, 0x7ad4, 0xf2ad,
	0xe1b0, 0xa5d4, 0x001a, 0xa8d4, 0x73d4, 0x001a, 0xe8b3, 0x001a,
	0xa9d4, 0xe7b0, 0x001a, 0xd9b0, 0xd6b0, 0x7ed4, 0xd3b0, 0x001a,
	0xa6d4, 0x001a, 0xdab0, 0xaad4, 0x001a, 0x74d4, 0xa4d4, 0xddb0,
	0x75d4, 0x78d4, 0x7dd4, 0x001a, 0x001a, 0xdeb0, 0xdcb0, 0xe8b0,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe3b0, 0x001a, 0xd7b0, 0xd2b1,
	0x001a, 0xd8b0, 0x79d4, 0xe5b0, 0xe0b0, 0xa3d4, 0xd5b0, 0x001a,
	0x001a, 0x001a, 0xd4b0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x71d4, 0x72d4, 0x6ad8,
	0x001a, 0x001a, 0x001a, 0xd7b3, 0xdab3, 0x75d8, 0xeeb3, 0x78d8,
	0xd8b3, 0x71d8, 0xdeb3, 0xe4b3, 0xbdb5, 0x001a, 0x001a, 0xe2b3,
	0x6ed8, 0xefb3, 0xdbb3, 0xe3b3, 0x76d8, 0xd7dc, 0x7bd8, 0x6fd8,
	0x001a, 0x66d8, 0x73d8, 0x6dd8, 0xe1b3, 0x79d8, 0x001a, 0x001a,
	0xddb3, 0xf1b3, 0xeab3, 0x001a, 0xdfb3, 0xdcb3, 0x001a, 0xe7b3,
	0x001a, 0x7ad8, 0x6cd8, 0x72d8, 0x74d8, 0x68d8, 0x77d8, 0xd9b3,
	0x67d8, 0x001a, 0xe0b3, 0xf0b3, 0xecb3, 0x69d8, 0xe6b3, 0x001a,
	0x001a, 0xedb3, 0xe9b3, 0xe5b3, 0x001a, 0x70d8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xebb3, 0x001a, 0x001a, 0x001a, 0xd5dc,
	0xd1dc, 0x001a, 0xe0dc, 0xcadc, 0xd3dc, 0xe5b6, 0xe6b6, 0xdeb6,
	0xdcdc, 0xe8b6, 0xcfdc, 0xcedc, 0xccdc, 0xdedc, 0xdcb6, 0xd8dc,
	0xcddc, 0xdfb6, 0xd6dc, 0xdab6, 0xd2dc, 0xd9dc, 0xdbdc, 0x001a,
	0x001a, 0xdfdc, 0xe3b6, 0xcbdc, 0xddb6, 0xd0dc, 0x001a, 0xd8b6,
	0x001a, 0xe4b6, 0xdadc, 0xe0b6, 0xe1b6, 0xe7b6, 0xdbb6, 0x5fa2,
	0xd9b6, 0xd4dc, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe2b6,
	0x001a, 0x001a, 0xdddc, 0x001a, 0x001a, 0x001a, 0xcdb9, 0xc8b9,
	0x001a, 0x55e1, 0x51e1, 0x001a, 0x4be1, 0xc2b9, 0xbeb9, 0x54e1,
	0xbfb9, 0x4ee1, 0x50e1, 0x001a, 0x53e1, 0x001a, 0xc4b9, 0x001a,
	0xcbb9, 0xc5b9, 0x001a, 0x001a, 0x49e1, 0xc6b9, 0xc7b9, 0x4ce1,
	0xccb9, 0x001a, 0x4ae1, 0x4fe1, 0xc3b9, 0x48e1, 0xc9b9, 0xc1b9,
	0x001a, 0x001a, 0x001a, 0xc0b9, 0x4de1, 0x52e1, 0x001a, 0xcab9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x47e1,
	0x001a, 0x4dbc, 0x47e5, 0x001a, 0x44e5, 0x001a, 0x47bc, 0x53bc,
	0x54bc, 0x001a, 0x4abc, 0x42e5, 0x4cbc, 0xf9e4, 0x52bc, 0x001a,
	0x46e5, 0x49bc, 0x48e5, 0x48bc, 0x001a, 0x43e5, 0x45e5, 0x4bbc,
	0x41e5, 0xfae4, 0xf7e4, 0x001a, 0x001a, 0x6bd8, 0xfde4, 0x001a,
	0xf6e4, 0xfce4, 0xfbe4, 0x001a, 0xf8e4, 0x001a, 0x4fbc, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4ebc, 0x001a, 0x001a, 0x001a, 0x50bc,
	0xfee4, 0xb2be, 0x40e5, 0x001a, 0x001a, 0x001a, 0x45e9, 0x001a,
	0xfde8, 0x001a, 0xbebe, 0x42e9, 0xb6be, 0xbabe, 0x41e9, 0x001a,
	0xb9be, 0xb5be, 0xb8be, 0xb3be, 0xbdbe, 0x43e9, 0xfee8, 0xbcbe,
	0xfce8, 0xbbbe, 0x44e9, 0x40e9, 0x51bc, 0x001a, 0xbfbe, 0x46e9,
	0xb7be, 0xb4be, 0x001a, 0x001a, 0x001a, 0x001a, 0xc6ec, 0xc8ec,
	0x7bc0, 0xc9ec, 0xc7ec, 0xc5ec, 0xc4ec, 0x7dc0, 0xc3ec, 0x7ec0,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc1ec, 0xc2ec, 0x7ac0, 0xa1c0,
	0x7cc0, 0x001a, 0x001a, 0xc0ec, 0x001a, 0x50c2, 0x001a, 0xbcef,
	0xbaef, 0xbfef, 0xbdef, 0x001a, 0xbbef, 0xbeef, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x60c3, 0xf2f1, 0xf3f1,
	0x56c4, 0x001a, 0xf4f1, 0xf0f1, 0xf5f1, 0xf1f1, 0x51c2, 0x001a,
	0x001a, 0x001a, 0xfef3, 0x41f4, 0x59c4, 0x40f4, 0x58c4, 0x57c4,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5ac4, 0xc5f5, 0xc6f5, 0x001a,
	0xdac4, 0xd9c4, 0xdbc4, 0xc4f5, 0x001a, 0xd8f6, 0xd7f6, 0x001a,
	0x6dc5, 0x6fc5, 0x6ec5, 0xd9f6, 0xc8c5, 0xa6f8, 0x001a, 0x001a,
	0x001a, 0xf1c5, 0x001a, 0xa5f8, 0xeef8, 0x001a, 0x001a, 0x49c9,
	0x001a, 0x001a, 0x7da5, 0x7ca5, 0x001a, 0x5fa6, 0x5ea6, 0xc7c9,
	0x5da6, 0xc6c9, 0x001a, 0x001a, 0x79a7, 0xa9ca, 0x001a, 0xa8ca,
	0x001a, 0x001a, 0x77a7, 0x7aa7, 0x001a, 0x001a, 0xa7ca, 0x001a,
	0x78a7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf0cb,
	0x001a, 0xf1cb, 0x54a9, 0x001a, 0x001a, 0x001a, 0x001a, 0xaaab,
	0x001a, 0x48d1, 0x49d1, 0x45ae, 0x46ae, 0x001a, 0x001a, 0xacd4,
	0xe9b0, 0xebb0, 0xabd4, 0xeab0, 0x7cd8, 0xf2b3, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe9b6, 0xeab6, 0xe1dc, 0x001a, 0xcfb9, 0x001a,
	0xceb9, 0x001a, 0x49e5, 0x48e9, 0x47e9, 0x001a, 0x6bf9, 0x67a4,
	0x59c9, 0x001a, 0x6ec9, 0x6fc9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x62a6, 0x66a6, 0xc9c9, 0x001a, 0x64a6, 0x63a6, 0xc8c9, 0x65a6,
	0x61a6, 0x001a, 0x001a, 0x60a6, 0xcac9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa6a7, 0x001a, 0x001a, 0xa3a7, 0x001a,
	0x7da7, 0xaaca, 0x001a, 0x001a, 0x001a, 0xabca, 0x001a, 0xa1a7,
	0x001a, 0xadca, 0x7ba7, 0xaeca, 0xacca, 0x7ea7, 0xa2a7, 0xa5a7,
	0xa4a7, 0x7ca7, 0xafca, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x59a9, 0xfecb, 0x001a, 0x5ba9, 0x001a, 0x5aa9, 0x001a,
	0x40cc, 0x58a9, 0x57a9, 0xf5cb, 0x001a, 0xf4cb, 0x001a, 0xf2cb,
	0xf7cb, 0xf6cb, 0xf3cb, 0xfccb, 0xfdcb, 0xfacb, 0xf8cb, 0x56a9,
	0x001a, 0x001a, 0x001a, 0xfbcb, 0x5ca9, 0x41cc, 0x001a, 0x001a,
	0xf9cb, 0x001a, 0xabab, 0x55a9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xacab, 0x54ce, 0x001a, 0x001a, 0x5ace,
	0x001a, 0x001a, 0x001a, 0xb2ab, 0x58ce, 0x5ece, 0x001a, 0x55ce,
	0x59ce, 0x5bce, 0x5dce, 0x57ce, 0x001a, 0x56ce, 0x51ce, 0x52ce,
	0xadab, 0x001a, 0xafab, 0xaeab, 0x53ce, 0x5cce, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb1ab, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x50ce, 0x53d1, 0x001a,
	0x52d1, 0x57d1, 0x4ed1, 0x001a, 0x51d1, 0x50d1, 0x001a, 0x54d1,
	0x001a, 0x58d1, 0x47ae, 0x4aae, 0x001a, 0x001a, 0x4fd1, 0x55d1,
	0x001a, 0x001a, 0x001a, 0x49ae, 0x4ad1, 0x001a, 0xb0ab, 0xbad4,
	0x56d1, 0x001a, 0x4dd1, 0x001a, 0x48ae, 0x4cd1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xb1d4, 0x001a, 0x001a, 0xecb0,
	0xf0b0, 0xc1d4, 0xafd4, 0xbdd4, 0xf1b0, 0xbfd4, 0x001a, 0xc5d4,
	0x001a, 0xc9d4, 0x001a, 0x001a, 0xc0d4, 0xb4d4, 0xbcd4, 0x001a,
	0xcad4, 0xc8d4, 0xbed4, 0xb9d4, 0xb2d4, 0xa6d8, 0xb0d4, 0xf5b0,
	0xb7d4, 0xf6b0, 0xf2b0, 0xadd4, 0xc3d4, 0xb5d4, 0x001a, 0x001a,
	0xb3d4, 0xc6d4, 0xf3b0, 0x001a, 0xccd4, 0xedb0, 0xefb0, 0xbbd4,
	0xb6d4, 0x4bae, 0xeeb0, 0xb8d4, 0xc7d4, 0xcbd4, 0xc2d4, 0x001a,
	0xc4d4, 0x001a, 0x001a, 0x001a, 0xaed4, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa1d8, 0x001a, 0xaad8, 0xa9d8, 0xfab3, 0xa2d8, 0x001a,
	0xfbb3, 0xf9b3, 0x001a, 0xa4d8, 0xf6b3, 0xa8d8, 0x001a, 0xa3d8,
	0xa5d8, 0x7dd8, 0xf4b3, 0x001a, 0xb2d8, 0xb1d8, 0xaed8, 0xf3b3,
	0xf7b3, 0xf8b3, 0x4bd1, 0xabd8, 0xf5b3, 0xf4b0, 0xadd8, 0x7ed8,
	0xb0d8, 0xafd8, 0x001a, 0xb3d8, 0x001a, 0xefdc, 0x001a, 0xacd8,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa7d8, 0xe7dc, 0xf4b6, 0xf7b6, 0xf2b6, 0xe6dc, 0xeadc, 0xe5dc,
	0x001a, 0xecb6, 0xf6b6, 0xe2dc, 0xf0b6, 0xe9dc, 0x001a, 0xeeb6,
	0xedb6, 0xecdc, 0xefb6, 0xeedc, 0x001a, 0xebdc, 0xebb6, 0x001a,
	0x001a, 0x001a, 0xf5b6, 0xf0dc, 0xe4dc, 0xeddc, 0x001a, 0x001a,
	0xe3dc, 0x001a, 0x001a, 0xf1b6, 0x001a, 0xf3b6, 0x001a, 0xe8dc,
	0x001a, 0xf1dc, 0x001a, 0x001a, 0x5de1, 0xd0b9, 0x63e1, 0x001a,
	0x001a, 0xd5b9, 0x5fe1, 0x66e1, 0x57e1, 0xd7b9, 0xd1b9, 0x5ce1,
	0x55bc, 0x5be1, 0x64e1, 0xd2b9, 0x001a, 0xd6b9, 0x5ae1, 0x60e1,
	0x65e1, 0x56e1, 0xd4b9, 0x5ee1, 0x001a, 0x001a, 0x62e1, 0x68e1,
	0x58e1, 0x61e1, 0x001a, 0xd3b9, 0x67e1, 0x001a, 0x001a, 0x001a,
	0x59e1, 0x001a, 0x001a, 0x001a, 0x59bc, 0x4be5, 0x57bc, 0x56bc,
	0x4de5, 0x52e5, 0x001a, 0x4ee5, 0x001a, 0x51e5, 0x5cbc, 0x001a,
	0xa5be, 0x5bbc, 0x001a, 0x4ae5, 0x50e5, 0x001a, 0x5abc, 0x4fe5,
	0x001a, 0x4ce5, 0x001a, 0x58bc, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x4de9, 0xd9f9, 0x4fe9, 0x4ae9, 0xc1be, 0x4ce9,
	0x001a, 0xc0be, 0x4ee9, 0x001a, 0x001a, 0xc3be, 0x50e9, 0xc2be,
	0x49e9, 0x4be9, 0x001a, 0x001a, 0x001a, 0x001a, 0xa5c0, 0xccec,
	0x001a, 0xa4c0, 0xcdec, 0xa3c0, 0xcbec, 0xa2c0, 0xcaec, 0x001a,
	0x53c2, 0x52c2, 0xf6f1, 0xf8f1, 0x001a, 0xf7f1, 0x61c3, 0x62c3,
	0x001a, 0x001a, 0x63c3, 0x42f4, 0x5bc4, 0x001a, 0x001a, 0xd3f7,
	0xd2f7, 0xf2c5, 0x001a, 0x68a4, 0xd0a4, 0x001a, 0x001a, 0xa7a7,
	0x001a, 0x001a, 0x001a, 0x001a, 0x5fce, 0x001a, 0x001a, 0x001a,
	0x001a, 0xfcb3, 0xfdb3, 0x001a, 0xf2dc, 0xd8b9, 0x69e1, 0x53e5,
	0x001a, 0x001a, 0x001a, 0x5ac9, 0x001a, 0x001a, 0xb0ca, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x42cc, 0x60ce, 0x59d1, 0x4cae,
	0x001a, 0x001a, 0xf9f1, 0x001a, 0xdcc4, 0x69a4, 0x7ea5, 0x70c9,
	0x001a, 0x67a6, 0x68a6, 0x001a, 0x5da9, 0x001a, 0x001a, 0x001a,
	0xf7b0, 0x001a, 0xdab9, 0x001a, 0xdbb9, 0xd9b9, 0x001a, 0x6aa4,
	0x001a, 0xd1a4, 0xd3a4, 0xd2a4, 0x5bc9, 0xd4a4, 0xa1a5, 0x71c9,
	0x001a, 0xa2a5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x69a6,
	0x6aa6, 0x001a, 0x001a, 0x001a, 0xcbc9, 0x001a, 0xa8a7, 0x001a,
	0xb1ca, 0x001a, 0x001a, 0x001a, 0x61a9, 0x43cc, 0x001a, 0x5fa9,
	0x60a9, 0x5ea9, 0x5ad1, 0x001a, 0x001a, 0x001a, 0xb6ab, 0xb5ab,
	0xb7ab, 0xb4ab, 0x001a, 0x61ce, 0x62a9, 0xb3ab, 0x001a, 0x4dae,
	0x4eae, 0x001a, 0x4fae, 0x001a, 0xcdd4, 0x001a, 0x001a, 0x001a,
	0xfeb3, 0xb4d8, 0xf8b0, 0x001a, 0x001a, 0x001a, 0x001a, 0xf8b6,
	0x001a, 0xddb9, 0xdcb9, 0x6ae1, 0x001a, 0x5dbc, 0xc4be, 0x001a,
	0xc0ef, 0xdaf6, 0xd4f7, 0x6ba4, 0xa3a5, 0x001a, 0xa4a5, 0xd1c9,
	0x6ca6, 0x6fa6, 0x001a, 0xcfc9, 0xcdc9, 0x6ea6, 0xd0c9, 0xd2c9,
	0xccc9, 0x71a6, 0x70a6, 0x6da6, 0x6ba6, 0xcec9, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb3a7, 0x001a, 0x001a, 0xb0a7, 0xb6ca, 0xb9ca,
	0xb8ca, 0x001a, 0xaaa7, 0xb2a7, 0x001a, 0x001a, 0xafa7, 0xb5ca,
	0xb3ca, 0xaea7, 0x001a, 0x001a, 0x001a, 0xa9a7, 0xaca7, 0x001a,
	0xb4ca, 0xbbca, 0xb7ca, 0xada7, 0xb1a7, 0xb4a7, 0xb2ca, 0xbaca,
	0xaba7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x67a9, 0x6fa9,
	0x001a, 0x4fcc, 0x48cc, 0x70a9, 0x53cc, 0x44cc, 0x4bcc, 0x001a,
	0x001a, 0x66a9, 0x45cc, 0x64a9, 0x4ccc, 0x50cc, 0x63a9, 0x001a,
	0x51cc, 0x4acc, 0x001a, 0x4dcc, 0x001a, 0x72a9, 0x69a9, 0x54cc,
	0x52cc, 0x001a, 0x6ea9, 0x6ca9, 0x49cc, 0x6ba9, 0x47cc, 0x46cc,
	0x6aa9, 0x68a9, 0x71a9, 0x6da9, 0x65a9, 0x001a, 0x4ecc, 0x001a,
	0xb9ab, 0x001a, 0xc0ab, 0x6fce, 0xb8ab, 0x67ce, 0x63ce, 0x001a,
	0x73ce, 0x62ce, 0x001a, 0xbbab, 0x6cce, 0xbeab, 0xc1ab, 0x001a,
	0xbcab, 0x70ce, 0xbfab, 0x001a, 0x56ae, 0x76ce, 0x64ce, 0x001a,
	0x001a, 0x66ce, 0x6dce, 0x71ce, 0x75ce, 0x72ce, 0x6bce, 0x6ece,
	0x001a, 0x001a, 0x68ce, 0xc3ab, 0x6ace, 0x69ce, 0x74ce, 0xbaab,
	0x65ce, 0xc2ab, 0x001a, 0xbdab, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5cae, 0x62d1, 0x001a, 0x5bae, 0x001a, 0x001a, 0x60d1,
	0x001a, 0x50ae, 0x001a, 0x55ae, 0x001a, 0x5fd1, 0x5cd1, 0x61d1,
	0x51ae, 0x5bd1, 0x001a, 0x54ae, 0x52ae, 0x001a, 0x63d1, 0x53ae,
	0x57ae, 0x001a, 0x001a, 0x58ae, 0x001a, 0x5aae, 0x001a, 0x001a,
	0x001a, 0x59ae, 0x001a, 0x001a, 0x001a, 0x5dd1, 0x5ed1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x64d1, 0x001a, 0xd4d4, 0xf9b0, 0xc2d8,
	0xd3d4, 0xe6d4, 0x001a, 0x001a, 0x40b1, 0x001a, 0xe4d4, 0x001a,
	0xfeb0, 0xfab0, 0xedd4, 0xddd4, 0xe0d4, 0x001a, 0x43b1, 0xead4,
	0xe2d4, 0xfbb0, 0x44b1, 0x001a, 0xe7d4, 0xe5d4, 0x001a, 0x001a,
	0xd6d4, 0xebd4, 0xdfd4, 0xdad4, 0x001a, 0xd0d4, 0xecd4, 0xdcd4,
	0xcfd4, 0x001a, 0x42b1, 0xe1d4, 0xeed4, 0xded4, 0xd2d4, 0xd7d4,
	0xced4, 0x001a, 0x41b1, 0x001a, 0xdbd4, 0xd8d4, 0xfcb0, 0xd1d4,
	0x001a, 0xe9d4, 0xfdb0, 0x001a, 0xd9d4, 0xd5d4, 0x001a, 0x001a,
	0xe8d4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x40b4,
	0xbbd8, 0x001a, 0xb8d8, 0xc9d8, 0xbdd8, 0xcad8, 0x001a, 0x42b4,
	0x001a, 0x001a, 0x001a, 0xc6d8, 0xc3d8, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc4d8, 0xc7d8, 0xcbd8, 0x001a, 0xe3d4, 0xcdd8,
	0x47dd, 0x001a, 0x43b4, 0xced8, 0xb6d8, 0xc0d8, 0x001a, 0xc5d8,
	0x001a, 0x001a, 0x41b4, 0x44b4, 0xccd8, 0xcfd8, 0xbad8, 0xb7d8,
	0x001a, 0x001a, 0xb9d8, 0x001a, 0x001a, 0xbed8, 0xbcd8, 0x45b4,
	0x001a, 0xc8d8, 0x001a, 0x001a, 0xbfd8, 0x001a, 0xc1d8, 0xb5d8,
	0xfadc, 0xf8dc, 0x42b7, 0x40b7, 0x43dd, 0xf9dc, 0x44dd, 0x40dd,
	0xf7dc, 0x46dd, 0xf6dc, 0xfddc, 0xfeb6, 0xfdb6, 0xfcb6, 0xfbdc,
	0x41dd, 0xf9b6, 0x41b7, 0x001a, 0xf4dc, 0x001a, 0xfedc, 0xf3dc,
	0xfcdc, 0xfab6, 0x42dd, 0xf5dc, 0xfbb6, 0x45dd, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6ee1, 0xe2b9, 0xe1b9,
	0xe3b9, 0x7ae1, 0x70e1, 0x76e1, 0x6be1, 0x79e1, 0x78e1, 0x7ce1,
	0x75e1, 0xdeb9, 0x74e1, 0xe4b9, 0x001a, 0x6de1, 0xdfb9, 0x001a,
	0x7be1, 0xe0b9, 0x6fe1, 0x72e1, 0x77e1, 0x71e1, 0x6ce1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x73e1, 0x55e5, 0x61bc, 0x58e5, 0x57e5,
	0x5ae5, 0x5ce5, 0xdcf9, 0x5fbc, 0x001a, 0x56e5, 0x001a, 0x54e5,
	0x001a, 0x5de5, 0x5be5, 0x59e5, 0x001a, 0x5fe5, 0x001a, 0x5ee5,
	0x63bc, 0x5ebc, 0x001a, 0x60bc, 0x62bc, 0x001a, 0x001a, 0x60e5,
	0x57e9, 0x001a, 0x001a, 0x56e9, 0x55e9, 0x001a, 0x58e9, 0x51e9,
	0x001a, 0x52e9, 0x5ae9, 0x53e9, 0x001a, 0xc5be, 0x5ce9, 0x001a,
	0x5be9, 0x54e9, 0x001a, 0xd1ec, 0xa8c0, 0xcfec, 0xd4ec, 0xd3ec,
	0x59e9, 0x001a, 0xa7c0, 0x001a, 0xd2ec, 0xceec, 0xd6ec, 0xd5ec,
	0xa6c0, 0x001a, 0xd0ec, 0x001a, 0xc6be, 0x001a, 0x001a, 0x001a,
	0x54c2, 0x001a, 0x001a, 0x001a, 0xc1ef, 0xfaf1, 0xfbf1, 0xfcf1,
	0x5cc4, 0x001a, 0x001a, 0x5dc4, 0x001a, 0x43f4, 0x001a, 0xc8f5,
	0xc7f5, 0x001a, 0x001a, 0xdbf6, 0xdcf6, 0xd5f7, 0xa7f8, 0x001a,
	0x6ca4, 0x6da4, 0x001a, 0x6ea4, 0xd5a4, 0xa5a5, 0xd3c9, 0x72a6,
	0x73a6, 0x001a, 0xb7a7, 0xb8a7, 0xb6a7, 0xb5a7, 0x001a, 0x73a9,
	0x001a, 0x001a, 0x55cc, 0x75a9, 0x74a9, 0x56cc, 0x001a, 0x001a,
	0x001a, 0xc4ab, 0x001a, 0x5dae, 0x65d1, 0x001a, 0xf0d4, 0x001a,
	0x45b1, 0x47b4, 0xefd4, 0x46b4, 0x001a, 0xe5b9, 0x001a, 0x7de1,
	0xc7be, 0x001a, 0xa9c0, 0xd7ec, 0x001a, 0x5ec4, 0x001a, 0x70c5,
	0x001a, 0x72c9, 0x001a, 0xa6a5, 0x73c9, 0x76a6, 0x001a, 0x74a6,
	0x75a6, 0x77a6, 0x001a, 0xbaa7, 0xb9a7, 0x001a, 0xbcca, 0xbba7,
	0x001a, 0x001a, 0xbdca, 0x57cc, 0x001a, 0x58cc, 0x001a, 0x76a9,
	0x78a9, 0x7aa9, 0x77a9, 0x7ba9, 0x79a9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc8ab, 0xc5ab, 0xc7ab, 0xc9ab, 0xc6ab, 0x66d1,
	0x77ce, 0x001a, 0x001a, 0x001a, 0x68d1, 0x67d1, 0x63ae, 0x001a,
	0x5fae, 0x001a, 0x001a, 0x60ae, 0x62ae, 0x64ae, 0x61ae, 0x001a,
	0x66ae, 0x65ae, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ab1,
	0xf2d4, 0xf1d4, 0x49b1, 0x001a, 0x48b1, 0x47b1, 0x4bb1, 0x46b1,
	0x001a, 0x001a, 0xd5d8, 0xd2d8, 0x49b4, 0xd1d8, 0xd6d8, 0x001a,
	0x4bb4, 0xd4d8, 0x48b4, 0x4ab4, 0xd3d8, 0x001a, 0x48dd, 0x001a,
	0x49dd, 0x4add, 0x001a, 0x001a, 0x001a, 0x001a, 0xe6b9, 0xeeb9,
	0x7ee1, 0xe8b9, 0xecb9, 0xa1e1, 0xedb9, 0xe9b9, 0xeab9, 0xe7b9,
	0xebb9, 0x66bc, 0xd0d8, 0x67bc, 0x65bc, 0x001a, 0x64bc, 0x5de9,
	0xc8be, 0xd8ec, 0xd9ec, 0x001a, 0x001a, 0x64c3, 0x5fc4, 0x001a,
	0x6fa4, 0x001a, 0x78a6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xcaab, 0x001a, 0x69d1, 0x67ae, 0x001a, 0x001a, 0x4eb1,
	0x4db1, 0x4cb1, 0x4cb4, 0x4db4, 0xd7d8, 0xefb9, 0xc9be, 0x70a4,
	0x5cc9, 0xd6a4, 0x74c9, 0x001a, 0x001a, 0xd4c9, 0x79a6, 0x001a,
	0x001a, 0x001a, 0x7ca9, 0x001a, 0x001a, 0x001a, 0x001a, 0x4bdd,
	0x001a, 0x001a, 0x71a4, 0x001a, 0xd7a4, 0xd5c9, 0x001a, 0x001a,
	0xbeca, 0x001a, 0xbfca, 0x001a, 0xbca7, 0x001a, 0x001a, 0x001a,
	0xd8d8, 0x4eb4, 0x001a, 0x4cdd, 0x001a, 0x001a, 0x001a, 0xaac0,
	0x72a4, 0xa8a4, 0xd8a4, 0x75c9, 0xa7a5, 0x001a, 0xc0a7, 0xbfa7,
	0xbda7, 0xbea7, 0x001a, 0x001a, 0x59cc, 0x7ea9, 0xa1a9, 0x5acc,
	0x7da9, 0x001a, 0x001a, 0xceab, 0x78ce, 0xcdab, 0xcbab, 0xccab,
	0x6aae, 0x68ae, 0x001a, 0x001a, 0x6bd1, 0x69ae, 0x6ad1, 0x001a,
	0x5eae, 0xf3d4, 0x001a, 0x001a, 0x50b1, 0x51b1, 0x001a, 0x001a,
	0x4fb1, 0x001a, 0xf0b9, 0xa2e1, 0x68bc, 0x69bc, 0x001a, 0x61e5,
	0xabc0, 0xc2ef, 0xc3ef, 0x001a, 0xddc4, 0xa8f8, 0x4bc9, 0xd9a4,
	0x001a, 0x73a4, 0x001a, 0x77c9, 0x76c9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7aa6, 0xd7c9, 0xd8c9, 0xd6c9, 0x001a, 0xd9c9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc7ca, 0x001a,
	0xc2ca, 0xc4ca, 0xc6ca, 0xc3ca, 0xc4a7, 0xc0ca, 0x001a, 0xc1ca,
	0xc1a7, 0xc2a7, 0xc5ca, 0xc8ca, 0xc3a7, 0xc9ca, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x68cc, 0x001a, 0x62cc,
	0x5dcc, 0xa3a9, 0x65cc, 0x63cc, 0x5ccc, 0x69cc, 0x6ccc, 0x67cc,
	0x60cc, 0xa5a9, 0x66cc, 0xa6a9, 0x61cc, 0x64cc, 0x5bcc, 0x5fcc,
	0x6bcc, 0xa7a9, 0x001a, 0xa8a9, 0x001a, 0x5ecc, 0x6acc, 0xa2a9,
	0xa4a9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xabce, 0xa4ce,
	0xaace, 0xa3ce, 0xa5ce, 0x7dce, 0x7bce, 0x001a, 0xacce, 0xa9ce,
	0x79ce, 0x001a, 0xd0ab, 0xa7ce, 0xa8ce, 0x001a, 0xa6ce, 0x7cce,
	0x7ace, 0xcfab, 0xa2ce, 0x7ece, 0x001a, 0x001a, 0xa1ce, 0xadce,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6fae, 0x001a, 0x6eae, 0x001a, 0x6cd1, 0x6bae, 0x6ed1, 0x001a,
	0x70ae, 0x6fd1, 0x001a, 0x001a, 0x73ae, 0x001a, 0x71ae, 0x70d1,
	0xaece, 0x72d1, 0x001a, 0x6dae, 0x001a, 0x6cae, 0x001a, 0x6dd1,
	0x71d1, 0x72ae, 0x001a, 0x001a, 0x001a, 0x001a, 0x53b1, 0x52b1,
	0x001a, 0x001a, 0x001a, 0xf5d4, 0xf9d4, 0xfbd4, 0x54b1, 0xfed4,
	0x001a, 0x58b1, 0x41d5, 0x001a, 0x5ab1, 0x001a, 0x56b1, 0x5eb1,
	0x001a, 0x5bb1, 0xf7d4, 0x55b1, 0x001a, 0xf6d4, 0xf4d4, 0x43d5,
	0xf8d4, 0x001a, 0x57b1, 0x42d5, 0x5cb1, 0xfdd4, 0xfcd4, 0x5db1,
	0xfad4, 0x59b1, 0x001a, 0x001a, 0x001a, 0x001a, 0x44d5, 0x001a,
	0x40d5, 0xe7d8, 0xeed8, 0xe3d8, 0x51b4, 0xdfd8, 0xefd8, 0xd9d8,
	0xecd8, 0xead8, 0xe4d8, 0x001a, 0xedd8, 0xe6d8, 0x001a, 0xded8,
	0xf0d8, 0xdcd8, 0xe9d8, 0xdad8, 0x001a, 0xf1d8, 0x001a, 0x52b4,
	0x001a, 0xebd8, 0x4fdd, 0xddd8, 0x4fb4, 0x001a, 0xe1d8, 0x001a,
	0x50b4, 0xe0d8, 0xe5d8, 0x001a, 0x001a, 0xe2d8, 0x001a, 0x001a,
	0x001a, 0xe8d8, 0x001a, 0x001a, 0x001a, 0x001a, 0x53dd, 0x001a,
	0x001a, 0x001a, 0x56dd, 0x4edd, 0x001a, 0x50dd, 0x001a, 0x55dd,
	0x54dd, 0x43b7, 0x001a, 0xdbd8, 0x52dd, 0x001a, 0x001a, 0x44b7,
	0x001a, 0x4ddd, 0x51dd, 0x001a, 0x001a, 0x001a, 0x001a, 0xa9e1,
	0x001a, 0xb0e1, 0xa7e1, 0x001a, 0xaee1, 0xa5e1, 0xade1, 0xb1e1,
	0xa4e1, 0xa8e1, 0xa3e1, 0x001a, 0xf1b9, 0x001a, 0xa6e1, 0xf2b9,
	0xace1, 0xabe1, 0xaae1, 0x001a, 0x001a, 0xafe1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x65e5, 0x67e5, 0x6bbc, 0x68e5, 0x001a, 0x63e5,
	0x001a, 0x62e5, 0x6ce5, 0x001a, 0x6ae5, 0x6abc, 0x6de5, 0x64e5,
	0x69e5, 0x6be5, 0x66e5, 0x001a, 0x001a, 0x001a, 0x001a, 0x61e9,
	0x66e9, 0x60e9, 0x65e9, 0x001a, 0x5ee9, 0x68e9, 0x64e9, 0x69e9,
	0x63e9, 0x5fe9, 0x67e9, 0x001a, 0x6ae9, 0x62e9, 0x001a, 0xdaec,
	0xafc0, 0x001a, 0xadc0, 0x001a, 0xacc0, 0xaec0, 0x001a, 0x001a,
	0xc4ef, 0x001a, 0x72f1, 0xfdf1, 0x001a, 0x001a, 0x44f4, 0x45f4,
	0x001a, 0x60c4, 0x001a, 0xc9f5, 0x001a, 0xdec4, 0x001a, 0xcaf5,
	0x001a, 0xdef6, 0x72c5, 0x001a, 0x71c5, 0xddf6, 0xc9c5, 0x001a,
	0xd6f7, 0x001a, 0x001a, 0x001a, 0x001a, 0x74a4, 0x7ba6, 0xdac9,
	0xcaca, 0xb5a8, 0x5fb1, 0x001a, 0x001a, 0x75a4, 0xaaa5, 0xa9a5,
	0xa8a5, 0x001a, 0x001a, 0xc5a7, 0x001a, 0x001a, 0x74ae, 0x001a,
	0x57dd, 0x76a4, 0x77a4, 0x78a4, 0xdaa4, 0x001a, 0x001a, 0xd1ab,
	0x001a, 0xafce, 0x001a, 0x001a, 0x001a, 0x53b4, 0x79a4, 0x5dc9,
	0x001a, 0x001a, 0xaba5, 0xaca5, 0x78c9, 0x001a, 0x7ca6, 0x001a,
	0x001a, 0x001a, 0xcbca, 0x001a, 0xc6a7, 0x001a, 0xccca, 0x001a,
	0x001a, 0xaea9, 0x001a, 0x001a, 0x6ecc, 0xaca9, 0xaba9, 0x6dcc,
	0xa9a9, 0x6fcc, 0xaaa9, 0xada9, 0x001a, 0xd2ab, 0x001a, 0xd4ab,
	0xb3ce, 0xb0ce, 0xb1ce, 0xb2ce, 0xb4ce, 0xd3ab, 0x001a, 0x001a,
	0x74d1, 0x73d1, 0x001a, 0x76ae, 0x001a, 0x75ae, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x62b1, 0x46d5, 0x001a, 0x61b1, 0x63b1,
	0x60b1, 0x001a, 0x001a, 0x001a, 0x001a, 0x55b4, 0x45d5, 0x001a,
	0x56b4, 0xf3d8, 0x001a, 0x57b4, 0xf2d8, 0x54b4, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5add, 0x5cdd, 0x45b7, 0x5bdd, 0x59dd, 0x58dd,
	0x001a, 0x001a, 0x001a, 0xb4e1, 0xf7b9, 0xf5b9, 0x001a, 0xf6b9,
	0xb2e1, 0xb3e1, 0x001a, 0xf3b9, 0x71e5, 0x6fe5, 0x001a, 0x6dbc,
	0x70e5, 0x6ebc, 0x6cbc, 0xf4b9, 0x001a, 0x001a, 0x6de9, 0x6be9,
	0x6ce9, 0x6ee5, 0xdcec, 0xb0c0, 0xdbec, 0xc5ef, 0xc6ef, 0x6ee9,
	0xfef1, 0x001a, 0x7aa4, 0xada5, 0x7ea6, 0xdbc9, 0x7da6, 0x001a,
	0xafa9, 0x46b7, 0x001a, 0xdba4, 0xaea5, 0xd5ab, 0x58b4, 0x001a,
	0x79c9, 0x001a, 0x7ac9, 0x001a, 0xdcc9, 0x001a, 0x001a, 0xc8a7,
	0xd0ca, 0xceca, 0xc9a7, 0xcdca, 0xcfca, 0xd1ca, 0x001a, 0xc7a7,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb3a9, 0xb4a9, 0xb1a9,
	0x001a, 0x001a, 0xb0a9, 0xb8ce, 0xb2a9, 0x001a, 0x001a, 0x001a,
	0xd6ab, 0x001a, 0xb7ce, 0xb9ce, 0xb6ce, 0xbace, 0xd7ab, 0x79ae,
	0x75d1, 0x001a, 0x77d1, 0x77ae, 0x78d1, 0x78ae, 0x76d1, 0x001a,
	0xb5ce, 0x47d5, 0x4ad5, 0x4bd5, 0x48d5, 0x67b1, 0x66b1, 0x64b1,
	0x65b1, 0x49d5, 0x001a, 0x001a, 0x001a, 0x001a, 0x68b1, 0x001a,
	0x001a, 0x5ab4, 0x5bb4, 0x001a, 0x5cb4, 0x5ddd, 0x5fdd, 0x61dd,
	0x48b7, 0x47b7, 0x59b4, 0x60dd, 0x5edd, 0x001a, 0xb8e1, 0x001a,
	0x001a, 0xb6e1, 0xbce1, 0xf8b9, 0xbde1, 0xbae1, 0xf9b9, 0xb7e1,
	0xb5e1, 0xbbe1, 0x70bc, 0x73e5, 0xb9e1, 0x72bc, 0x74e5, 0x71bc,
	0x74bc, 0x75e5, 0x6fbc, 0x73bc, 0x001a, 0x73e9, 0x71e9, 0x70e9,
	0x72e9, 0x6fe9, 0x001a, 0x001a, 0x66c3, 0x001a, 0x46f4, 0x47f4,
	0x001a, 0xcbf5, 0xdff6, 0x55c6, 0x001a, 0x001a, 0xb5a9, 0xcaa7,
	0x001a, 0x001a, 0xd8ab, 0x001a, 0x001a, 0x001a, 0x7ba4, 0xdca4,
	0x001a, 0xafa5, 0xddc9, 0x001a, 0xcba7, 0xd2ca, 0x001a, 0xbbce,
	0xd9ab, 0x001a, 0xfab9, 0x7ca4, 0x001a, 0x001a, 0x001a, 0xa1a6,
	0x001a, 0x001a, 0x49b7, 0x7da4, 0xdda4, 0xdea4, 0x001a, 0xb1a5,
	0xb0a5, 0x001a, 0xdec9, 0xa2a6, 0x001a, 0xd3ca, 0x001a, 0xcca7,
	0x001a, 0x001a, 0x71cc, 0x72cc, 0x73cc, 0x001a, 0xb6a9, 0xb7a9,
	0x70cc, 0xb8a9, 0x001a, 0x001a, 0x001a, 0xdaab, 0xbcce, 0x001a,
	0x7ad1, 0x7aae, 0x001a, 0x79d1, 0x001a, 0x69b1, 0x4cd5, 0x6ab1,
	0x4dd5, 0x001a, 0x001a, 0x001a, 0x5db4, 0x001a, 0x001a, 0x001a,
	0x62dd, 0x001a, 0x001a, 0xbfe1, 0xbee1, 0x001a, 0xfbb9, 0x001a,
	0x75bc, 0x76e5, 0xcabe, 0x74e9, 0xb1c0, 0x001a, 0x73c5, 0xd8f7,
	0x001a, 0x001a, 0x001a, 0x001a, 0x74cc, 0x001a, 0xbdce, 0x6bb1,
	0xf4d8, 0x4ab7, 0x001a, 0x001a, 0x001a, 0x55c2, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcea7, 0x001a, 0xcda7, 0xdbab, 0x001a, 0x7bd1,
	0x001a, 0x6db1, 0x43b3, 0x6eb1, 0x6cb1, 0x5eb4, 0x001a, 0xc0e1,
	0xfcb9, 0x76bc, 0x001a, 0x4cc9, 0xdfc9, 0x001a, 0xd5ca, 0xcfa7,
	0xd4ca, 0xd0a7, 0x001a, 0x001a, 0xbca9, 0x77cc, 0x76cc, 0xbba9,
	0xb9a9, 0xbaa9, 0x75cc, 0x001a, 0x001a, 0xddab, 0xbece, 0xe0ab,
	0xdcab, 0xe2ab, 0xdeab, 0xdfab, 0xe1ab, 0x001a, 0x001a, 0x001a,
	0x7dae, 0x7cae, 0x7bae, 0x001a, 0x001a, 0x001a, 0x4fd5, 0x6fb1,
	0x72b1, 0x70b1, 0x001a, 0x4ed5, 0x75b1, 0x001a, 0x71b1, 0x50d5,
	0x74b1, 0x73b1, 0x001a, 0x001a, 0x001a, 0xf6d8, 0xf5d8, 0x001a,
	0x61b4, 0x5fb4, 0x60b4, 0xf7d8, 0x4bb7, 0x64dd, 0x4cb7, 0x63dd,
	0x001a, 0x001a, 0x77e5, 0x001a, 0x001a, 0x78bc, 0xc1e1, 0x77bc,
	0x001a, 0xfdb9, 0x001a, 0xdeec, 0x75e9, 0xb2c0, 0xddec, 0x40f2,
	0x48f4, 0x49f4, 0x001a, 0xdfa4, 0x001a, 0xb2a5, 0x001a, 0x001a,
	0x001a, 0x7bc9, 0x001a, 0x001a, 0xd2a7, 0xd4a7, 0x001a, 0xe2c9,
	0xd8ca, 0xd7ca, 0xd6ca, 0x001a, 0xe1c9, 0xe0c9, 0xa4a6, 0xd3a7,
	0xd1a7, 0xa3a6, 0x001a, 0x001a, 0x001a, 0xbda9, 0x78cc, 0x001a,
	0xbea9, 0xddca, 0x001a, 0xdfca, 0xdeca, 0x79cc, 0x001a, 0x001a,
	0xdaca, 0x001a, 0xd8a7, 0xd6a7, 0x001a, 0xd9ca, 0xdbca, 0xe1ca,
	0x001a, 0xd5a7, 0x001a, 0xdcca, 0xe5ca, 0xc0a9, 0x001a, 0xe2ca,
	0xd7a7, 0x001a, 0xe0ca, 0xe3ca, 0x001a, 0xbfa9, 0x001a, 0xc1a9,
	0xe4ca, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xafcc, 0xa2cc, 0x7ecc, 0xaecc, 0xa9cc, 0xe7ab, 0xc2a9,
	0xaacc, 0xadcc, 0xe3ab, 0xaccc, 0xc3a9, 0xc8a9, 0xc6a9, 0xa3cc,
	0x001a, 0x7ccc, 0xa5cc, 0xcda9, 0xb0cc, 0xe4ab, 0xa6cc, 0x001a,
	0xe5ab, 0xc9a9, 0xa8cc, 0x001a, 0xcdce, 0xe6ab, 0x7bcc, 0xcaa9,
	0xe8ab, 0xcba9, 0xc7a9, 0xcca9, 0xa7cc, 0x7acc, 0xabcc, 0xc4a9,
	0x001a, 0x001a, 0x7dcc, 0xa4cc, 0xa1cc, 0xc5a9, 0x001a, 0xbfce,
	0x001a, 0xc0ce, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcace, 0xa1d1, 0xcbce, 0xeeab, 0xcece, 0xc4ce, 0xedab, 0xc6ce,
	0x001a, 0xc7ce, 0x001a, 0x001a, 0xc9ce, 0xe9ab, 0x001a, 0x001a,
	0xa3ae, 0x001a, 0xdaf9, 0xc5ce, 0xc1ce, 0xa4ae, 0x001a, 0x001a,
	0xcfce, 0x7eae, 0x7dd1, 0xc8ce, 0x001a, 0x7cd1, 0xc3ce, 0xccce,
	0x001a, 0x001a, 0xecab, 0xa1ae, 0xf2ab, 0xa2ae, 0xd0ce, 0x7ed1,
	0xebab, 0xa6ae, 0xf1ab, 0xf0ab, 0xefab, 0xa5ae, 0xd1ce, 0xa7ae,
	0xeaab, 0x001a, 0xc2ce, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x76b1,
	0xa4d1, 0xa6d1, 0x001a, 0xa8d1, 0xa8ae, 0xaeae, 0x53d5, 0xacd1,
	0xa3d1, 0x78b1, 0x51d5, 0x001a, 0xadae, 0xabae, 0xaed1, 0x001a,
	0x52d5, 0x001a, 0xa5d1, 0x001a, 0xacae, 0xa9d1, 0xafae, 0xabd1,
	0x001a, 0x001a, 0xaaae, 0xaad1, 0xadd1, 0xa7d1, 0x001a, 0xa9ae,
	0x79b1, 0x001a, 0xa2d1, 0x77b1, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7ab1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x55d5, 0x5ed5, 0x64b4, 0x001a, 0x7cb1, 0xa3b1, 0x65b4, 0x60d5,
	0xaab1, 0xf9d8, 0x56d5, 0xa2b1, 0xa5b1, 0x7eb1, 0x54d5, 0x62d5,
	0x65d5, 0x49d9, 0x001a, 0x63d5, 0xfdd8, 0xa1b1, 0xa8b1, 0xacb1,
	0x5dd5, 0xf8d8, 0x61d5, 0x7bb1, 0xfad8, 0x64d5, 0xfcd8, 0x59d5,
	0x001a, 0x62b4, 0x001a, 0x57d5, 0x58d5, 0xa7b1, 0x001a, 0x001a,
	0xa6b1, 0x5bd5, 0xabb1, 0x5fd5, 0xa4b1, 0x5cd5, 0x001a, 0xa9b1,
	0x66b4, 0x63b4, 0xfbd8, 0x001a, 0x5ad5, 0x001a, 0x7db1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6bb4, 0x6fb4, 0x40d9, 0x51b7, 0x6db4, 0x44d9, 0x71b4, 0x65dd,
	0x46d9, 0x53b7, 0x69b4, 0x6cb4, 0x47d9, 0x001a, 0x48d9, 0x4ed9,
	0x73b4, 0x54b7, 0x001a, 0x4ad9, 0x4fd9, 0x43d9, 0x5eb7, 0x001a,
	0x55b7, 0x72b4, 0x41d9, 0x50d9, 0x001a, 0x5db7, 0x70b4, 0x4eb7,
	0x4dd9, 0x001a, 0x74b4, 0x45d9, 0xfed8, 0x6ab4, 0x42d9, 0x001a,
	0x4bd9, 0x001a, 0x4db7, 0x52b7, 0x67b4, 0x4cd9, 0x001a, 0x50b7,
	0x001a, 0x001a, 0x001a, 0x68b4, 0x001a, 0x001a, 0x001a, 0x5cb7,
	0xc3e1, 0x70dd, 0x001a, 0x68dd, 0xc2e1, 0x001a, 0x6cdd, 0x6edd,
	0x001a, 0x001a, 0x6bdd, 0x001a, 0x5bb7, 0x001a, 0x6add, 0x5fb7,
	0x001a, 0x001a, 0x001a, 0xd2e1, 0x001a, 0x001a, 0x5ab7, 0x40ba,
	0x71dd, 0xc4e1, 0x001a, 0x001a, 0x58b7, 0x69dd, 0x6ddd, 0xfeb9,
	0x4fb7, 0x66dd, 0x67dd, 0x41ba, 0x57b7, 0x59b7, 0x56b7, 0x6fdd,
	0x001a, 0x001a, 0xc8e1, 0xc9e1, 0xcee1, 0x7dbc, 0xd5e1, 0x001a,
	0x47ba, 0x001a, 0x46ba, 0xd0e1, 0x001a, 0x7cbc, 0xc5e1, 0x45ba,
	0x001a, 0xd4e1, 0x43ba, 0x44ba, 0x001a, 0xd1e1, 0xaae5, 0x7abc,
	0x6eb4, 0x001a, 0xd3e1, 0xa3bc, 0xcbe1, 0x001a, 0x7bbc, 0x001a,
	0xa2bc, 0xc6e1, 0xcae1, 0xc7e1, 0xcde1, 0x48ba, 0x79bc, 0x42ba,
	0x001a, 0x7ae5, 0xcfe1, 0x001a, 0xa1bc, 0x001a, 0xa4bc, 0x001a,
	0xcce1, 0x001a, 0x7ebc, 0x79e5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x7ee5, 0xcebe, 0x78e5, 0xa3e9, 0xa9e5, 0xa8bc, 0x001a,
	0xa6bc, 0xccbe, 0xa6e5, 0xa2e5, 0xacbc, 0x001a, 0x78e9, 0x001a,
	0x001a, 0x001a, 0xaabc, 0xa1e5, 0x001a, 0x76e9, 0x001a, 0xa5e5,
	0x001a, 0xa8e5, 0x7de5, 0x001a, 0xabbc, 0x001a, 0x001a, 0xa5bc,
	0x77e9, 0xcdbe, 0xa7e5, 0xa7bc, 0xa9bc, 0xa4e5, 0xadbc, 0xa3e5,
	0x7ce5, 0x7be5, 0xcbbe, 0xabe5, 0x7ae9, 0xe0ec, 0xd0be, 0x001a,
	0xa2e9, 0x001a, 0x7ee9, 0x001a, 0xe1ec, 0x001a, 0xd1be, 0xa1e9,
	0x001a, 0x7ce9, 0xb4c0, 0xdfec, 0x001a, 0x79e9, 0x7be9, 0xb5c0,
	0xd3be, 0xb3c0, 0xd2be, 0xb7c0, 0x7de9, 0xcfbe, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcfef, 0x001a,
	0xc7ef, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe7ec, 0xc8ef,
	0xe3ec, 0x001a, 0x001a, 0x56c2, 0xe5ec, 0xe4ec, 0xb6c0, 0xe2ec,
	0xe6ec, 0xd0ef, 0xccef, 0xceef, 0x001a, 0xc9ef, 0xcaef, 0x001a,
	0xcdef, 0xcbef, 0x67c3, 0x001a, 0x001a, 0x6ac3, 0x69c3, 0x68c3,
	0x61c4, 0x4af4, 0x62c4, 0x41f2, 0xdfc4, 0xccf5, 0xe0c4, 0x74c5,
	0xcac5, 0xd9f7, 0x001a, 0xdaf7, 0xdbf7, 0x001a, 0x001a, 0xbaf9,
	0xe0a4, 0x7cc9, 0xb3a5, 0x001a, 0xa6a6, 0xa7a6, 0xa5a6, 0x001a,
	0xa8a6, 0xdaa7, 0xd9a7, 0x001a, 0xb1cc, 0xcfa9, 0xcea9, 0x001a,
	0x001a, 0xafd1, 0xadb1, 0xaeb1, 0x001a, 0x001a, 0x001a, 0x75b4,
	0x72dd, 0x60b7, 0x61b7, 0x74dd, 0x76dd, 0x75dd, 0x001a, 0xd7e1,
	0x001a, 0xd6e1, 0x49ba, 0xd8e1, 0x001a, 0xace5, 0xaebc, 0x001a,
	0xd4be, 0x001a, 0xb8c0, 0x57c2, 0xb9c0, 0x001a, 0xe1a4, 0x001a,
	0x001a, 0x001a, 0xe6ca, 0x001a, 0x001a, 0xb2cc, 0xd1a9, 0xd0a9,
	0xd2a9, 0xf3ab, 0xd2ce, 0xd3ce, 0x001a, 0x001a, 0xb0d1, 0xb0ae,
	0xafb1, 0x76b4, 0x51d9, 0xe2a4, 0x001a, 0x7ea4, 0xe3a4, 0x001a,
	0x7dc9, 0xb7a5, 0xb6a5, 0xb4a5, 0xb5a5, 0x001a, 0x001a, 0x001a,
	0xaba6, 0xe9c9, 0xebc9, 0xaaa6, 0xe3c9, 0x001a, 0xe4c9, 0x001a,
	0xeac9, 0xe6c9, 0xe8c9, 0xa9a6, 0xe5c9, 0xecc9, 0xe7c9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe1a7, 0xeaa7, 0xe8a7,
	0xf0ca, 0xedca, 0xf5ca, 0xe6a7, 0xf6ca, 0x001a, 0xdfa7, 0xf3ca,
	0x001a, 0xe5a7, 0xefca, 0xeeca, 0xe3a7, 0xf4ca, 0xe4a7, 0xd3a9,
	0xdea7, 0xf1ca, 0x001a, 0xe7ca, 0xdba7, 0x001a, 0xeea7, 0xecca,
	0xf2ca, 0xe0a7, 0xe2a7, 0x001a, 0xe8ca, 0x001a, 0xe9ca, 0xeaca,
	0x001a, 0xeda7, 0xe7a7, 0xeca7, 0xebca, 0xeba7, 0xdda7, 0xdca7,
	0xe9a7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe1a9, 0xbecc, 0xb7cc, 0xdca9, 0xefa9, 0xb3cc, 0xbacc, 0xbccc,
	0xbfcc, 0xeaa9, 0x001a, 0xbbcc, 0xb4cc, 0xe8a9, 0xb8cc, 0x001a,
	0xc0cc, 0xd9a9, 0x001a, 0xbdcc, 0xe3a9, 0xe2a9, 0xb6cc, 0xd7a9,
	0x001a, 0x001a, 0xd8a9, 0x001a, 0xd6a9, 0x001a, 0xeea9, 0xe6a9,
	0xe0a9, 0xd4a9, 0xb9cc, 0xdfa9, 0xd5a9, 0xe7a9, 0xf0a9, 0xd4ce,
	0xe4a9, 0xb5cc, 0xdaa9, 0xdda9, 0xdea9, 0x001a, 0xeca9, 0xeda9,
	0xeba9, 0xe5a9, 0xe9a9, 0xdba9, 0xf4ab, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdace, 0x41ac, 0xf8ab, 0xfaab, 0x40ac,
	0xe6ce, 0xfdab, 0xb1d1, 0xb1ae, 0x43ac, 0xd7ce, 0xdfce, 0xfeab,
	0xdece, 0xdbce, 0xe3ce, 0xe5ce, 0xf7ab, 0xfbab, 0x42ac, 0xb3ae,
	0xe0ce, 0xf9ab, 0x45ac, 0xd9ce, 0x001a, 0x001a, 0x001a, 0xfcab,
	0xb2ae, 0xf6ab, 0x001a, 0xd6ce, 0xddce, 0xd5ce, 0xd8ce, 0xdcce,
	0xb2d1, 0x44ac, 0x001a, 0xe1ce, 0xe2ce, 0xe4ce, 0xf5ab, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc1ae, 0xbed1, 0xbfae, 0xc0ae, 0xb4d1, 0xc4d1, 0x001a, 0xb6ae,
	0x001a, 0x001a, 0x66d5, 0xc6d1, 0xc0d1, 0x001a, 0xb7d1, 0x001a,
	0xc9d1, 0xbad1, 0xbcae, 0x7dd5, 0xbdd1, 0xbeae, 0xb5ae, 0x001a,
	0xcbd1, 0xbfd1, 0xb8ae, 0xb8d1, 0xb5d1, 0xb6d1, 0xb9ae, 0xc5d1,
	0xccd1, 0xbbae, 0xbcd1, 0xbbd1, 0xc3ae, 0xc2ae, 0xb4ae, 0xbaae,
	0xbdae, 0xc8d1, 0x001a, 0x001a, 0xc2d1, 0xb7ae, 0xb3d1, 0xcad1,
	0xc1d1, 0xc3d1, 0xc7d1, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x67d5, 0x001a, 0xb7b1,
	0xcbb1, 0xcab1, 0x001a, 0xbfb1, 0x001a, 0x79d5, 0x75d5, 0x72d5,
	0xa6d5, 0xbab1, 0xb2b1, 0x001a, 0x001a, 0x77d5, 0xa8b4, 0xb6b1,
	0xa1d5, 0x001a, 0xccb1, 0xc9b1, 0x7bd5, 0x6ad5, 0x001a, 0x001a,
	0xc8b1, 0xa3d5, 0x69d5, 0xbdb1, 0xc1b1, 0xa2d5, 0x001a, 0x73d5,
	0xc2b1, 0xbcb1, 0x68d5, 0x001a, 0x78b4, 0xa5d5, 0x71d5, 0xc7b1,
	0x74d5, 0xa4d5, 0xc6b1, 0x001a, 0x52d9, 0x001a, 0xb3b1, 0x6fd5,
	0xb8b1, 0xc3b1, 0x001a, 0xbeb1, 0x78d5, 0x6ed5, 0x6cd5, 0x7ed5,
	0xb0b1, 0xc4b1, 0xb4b1, 0x77b4, 0x7cd5, 0xb5b1, 0x001a, 0xb1b1,
	0xc0b1, 0xbbb1, 0xb9b1, 0x70d5, 0xc5b1, 0x6dd5, 0x7ad5, 0x76d5,
	0x54d9, 0x53d9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6bd5, 0x64d9, 0x001a,
	0x7ab4, 0x001a, 0x6ad9, 0x59d9, 0x67d9, 0x77dd, 0x7db4, 0x6bd9,
	0x6ed9, 0x7cb4, 0x5cd9, 0x6dd9, 0x6cd9, 0x7eb4, 0x55d9, 0x79b4,
	0xa3b4, 0x001a, 0xa1b4, 0x69d9, 0x001a, 0x5fd9, 0xa5b4, 0x70d9,
	0x68d9, 0x71d9, 0xadb4, 0xabb4, 0x66d9, 0x65d9, 0x001a, 0x63d9,
	0x5dd9, 0xa4b4, 0x001a, 0xa2b4, 0xb9d1, 0x56d9, 0x001a, 0xb7dd,
	0x57d9, 0x7bb4, 0xaab4, 0x79dd, 0x001a, 0xa6b4, 0xa7b4, 0x58d9,
	0x6fd9, 0x78dd, 0x60d9, 0x5bd9, 0xa9b4, 0x61d9, 0x5ed9, 0x001a,
	0x001a, 0xaeb4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x70b7, 0x001a,
	0x001a, 0x7cdd, 0xb1dd, 0xb6dd, 0xaadd, 0x6cb7, 0xbbdd, 0x69b7,
	0x7add, 0x001a, 0x7bdd, 0x62b7, 0x6bb7, 0xa4dd, 0x6eb7, 0x6fb7,
	0xa5dd, 0x001a, 0xb2dd, 0xb8dd, 0x6ab7, 0x001a, 0x64b7, 0xa3dd,
	0x7ddd, 0xbadd, 0xa8dd, 0xa9dd, 0x7edd, 0xb4dd, 0xabdd, 0xb5dd,
	0xaddd, 0x001a, 0x65b7, 0xd9e1, 0x68b7, 0x66b7, 0xb9dd, 0xb0dd,
	0xacdd, 0x001a, 0x001a, 0xa1dd, 0x53ba, 0xafdd, 0x6db7, 0xa7dd,
	0x001a, 0xa6dd, 0x001a, 0x001a, 0x001a, 0x67b7, 0x63b7, 0xeee1,
	0xb3dd, 0xaedd, 0x001a, 0xa2dd, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe9e1, 0x001a, 0xdae1, 0xe5e1, 0x001a,
	0xece1, 0x51ba, 0xacb4, 0xeae1, 0x4cba, 0x001a, 0x001a, 0x001a,
	0x4bba, 0xf1e1, 0x001a, 0xdbe1, 0xe8e1, 0xdce1, 0xe7e1, 0x4fba,
	0xebe1, 0x62d9, 0x001a, 0x001a, 0x001a, 0xf2e1, 0xe3e1, 0x52ba,
	0xbae5, 0xafbc, 0x001a, 0xf0e1, 0xefe1, 0x54ba, 0xade5, 0xb0bc,
	0xaee5, 0x001a, 0xdfe1, 0xe0e1, 0xdde1, 0xe2e1, 0xdee1, 0xf3e1,
	0x4eba, 0xb1bc, 0x50ba, 0x55ba, 0x001a, 0xe1e1, 0x001a, 0xede1,
	0x001a, 0x001a, 0xe6e1, 0x001a, 0x001a, 0xb1e5, 0x001a, 0x4aba,
	0xb4bc, 0xaae9, 0xb6e5, 0xb5e5, 0xb7e5, 0x001a, 0x001a, 0xb4e5,
	0xb5bc, 0x001a, 0xbbbc, 0xb8bc, 0x001a, 0xb9bc, 0xafe5, 0xb2e5,
	0xbce5, 0xc1bc, 0xbfbc, 0x001a, 0xb3e5, 0x5ad9, 0xb2bc, 0xb9e5,
	0xb0e5, 0x001a, 0xc2bc, 0xb8e5, 0x4dba, 0xb7bc, 0xe4e1, 0x001a,
	0x001a, 0xbabc, 0x001a, 0xbebc, 0xc0bc, 0xbdbc, 0xbcbc, 0x001a,
	0xb6bc, 0xbbe5, 0xb3bc, 0xc3bc, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xd8be, 0xd9be, 0xa9e9, 0xe2be, 0xdfbe,
	0x001a, 0xd6be, 0xddbe, 0xabe9, 0xdbbe, 0xd5be, 0x001a, 0xdcbe,
	0x001a, 0xa8e9, 0xbbc0, 0xd7be, 0x001a, 0xdebe, 0xbac0, 0xa7e9,
	0xa6e9, 0x001a, 0xe0be, 0x001a, 0xe1be, 0x001a, 0xa5e9, 0xa4e9,
	0xbcc0, 0xaee9, 0xdabe, 0xace9, 0x001a, 0x001a, 0x001a, 0x001a,
	0xbdc0, 0x001a, 0xc2c0, 0xeaec, 0xecec, 0x001a, 0xbfc0, 0x001a,
	0xedec, 0xe9ec, 0x001a, 0xebec, 0xc0c0, 0xc3c0, 0x001a, 0xe8ec,
	0xbec0, 0xc1c0, 0x59c2, 0xade9, 0x58c2, 0x001a, 0x001a, 0x5ec2,
	0xd4ef, 0x001a, 0x5cc2, 0x5dc2, 0xd7ef, 0xd3ef, 0x5ac2, 0xd1ef,
	0x6bc3, 0xd5ef, 0x001a, 0xd6ef, 0xd2ef, 0x001a, 0x5bc2, 0x42f2,
	0x001a, 0x45f2, 0x001a, 0x001a, 0x46f2, 0x44f2, 0x47f2, 0x6cc3,
	0x43f2, 0x001a, 0x001a, 0x4ef4, 0x64c4, 0x4df4, 0x4cf4, 0x4bf4,
	0x63c4, 0x65c4, 0x001a, 0xcdf5, 0xe2c4, 0xe1c4, 0x001a, 0x001a,
	0xe1f6, 0xe0f6, 0xe3f6, 0xcbc5, 0x75c5, 0xddf7, 0xe2f6, 0x001a,
	0x001a, 0xdcf7, 0xcdc5, 0xccc5, 0xf3c5, 0xa9f8, 0xeff8, 0xe4a4,
	0x001a, 0x001a, 0x72d9, 0xafe9, 0x001a, 0x001a, 0xaca6, 0xf7ca,
	0xf1a7, 0xefa7, 0x001a, 0xf0a7, 0x001a, 0xc1cc, 0xf1a9, 0x46ac,
	0x001a, 0xe7ce, 0x001a, 0xe8ce, 0x001a, 0x47ac, 0xced1, 0x001a,
	0xc4ae, 0xc5ae, 0xcdd1, 0x001a, 0x001a, 0x001a, 0x001a, 0xd3b1,
	0x001a, 0xcfb1, 0x001a, 0xa7d5, 0xd6b1, 0xd5b1, 0xceb1, 0xd1b1,
	0xd4b1, 0xd0b1, 0x001a, 0x001a, 0x76d9, 0xcdb1, 0xafb4, 0x001a,
	0x001a, 0x001a, 0xb1b4, 0xb2b4, 0x75d9, 0x78d9, 0xb0b4, 0x73d9,
	0x77d9, 0x001a, 0x74d9, 0x001a, 0x71b7, 0x001a, 0x001a, 0xbcdd,
	0x001a, 0x001a, 0x56ba, 0xf4e1, 0xe3be, 0xc4bc, 0xbde5, 0xc5bc,
	0xc6bc, 0xbfe5, 0xbee5, 0xc0e5, 0xb1e9, 0x001a, 0x001a, 0xb0e9,
	0xefec, 0xeeec, 0xc4c0, 0xc5c0, 0x48f2, 0x001a, 0x001a, 0xe5a4,
	0x001a, 0x001a, 0x001a, 0x001a, 0x79d9, 0x001a, 0x001a, 0x001a,
	0xb4b4, 0xb3b4, 0xbddd, 0x001a, 0xd8ef, 0xe3c4, 0xdef7, 0xe6a4,
	0x001a, 0xc6ae, 0x001a, 0xd8b1, 0xd7b1, 0x7ad9, 0x7bd9, 0x72b7,
	0xf5e1, 0x57ba, 0xb2e9, 0x001a, 0xe7a4, 0xb8a5, 0x001a, 0xf2a9,
	0xc2cc, 0x001a, 0xe9ce, 0x48ac, 0xd9b1, 0x001a, 0x7cd9, 0xb5b4,
	0x73b7, 0x001a, 0xc1e5, 0xc2e5, 0x001a, 0x001a, 0xf0ec, 0x5fc2,
	0xf0f8, 0xe8a4, 0x001a, 0xc3cc, 0xf3a9, 0x49ac, 0x001a, 0xeace,
	0x001a, 0xc7ae, 0xd2d1, 0xd0d1, 0xd1d1, 0xc8ae, 0xcfd1, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdbb1, 0xdcb1, 0xa8d5, 0xddb1, 0xdab1,
	0x7dd9, 0x001a, 0x7ed9, 0xbedd, 0x001a, 0x001a, 0x59ba, 0x58ba,
	0x001a, 0x001a, 0xf1ec, 0xd9ef, 0x001a, 0x4af2, 0x49f2, 0x4ff4,
	0x001a, 0x5ec9, 0x4aac, 0x001a, 0x001a, 0xe9a4, 0xb9a5, 0x001a,
	0xaea6, 0xada6, 0x001a, 0x001a, 0xafa6, 0xb0a6, 0xeec9, 0xedc9,
	0xf8ca, 0xf2a7, 0xfbca, 0xfaca, 0xf9ca, 0xfcca, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf4a9, 0xc9cc, 0xc5cc, 0xcecc, 0x001a, 0x001a,
	0xfba9, 0x001a, 0xf9a9, 0xcacc, 0xc6cc, 0xcdcc, 0xf8a9, 0x40aa,
	0xc8cc, 0xc4cc, 0xfea9, 0xcbcc, 0xf7a9, 0xcccc, 0xfaa9, 0xfca9,
	0xd0cc, 0xcfcc, 0xc7cc, 0xf6a9, 0xf5a9, 0xfda9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xefce, 0xf5ce, 0x001a, 0x50ac,
	0x4dac, 0xecce, 0xf1ce, 0x001a, 0x53ac, 0x4bac, 0xf0ce, 0x4eac,
	0x51ac, 0x001a, 0x001a, 0xf3ce, 0x001a, 0x4cac, 0xf8ce, 0x4fac,
	0x001a, 0x52ac, 0xedce, 0xf2ce, 0xf6ce, 0xeece, 0xebce, 0x001a,
	0x001a, 0xf7ce, 0xf4ce, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd0ae, 0xc9ae, 0xccae, 0x001a, 0xcfae, 0x001a, 0xd5d1,
	0x001a, 0xcaae, 0xd3d1, 0x001a, 0xceae, 0x001a, 0x001a, 0xcbae,
	0x001a, 0xd6d1, 0xcdae, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xacd5, 0xdfb1, 0xabd5, 0xadd5, 0xdeb1, 0xe3b1, 0xd4d1,
	0x001a, 0xaad5, 0xaed5, 0x001a, 0xe0b1, 0xa9d5, 0xe2b1, 0x001a,
	0xe1b1, 0x001a, 0xa7d9, 0x001a, 0xa2d9, 0x001a, 0xb6b4, 0xbab4,
	0xb7b4, 0xa5d9, 0xa8d9, 0x001a, 0xb8b4, 0x001a, 0xb9b4, 0xbeb4,
	0xc7dd, 0xa6d9, 0xbcb4, 0xa3d9, 0xa1d9, 0x001a, 0xbdb4, 0x001a,
	0xa4d9, 0x001a, 0x001a, 0x001a, 0x79b7, 0x001a, 0xbfdd, 0x76b7,
	0x77b7, 0x75b7, 0xc4dd, 0xc3dd, 0xc0dd, 0x7bb7, 0x001a, 0x001a,
	0xc2dd, 0xbbb4, 0x001a, 0x001a, 0xc6dd, 0xc1dd, 0x78b7, 0x74b7,
	0x7ab7, 0xc5dd, 0x001a, 0x001a, 0x001a, 0x5cba, 0x001a, 0xf8e1,
	0xf7e1, 0xf6e1, 0x5aba, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x5bba, 0xc5e5, 0xc8e5, 0xc8bc, 0x001a, 0x001a, 0xc7bc, 0xc9e5,
	0xc4e5, 0xcabc, 0xc6e5, 0x001a, 0xc9bc, 0xc3e5, 0x001a, 0xc7e5,
	0xe9be, 0xe6be, 0xbbe9, 0xbae9, 0x001a, 0xb9e9, 0xb4e9, 0x001a,
	0xb5e9, 0x001a, 0x001a, 0x001a, 0xe7be, 0x001a, 0xe4be, 0xe8be,
	0xb3e9, 0xe5be, 0xb6e9, 0xb7e9, 0xbce9, 0x001a, 0x001a, 0xb8e9,
	0x001a, 0x001a, 0xf2ec, 0x001a, 0x001a, 0x001a, 0xc7c0, 0x001a,
	0xdcef, 0xc6c0, 0xdaef, 0xdbef, 0x60c2, 0x6ec3, 0x4bf2, 0x001a,
	0x6dc3, 0x001a, 0x001a, 0x51f4, 0x52f4, 0x001a, 0x66c4, 0x001a,
	0x50f4, 0xe4c4, 0x001a, 0xdff7, 0xcec5, 0xaaf8, 0xabf8, 0x001a,
	0xeaa4, 0x001a, 0xb1a6, 0xb2a6, 0xf3a7, 0x001a, 0xd1cc, 0x54ac,
	0xd1ae, 0xe4b1, 0x001a, 0x001a, 0xd2b0, 0x001a, 0xbfb4, 0xc0b4,
	0xccb3, 0xa9d9, 0x001a, 0x7cb7, 0xfae1, 0xf9e1, 0x001a, 0x001a,
	0xeba4, 0xb3a6, 0xd2cc, 0x42aa, 0x001a, 0x41aa, 0x001a, 0xf9ce,
	0xface, 0x001a, 0xd7d1, 0xd8d1, 0xd2ae, 0xd3ae, 0x001a, 0xd4ae,
	0xafd5, 0x001a, 0x001a, 0xe6b1, 0x001a, 0xc2b4, 0x001a, 0xc1b4,
	0xc8dd, 0x7adf, 0xfbe1, 0xbde9, 0x001a, 0x001a, 0x61c2, 0x67c4,
	0xeca4, 0x001a, 0xbca5, 0xbda5, 0xbba5, 0xbea5, 0xbaa5, 0x001a,
	0x001a, 0xb6a6, 0x001a, 0xf6c9, 0xb5a6, 0xb7a6, 0x001a, 0x001a,
	0xf1c9, 0xf0c9, 0xf3c9, 0xf2c9, 0xf5c9, 0xb4a6, 0xefc9, 0xf4c9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xfdca, 0xfda7, 0xfeca,
	0x43cb, 0xfca7, 0x001a, 0x47cb, 0x42cb, 0x45cb, 0xf5a7, 0xf6a7,
	0xf7a7, 0xf8a7, 0x001a, 0x40a8, 0x001a, 0x41cb, 0xfaa7, 0x41a8,
	0x001a, 0x40cb, 0x46cb, 0x001a, 0xf9a7, 0x44cb, 0xfba7, 0xf4a7,
	0xfea7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x57aa, 0x001a, 0xd4cc, 0x43aa, 0x001a, 0x4daa,
	0x4eaa, 0x46aa, 0x58aa, 0x48aa, 0xdccc, 0x53aa, 0xd7cc, 0x49aa,
	0xe6cc, 0xe7cc, 0xdfcc, 0xd8cc, 0x56aa, 0xe4cc, 0x51aa, 0x4faa,
	0x001a, 0xe5cc, 0x001a, 0xe3cc, 0xdbcc, 0xd3cc, 0xdacc, 0x4aaa,
	0x001a, 0x50aa, 0x001a, 0x44aa, 0xdecc, 0xddcc, 0xd5cc, 0x001a,
	0x52aa, 0xe1cc, 0xd6cc, 0x55aa, 0xe8cc, 0x45aa, 0x001a, 0x4caa,
	0xd9cc, 0xe2cc, 0x54aa, 0x001a, 0x47aa, 0x4baa, 0x001a, 0xe0cc,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5bcf, 0x5cac,
	0x69ac, 0x001a, 0x56cf, 0x4ccf, 0x62ac, 0x4acf, 0x5bac, 0x45cf,
	0x65ac, 0x52cf, 0xfece, 0x41cf, 0x001a, 0x001a, 0x001a, 0x001a,
	0x44cf, 0xfbce, 0x51cf, 0x61cf, 0x60ac, 0x46cf, 0x58cf, 0x001a,
	0xfdce, 0x5fcf, 0x60cf, 0x63cf, 0x5acf, 0x4bcf, 0x53cf, 0x66ac,
	0x59ac, 0x61ac, 0x6dac, 0x56ac, 0x58ac, 0x001a, 0x001a, 0x001a,
	0x43cf, 0x6aac, 0x63ac, 0x5dcf, 0x40cf, 0x6cac, 0x67ac, 0x49cf,
	0x001a, 0x001a, 0x6bac, 0x50cf, 0x48cf, 0x64ac, 0x5ccf, 0x54cf,
	0x001a, 0x5eac, 0x62cf, 0x47cf, 0x5aac, 0x59cf, 0x4fcf, 0x5fac,
	0x55cf, 0x57ac, 0xfcce, 0x68ac, 0xe3ae, 0x5dac, 0x4ecf, 0x4dcf,
	0x42cf, 0x001a, 0x5ecf, 0x001a, 0x57cf, 0x001a, 0x001a, 0x55ac,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xecd1, 0xeaae, 0xedd1, 0x001a, 0xe1d1, 0xdfae,
	0xebae, 0x001a, 0xdad1, 0x001a, 0xe3d1, 0xebd1, 0x001a, 0xd9d1,
	0xf4d1, 0xd5ae, 0x001a, 0x001a, 0x001a, 0xf3d1, 0xeed1, 0x001a,
	0xefd1, 0xddae, 0xe8ae, 0xe5d1, 0x001a, 0xe6d1, 0xf0d1, 0xe7d1,
	0x001a, 0xe2d1, 0xdcd1, 0xddd1, 0xead1, 0xe4d1, 0x001a, 0x001a,
	0xd6ae, 0xdaae, 0xf2d1, 0xded1, 0xe6ae, 0xe2ae, 0x001a, 0x001a,
	0xe5ae, 0xecae, 0xdbae, 0xe7ae, 0xe9d1, 0xe9ae, 0xd8ae, 0x001a,
	0xd7ae, 0xdbd1, 0x001a, 0xdfd1, 0xe0ae, 0xf1d1, 0xe8d1, 0xe0d1,
	0xe4ae, 0xe1ae, 0x001a, 0xd9ae, 0xdcae, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xc4d5, 0x001a, 0xb4d5, 0xb5d5, 0xb9d5,
	0x001a, 0xc8d5, 0xc5d5, 0x001a, 0xbed5, 0xbdd5, 0xedb1, 0xc1d5,
	0xd0d5, 0xb0d5, 0x001a, 0xd1d5, 0xc3d5, 0xd5d5, 0xc9d5, 0xecb1,
	0xc7d5, 0xe7b1, 0xfcb1, 0xf2b1, 0x001a, 0xf6b1, 0xf5b1, 0xb1d5,
	0x001a, 0xced5, 0xd4d5, 0xccd5, 0xd3d5, 0x001a, 0x001a, 0xc0d5,
	0xb2d5, 0xd2d5, 0xc2d5, 0xeab1, 0xf7b1, 0x001a, 0xcbd5, 0xf0b1,
	0x001a, 0x001a, 0x001a, 0xcad5, 0xb3d5, 0xf8b1, 0x001a, 0xfab1,
	0xcdd5, 0xfbb1, 0xe9b1, 0xbad5, 0xcfd5, 0x001a, 0x001a, 0xefb1,
	0xf9b1, 0xbcd5, 0xc6d5, 0xb7d5, 0xbbd5, 0xf4b1, 0xb6d5, 0xe8b1,
	0xf1b1, 0xeeb1, 0xbfd5, 0xdeae, 0xc0d9, 0xebb1, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xf3b1, 0x001a, 0xc3d9, 0xd9d9,
	0xced9, 0xd6b4, 0x001a, 0xd1b4, 0xbdd9, 0xd2b4, 0xcdd9, 0x001a,
	0xc6d9, 0xd3d9, 0xceb4, 0xabd9, 0xd5d9, 0xc4b4, 0xb3d9, 0xc7b4,
	0xc6b4, 0x001a, 0xd7b4, 0x001a, 0xadd9, 0xcfd9, 0xd0d9, 0xc9b4,
	0xc5b4, 0xbbd9, 0x001a, 0xd0b4, 0xb6d9, 0x001a, 0xd1d9, 0xccb4,
	0xc9d9, 0xd6d9, 0xb0d9, 0xb5d9, 0xafd9, 0x001a, 0xcbb4, 0xc2d9,
	0xdedd, 0xb1d9, 0xcfb4, 0xbad9, 0xd2d9, 0xcab4, 0xb7d9, 0xb4d9,
	0xc5d9, 0xcdb4, 0xc3b4, 0xd9b4, 0xc8d9, 0xc7d9, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xacd9, 0xc8b4, 0xd4d9, 0xbcd9,
	0xbed9, 0x001a, 0xcbd9, 0xcad9, 0xaad9, 0xd3b4, 0xd5b4, 0xb2d9,
	0xb9d9, 0xc1d9, 0xd4b4, 0xb8d9, 0xc4d9, 0xd7d9, 0x001a, 0xccd9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd8d9, 0x001a, 0x001a,
	0x001a, 0x001a, 0xaed9, 0x001a, 0x001a, 0x001a, 0x001a, 0xf2dd,
	0xa6b7, 0x001a, 0xf0dd, 0xdbdd, 0xe0dd, 0xd9dd, 0x001a, 0xecdd,
	0xcbdd, 0xd2dd, 0x001a, 0xeadd, 0xf4dd, 0xdcdd, 0x001a, 0xcfdd,
	0xe2dd, 0xe7dd, 0xd3dd, 0x001a, 0xe4dd, 0xd0dd, 0x001a, 0x001a,
	0xd7dd, 0xd8dd, 0xa8b7, 0xebdd, 0xe9dd, 0x001a, 0xccdd, 0xeedd,
	0x001a, 0xefdd, 0xf1dd, 0xacb7, 0xa4b7, 0x001a, 0xb8d5, 0xd4dd,
	0xe6dd, 0xd5dd, 0xa1b7, 0xb1b7, 0xeddd, 0xafb7, 0xabb7, 0xcadd,
	0xa3b7, 0x001a, 0xcddd, 0xb0b7, 0x001a, 0xdddd, 0xc9dd, 0x001a,
	0xa9b7, 0xe1dd, 0xd1dd, 0xaab7, 0xdadd, 0x7eb7, 0xd8b4, 0xe3dd,
	0xbfd9, 0xcedd, 0x001a, 0x001a, 0xe8dd, 0xa5b7, 0xe5dd, 0xa2b7,
	0xdfdd, 0xadb7, 0xd6dd, 0xf3dd, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa7b7, 0xc6de, 0x001a, 0x001a, 0xaeb7, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ae2, 0x48e2, 0x001a,
	0x5ee2, 0x46e2, 0x001a, 0x58e2, 0x7db7, 0x5fba, 0x42e2, 0x5de2,
	0x001a, 0x47e2, 0x55e2, 0x64ba, 0x5dba, 0x001a, 0x5be2, 0x001a,
	0x40e2, 0x5ae2, 0x001a, 0x6fba, 0x51e2, 0x61e2, 0x6dba, 0x49e2,
	0x5eba, 0x4be2, 0x59e2, 0x67ba, 0x44e2, 0x6bba, 0x61ba, 0x4de2,
	0x43e2, 0xfce1, 0x001a, 0x57e2, 0x68ba, 0x60e2, 0xfde1, 0x65ba,
	0x001a, 0x53e2, 0x001a, 0x66ba, 0x45e2, 0x50e2, 0x4ce2, 0x4ee2,
	0x001a, 0x60ba, 0x5fe2, 0x6eba, 0x4fe2, 0x001a, 0x62e2, 0x001a,
	0x001a, 0xfee1, 0x54e2, 0x63ba, 0x6cba, 0x6aba, 0x41e2, 0x56e2,
	0x69ba, 0x001a, 0x001a, 0x62ba, 0x52e2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x5ce2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd5e5, 0x001a, 0xd1e5, 0xcde5, 0xe1e5, 0xdee5,
	0xcdbc, 0x001a, 0x001a, 0xe5e5, 0xd4e5, 0xd8bc, 0xdbe5, 0x001a,
	0x001a, 0xd0e5, 0xdae5, 0xd5bc, 0xeee5, 0x001a, 0xebe5, 0xdde5,
	0xcee5, 0x001a, 0x001a, 0xe2e5, 0xe4e5, 0xd1bc, 0xd8e5, 0xd3e5,
	0xcae5, 0xcebc, 0xd6bc, 0x001a, 0xe7e5, 0xd7bc, 0xcbe5, 0xede5,
	0xe0e5, 0xe6e5, 0xd4bc, 0x001a, 0x001a, 0xe3e5, 0x001a, 0xeae5,
	0x001a, 0xd9bc, 0x001a, 0xd3bc, 0xdce5, 0xcfe5, 0xefe5, 0xcce5,
	0xe8e5, 0xd0bc, 0x001a, 0xd6e5, 0x001a, 0xd7e5, 0xcfbc, 0xccbc,
	0xd2e5, 0xd2bc, 0x001a, 0xcbbc, 0x001a, 0xe9e5, 0xece5, 0xd9e5,
	0xcae9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc2e9, 0x001a, 0xbee9, 0xf6be, 0x001a, 0x001a,
	0xebbe, 0xf0be, 0xecbe, 0xcce9, 0xd7e9, 0xeabe, 0xc4e9, 0xcde9,
	0xdfe5, 0xcee9, 0x001a, 0x001a, 0xf1be, 0x001a, 0xdde9, 0xf5be,
	0xf8be, 0xc0e9, 0x001a, 0xf4be, 0x001a, 0xdbe9, 0xdce9, 0xd2e9,
	0xd1e9, 0xc9e9, 0x001a, 0x001a, 0xd3e9, 0xdae9, 0xd9e9, 0x001a,
	0xefbe, 0xedbe, 0xcbe9, 0xc8e9, 0x001a, 0xc5e9, 0xd8e9, 0xf7be,
	0xd6e9, 0xf3be, 0xf2be, 0x001a, 0xd0e9, 0x001a, 0xbfe9, 0xc1e9,
	0xc3e9, 0xd5e9, 0xcfe9, 0xeebe, 0x001a, 0xc6e9, 0x001a, 0xd4e9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc7e9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcfc0, 0x45ed,
	0xc8c0, 0xf5ec, 0x001a, 0x41ed, 0xcac0, 0x48ed, 0x001a, 0xfcec,
	0x001a, 0xf7ec, 0x001a, 0x001a, 0x49ed, 0xf3ec, 0xfeec, 0x001a,
	0xd1c0, 0x44ed, 0x4aed, 0xfdec, 0xc9c0, 0x40ed, 0xf4ec, 0xd0c0,
	0x001a, 0x001a, 0x47ed, 0xf9ec, 0xccc0, 0x001a, 0xfbec, 0xf8ec,
	0xd2c0, 0xfaec, 0xcbc0, 0xcec0, 0x43ed, 0xf6ec, 0x46ed, 0x001a,
	0x42ed, 0x001a, 0x001a, 0x001a, 0x63c2, 0xe7ef, 0x68c2, 0x69c2,
	0x001a, 0x001a, 0x001a, 0x62c2, 0xe6ef, 0x001a, 0xe3ef, 0xe4ef,
	0x66c2, 0xdeef, 0xe2ef, 0x65c2, 0x001a, 0xdfef, 0x001a, 0x001a,
	0x001a, 0x001a, 0x67c2, 0x64c2, 0x001a, 0xddef, 0xe1ef, 0xe5ef,
	0x001a, 0x001a, 0x001a, 0x51f2, 0x4ef2, 0x57f2, 0x001a, 0x56f2,
	0x54f2, 0x4ff2, 0x001a, 0x72c3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x50f2, 0x71c3, 0xcdc0, 0x53f2, 0x70c3, 0x58f2, 0x52f2,
	0x4df2, 0xe0ef, 0x001a, 0x001a, 0x001a, 0x6fc3, 0x001a, 0x4cf2,
	0x56f4, 0x001a, 0x55f4, 0x55f2, 0x68c4, 0x001a, 0x59f4, 0x5af4,
	0x54f4, 0x58f4, 0x001a, 0x53f4, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd1f5, 0x57f4, 0xe7c4, 0xe5c4, 0xcff5, 0x001a, 0x001a, 0x001a,
	0xd2f5, 0x001a, 0xcef5, 0xd0f5, 0xe6c4, 0x001a, 0x001a, 0x001a,
	0xe5f6, 0xe6f6, 0x76c5, 0xe4f6, 0x001a, 0x001a, 0x001a, 0xe2f7,
	0xcfc5, 0xe0f7, 0xe1f7, 0xacf8, 0x001a, 0x001a, 0x56c6, 0xf3f8,
	0xf1f8, 0xf2f8, 0xf4f8, 0x001a, 0x001a, 0x001a, 0xbbf9, 0x001a,
	0xeda4, 0xb8a6, 0x001a, 0x59aa, 0x001a, 0xe9cc, 0x001a, 0x001a,
	0x64cf, 0x001a, 0x001a, 0x001a, 0xf5d1, 0xf7d1, 0x001a, 0xf6d1,
	0x001a, 0xf8d1, 0xfdb1, 0xd7d5, 0xf9d1, 0x001a, 0xd6d5, 0xd8d5,
	0xd9d5, 0xdad9, 0xdbb4, 0xdbd9, 0xddd9, 0xdcb4, 0xdab4, 0xdcd9,
	0x001a, 0xfadd, 0xf8dd, 0xf7dd, 0x001a, 0xf6dd, 0xf5dd, 0xb2b7,
	0xf9dd, 0x70ba, 0x63e2, 0x65e2, 0x71ba, 0x64e2, 0xdbbc, 0x001a,
	0xdabc, 0xf0e5, 0x001a, 0x001a, 0xdfe9, 0xdee9, 0xe0e9, 0x001a,
	0x001a, 0xf9be, 0x001a, 0x4bed, 0xd3c0, 0x001a, 0xe8ef, 0x6ac2,
	0x59f2, 0x77c5, 0xeea4, 0xbfa5, 0xb9a6, 0x42a8, 0x5aaa, 0x5baa,
	0x001a, 0x001a, 0x6eac, 0x001a, 0x001a, 0xfad1, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb3b7, 0x001a, 0x001a, 0x001a, 0xd1e6, 0xfabe,
	0x6bc2, 0xefa4, 0x001a, 0xbaa6, 0x001a, 0x001a, 0xebcc, 0x5caa,
	0xeacc, 0x001a, 0x65cf, 0x6fac, 0x66cf, 0x001a, 0x70ac, 0x001a,
	0xfcd1, 0xeeae, 0xedae, 0x001a, 0xded5, 0xdcd5, 0xddd5, 0xdbd5,
	0x001a, 0xdad5, 0x001a, 0x001a, 0xded9, 0xe1d9, 0xdeb4, 0xdfd9,
	0xddb4, 0xe0d9, 0x001a, 0xfbdd, 0x001a, 0x001a, 0x66e2, 0x67e2,
	0x68e2, 0x001a, 0xf3e5, 0xf2e5, 0xdcbc, 0xf1e5, 0xf4e5, 0xe1e9,
	0x001a, 0x001a, 0xe2e9, 0xe3e9, 0x001a, 0x4ced, 0xd4c0, 0x6cc2,
	0x5af2, 0x001a, 0xe8c4, 0x5fc9, 0x001a, 0x71ac, 0x67cf, 0xefae,
	0x001a, 0x001a, 0xfeb1, 0x001a, 0xdfb4, 0xe2d9, 0x001a, 0xb5b7,
	0xb4b7, 0x001a, 0x001a, 0x69e2, 0x6ae2, 0xddbc, 0xdebc, 0xe5e9,
	0xe4e9, 0xe9ef, 0xe3f7, 0xf0a4, 0x60c9, 0xc0a5, 0x001a, 0x43a8,
	0x48cb, 0x001a, 0x72ac, 0xb6b7, 0xf1a4, 0x001a, 0x68cf, 0x73ac,
	0x69cf, 0x001a, 0xd5c0, 0xf2a4, 0x001a, 0x001a, 0xeccc, 0x001a,
	0x6acf, 0x001a, 0x42d2, 0x41d2, 0xfed1, 0x001a, 0xfdd1, 0x43d2,
	0x40d2, 0x001a, 0x001a, 0x40b2, 0x41b2, 0x001a, 0x001a, 0xe0b4,
	0xe3d9, 0x001a, 0xe4d9, 0xe5d9, 0x001a, 0x001a, 0x001a, 0x41de,
	0x42de, 0x40de, 0x001a, 0xfddd, 0xfedd, 0xb7b7, 0x6be2, 0xf7e5,
	0xf6e5, 0xf5e5, 0xf8e5, 0xe7e9, 0xe6e9, 0xfbbe, 0xe8e9, 0x001a,
	0xd6c0, 0x4ded, 0x001a, 0xeaef, 0x5bf2, 0xe7f6, 0x001a, 0xf3a4,
	0xc2a5, 0xc1a5, 0x001a, 0x5daa, 0x61c9, 0x7ec9, 0xbba6, 0x001a,
	0xf7c9, 0x49cb, 0x4acb, 0x5eaa, 0x001a, 0xedcc, 0x001a, 0x74ac,
	0x6bcf, 0x6ccf, 0x001a, 0xf0ae, 0xf4ae, 0x44d2, 0xf3ae, 0xf1ae,
	0xf2ae, 0x001a, 0xdfd5, 0x42b2, 0xe3b4, 0x001a, 0xe1b4, 0xe2b4,
	0xe6d9, 0x001a, 0x001a, 0x72ba, 0xf4a4, 0x001a, 0xa1c9, 0x001a,
	0xc3a5, 0x001a, 0x001a, 0xa4c9, 0x001a, 0x001a, 0xc6a5, 0xa3c9,
	0xc5a5, 0xc4a5, 0x44a8, 0xa2c9, 0x001a, 0x001a, 0xf8c9, 0x001a,
	0x001a, 0x001a, 0xfcc9, 0xfec9, 0x40ca, 0xc5a6, 0xc6a6, 0xfbc9,
	0xc1a6, 0x001a, 0xf9c9, 0x001a, 0xfdc9, 0xc2a6, 0x001a, 0xbda6,
	0x001a, 0xbea6, 0x001a, 0xc4a6, 0xfac9, 0xbca6, 0x45a8, 0xbfa6,
	0xc0a6, 0xc3a6, 0x001a, 0x001a, 0x001a, 0x5bcb, 0x59cb, 0x4ccb,
	0x51a8, 0x53cb, 0x4ca8, 0x4dcb, 0x001a, 0x55cb, 0x001a, 0x52cb,
	0x4fa8, 0x51cb, 0x56a8, 0x5acb, 0x58a8, 0x001a, 0x5aa8, 0x001a,
	0x4bcb, 0x001a, 0x4da8, 0x5ccb, 0x001a, 0x54a8, 0x57a8, 0x001a,
	0x45cd, 0x47a8, 0x5ea8, 0x55a8, 0x4ecb, 0x4aa8, 0x59a8, 0x56cb,
	0x48a8, 0x49a8, 0x43cd, 0x4fcb, 0x50a8, 0x5ba8, 0x5dcb, 0x50cb,
	0x4ea8, 0x001a, 0x53a8, 0xeecc, 0x5ca8, 0x57cb, 0x52a8, 0x001a,
	0x5da8, 0x46a8, 0x54cb, 0x4ba8, 0x58cb, 0x44cd, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x6aaa, 0x7aaa, 0xf5cc, 0x71aa, 0x001a,
	0x4bcd, 0x62aa, 0x001a, 0x65aa, 0x42cd, 0x001a, 0xf3cc, 0xf7cc,
	0x6daa, 0x6faa, 0xfacc, 0x76aa, 0x68aa, 0x66aa, 0x67aa, 0x75aa,
	0x47cd, 0x70aa, 0xf9cc, 0xfbcc, 0x6eaa, 0x73aa, 0xfccc, 0x4acd,
	0x001a, 0x75ac, 0x79aa, 0x001a, 0x63aa, 0x49cd, 0x001a, 0x4dcd,
	0xf8cc, 0x4fcd, 0x40cd, 0x6caa, 0xf4cc, 0x6baa, 0x7daa, 0x72aa,
	0x001a, 0xf2cc, 0x75cf, 0x78aa, 0x7caa, 0x41cd, 0x46cd, 0x001a,
	0x7eaa, 0x77aa, 0x69aa, 0x5faa, 0x001a, 0x64aa, 0x001a, 0xf6cc,
	0x60aa, 0x4ecd, 0x001a, 0xf0cc, 0xefcc, 0xfdcc, 0xf1cc, 0x7baa,
	0xf5ae, 0x74aa, 0xfecc, 0x61aa, 0x001a, 0xa6ac, 0x001a, 0x001a,
	0x001a, 0x4ccd, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7ccf, 0xa1cf, 0x001a, 0xa4cf, 0x77cf, 0x001a, 0x001a, 0xa7cf,
	0xaacf, 0xaccf, 0x74cf, 0x76ac, 0x7bac, 0x49d2, 0xadac, 0xa5cf,
	0xadcf, 0x7bcf, 0x73cf, 0x001a, 0x001a, 0x001a, 0x64d2, 0x7eac,
	0xa2cf, 0x78cf, 0x7acf, 0xa5ac, 0x001a, 0x7dcf, 0x7dac, 0x70cf,
	0xa8cf, 0x001a, 0xabcf, 0x001a, 0x001a, 0x7aac, 0x001a, 0xa8ac,
	0x6dcf, 0xaaac, 0x78ac, 0xaeac, 0xa9cf, 0x6fcf, 0xabac, 0x5ed2,
	0x48cd, 0x7cac, 0x77ac, 0x76cf, 0x6ecf, 0xacac, 0xa4ac, 0xa3cf,
	0xa9ac, 0xa7ac, 0x79cf, 0xa1ac, 0x71cf, 0xa2ac, 0xa3ac, 0x72cf,
	0xa6cf, 0x79ac, 0x7ecf, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4cd2, 0xfdae, 0x43af, 0x001a, 0x001a, 0x001a, 0x55d2, 0x5bd2,
	0x57d2, 0x4ad2, 0x4dd2, 0x46d2, 0x47d2, 0x4aaf, 0xfaae, 0x56d2,
	0x5fd2, 0x45af, 0xf6ae, 0x001a, 0x40af, 0x4ed2, 0x42af, 0x4fd2,
	0x59d2, 0x001a, 0x001a, 0x001a, 0x44af, 0x68d2, 0x48d2, 0xfcae,
	0xfbae, 0x48af, 0x45d2, 0x66d2, 0x5ad2, 0x67d2, 0x61d2, 0x53d2,
	0x62d2, 0x001a, 0x5cd2, 0x65d2, 0x63d2, 0x49af, 0x54d2, 0xf9ae,
	0xf8ae, 0x41af, 0x47af, 0x60d2, 0x46af, 0x51d2, 0x43b2, 0x001a,
	0x69d2, 0x50d2, 0x4bd2, 0xfeae, 0x4baf, 0xf7ae, 0x001a, 0x58d2,
	0x5dd2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x65b2, 0xe1d5, 0xe5d5, 0x001a, 0x52b2, 0x50b2,
	0x001a, 0x001a, 0x47b2, 0xe3d5, 0xe2d5, 0x5bb2, 0x001a, 0xe8d5,
	0x55b2, 0x001a, 0xfad5, 0x47d6, 0x44b2, 0xf7d5, 0xf0d5, 0x67b2,
	0xe0d5, 0x001a, 0xfcd5, 0x001a, 0x64b2, 0x58b2, 0x63b2, 0x4eb2,
	0xecd5, 0xfed5, 0xf6d5, 0x4fb2, 0x49b2, 0x45d6, 0x001a, 0xfdd5,
	0x40d6, 0x51b2, 0x59b2, 0x42d6, 0xead5, 0xfbd5, 0xefd5, 0x44d6,
	0x5eb2, 0x46b2, 0x5cb2, 0xf4d5, 0xf2d5, 0xf3d5, 0x53b2, 0xeed5,
	0xedd5, 0x48b2, 0xe7d5, 0x46d6, 0x4ab2, 0xf1d5, 0x68b2, 0x001a,
	0x62b2, 0xe6d5, 0x5fb2, 0x5db2, 0x66b2, 0xf8d5, 0x61b2, 0x52d2,
	0xf9d5, 0x60b2, 0x41d6, 0x45b2, 0xf5d5, 0x57b2, 0xe9d5, 0x56b2,
	0x001a, 0x54b2, 0x4cb2, 0x4bb2, 0xe7d9, 0x43d6, 0x001a, 0x001a,
	0xebd5, 0x001a, 0x001a, 0xfcd9, 0x001a, 0x4db2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x41b5, 0x5ab2, 0xeeb4, 0xf6d9, 0xfcb4, 0x001a, 0xead9,
	0xebb4, 0xe7b4, 0x49da, 0xedb4, 0xf1b4, 0xecb4, 0xf5b4, 0x4dda,
	0x44da, 0x001a, 0x001a, 0xf1d9, 0xfab4, 0xf4b4, 0xfdd9, 0xe4b4,
	0x4ada, 0x43da, 0xe8b4, 0xf7d9, 0xf7b4, 0x55da, 0x56da, 0x001a,
	0xe5b4, 0x48da, 0xf9b4, 0xfbd9, 0xedd9, 0xeed9, 0xfdb4, 0xf2d9,
	0xf9d9, 0xf3d9, 0x001a, 0xfbb4, 0x44b5, 0xefd9, 0xe8d9, 0xe9d9,
	0x001a, 0xebd9, 0xeab4, 0xf8d9, 0x001a, 0xf8b4, 0x42b5, 0x001a,
	0x001a, 0xfad9, 0x53da, 0x4bda, 0xe6b4, 0x51da, 0xf2b4, 0x001a,
	0xf0b4, 0x001a, 0x57da, 0xefb4, 0x41da, 0xf4d9, 0xfed9, 0x47b5,
	0x45da, 0x42da, 0xf0d9, 0x43b5, 0x4fda, 0x4cda, 0x54da, 0xe9b4,
	0x40da, 0x46b5, 0x001a, 0x47da, 0x001a, 0x001a, 0xf3b4, 0xf6b4,
	0x001a, 0x46da, 0x45b5, 0xf5d9, 0xe4d5, 0x001a, 0x001a, 0x50da,
	0x4eda, 0x52da, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xecd9, 0x40b5, 0x001a, 0x001a, 0x001a, 0x61de, 0x60de, 0x46de,
	0xbdb7, 0x001a, 0x5fde, 0x49de, 0x4ade, 0x001a, 0xc7b7, 0x68de,
	0xc2b7, 0x5ede, 0x001a, 0x43de, 0xc8b7, 0xbeb7, 0x52de, 0x48de,
	0x4bde, 0x63de, 0xb8b7, 0x6ade, 0x62de, 0xc1b7, 0x57de, 0xccb7,
	0x001a, 0x001a, 0xcbb7, 0xc5b7, 0x001a, 0x001a, 0x69de, 0xb9b7,
	0x55de, 0x4cde, 0x59de, 0x65de, 0xcdb7, 0x001a, 0xbbb7, 0x54de,
	0x001a, 0x4dde, 0xc4b7, 0x001a, 0xc3b7, 0x50de, 0x5ade, 0x64de,
	0x47de, 0x51de, 0xbcb7, 0x5bde, 0xc9b7, 0xc0b7, 0x4ede, 0xbfb7,
	0x45de, 0x53de, 0x67de, 0xfeb4, 0xb0ba, 0x56de, 0x6ce2, 0x58de,
	0x66de, 0xc6b7, 0x4fde, 0xbab7, 0xcab7, 0xf0bc, 0x44de, 0x001a,
	0x5dde, 0x001a, 0x001a, 0x001a, 0x5cde, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xaae2, 0xadba, 0x7de2, 0xa4e2, 0xa2ba,
	0x001a, 0x6ee2, 0xafba, 0x001a, 0x77ba, 0x6de2, 0xb0e2, 0xb1ba,
	0x71e2, 0xa3e2, 0x001a, 0x73e2, 0xb3e2, 0xafe2, 0x75ba, 0xa1ba,
	0x53e6, 0xaeba, 0x7dba, 0x6fe2, 0x001a, 0xaee2, 0xa3ba, 0xabe2,
	0xb8e2, 0x75e2, 0x7ee2, 0x001a, 0x001a, 0xb6e2, 0xace2, 0x7cba,
	0x001a, 0x001a, 0x7ce2, 0x76ba, 0x74ba, 0xa8ba, 0x001a, 0x001a,
	0x7ae2, 0x77e2, 0x78e2, 0x001a, 0xb2e2, 0x001a, 0xb7e2, 0xb5e2,
	0x7aba, 0xb9e2, 0x7eba, 0xa7ba, 0x001a, 0x70e2, 0xfae5, 0x79e2,
	0x001a, 0x78ba, 0xacba, 0xa9ba, 0x7bba, 0xa5e2, 0x74e2, 0xaaba,
	0xa7e2, 0xa4ba, 0xa6ba, 0x73ba, 0x001a, 0xa9e2, 0xa1e2, 0x72e2,
	0xa5ba, 0xb1e2, 0xb4e2, 0x7be2, 0xa8e2, 0x001a, 0x79ba, 0xdfbc,
	0xa6e2, 0xf9e5, 0x001a, 0xade2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x76e2, 0x44e6,
	0x4ee6, 0xe2bc, 0x4de6, 0x59e6, 0xe4bc, 0x4be6, 0x001a, 0x4fe6,
	0xefbc, 0x001a, 0x46e6, 0xe7bc, 0x001a, 0x52e6, 0xf0e9, 0xf3bc,
	0xf2bc, 0x54e6, 0x43e6, 0x5ee6, 0xedbc, 0x001a, 0xe3bc, 0x57e6,
	0x001a, 0x5be6, 0x60e6, 0x55e6, 0x49e6, 0xe6bc, 0xe9bc, 0xf1bc,
	0xecbc, 0x001a, 0x4ce6, 0xa2e2, 0x001a, 0x001a, 0x48e6, 0x5fe6,
	0xe8bc, 0x001a, 0xebbc, 0x61e6, 0xe0bc, 0x56e6, 0xfbe5, 0x5ce6,
	0xdfc0, 0x001a, 0x4ae6, 0x001a, 0xe1bc, 0x45e6, 0xe5bc, 0xfce5,
	0xabba, 0x41e6, 0x001a, 0x5ae6, 0x42e6, 0x40e6, 0xeabc, 0x001a,
	0x58e6, 0x001a, 0xfee5, 0x51e6, 0x50e6, 0x5de6, 0x47e6, 0xeebc,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf3e9, 0x001a,
	0x49bf, 0xfebe, 0x40ea, 0xebe9, 0x41bf, 0xf7e9, 0x48bf, 0x43bf,
	0xf5e9, 0x4fed, 0xfbe9, 0x42ea, 0xfae9, 0xe9e9, 0xf8e9, 0x44ea,
	0x46ea, 0xfdbe, 0x45ea, 0x44bf, 0x4abf, 0x001a, 0x47bf, 0x001a,
	0xfee9, 0x46bf, 0xf9e9, 0x001a, 0xede9, 0xf2e9, 0x001a, 0xfde9,
	0x45bf, 0x42bf, 0xfcbe, 0x40bf, 0xf1e9, 0x001a, 0xfde5, 0xece9,
	0xefe9, 0x41ea, 0xf4e9, 0xeae9, 0x4eed, 0x43ea, 0xeee9, 0xfce9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x51ed, 0xe3c0, 0x001a, 0x001a,
	0xd7c0, 0x001a, 0x001a, 0xdbc0, 0x53ed, 0x59ed, 0x57ed, 0xd9c0,
	0xdac0, 0xe1c0, 0x5aed, 0x52ed, 0xdcc0, 0x001a, 0x56ed, 0x55ed,
	0x5bed, 0xe2c0, 0x001a, 0xddc0, 0xe0c0, 0x54ed, 0xe4c0, 0xdec0,
	0xe5c0, 0xd8c0, 0x58ed, 0x001a, 0x50ed, 0x001a, 0x001a, 0xf7ef,
	0x001a, 0x001a, 0x71c2, 0xf4ef, 0xf6ef, 0x001a, 0x6fc2, 0xf2ef,
	0xf3ef, 0xeeef, 0x001a, 0x001a, 0xf6e9, 0xefef, 0x70c2, 0xebef,
	0x001a, 0x6dc2, 0xf8ef, 0x6ec2, 0xecef, 0xedef, 0xf1ef, 0x73c2,
	0x001a, 0x72c2, 0x001a, 0x001a, 0xf0ef, 0x78c3, 0x5ff2, 0x65f2,
	0x79c3, 0x5cf2, 0x76c3, 0x73c3, 0x67f2, 0x77c3, 0x001a, 0x74c3,
	0x5ef2, 0x61f2, 0x62f2, 0x63f2, 0x66f2, 0x001a, 0xf5ef, 0x5df2,
	0x75c3, 0x64f2, 0x68f2, 0x60f2, 0x001a, 0x001a, 0x001a, 0x5df4,
	0x6ac4, 0x60f4, 0x6bc4, 0x68f4, 0x5ff4, 0x5cf4, 0x001a, 0x5ef4,
	0x62f4, 0x65f4, 0x64f4, 0x67f4, 0x5bf4, 0x001a, 0x69c4, 0x63f4,
	0x66f4, 0x69f4, 0x61f4, 0xd3f5, 0xd4f5, 0xd8f5, 0xd9f5, 0x001a,
	0xd6f5, 0xd7f5, 0xd5f5, 0x001a, 0xe9c4, 0x001a, 0x001a, 0x001a,
	0x001a, 0x78c5, 0xebf6, 0x001a, 0x001a, 0xe8f6, 0xe9f6, 0xeaf6,
	0x79c5, 0x001a, 0xe5f7, 0xe4f7, 0x001a, 0xaff8, 0xf4c5, 0xadf8,
	0xb0f8, 0xaef8, 0xf5f8, 0x57c6, 0x65c6, 0xa3f9, 0x6cf9, 0x001a,
	0xa2f9, 0xd0f9, 0xd1f9, 0xf5a4, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc7a6, 0x41ca, 0x001a, 0x001a, 0x5ecb, 0x001a, 0x5fa8, 0x001a,
	0x62a8, 0x001a, 0x5fcb, 0x001a, 0x60a8, 0x61a8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x58cd, 0x5acd, 0x55cd, 0x52cd, 0x54cd, 0x001a,
	0x001a, 0x001a, 0xa4aa, 0x001a, 0x001a, 0x001a, 0xa2aa, 0x001a,
	0x001a, 0x56cd, 0xa3aa, 0x53cd, 0x50cd, 0xa1aa, 0x57cd, 0x001a,
	0x51cd, 0xa5aa, 0x59cd, 0x001a, 0x001a, 0x001a, 0x001a, 0xafcf,
	0x001a, 0xb3cf, 0x001a, 0x001a, 0xb7ac, 0x001a, 0x001a, 0x001a,
	0x001a, 0xb6cf, 0x001a, 0xafac, 0xb2ac, 0xb4ac, 0xb6ac, 0xb3ac,
	0xb2cf, 0xb1cf, 0x001a, 0xb1ac, 0xb4cf, 0xb5cf, 0x001a, 0xaecf,
	0xb5ac, 0x001a, 0xb0ac, 0x001a, 0x001a, 0x001a, 0xb0cf, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x77d2, 0x78d2, 0x79d2,
	0x50af, 0x001a, 0x4caf, 0x6ed2, 0x001a, 0x76d2, 0x7bd2, 0x51af,
	0x001a, 0x6cd2, 0x72d2, 0x6bd2, 0x75d2, 0x001a, 0x001a, 0x71d2,
	0x4daf, 0x4faf, 0x7ad2, 0x001a, 0x6ad2, 0x6dd2, 0x73d2, 0x001a,
	0x74d2, 0x7cd2, 0x70d2, 0x001a, 0x4eaf, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6db2,
	0x4ed6, 0x001a, 0x001a, 0x50d6, 0x4cd6, 0x001a, 0x58d6, 0x4ad6,
	0x57d6, 0x69b2, 0x48d6, 0x5bda, 0x52d6, 0x6cb2, 0x001a, 0x53d6,
	0x56d6, 0x001a, 0x5ad6, 0x001a, 0x4fd6, 0x001a, 0x54d6, 0x001a,
	0x001a, 0x6ab2, 0x6bb2, 0x59d6, 0x4dd6, 0x49d6, 0x5bd6, 0x001a,
	0x51d6, 0x001a, 0x001a, 0x55d6, 0x001a, 0x001a, 0x001a, 0x4bd6,
	0x001a, 0x48b5, 0x49b5, 0x65da, 0x4fb5, 0x001a, 0x59da, 0x62da,
	0x58da, 0x4cb5, 0x60da, 0x5eda, 0x001a, 0x5fda, 0x4ab5, 0x001a,
	0x63da, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5cda, 0x5ada,
	0x4bb5, 0x5dda, 0x61da, 0x001a, 0x001a, 0x001a, 0x4db5, 0x001a,
	0x001a, 0x001a, 0x64da, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x70de, 0x77de, 0x79de, 0xa1de, 0x001a, 0xdab7, 0x6bde,
	0x001a, 0xd2b7, 0x001a, 0x7ade, 0xd7b7, 0xa2de, 0xceb7, 0x001a,
	0x7dde, 0x001a, 0x6dde, 0x7ede, 0x6cde, 0x001a, 0xdcb7, 0x001a,
	0x78de, 0xcfb7, 0xa3de, 0x001a, 0xd4b7, 0x71de, 0xd9b7, 0x7cde,
	0x6fde, 0x76de, 0x72de, 0x6ede, 0xd1b7, 0xd8b7, 0xd6b7, 0xd3b7,
	0xdbb7, 0xd0b7, 0x75de, 0x001a, 0xd5b7, 0x001a, 0x4eb5, 0x001a,
	0x7bde, 0x001a, 0x73de, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x74de, 0x001a, 0x001a, 0xc1e2, 0x001a, 0xb4ba, 0x001a, 0x001a,
	0xbde2, 0xc3e2, 0xbfe2, 0x001a, 0xb6ba, 0xbee2, 0xc2e2, 0xbae2,
	0x001a, 0xbce2, 0xb5ba, 0x001a, 0x001a, 0x001a, 0x001a, 0xc0e2,
	0xbbe2, 0x001a, 0xb7ba, 0x001a, 0xb2ba, 0x001a, 0x001a, 0xc4e2,
	0x001a, 0xb3ba, 0x67e6, 0x64e6, 0x70e6, 0x6ae6, 0x6ce6, 0xf4bc,
	0x66e6, 0x6ee6, 0x001a, 0x001a, 0x6de6, 0x6be6, 0x001a, 0x71e6,
	0xf7bc, 0x68e6, 0x6fe6, 0x001a, 0xf5bc, 0x001a, 0x001a, 0x63e6,
	0x65e6, 0xf6bc, 0x62e6, 0x72e6, 0x001a, 0x69e6, 0x001a, 0x001a,
	0x4aea, 0x51bf, 0x001a, 0x001a, 0x55ea, 0x53ea, 0x4bbf, 0x49ea,
	0x4cea, 0x4dea, 0x48ea, 0x55bf, 0x56bf, 0x47ea, 0x56ea, 0x51ea,
	0x4fbf, 0x4cbf, 0x50ea, 0x4eea, 0x001a, 0x001a, 0x52bf, 0x52ea,
	0x4dbf, 0x001a, 0x4ebf, 0x001a, 0x4fea, 0x50bf, 0x4bea, 0x001a,
	0x54ea, 0x53bf, 0x57ea, 0x58ea, 0x54bf, 0x001a, 0x001a, 0xe7c0,
	0xeec0, 0x5ced, 0x62ed, 0x001a, 0x60ed, 0xeac0, 0xe9c0, 0xe6c0,
	0x5eed, 0x001a, 0x001a, 0x001a, 0xecc0, 0xebc0, 0xe8c0, 0x001a,
	0x61ed, 0x5ded, 0x5fed, 0x001a, 0xedc0, 0x001a, 0x001a, 0x001a,
	0x77c2, 0xfbef, 0x001a, 0x74c2, 0x75c2, 0xfdef, 0x76c2, 0xfaef,
	0x001a, 0xf9ef, 0x6cf2, 0xfcef, 0x001a, 0x6df2, 0x7ac3, 0x6bf2,
	0x001a, 0x001a, 0x6af2, 0x001a, 0x69f2, 0x7bc3, 0x001a, 0x001a,
	0x6cc4, 0x001a, 0x001a, 0x6af4, 0x6bf4, 0x001a, 0x001a, 0x001a,
	0x001a, 0xdcf5, 0xdbf5, 0xeac4, 0x001a, 0xdaf5, 0xecf6, 0xedf6,
	0x001a, 0x001a, 0xe6f7, 0xb1f8, 0x001a, 0x001a, 0xf6f8, 0xbcf9,
	0x79c6, 0xc6f9, 0xf6a4, 0x001a, 0xa6aa, 0xa7aa, 0x001a, 0x001a,
	0xb8ac, 0x001a, 0x001a, 0x001a, 0x001a, 0xefc0, 0xf7a4, 0x001a,
	0xa8aa, 0x52af, 0xddb7, 0xf8a4, 0x001a, 0x6eb2, 0xb8ba, 0x62c9,
	0x001a, 0xb7cf, 0x7dd2, 0x001a, 0xc5e2, 0x001a, 0xf0c0, 0xf9a4,
	0xa9aa, 0xb8cf, 0xb9cf, 0x66da, 0x50b5, 0x001a, 0x001a, 0xa4de,
	0x001a, 0x001a, 0xdeb7, 0xc6e2, 0x001a, 0x001a, 0xf8bc, 0x001a,
	0x7cc3, 0xfaa4, 0x67da, 0xfba4, 0x001a, 0xc9a6, 0x42ca, 0xc8a6,
	0x65a8, 0x64a8, 0x63a8, 0x60cb, 0x001a, 0x001a, 0x001a, 0xaaaa,
	0x001a, 0xabaa, 0x5bcd, 0x001a, 0xbacf, 0x001a, 0xbdcf, 0xbaac,
	0xbbcf, 0x001a, 0xb9ac, 0xbccf, 0xbbac, 0x001a, 0xa2d2, 0xa1d2,
	0x7ed2, 0x53af, 0x001a, 0x5dd6, 0x5ed6, 0x6fb2, 0x5cd6, 0x5fd6,
	0x52b5, 0x70b2, 0x001a, 0x001a, 0x51b5, 0x6bda, 0x6ada, 0x001a,
	0x68da, 0x69da, 0x001a, 0x6cda, 0xa6de, 0xa5de, 0xa9de, 0x001a,
	0xa8de, 0xa7de, 0xb9ba, 0xc9e2, 0x001a, 0xc8e2, 0xbaba, 0xc7e2,
	0x73e6, 0x001a, 0x74e6, 0xf9bc, 0x001a, 0x59ea, 0x5aea, 0x001a,
	0x001a, 0x72f2, 0x7dc3, 0x71f2, 0x70f2, 0x6ef2, 0x6ff2, 0xebc4,
	0x6cf4, 0xeef6, 0xf7f8, 0x001a, 0xfca4, 0x001a, 0xa5c9, 0xc7a5,
	0xa6c9, 0x001a, 0x001a, 0x001a, 0x43ca, 0x44ca, 0x001a, 0x001a,
	0x001a, 0x001a, 0x66cb, 0x001a, 0x001a, 0x62cb, 0x001a, 0x61cb,
	0xacaa, 0x65cb, 0x67a8, 0x63cb, 0x66a8, 0x67cb, 0x64cb, 0x001a,
	0x001a, 0x5fcd, 0xbecf, 0x5dcd, 0x64cd, 0x001a, 0xadaa, 0x001a,
	0xb0aa, 0x65cd, 0x61cd, 0x001a, 0x62cd, 0x001a, 0x5ccd, 0xafaa,
	0x5ecd, 0xaeaa, 0x63cd, 0x001a, 0x60cd, 0x001a, 0x001a, 0xc2cf,
	0xbdac, 0xbeac, 0x001a, 0xc5cf, 0xbfcf, 0x001a, 0xc4cf, 0x001a,
	0xc0cf, 0xbcac, 0xc3cf, 0xc1cf, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa8d2, 0xa5d2, 0x001a, 0xa7d2, 0x58af,
	0x57af, 0x55af, 0xa4d2, 0xa9d2, 0x54af, 0x56af, 0xa6d2, 0x67d6,
	0xa3d2, 0xaad2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x62d6,
	0x66d6, 0x001a, 0x65d6, 0x6eda, 0x79da, 0x001a, 0x001a, 0x68d6,
	0x001a, 0x63d6, 0x6dda, 0x74b2, 0x001a, 0x001a, 0x73b2, 0x61d6,
	0x64d6, 0x75b2, 0x001a, 0x72b2, 0x71b2, 0x60d6, 0x69d6, 0x001a,
	0x001a, 0x001a, 0x70da, 0x77da, 0x001a, 0x54b5, 0x76da, 0x73da,
	0x001a, 0x56b5, 0x001a, 0x001a, 0x001a, 0x75da, 0x001a, 0x001a,
	0x6fda, 0x71da, 0x74da, 0x72da, 0x55b5, 0x78da, 0x53b5, 0xdfb7,
	0x001a, 0x001a, 0xadde, 0xacde, 0xaade, 0x001a, 0xe2b7, 0xe1b7,
	0xaede, 0x001a, 0xabde, 0xcae2, 0xbbba, 0xe0b7, 0x001a, 0x001a,
	0x001a, 0xb0de, 0xafde, 0x001a, 0xcde2, 0xcbe2, 0xfabc, 0x001a,
	0xbcba, 0xcce2, 0x76e6, 0x001a, 0x001a, 0x001a, 0x001a, 0xfbbc,
	0x75e6, 0x7ee6, 0x7de6, 0x7be6, 0x001a, 0x7ae6, 0x77e6, 0x78e6,
	0x79e6, 0x7ce6, 0xa1e6, 0x001a, 0x001a, 0x5fea, 0x5cea, 0x5dea,
	0x57bf, 0x5bea, 0x61ea, 0x60ea, 0x5eea, 0x001a, 0x64ed, 0x65ed,
	0xf1c0, 0x001a, 0xf2c0, 0x63ed, 0x001a, 0x79c2, 0xfeef, 0x78c2,
	0x7ec3, 0x001a, 0xa1c3, 0x6dc4, 0x6ef4, 0x6df4, 0xddf5, 0xeff6,
	0x7ac5, 0xe8f7, 0xe7f7, 0xe9f7, 0xc8a5, 0xc6cf, 0x59af, 0x76b2,
	0x6ad6, 0xc9a5, 0xa7c9, 0xfda4, 0x001a, 0x001a, 0x45ca, 0x001a,
	0x001a, 0x001a, 0x6ccb, 0x6acb, 0x6bcb, 0x68cb, 0x68a8, 0x69cb,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6dcd, 0x001a, 0xb3aa,
	0x6bcd, 0x67cd, 0x6acd, 0x001a, 0x66cd, 0xb5aa, 0x69cd, 0x001a,
	0xb2aa, 0xb1aa, 0x001a, 0xb4aa, 0x6ccd, 0x68cd, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc2ac, 0xc5ac, 0xcecf, 0xcdcf, 0xcccf, 0xbfac,
	0xd5cf, 0xcbcf, 0x001a, 0xc1ac, 0xafd2, 0x001a, 0xd2cf, 0xd0cf,
	0xc4ac, 0x001a, 0xc8cf, 0xd3cf, 0x001a, 0xcacf, 0xd4cf, 0xd1cf,
	0xc9cf, 0x001a, 0xc0ac, 0xd6cf, 0xc7cf, 0xc3ac, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb4d2, 0xabd2, 0xb6d2, 0x001a, 0xaed2, 0xb9d2,
	0xbad2, 0xacd2, 0xb8d2, 0xb5d2, 0xb3d2, 0xb7d2, 0x5faf, 0x001a,
	0x5daf, 0x001a, 0x001a, 0xb1d2, 0x001a, 0xadd2, 0x001a, 0xb0d2,
	0xbbd2, 0xb2d2, 0x5eaf, 0xcfcf, 0x001a, 0x5aaf, 0x5caf, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x78d6, 0x6dd6, 0x6bd6, 0x001a,
	0x6cd6, 0x001a, 0x73d6, 0x001a, 0x74d6, 0x70d6, 0x7bb2, 0x75d6,
	0x72d6, 0x6fd6, 0x001a, 0x79b2, 0x6ed6, 0x77b2, 0x7ab2, 0x71d6,
	0x79d6, 0x5baf, 0x78b2, 0x77d6, 0x76d6, 0x7cb2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7eda, 0x001a,
	0x001a, 0x001a, 0xa1da, 0x60b5, 0x001a, 0xa7da, 0x001a, 0x001a,
	0xa9da, 0xa2da, 0x5ab5, 0xa6da, 0xa5da, 0x5bb5, 0x61b5, 0x001a,
	0x62b5, 0xa8da, 0x58b5, 0x7dda, 0x7bda, 0xa3da, 0x7ada, 0x5fb5,
	0x7cda, 0xa4da, 0xaada, 0x59b5, 0x5eb5, 0x5cb5, 0x5db5, 0x001a,
	0x001a, 0x001a, 0x57b5, 0x001a, 0x001a, 0x001a, 0x001a, 0xe9b7,
	0xb7de, 0xe8b7, 0xbbde, 0x001a, 0xb1de, 0x001a, 0xbcde, 0x001a,
	0x001a, 0x001a, 0xb2de, 0xb3de, 0x001a, 0xbdde, 0xbade, 0xb8de,
	0xb9de, 0xb5de, 0xb4de, 0x001a, 0xbede, 0xe5b7, 0x001a, 0xb6de,
	0x001a, 0xeab7, 0xe4b7, 0xebb7, 0xecb7, 0x001a, 0xe7b7, 0xe6b7,
	0x001a, 0x001a, 0xcee2, 0xbeba, 0xbdba, 0x001a, 0x001a, 0xd3e2,
	0x001a, 0xfcbc, 0xbfba, 0x001a, 0x001a, 0xc1ba, 0xd4e2, 0xe3b7,
	0xc0ba, 0xd0e2, 0xd2e2, 0xcfe2, 0x001a, 0xd1e2, 0x001a, 0x001a,
	0x001a, 0xabe6, 0x001a, 0x001a, 0xaae6, 0xa7e6, 0x40bd, 0x62ea,
	0x41bd, 0xa6e6, 0x001a, 0xfebc, 0x001a, 0xa8e6, 0xa5e6, 0xa2e6,
	0xa9e6, 0xa3e6, 0xa4e6, 0xfdbc, 0x001a, 0x001a, 0x001a, 0x001a,
	0x69ed, 0x001a, 0x66ea, 0x001a, 0x65ea, 0x67ea, 0x001a, 0x66ed,
	0x5abf, 0x001a, 0x63ea, 0x001a, 0x58bf, 0x001a, 0x5cbf, 0x5bbf,
	0x64ea, 0x68ea, 0x001a, 0x59bf, 0x001a, 0x6ded, 0xf5c0, 0x7ac2,
	0xf6c0, 0xf3c0, 0x6aed, 0x68ed, 0x001a, 0x6bed, 0x001a, 0x6eed,
	0xf4c0, 0x6ced, 0x67ed, 0x001a, 0x001a, 0x42f0, 0x45f0, 0x75f2,
	0x40f0, 0x001a, 0x6ff4, 0x46f0, 0x001a, 0xa2c3, 0x44f0, 0x7bc2,
	0x41f0, 0x43f0, 0x47f0, 0x76f2, 0x001a, 0x74f2, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa3c3, 0x73f2, 0x001a, 0x001a, 0x001a, 0x6ec4,
	0x001a, 0x001a, 0x001a, 0x001a, 0xedc4, 0xf1f6, 0xecc4, 0xf3f6,
	0xf0f6, 0xf2f6, 0xd0c5, 0xb2f8, 0xcaa5, 0x6ecd, 0xbcd2, 0xbdd2,
	0x7db2, 0xbfde, 0x5dbf, 0xa4c3, 0x7bc5, 0xb3f8, 0xcba5, 0x001a,
	0x6fcd, 0x60a2, 0x001a, 0x001a, 0xd7cf, 0x001a, 0xd8cf, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbed2, 0xbfd2, 0x7eb2, 0xa1b2,
	0x001a, 0x001a, 0x001a, 0xabda, 0x001a, 0xc2de, 0xc1de, 0xc0de,
	0xd5e2, 0x001a, 0xd6e2, 0xd7e2, 0xc2ba, 0x001a, 0x001a, 0xade6,
	0xace6, 0x001a, 0x001a, 0x69ea, 0x5ebf, 0x5fbf, 0x001a, 0x72ed,
	0x6fed, 0x70ed, 0x71ed, 0x49f0, 0x48f0, 0x7cc2, 0x77f2, 0xdef5,
	0xcca5, 0x001a, 0xc6ac, 0x001a, 0xa2b2, 0xc3de, 0x001a, 0xcda5,
	0x001a, 0xc0d2, 0xa3b2, 0x001a, 0x001a, 0x63b5, 0x64b5, 0x001a,
	0xcea5, 0xcfa5, 0x46ca, 0x6aa8, 0x69a8, 0xc7ac, 0xd9cf, 0xacda,
	0xd0a5, 0xd1a5, 0xd2a5, 0xd3a5, 0x001a, 0x001a, 0x001a, 0x6ba8,
	0x6ca8, 0x6ecb, 0x6dcb, 0x001a, 0x001a, 0xb6aa, 0x72cd, 0x70cd,
	0x71cd, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdacf,
	0xdbcf, 0x001a, 0x001a, 0xcbac, 0xc9ac, 0x001a, 0xcaac, 0xc8ac,
	0x001a, 0x001a, 0x001a, 0x001a, 0x60af, 0x001a, 0x001a, 0x001a,
	0x001a, 0x64af, 0x63af, 0xc1d2, 0x62af, 0x61af, 0x001a, 0xc2d2,
	0x001a, 0x001a, 0xa6b2, 0x7bd6, 0x7ad6, 0xa4b2, 0xa5b2, 0x001a,
	0x001a, 0x001a, 0x66b5, 0x65b5, 0xaeda, 0x001a, 0x001a, 0xadda,
	0xa7b2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xedb7, 0xc5de,
	0xeeb7, 0xc4de, 0x001a, 0x001a, 0x001a, 0xd8e2, 0xaee6, 0x42bd,
	0x6aea, 0x001a, 0x001a, 0x001a, 0x73ed, 0x001a, 0xa6c3, 0xa5c3,
	0x001a, 0x001a, 0x7cc5, 0xd4a5, 0x73cd, 0x001a, 0x001a, 0xa8b2,
	0xd9e2, 0xc3ba, 0x001a, 0x001a, 0x6fcb, 0x70cb, 0x001a, 0x001a,
	0x74cd, 0xb8aa, 0xb9aa, 0x001a, 0x001a, 0xb7aa, 0x001a, 0x001a,
	0x001a, 0x001a, 0xcfac, 0xd0ac, 0xcdac, 0xceac, 0x001a, 0xdccf,
	0x001a, 0x001a, 0xddcf, 0xccac, 0x001a, 0x001a, 0x001a, 0x001a,
	0xc3d2, 0x001a, 0x68af, 0x69af, 0x001a, 0xabb2, 0xc9d2, 0x001a,
	0x6eaf, 0x6caf, 0xcad2, 0xc5d2, 0x6baf, 0x6aaf, 0x65af, 0xc8d2,
	0xc7d2, 0xc4d2, 0x6daf, 0x001a, 0xc6d2, 0x66af, 0x001a, 0x67af,
	0x001a, 0x001a, 0xacb2, 0xa1d6, 0xa2d6, 0xadb2, 0x7cd6, 0x7ed6,
	0xa4d6, 0xa3d6, 0x7dd6, 0x001a, 0xa9b2, 0xaab2, 0x001a, 0xb6da,
	0x6bb5, 0x6ab5, 0xb0da, 0x68b5, 0x001a, 0xb3da, 0x6cb5, 0xb4da,
	0x6db5, 0xb1da, 0x67b5, 0x69b5, 0xb5da, 0x001a, 0xb2da, 0xafda,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xd2de, 0x001a, 0xc7de,
	0xf0b7, 0xf3b7, 0xf2b7, 0xf7b7, 0xf6b7, 0xd3de, 0xd1de, 0xcade,
	0xcede, 0xcdde, 0xf4b7, 0xd0de, 0xccde, 0xd4de, 0xcbde, 0xf5b7,
	0xefb7, 0xf1b7, 0x001a, 0xc9de, 0x001a, 0x001a, 0x001a, 0x001a,
	0xdbe2, 0xc7ba, 0xdfe2, 0xc6ba, 0xdce2, 0xc5ba, 0x001a, 0xc8de,
	0xcfde, 0xdee2, 0x001a, 0xc8ba, 0xe0e2, 0xdde2, 0xdae2, 0x001a,
	0x001a, 0xb1e6, 0xb5e6, 0xb7e6, 0xb3e6, 0xb2e6, 0xb0e6, 0x45bd,
	0x43bd, 0x48bd, 0x49bd, 0xb4e6, 0x46bd, 0xafe6, 0x47bd, 0xc4ba,
	0xb6e6, 0x44bd, 0x001a, 0x001a, 0x001a, 0x6cea, 0x001a, 0x6bea,
	0x73ea, 0x6dea, 0x72ea, 0x6fea, 0x60bf, 0x71ea, 0x001a, 0x001a,
	0x61bf, 0x001a, 0x62bf, 0x001a, 0x70ea, 0x6eea, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf8c0, 0x74ed, 0x001a, 0x001a, 0xf7c0, 0x77ed,
	0x75ed, 0x76ed, 0x001a, 0x001a, 0xf9c0, 0x001a, 0x001a, 0x001a,
	0x4df0, 0x001a, 0xa1c2, 0x4ef0, 0x001a, 0x001a, 0x7dc2, 0x4ff0,
	0x7ec2, 0x4cf0, 0x50f0, 0x001a, 0x4af0, 0x001a, 0x001a, 0xa7c3,
	0x78f2, 0xa8c3, 0x6fc4, 0x001a, 0x4bf0, 0x70c4, 0x001a, 0x001a,
	0x001a, 0xeec4, 0xdff5, 0x001a, 0x7ec5, 0xf4f6, 0x7dc5, 0x001a,
	0xeaf7, 0xf5c5, 0xf6c5, 0x001a, 0x001a, 0xccf9, 0x001a, 0x001a,
	0xd1ac, 0xdecf, 0x001a, 0x6eb5, 0x6fb5, 0xd5a5, 0xcaa6, 0x47ca,
	0x001a, 0x71cb, 0x6da8, 0x001a, 0xbaaa, 0x001a, 0xd2ac, 0xd3ac,
	0xd4ac, 0xa6d6, 0xcbd2, 0x6faf, 0x001a, 0x001a, 0xaeb2, 0xa5d6,
	0x001a, 0x001a, 0xb8da, 0x71b5, 0x001a, 0xb7da, 0x70b5, 0x001a,
	0x001a, 0xd5de, 0x4abd, 0xbbe6, 0xb8e6, 0xb9e6, 0xbae6, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x78ed, 0x001a, 0x51f0, 0x001a,
	0x001a, 0x001a, 0x71f4, 0x70f4, 0x001a, 0xf5f6, 0xd6a5, 0x75cd,
	0x70af, 0x001a, 0x001a, 0x001a, 0x72b5, 0xd6de, 0x001a, 0x001a,
	0xe1e2, 0x001a, 0x4bbd, 0x74ea, 0x001a, 0x52f0, 0x72f4, 0xd7a5,
	0x001a, 0x001a, 0xbbaa, 0xd7ac, 0xdfcf, 0xd8ac, 0xd6ac, 0x001a,
	0xd5ac, 0xccd2, 0x71af, 0x001a, 0x001a, 0x72af, 0x73af, 0x001a,
	0x001a, 0x001a, 0xb0b2, 0xa7d6, 0xafb2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xb9da, 0xb1b2, 0x73b5, 0xd7de, 0xf8b7, 0xf9b7,
	0x001a, 0xc9ba, 0x001a, 0xcaba, 0x4cbd, 0x64bf, 0x75ea, 0x63bf,
	0x001a, 0x79ed, 0xfac0, 0x001a, 0x53f0, 0x73f4, 0xd8a5, 0x6ea8,
	0x78cd, 0x77cd, 0xbcaa, 0x76cd, 0xbdaa, 0x79cd, 0x001a, 0xe5cf,
	0xdbac, 0xdaac, 0xe7cf, 0xe6cf, 0xdfac, 0x001a, 0xdeac, 0x001a,
	0x001a, 0xd9ac, 0x001a, 0xe1cf, 0xe2cf, 0xe3cf, 0x001a, 0xe0ac,
	0xe0cf, 0xdcac, 0xe4cf, 0xddac, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcfd2, 0xd3d2, 0xd1d2, 0xd0d2, 0x001a, 0xd4d2, 0x001a, 0x001a,
	0x001a, 0xd5d2, 0xd6d2, 0xced2, 0x001a, 0xcdd2, 0x001a, 0x75af,
	0x76af, 0x001a, 0xd7d2, 0xd2d2, 0x001a, 0xb0d6, 0x001a, 0xd8d2,
	0x77af, 0x74af, 0x001a, 0x001a, 0x001a, 0xaad6, 0x001a, 0xa9d6,
	0x001a, 0xabd6, 0xacd6, 0xaed6, 0xadd6, 0xb2d6, 0xb5b2, 0xb2b2,
	0xb6b2, 0xa8d6, 0xb7b2, 0xb1d6, 0xb4b2, 0xafd6, 0xb3b2, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbcda, 0xbeda, 0xbada, 0xbbda,
	0x001a, 0x001a, 0xbfda, 0xc1da, 0xc2da, 0xbdda, 0xc0da, 0x74b5,
	0x001a, 0x001a, 0xdbde, 0x001a, 0xe0de, 0xd8de, 0xdcde, 0x001a,
	0x001a, 0xe1de, 0xddde, 0xfab7, 0x43b8, 0x001a, 0xfdb7, 0xd9de,
	0xdade, 0xceba, 0x46b8, 0xfeb7, 0x001a, 0x44b8, 0xfcb7, 0xdfde,
	0x45b8, 0xdede, 0x41b8, 0xfbb7, 0x42b8, 0xe2de, 0xe6e2, 0xe8e2,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x40b8, 0x001a, 0x001a, 0xe3e2, 0xccba, 0xe9e2, 0xcdba,
	0xe7e2, 0xe2e2, 0xe5e2, 0xeae2, 0xcbba, 0xe4e2, 0x001a, 0x4ebd,
	0xbfe6, 0xbee6, 0x001a, 0x51bd, 0x4fbd, 0xbce6, 0x4dbd, 0xbde6,
	0x001a, 0x50bd, 0x001a, 0x001a, 0x001a, 0x7dea, 0x001a, 0xa1ea,
	0x001a, 0x7eea, 0x76ea, 0x7aea, 0x79ea, 0x77ea, 0x66bf, 0x67bf,
	0x65bf, 0x78ea, 0x7bea, 0x7cea, 0x001a, 0x68bf, 0x001a, 0x40c1,
	0xa3ed, 0x001a, 0xfcc0, 0x7bed, 0xfec0, 0x41c1, 0x001a, 0x001a,
	0xfdc0, 0xa2ed, 0x7ced, 0xfbc0, 0xa1ed, 0x7aed, 0x7eed, 0x7ded,
	0x001a, 0x001a, 0x55f0, 0xa4c2, 0xa5c2, 0xa2c2, 0x001a, 0xa3c2,
	0x001a, 0x001a, 0x54f0, 0x001a, 0x7bf2, 0x001a, 0x001a, 0xa9c3,
	0x001a, 0x79f2, 0x7af2, 0x001a, 0x74f4, 0x77f4, 0x75f4, 0x76f4,
	0xe0f5, 0x001a, 0x001a, 0xefc4, 0xebf7, 0xb4f8, 0x001a, 0xf7c5,
	0xf8f8, 0xf9f8, 0x66c6, 0xd9a5, 0xe1ac, 0x001a, 0xc3da, 0x001a,
	0xe3de, 0x001a, 0xdaa5, 0x6fa8, 0x001a, 0xbeaa, 0x001a, 0xe8cf,
	0xe9cf, 0x78af, 0x001a, 0x001a, 0xc4da, 0x75b5, 0x47b8, 0x42c1,
	0xa4ed, 0x7cf2, 0x78f4, 0xdba5, 0x001a, 0x001a, 0x001a, 0xa1cd,
	0x7acd, 0x7ccd, 0x7ecd, 0x7dcd, 0x7bcd, 0xbfaa, 0x001a, 0x001a,
	0x001a, 0x001a, 0xe2ac, 0xf2cf, 0x001a, 0xedcf, 0xeacf, 0x001a,
	0x001a, 0xf1cf, 0x001a, 0x001a, 0xe4ac, 0xe5ac, 0xf0cf, 0xefcf,
	0xeecf, 0xebcf, 0xeccf, 0xf3cf, 0xe3ac, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x7caf, 0x001a, 0xa4af,
	0xa3af, 0xe1d2, 0xdbd2, 0xd9d2, 0x001a, 0xa1af, 0xb9d6, 0x7aaf,
	0xded2, 0xe2d2, 0xe4d2, 0xe0d2, 0xdad2, 0xa2af, 0xdfd2, 0xddd2,
	0x79af, 0xe5d2, 0xa5af, 0xe3d2, 0x7daf, 0xdcd2, 0x001a, 0x7eaf,
	0x7baf, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb9b2, 0x001a, 0xbad6, 0x001a, 0x001a,
	0xb3d6, 0xb5d6, 0xb7d6, 0x001a, 0xb8d6, 0xb6d6, 0xbab2, 0x001a,
	0xbbd6, 0x001a, 0xb4d6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc8da, 0x76b5, 0xd0da, 0x001a,
	0xc5da, 0x001a, 0xd1da, 0x001a, 0xc6da, 0xc7da, 0x001a, 0x001a,
	0xcfda, 0xceda, 0xcbda, 0xb8b2, 0x77b5, 0xc9da, 0xccda, 0x78b5,
	0xcdda, 0xcada, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xeede, 0x001a, 0xf2de, 0x4eb8, 0x001a, 0xf0e2, 0x51b8,
	0xf0de, 0xd6f9, 0x001a, 0xedde, 0xe8de, 0xeade, 0xebde, 0xe4de,
	0x001a, 0x4db8, 0x001a, 0x001a, 0x4cb8, 0x001a, 0x48b8, 0xe7de,
	0x001a, 0x4fb8, 0x001a, 0x50b8, 0xe6de, 0xe9de, 0xf1de, 0x4ab8,
	0x4bb8, 0xefde, 0xe5de, 0x001a, 0x001a, 0x001a, 0xf2e2, 0xd0ba,
	0xf4e2, 0xecde, 0xf6e2, 0xd4ba, 0xf7e2, 0xf3e2, 0x001a, 0xd1ba,
	0xefe2, 0xd3ba, 0xece2, 0xf1e2, 0xf5e2, 0xeee2, 0x001a, 0x001a,
	0x49b8, 0x001a, 0xebe2, 0xd2ba, 0xede2, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x54bd, 0xc1e6, 0x58bd, 0x001a, 0x56bd, 0x001a,
	0x001a, 0xcfba, 0x001a, 0xc8e6, 0xc9e6, 0x53bd, 0x001a, 0x001a,
	0xc7e6, 0xcae6, 0x55bd, 0x52bd, 0xc3e6, 0xc0e6, 0xc5e6, 0xc2e6,
	0x59bd, 0xc4e6, 0x001a, 0x001a, 0xc6e6, 0x57bd, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6abf, 0xa8ea, 0x001a, 0xa2ea, 0xa6ea, 0xacea,
	0xadea, 0xa9ea, 0xaaea, 0xa7ea, 0x001a, 0xa4ea, 0x001a, 0x6cbf,
	0x69bf, 0xa3ea, 0xa5ea, 0x001a, 0x6bbf, 0xabea, 0x001a, 0x46c1,
	0x001a, 0x001a, 0xaaed, 0xa5ed, 0x45c1, 0x001a, 0x001a, 0x43c1,
	0x001a, 0xaced, 0x44c1, 0xa8ed, 0xa9ed, 0xa6ed, 0xaded, 0x56f0,
	0x001a, 0x47c1, 0xa7ed, 0x001a, 0xaeed, 0xabed, 0x001a, 0x001a,
	0x001a, 0x5af0, 0x001a, 0x001a, 0x57f0, 0x001a, 0xa6c2, 0x001a,
	0x5bf0, 0x5df0, 0x5cf0, 0x58f0, 0x59f0, 0x001a, 0x001a, 0xa3f2,
	0x001a, 0xaac3, 0x001a, 0x7ef2, 0xa2f2, 0x7df2, 0xa4f2, 0x001a,
	0x001a, 0xa1f2, 0x001a, 0x7af4, 0x7df4, 0x79f4, 0x71c4, 0x7bf4,
	0x7cf4, 0x7ef4, 0x72c4, 0x74c4, 0x73c4, 0xe1f5, 0x001a, 0xe3f5,
	0x001a, 0xe2f5, 0x001a, 0x001a, 0x001a, 0xf6f6, 0x001a, 0x001a,
	0xb5f8, 0xfaf8, 0xdca5, 0x001a, 0x001a, 0x72cb, 0xc0aa, 0xa3cd,
	0xc1aa, 0xc2aa, 0xa2cd, 0x001a, 0xf8cf, 0xf7cf, 0xe6ac, 0xe9ac,
	0xe8ac, 0xe7ac, 0xf4cf, 0xf6cf, 0xf5cf, 0x001a, 0x001a, 0xe8d2,
	0xa7af, 0xecd2, 0xebd2, 0xead2, 0xe6d2, 0xa6af, 0xaaaf, 0xadaf,
	0x001a, 0x001a, 0xaeaf, 0xe7d2, 0xe9d2, 0xacaf, 0xabaf, 0xa9af,
	0xa8af, 0xc2d6, 0x001a, 0xc0d6, 0xbcd6, 0xbbb2, 0x001a, 0xbdd6,
	0xbcb2, 0xbed6, 0xbfd6, 0xc1d6, 0x001a, 0xbdb2, 0x001a, 0x001a,
	0xd5da, 0x001a, 0xd4da, 0xd3da, 0xd2da, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf6de, 0x52b8, 0x001a, 0xf3de, 0xf5de, 0x001a, 0x53b8,
	0x001a, 0x54b8, 0xf4de, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x41e3, 0x001a, 0xf9e2, 0xfae2, 0x001a, 0xd7ba, 0xd5ba, 0xd6ba,
	0x43e3, 0x001a, 0x42e3, 0xfee2, 0xfde2, 0xfce2, 0xfbe2, 0x40e3,
	0xf8e2, 0x001a, 0xcbe6, 0xd0e6, 0xcee6, 0x001a, 0x001a, 0x001a,
	0xcde6, 0xcce6, 0xcfe6, 0x001a, 0xaeea, 0x001a, 0x6dbf, 0x48c1,
	0xb0ed, 0x001a, 0x49c1, 0xafed, 0x5ff0, 0x5ef0, 0xa7c2, 0x001a,
	0xa5f2, 0xabc3, 0xa1f4, 0xa1c5, 0xf7f6, 0x001a, 0xb7f8, 0xb6f8,
	0xa8c9, 0xeaac, 0xebac, 0xc3d6, 0x001a, 0x56b8, 0xdda5, 0x72a8,
	0x71a8, 0x70a8, 0x001a, 0x001a, 0x001a, 0xa4cd, 0x001a, 0x001a,
	0xc4aa, 0xc3aa, 0x001a, 0xeeac, 0x001a, 0xfacf, 0xfdcf, 0xfbcf,
	0x001a, 0xecac, 0xedac, 0x001a, 0x001a, 0xf9cf, 0xfccf, 0x001a,
	0xb5af, 0x001a, 0x001a, 0x001a, 0xf3d2, 0xf5d2, 0xf4d2, 0xb2af,
	0xefd2, 0x001a, 0x001a, 0xb0af, 0xafaf, 0x001a, 0xb3af, 0xb1af,
	0x001a, 0xb4af, 0xf2d2, 0xedd2, 0xeed2, 0xf1d2, 0xf0d2, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc6d6, 0xc7d6,
	0xc5d6, 0x001a, 0xc4d6, 0xbeb2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7db5, 0x001a, 0xd6da, 0xd8da, 0xdada, 0x7cb5, 0x001a, 0x001a,
	0x7ab5, 0x001a, 0xd7da, 0x7bb5, 0xd9da, 0x79b5, 0x001a, 0x001a,
	0x41df, 0xf7de, 0xfade, 0xfede, 0x5ab8, 0xfcde, 0x001a, 0xfbde,
	0xf8de, 0xf9de, 0x58b8, 0x40df, 0x57b8, 0x001a, 0x5cb8, 0x5bb8,
	0x59b8, 0x001a, 0xfdde, 0x001a, 0x001a, 0x001a, 0x49e3, 0x001a,
	0x48e3, 0x001a, 0x001a, 0x44e3, 0x001a, 0x001a, 0xd8ba, 0x47e3,
	0x46e3, 0xd9ba, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x5ebd,
	0x001a, 0xd2e6, 0x001a, 0x5fbd, 0x5bbd, 0x5dbd, 0x001a, 0x5abd,
	0x5cbd, 0x001a, 0x001a, 0x001a, 0xafea, 0x001a, 0x70bf, 0xb1ea,
	0xb0ea, 0x001a, 0x45e3, 0x72bf, 0x71bf, 0x6ebf, 0x6fbf, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xb5ed, 0x001a, 0xb3ed, 0x4ac1,
	0xb4ed, 0x001a, 0xb6ed, 0xb2ed, 0xb1ed, 0x001a, 0x001a, 0x60f0,
	0xaac2, 0xa8c2, 0xa9c2, 0x001a, 0x001a, 0x001a, 0x001a, 0xa6f2,
	0xa7f2, 0xadc3, 0x001a, 0xacc3, 0xa3f4, 0xa4f4, 0xa2f4, 0x001a,
	0xf8f6, 0xf9f6, 0x001a, 0x001a, 0xdea5, 0x48ca, 0x73a8, 0x001a,
	0xa5cd, 0xc6aa, 0xc5aa, 0xa6cd, 0x001a, 0x001a, 0x40d0, 0xefac,
	0xfecf, 0xf0ac, 0x001a, 0x001a, 0xb6af, 0xf8d2, 0xf6d2, 0xfcd2,
	0xb7af, 0xf7d2, 0xfbd2, 0xf9d2, 0xfad2, 0x001a, 0x001a, 0xc8d6,
	0xcad6, 0x001a, 0xbfb2, 0x001a, 0xc9d6, 0xc0b2, 0xa2b5, 0xa1b5,
	0x7eb5, 0xdbda, 0x001a, 0x001a, 0x001a, 0x001a, 0x44df, 0x5db8,
	0x5eb8, 0x001a, 0x43df, 0x42df, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4ae3, 0xdbba, 0xdaba, 0x4be3, 0x4ce3, 0x001a, 0x61bd, 0x60bd,
	0x001a, 0xb5ea, 0xd3e6, 0xd5e6, 0xd4e6, 0xb4ea, 0xb2ea, 0xb6ea,
	0xb3ea, 0x001a, 0x73bf, 0x001a, 0x001a, 0x001a, 0xb7ed, 0x4bc1,
	0xb8ed, 0xb9ed, 0x001a, 0x001a, 0xabc2, 0xacc2, 0x001a, 0x75c4,
	0x001a, 0x001a, 0xd1c5, 0xdfa5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x41d0, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xfdd2, 0xb8af, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbab3,
	0xb9b3, 0x001a, 0x001a, 0xa4b5, 0xddda, 0xa3b5, 0xdcda, 0x001a,
	0x001a, 0x001a, 0x001a, 0x45df, 0x001a, 0xdcba, 0x4de3, 0xddba,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x76c4, 0xa5f4,
	0x001a, 0xcba6, 0xc7aa, 0xa7cd, 0x001a, 0xf2ac, 0x001a, 0xf1ac,
	0x42d0, 0x43d0, 0x001a, 0x001a, 0x40d3, 0x42d3, 0xb9af, 0x001a,
	0x44d3, 0x47d3, 0x45d3, 0x001a, 0x001a, 0x001a, 0x46d3, 0x43d3,
	0xfed2, 0xbaaf, 0x48d3, 0x41d3, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd3d6, 0xc6b2, 0xdcd6, 0xc3b2, 0x001a, 0xd5d6, 0xc7b2, 0x001a,
	0xc1b2, 0x001a, 0xd0d6, 0xddd6, 0xd1d6, 0xced6, 0xc5b2, 0x001a,
	0xc2b2, 0x001a, 0xd4d6, 0xd7d6, 0xc4b2, 0xd8d6, 0xc8b2, 0xd9d6,
	0xcfd6, 0xd6d6, 0xdad6, 0xd2d6, 0xcdd6, 0xcbd6, 0x001a, 0x001a,
	0xdbd6, 0x001a, 0x001a, 0xdfda, 0x001a, 0x001a, 0x001a, 0x001a,
	0xe4da, 0x001a, 0x001a, 0x001a, 0xe0da, 0xe6da, 0xa7b5, 0xccd6,
	0xe1da, 0xa5b5, 0xdeda, 0xacb5, 0xe2da, 0xabb5, 0xe3da, 0xadb5,
	0xa8b5, 0xaeb5, 0xa9b5, 0x001a, 0xaab5, 0x001a, 0xa6b5, 0x001a,
	0xe5da, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x61b8, 0x50df, 0x001a, 0x53df, 0x47df, 0x4cdf, 0x46df, 0x63b8,
	0x001a, 0x4adf, 0x001a, 0x001a, 0x001a, 0x48df, 0x62b8, 0x001a,
	0x4fdf, 0x4edf, 0x4bdf, 0x4ddf, 0x49df, 0xe1ba, 0x52df, 0x5fb8,
	0x51df, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5de3, 0x001a, 0xe8ba, 0x58e3, 0x001a, 0xe7ba,
	0x4ee3, 0x001a, 0x50e3, 0xe0ba, 0x55e3, 0x54e3, 0x57e3, 0xe5ba,
	0x52e3, 0x51e3, 0x001a, 0x001a, 0xe4ba, 0xdfba, 0x53e3, 0xe2ba,
	0x59e3, 0x5be3, 0x001a, 0x56e3, 0x4fe3, 0xe3ba, 0x001a, 0x001a,
	0x69bd, 0xdeba, 0x001a, 0x001a, 0x5ce3, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xd9e6, 0x62bd, 0x001a, 0xdbe6,
	0x001a, 0x63bd, 0x001a, 0x001a, 0x65bd, 0xdee6, 0x001a, 0xd6e6,
	0xe6ba, 0xdce6, 0x001a, 0x001a, 0x001a, 0x001a, 0xd8e6, 0x001a,
	0x60b8, 0x68bd, 0x001a, 0x001a, 0x64bd, 0x001a, 0x66bd, 0x67bd,
	0x001a, 0x76bf, 0xdde6, 0xd7e6, 0x6abd, 0x001a, 0xdae6, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc0ea, 0xbbea, 0x001a, 0x001a,
	0xc5ea, 0x74bf, 0xbdea, 0x78bf, 0xc3ea, 0xbaea, 0xb7ea, 0xc6ea,
	0x51c1, 0x79bf, 0xc2ea, 0xb8ea, 0x77bf, 0xbcea, 0x7bbf, 0xb9ea,
	0xbeea, 0x7abf, 0xc1ea, 0xc4ea, 0x001a, 0x001a, 0x001a, 0x001a,
	0xcbed, 0xcced, 0xbced, 0xc3ed, 0xc1ed, 0x001a, 0x001a, 0x4fc1,
	0xc8ed, 0xbfea, 0x001a, 0xbfed, 0x001a, 0xc9ed, 0x4ec1, 0xbeed,
	0xbded, 0xc7ed, 0xc4ed, 0xc6ed, 0x001a, 0xbaed, 0xcaed, 0x4cc1,
	0x001a, 0xc5ed, 0xceed, 0xc2ed, 0x50c1, 0x4dc1, 0xc0ed, 0xbbed,
	0xcded, 0x75bf, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x63f0, 0x001a, 0x001a, 0x61f0, 0x67f0, 0xb0c2, 0x65f0,
	0x64f0, 0xb2c2, 0x6af0, 0xb1c2, 0x001a, 0x6bf0, 0x68f0, 0xaec2,
	0x69f0, 0x62f0, 0xafc2, 0xadc2, 0xabf2, 0x66f0, 0x001a, 0x001a,
	0x6cf0, 0x001a, 0x001a, 0xa8f2, 0x001a, 0x001a, 0x001a, 0xb2c3,
	0xb0c3, 0xaaf2, 0x001a, 0xacf2, 0xa9f2, 0xb1c3, 0xaec3, 0xafc3,
	0xb3c3, 0x001a, 0x001a, 0x78c4, 0x001a, 0xaaf4, 0x001a, 0xa9f4,
	0xa7f4, 0xa6f4, 0xa8f4, 0x001a, 0x77c4, 0x79c4, 0x001a, 0x001a,
	0xf0c4, 0x001a, 0x001a, 0xe5f5, 0xe4f5, 0x001a, 0x001a, 0xfaf6,
	0x001a, 0xfcf6, 0xfef6, 0xfdf6, 0xfbf6, 0x001a, 0x001a, 0xa3c5,
	0xa2c5, 0x001a, 0x001a, 0xd3c5, 0xd2c5, 0xd4c5, 0xedf7, 0xecf7,
	0x001a, 0xfbf8, 0xb8f8, 0xfcf8, 0x58c6, 0x001a, 0x59c6, 0x6df9,
	0x001a, 0x001a, 0x7ec6, 0xcca6, 0x001a, 0xa8cd, 0x001a, 0x001a,
	0x45d0, 0x46d0, 0x44d0, 0x001a, 0x001a, 0xf3ac, 0x001a, 0x47d0,
	0x48d0, 0x49d0, 0x001a, 0x001a, 0x49d3, 0x4fd3, 0x001a, 0x001a,
	0x4dd3, 0xbbaf, 0x4bd3, 0x001a, 0x4cd3, 0x4ed3, 0x001a, 0x001a,
	0x001a, 0x4ad3, 0xc9b2, 0x001a, 0xded6, 0xcbb2, 0xe0d6, 0xcab2,
	0xdfd6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe8da, 0xafb5,
	0x001a, 0xeada, 0xe7da, 0xe1d6, 0x001a, 0xb0b5, 0x001a, 0xdbf9,
	0xe9da, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x56df,
	0x001a, 0x64b8, 0x54df, 0x65b8, 0x55df, 0x66b8, 0x001a, 0x001a,
	0x001a, 0xe9ba, 0x61e3, 0x5ee3, 0x60e3, 0xeaba, 0xebba, 0x5fe3,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdfe6, 0x001a, 0x001a,
	0xe0e6, 0x001a, 0x6bbd, 0xe2e6, 0xe1e6, 0x001a, 0x61a2, 0x001a,
	0xcaea, 0xcbea, 0xc7ea, 0x001a, 0xc8ea, 0x7cbf, 0x7dbf, 0xc9ea,
	0x001a, 0x57c1, 0x001a, 0x001a, 0x53c1, 0x58c1, 0x54c1, 0x56c1,
	0x52c1, 0x001a, 0x55c1, 0x001a, 0x001a, 0x001a, 0x001a, 0xb3c2,
	0xcfed, 0x001a, 0xaef2, 0x001a, 0xadf2, 0x001a, 0xabf4, 0x7ac4,
	0x7bc4, 0x41f7, 0xe6f5, 0x001a, 0x40f7, 0x001a, 0xfdf8, 0xa4f9,
	0xcda6, 0x001a, 0x001a, 0x74a8, 0x001a, 0xa9cd, 0xc8aa, 0x001a,
	0xf6ac, 0x4cd0, 0xf4ac, 0x4ad0, 0xf9ac, 0xf5ac, 0xfaac, 0xf8ac,
	0x4bd0, 0xf7ac, 0xbfaf, 0xbeaf, 0x5ad3, 0xc7af, 0x53d3, 0x59d3,
	0xc3af, 0x52d3, 0x58d3, 0x56d3, 0xc2af, 0xc4af, 0x55d3, 0xbdaf,
	0x54d3, 0xc8af, 0xc5af, 0xc9af, 0xc6af, 0x51d3, 0x50d3, 0x57d3,
	0xc0af, 0xbcaf, 0xc1af, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf0d6, 0xe9d6, 0x001a, 0xb5b5, 0xe8d6, 0x001a, 0xcfb2, 0xd6b2,
	0xd3b2, 0xd9b2, 0xd8b2, 0xd4b2, 0x001a, 0xe2d6, 0xe5d6, 0x001a,
	0xe4d6, 0xd0b2, 0xe6d6, 0xefd6, 0xd1b2, 0xe3d6, 0xecd6, 0xedd6,
	0xd2b2, 0xead6, 0xd7b2, 0xcdb2, 0xd5b2, 0xe7d6, 0xccb2, 0xebd6,
	0x001a, 0x001a, 0xeed6, 0x001a, 0x001a, 0x001a, 0xfbda, 0xf2da,
	0xb2b5, 0xf9da, 0xf6da, 0xeeda, 0xf7da, 0xb4b5, 0xefda, 0x001a,
	0xebda, 0x001a, 0x001a, 0x6cb8, 0xf4da, 0x001a, 0xb1b5, 0xfada,
	0x001a, 0xb8b5, 0xbab5, 0xedda, 0x001a, 0x001a, 0xb9b5, 0xf0da,
	0xb3b5, 0xf8da, 0xf1da, 0xf5da, 0x001a, 0xf3da, 0xb6b5, 0xecda,
	0xbbb5, 0xceb2, 0xb7b5, 0xbcb5, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x68b8, 0x5ddf, 0x5fdf, 0x61df, 0x65df, 0x001a, 0x5bdf,
	0x59df, 0x6ab8, 0x001a, 0x60df, 0x64df, 0x5cdf, 0x58df, 0x001a,
	0x57df, 0x001a, 0x001a, 0x001a, 0x62df, 0x5adf, 0x5edf, 0x6bb8,
	0x001a, 0x69b8, 0x66df, 0x67b8, 0x63df, 0x001a, 0x72e3, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xeeba, 0x6ae3, 0x78bd, 0x74e3,
	0xf1ba, 0x78e3, 0xf7ba, 0x65e3, 0x001a, 0x001a, 0x75e3, 0x62e3,
	0x001a, 0x77e3, 0x66e3, 0x001a, 0xfeba, 0xfbba, 0x76e3, 0x70e3,
	0xedba, 0xf5ba, 0xf4ba, 0x001a, 0xf3ba, 0xf9ba, 0x001a, 0x63e3,
	0xfaba, 0x71e3, 0xf6ba, 0xecba, 0x73e3, 0xefba, 0xf0ba, 0xf8ba,
	0x68e3, 0x67e3, 0x64e3, 0x001a, 0x6ce3, 0x69e3, 0x6de3, 0xfdba,
	0x001a, 0x79e3, 0xf2ba, 0x6ee3, 0x6fe3, 0x001a, 0x6be3, 0x001a,
	0x001a, 0x001a, 0xfcba, 0x001a, 0x001a, 0x001a, 0x001a, 0xe7e6,
	0x70bd, 0x79bd, 0x75bd, 0xe4e6, 0x001a, 0x72bd, 0x76bd, 0xf0e6,
	0x6cbd, 0xe8e6, 0x001a, 0x74bd, 0x001a, 0x001a, 0xebe6, 0xe6e6,
	0x73bd, 0x77bd, 0xe5e6, 0x001a, 0x71bd, 0x001a, 0xefe6, 0x6ebd,
	0xeee6, 0xede6, 0x7abd, 0x72e5, 0x6dbd, 0x001a, 0xece6, 0xe3e6,
	0x001a, 0x7bbd, 0xeae6, 0x6fbd, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe9e6, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa2bf, 0xa7bf, 0x7ebf, 0xd8ea, 0xcfea, 0xdbea, 0xd3ea, 0xd9ea,
	0xa8bf, 0xa1bf, 0xccea, 0xd2ea, 0xdcea, 0xd5ea, 0xdaea, 0xceea,
	0x001a, 0x001a, 0xd6ea, 0xa3bf, 0xd4ea, 0xa6bf, 0xa5bf, 0xd0ea,
	0xd1ea, 0xcdea, 0xd7ea, 0xa4bf, 0xdeea, 0xddea, 0x001a, 0x001a,
	0x001a, 0xdaed, 0xd6ed, 0x5fc1, 0x001a, 0xd0ed, 0x59c1, 0x69c1,
	0xdced, 0x61c1, 0x5dc1, 0xd3ed, 0x64c1, 0x67c1, 0xdeed, 0x5cc1,
	0xd5ed, 0x65c1, 0xe0ed, 0xdded, 0xd1ed, 0x60c1, 0x5ac1, 0x68c1,
	0xd8ed, 0x63c1, 0xd2ed, 0x5ec1, 0xdfed, 0x62c1, 0x5bc1, 0xd9ed,
	0x66c1, 0xd7ed, 0x001a, 0x001a, 0xdbed, 0x001a, 0x001a, 0x001a,
	0x6ef0, 0x74f0, 0xb9c2, 0x77f0, 0xb4c2, 0xb5c2, 0x6ff0, 0x76f0,
	0x71f0, 0xbac2, 0xb7c2, 0x001a, 0x6df0, 0x001a, 0xb6c2, 0x73f0,
	0x75f0, 0xb8c2, 0x72f0, 0x70f0, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb8f2, 0xb7c3, 0xb8c3, 0xb4c3, 0x001a, 0xb5c3, 0x001a, 0xb4f2,
	0xb2f2, 0x001a, 0xb6f2, 0xbac3, 0xb7f2, 0xb0f2, 0xaff2, 0xb3f2,
	0xb1f2, 0xb6c3, 0xb5f2, 0xacf4, 0x7ec4, 0x7dc4, 0xadf4, 0x001a,
	0xaff4, 0xaef4, 0xa1c4, 0x001a, 0x001a, 0x001a, 0xebf5, 0xe8f5,
	0xe9f5, 0x001a, 0xe7f5, 0xeaf5, 0xf2c4, 0xecf5, 0x001a, 0xf1c4,
	0x001a, 0x42f7, 0x001a, 0xd5c5, 0xd7c5, 0xeef7, 0xd6c5, 0xb9f8,
	0x40f9, 0x42f9, 0xfef8, 0x41f9, 0x6cc6, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xcea6, 0x001a,
	0xfbac, 0x6fd2, 0xcaaf, 0x001a, 0x001a, 0xdab2, 0xfcda, 0xfdda,
	0x001a, 0x001a, 0x001a, 0xdfea, 0x6ac1, 0xe1ed, 0x001a, 0x001a,
	0xbbc2, 0x001a, 0xbaf2, 0xb9f2, 0xa2c4, 0xedf5, 0x001a, 0x43f7,
	0xf8c5, 0x49ca, 0x001a, 0x001a, 0xc9aa, 0x75a8, 0x001a, 0x001a,
	0x4dd0, 0x001a, 0x001a, 0x60d3, 0x5bd3, 0x5fd3, 0x5dd3, 0xcbaf,
	0x5ed3, 0x5cd3, 0x001a, 0xf1d6, 0x001a, 0xfeda, 0x40db, 0x69df,
	0x6adf, 0x6eb8, 0x6fb8, 0x68df, 0x6bdf, 0x67df, 0x6db8, 0x001a,
	0x40bb, 0x001a, 0x70b8, 0x7ae3, 0x001a, 0x7cbd, 0xf1e6, 0x7dbd,
	0x001a, 0xa9bf, 0xe2ea, 0xe0ea, 0xe1ea, 0xe4ed, 0xe3ed, 0xe2ed,
	0x001a, 0x001a, 0x001a, 0xbbf2, 0x001a, 0xb9c3, 0xbcf2, 0x44f7,
	0xf9c5, 0xbaf8, 0xcfa6, 0xcbaa, 0xcaaa, 0x4fd0, 0xfcac, 0x001a,
	0x001a, 0x4ed0, 0x62d3, 0x001a, 0xccaf, 0xf2d6, 0x61d3, 0x001a,
	0x001a, 0x001a, 0xdcb2, 0xf5d6, 0xf3d6, 0xf4d6, 0xdbb2, 0x001a,
	0x42db, 0x43db, 0x41db, 0x001a, 0x73b8, 0x6ddf, 0x6cdf, 0x6edf,
	0x72b8, 0x71b8, 0x001a, 0x001a, 0xf2e6, 0xf4e6, 0x001a, 0x7ebd,
	0xf3e6, 0xe3ea, 0xaabf, 0x79f0, 0x001a, 0x78f0, 0xbbc3, 0xbdf2,
	0xbdc3, 0xbcc3, 0xb0f4, 0xeef5, 0xf3c4, 0xd0a6, 0x50d0, 0xfdac,
	0x65d3, 0xceaf, 0x64d3, 0x63d3, 0x001a, 0xcdaf, 0x001a, 0xfbd6,
	0x001a, 0xfdd6, 0xf6d6, 0xf7d6, 0xddb2, 0xf8d6, 0xdeb2, 0xfcd6,
	0xf9d6, 0xfad6, 0xdfb2, 0x001a, 0xbeb5, 0xbfb5, 0x001a, 0x44db,
	0x001a, 0x001a, 0x001a, 0x6fdf, 0x70df, 0x001a, 0x7ee3, 0x43bb,
	0x41bb, 0x42bb, 0x7be3, 0x7ce3, 0x001a, 0x7de3, 0xf9e6, 0x001a,
	0xfae6, 0xa1bd, 0xf7e6, 0xf6e6, 0xf8e6, 0xf5e6, 0xadbf, 0xe4ea,
	0xabbf, 0xacbf, 0xe6ed, 0x6bc1, 0xe5ed, 0xa8ef, 0x001a, 0x7af0,
	0x7bf0, 0xbcc2, 0x001a, 0xbdc2, 0x6cc1, 0xbef2, 0xbff2, 0xb1f4,
	0xa3c4, 0xd1a6, 0x001a, 0xd2a6, 0xfeac, 0xccaa, 0xcfaf, 0x51d0,
	0x001a, 0x001a, 0x001a, 0xc0b5, 0xd3a6, 0x41ad, 0x52d0, 0x53d0,
	0x40ad, 0x42ad, 0xd4a6, 0x001a, 0x54d0, 0xd1af, 0x66d3, 0xd3af,
	0xd0af, 0xd2af, 0x001a, 0x41d7, 0xe0b2, 0x001a, 0x40d7, 0xfed6,
	0x001a, 0x71df, 0x001a, 0x001a, 0xa1e3, 0x001a, 0xa2bd, 0x001a,
	0xaebf, 0xe6ea, 0xe5ea, 0x001a, 0xe7ed, 0x001a, 0x001a, 0x001a,
	0xeff5, 0x001a, 0x001a, 0xd5a6, 0x73cb, 0xaacd, 0x43ad, 0x55d0,
	0x001a, 0x68d3, 0x001a, 0x001a, 0x001a, 0xd4af, 0x67d3, 0xd5af,
	0x001a, 0x001a, 0x001a, 0x43d7, 0x001a, 0x001a, 0xe2b2, 0x42d7,
	0x44d7, 0x001a, 0xe1b2, 0x001a, 0x001a, 0x001a, 0x001a, 0x46db,
	0x47db, 0x45db, 0xc1b5, 0x001a, 0x001a, 0x001a, 0x74b8, 0x001a,
	0x75b8, 0x001a, 0x45bb, 0x001a, 0xa3e3, 0xa2e3, 0x44bb, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xfbe6, 0x001a, 0x001a, 0xfce6,
	0x001a, 0x001a, 0x001a, 0x001a, 0xe7ea, 0x001a, 0x001a, 0x70c1,
	0x6fc1, 0x6dc1, 0x6ec1, 0x71c1, 0x001a, 0x7cf0, 0xbfc2, 0xbec2,
	0xc0f2, 0xb2f4, 0x001a, 0x001a, 0x001a, 0xa5c5, 0xa4c5, 0xd6a6,
	0x001a, 0x001a, 0xfbd1, 0x001a, 0x77b8, 0xc2b5, 0x76b8, 0x46bb,
	0x001a, 0xd7a6, 0xa9c9, 0xd8a6, 0xd9a6, 0x001a, 0x001a, 0xabcd,
	0x76cb, 0x001a, 0x77cb, 0x77a8, 0x001a, 0x74cb, 0x76a8, 0x001a,
	0x79a8, 0x75cb, 0x7ba8, 0x7aa8, 0x78cb, 0x78a8, 0x001a, 0x001a,
	0x001a, 0xd1aa, 0xcfaa, 0xadcd, 0x001a, 0xceaa, 0x001a, 0x001a,
	0x001a, 0xd3aa, 0xd5aa, 0xd2aa, 0x001a, 0xb0cd, 0xaccd, 0xd6aa,
	0x001a, 0xd0aa, 0x7ca8, 0x001a, 0xd4aa, 0xafcd, 0x001a, 0x001a,
	0xaecd, 0x001a, 0xcdaa, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5bd0, 0x47ad, 0x48ad, 0x5dd0, 0x001a, 0x57d0,
	0x5ad0, 0x63d0, 0x61d0, 0x001a, 0x49ad, 0x67d0, 0x4cad, 0x64d0,
	0x5cd0, 0x59d0, 0x001a, 0x001a, 0x49db, 0x62d0, 0x44ad, 0x65d0,
	0x56d0, 0x5fd0, 0x46ad, 0x4bad, 0x60d0, 0x4fad, 0x4dad, 0x001a,
	0x58d0, 0x4aad, 0x001a, 0x5ed0, 0x4ead, 0x45ad, 0x66d0, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xdaaf, 0x001a, 0xe3af,
	0xd8af, 0xd6af, 0x6ad3, 0xdeaf, 0xdbaf, 0x6cd3, 0x001a, 0x001a,
	0xddaf, 0x6bd3, 0x69d3, 0x6ed3, 0xe2af, 0xe0af, 0x48db, 0x001a,
	0x6fd3, 0x6dd3, 0xd7af, 0x001a, 0x001a, 0xd9af, 0xdcaf, 0x001a,
	0xdfaf, 0x001a, 0xe1af, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ed7, 0xe4b2, 0x001a,
	0x45d7, 0x47d7, 0x001a, 0x48d7, 0x001a, 0x50d7, 0x4cd7, 0x4ad7,
	0x001a, 0x4dd7, 0x51d7, 0xe5b2, 0xe9b2, 0x46d7, 0x001a, 0x4fd7,
	0x001a, 0xe7b2, 0x001a, 0xe6b2, 0x4bd7, 0x49d7, 0x001a, 0xe3b2,
	0xe8b2, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc8b5, 0x51db, 0x001a, 0x001a, 0x4fdb, 0xcab5, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4adb, 0xa1df, 0x001a, 0xc9b5, 0x4edb,
	0x001a, 0x001a, 0x4bdb, 0xc5b5, 0xcbb5, 0x50db, 0xc7b5, 0x4ddb,
	0x47bb, 0xc6b5, 0x4cdb, 0xccb5, 0xc4b5, 0xc3b5, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x77df, 0x75df, 0x001a, 0x7bdf, 0x001a,
	0x73df, 0xa2df, 0x78df, 0x001a, 0x72df, 0x7bb8, 0xa3b8, 0x7ddf,
	0x001a, 0x76df, 0x001a, 0x7eb8, 0x001a, 0x001a, 0x7cb8, 0x7edf,
	0x79b8, 0x78b8, 0x79df, 0x7db8, 0xcdb5, 0x001a, 0x7cdf, 0x74df,
	0x7ab8, 0xa1b8, 0xa2b8, 0x001a, 0x001a, 0x001a, 0x001a, 0x4cbb,
	0x48bb, 0x001a, 0x4dbb, 0xa6e3, 0x001a, 0x001a, 0xa5e3, 0xa7e3,
	0x4abb, 0xa4e3, 0x4bbb, 0xaae3, 0xa9e3, 0xa8e3, 0x001a, 0x49bb,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x41e7, 0x001a, 0x44e7,
	0xa8bd, 0x43e7, 0xa7bd, 0xa3bd, 0xa4bd, 0xa5bd, 0x40e7, 0xfee6,
	0xa6bd, 0x001a, 0x42e7, 0xfde6, 0x001a, 0x001a, 0xe9ea, 0xf3ea,
	0xb1bf, 0xb0bf, 0x001a, 0xedea, 0xefea, 0x001a, 0xeaea, 0x001a,
	0xeeea, 0xe8ea, 0xf1ea, 0xafbf, 0xf0ea, 0xecea, 0x001a, 0xf2ea,
	0x001a, 0xebea, 0x74c1, 0xe8ed, 0xeeed, 0x78c1, 0x7ac1, 0x77c1,
	0x76c1, 0x001a, 0x75c1, 0x73c1, 0xe9ed, 0xeced, 0x72c1, 0xeded,
	0x001a, 0x79c1, 0xebed, 0x001a, 0xeaed, 0xc0c2, 0x001a, 0xc1c2,
	0xa1f0, 0x7df0, 0x7ef0, 0x001a, 0x001a, 0xc2f2, 0x001a, 0xc1f2,
	0xbec3, 0xb4f4, 0xa4c4, 0xb3f4, 0x001a, 0xf0f5, 0x45f7, 0xa6c5,
	0x43f9, 0x44f9, 0xd8c5, 0xdaa6, 0x001a, 0xd7aa, 0x52db, 0x4ebb,
	0x7bc1, 0xefed, 0xdba6, 0x001a, 0xe5af, 0xe4af, 0x53db, 0x001a,
	0x001a, 0x001a, 0xf4ea, 0xdca6, 0x50ad, 0x001a, 0x001a, 0x54db,
	0x55db, 0x56db, 0x4fbb, 0xb2bf, 0xdda6, 0x001a, 0xd8aa, 0x68d0,
	0xe6af, 0x70d3, 0xeab2, 0x001a, 0x57db, 0xa4b8, 0x001a, 0x50bb,
	0xb3bf, 0x7cc1, 0xc2c2, 0xb5f4, 0xdea6, 0xd9aa, 0x001a, 0x001a,
	0xe7af, 0x52d7, 0xceb5, 0x001a, 0x51bb, 0xabe3, 0x45e7, 0x001a,
	0x001a, 0x001a, 0x001a, 0xdfa6, 0xcfb5, 0xa3df, 0x52bb, 0xe0a6,
	0xb1cd, 0x69d0, 0x51ad, 0x001a, 0x001a, 0x72d3, 0x001a, 0x001a,
	0xeaaf, 0x001a, 0xe8af, 0xe9af, 0xebaf, 0x001a, 0x001a, 0x71d3,
	0x001a, 0x001a, 0x57d7, 0x54d7, 0x56d7, 0xebb2, 0xedb2, 0xecb2,
	0x53d7, 0xeeb2, 0x55d7, 0x001a, 0x58db, 0x59db, 0x001a, 0x5adb,
	0xa6df, 0x001a, 0xa7df, 0x001a, 0xa5df, 0xa8df, 0x001a, 0xa5b8,
	0x001a, 0xa4df, 0x001a, 0x53bb, 0x001a, 0x001a, 0x4ae7, 0x46e7,
	0x49e7, 0x4be7, 0x48e7, 0x47e7, 0x001a, 0xf5ea, 0xf6ea, 0xf7ea,
	0xb4bf, 0xb5bf, 0xf1ed, 0xf0ed, 0xf2ed, 0x001a, 0xa3f0, 0xa2f0,
	0x001a, 0xc4f2, 0x001a, 0xc5f2, 0xc3f2, 0x001a, 0xa5c4, 0x001a,
	0xb6f4, 0xb7f4, 0x001a, 0x46f7, 0xeff7, 0xbbf8, 0xe1a6, 0x7da8,
	0x001a, 0x7dc1, 0xe2a6, 0x001a, 0x58d7, 0x5bdb, 0x001a, 0x41c6,
	0x4aca, 0x001a, 0x001a, 0x001a, 0x4bca, 0x4dca, 0xe3a6, 0x4eca,
	0x4cca, 0x001a, 0x001a, 0xa2cb, 0xa3cb, 0x7bcb, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa1cb, 0xa1a8, 0x001a, 0xa2a8, 0x7ccb, 0x7acb,
	0x79cb, 0x7dcb, 0x7ea8, 0x7ecb, 0x6ad0, 0x001a, 0x001a, 0x001a,
	0xb6cd, 0xdcaa, 0xb5cd, 0xb7cd, 0x001a, 0xdbaa, 0xbccd, 0xdfaa,
	0xb2cd, 0xc0cd, 0xc6cd, 0xe6aa, 0xc3cd, 0xe3aa, 0x001a, 0xb9cd,
	0xbfcd, 0xc1cd, 0x001a, 0xb4cd, 0xe2aa, 0xddaa, 0xbacd, 0xe4aa,
	0xe7aa, 0xe1aa, 0x001a, 0xdaaa, 0xbecd, 0xb8cd, 0xc5cd, 0xe9aa,
	0xe5aa, 0xe0aa, 0xbdcd, 0xecaf, 0xbbcd, 0xdeaa, 0xe8aa, 0x001a,
	0xb3cd, 0x001a, 0xc2cd, 0xc4cd, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x62ad, 0x5cad, 0x64ad, 0x61ad, 0x71d0, 0x74d0, 0x5dad,
	0x001a, 0x6bd0, 0x001a, 0x56ad, 0x60ad, 0x001a, 0x63ad, 0x65ad,
	0xa2d0, 0x77d0, 0x001a, 0x55ad, 0xa1d0, 0x59ad, 0x57ad, 0x52ad,
	0x6fd0, 0x001a, 0x7ed0, 0x73d0, 0x76d0, 0xa5d0, 0x001a, 0x66ad,
	0x7dd0, 0x5ead, 0x78d0, 0xa4d0, 0x75d0, 0x79d0, 0x7cd0, 0x001a,
	0x001a, 0x6dd0, 0xa3d0, 0x7bd0, 0x001a, 0x001a, 0x6cd0, 0x001a,
	0x70d0, 0x5fad, 0x5aad, 0x53ad, 0x58ad, 0x54ad, 0x67ad, 0x6ed0,
	0xa5d3, 0x5bad, 0x001a, 0x001a, 0x7ad0, 0x41ce, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa8d3, 0xfaaf,
	0x001a, 0x76d3, 0x001a, 0xa3d3, 0x7dd3, 0x001a, 0xb2d3, 0x001a,
	0xaad3, 0x001a, 0x7ed3, 0x001a, 0xa9d3, 0x78d3, 0x7cd3, 0xb5d3,
	0xfdaf, 0xadd3, 0xa4d3, 0xedaf, 0xb3d3, 0x74d3, 0x001a, 0xacd3,
	0x001a, 0xfcaf, 0xf7af, 0x73d3, 0xf5af, 0xf4af, 0xf9af, 0xabd3,
	0xf1af, 0xf8af, 0x72d0, 0x5cdb, 0xa6d3, 0x001a, 0x001a, 0x7ad3,
	0xfbaf, 0x7bd3, 0xa1d3, 0xfeaf, 0x75d3, 0xafd3, 0x001a, 0xaed3,
	0xb6d3, 0xf3af, 0xf0af, 0xb4d3, 0xb0d3, 0xa7d3, 0xa2d3, 0xf6af,
	0xf2af, 0x77d3, 0xeeaf, 0xb1d3, 0xefaf, 0x001a, 0x79d3, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x5ed7, 0x60d7, 0x65d7, 0x79d7, 0xfcb2,
	0xf2b2, 0x001a, 0x5dd7, 0xfdb2, 0xfeb2, 0x68d7, 0x6fd7, 0x75d7,
	0x001a, 0x62d7, 0x001a, 0x69d7, 0x001a, 0x001a, 0x40b3, 0x77d7,
	0x72d7, 0xfab2, 0xf8b2, 0x6ed7, 0x6ad7, 0x5cd7, 0xefb2, 0x61d7,
	0x59d7, 0x001a, 0xf7b2, 0xf9b2, 0x66d7, 0x63d7, 0xf4b2, 0x73d7,
	0xf1b2, 0x64d7, 0x7ad7, 0x6cd7, 0x001a, 0x6bd7, 0xf0b2, 0x001a,
	0xfbb2, 0x001a, 0xf3b2, 0x5ad7, 0x5fd7, 0x70d7, 0x76d7, 0x41b3,
	0x5bd7, 0x67d7, 0x6dd7, 0xf6b2, 0x001a, 0x001a, 0x78d7, 0x71d7,
	0x74d7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf5b2, 0x001a, 0x6cdb,
	0x60db, 0xd7b5, 0x7ddb, 0xa7db, 0xaadb, 0xd5b5, 0x68db, 0xa3db,
	0x69db, 0x77db, 0xe2b5, 0x73db, 0xdfb5, 0x001a, 0x74db, 0x5ddb,
	0x001a, 0xa4db, 0x001a, 0x001a, 0xe8b5, 0xa1db, 0x75db, 0xacdb,
	0x70db, 0xc8df, 0x001a, 0xafdb, 0xe6b5, 0x6edb, 0x7adb, 0xe9b5,
	0xd4b5, 0x72db, 0xaddb, 0x6bdb, 0x64db, 0x6fdb, 0x001a, 0x63db,
	0x61db, 0xd0b5, 0xa5db, 0x6adb, 0xa8db, 0x001a, 0xa9db, 0xd8b5,
	0xddb5, 0xd9b5, 0xe1b5, 0x7edb, 0xdab5, 0x76db, 0x66db, 0x001a,
	0xd2b5, 0x5edb, 0xa2db, 0xabdb, 0x65db, 0xe0b5, 0xb0db, 0x71db,
	0x001a, 0x6ddb, 0x001a, 0xd1b5, 0xe5b5, 0x001a, 0x7cdb, 0xe7b5,
	0x001a, 0x78db, 0xdcb5, 0xd6b5, 0xdeb5, 0xd3b5, 0xe4b5, 0x79db,
	0x67db, 0x7bdb, 0x62db, 0xa6db, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xaedb, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x5fdb, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc7df, 0x001a, 0xdddf, 0x55b8, 0xccdf, 0x001a, 0xcadf,
	0xb5df, 0xa9b8, 0xc5df, 0xd9df, 0xc1df, 0xb1b8, 0xd8df, 0xbfdf,
	0xe3b5, 0xcfdf, 0xc0df, 0xd6df, 0xb0b8, 0xa8b8, 0x001a, 0xaadf,
	0xb2df, 0x001a, 0xcbdf, 0xc3df, 0xdcdf, 0xc6df, 0xb6b8, 0xd7df,
	0x001a, 0xadb8, 0x001a, 0xc9df, 0xd1df, 0xb6df, 0xd0df, 0x001a,
	0xe1df, 0xb1df, 0xd2df, 0x001a, 0xdfdf, 0x001a, 0xabdf, 0xdbb5,
	0x001a, 0xb9df, 0xb8df, 0xafb8, 0x001a, 0xbcdf, 0xbedf, 0xcddf,
	0xdedf, 0xb2b8, 0x001a, 0xb3b8, 0x001a, 0xb0df, 0xabb8, 0xb4df,
	0xdadf, 0xb4b8, 0x001a, 0xacb8, 0xaeb8, 0xb5b8, 0xe0df, 0xd3df,
	0xcedf, 0x001a, 0x001a, 0xbbdf, 0xbadf, 0xaab8, 0xacdf, 0xa7b8,
	0xc4df, 0xaddf, 0xc2df, 0x001a, 0x001a, 0xb7df, 0xdbdf, 0x001a,
	0x001a, 0x001a, 0xa6b8, 0x001a, 0x001a, 0x001a, 0xb3df, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xafdf, 0xd5df, 0xaedf,
	0x60bb, 0xd3e3, 0x001a, 0x001a, 0xc2e3, 0x001a, 0x001a, 0xace3,
	0xcae3, 0x58bb, 0xbbe3, 0xc5e3, 0x5bbb, 0xbee3, 0x59bb, 0xafe3,
	0xcde3, 0xaee3, 0xc1e3, 0x001a, 0xade3, 0x001a, 0x001a, 0xbfe3,
	0xc8e3, 0xc6e3, 0xbae3, 0xb5e3, 0xb3e3, 0x001a, 0xb4e3, 0xc7e3,
	0xd2e3, 0xbce3, 0x5abb, 0x001a, 0xb7e3, 0x001a, 0xcbe3, 0x001a,
	0x5dbb, 0xb6e3, 0xb0e3, 0xc0e3, 0x61bb, 0x001a, 0x001a, 0x55bb,
	0x5ebb, 0xb8e3, 0xb2e3, 0x001a, 0x57bb, 0xd4df, 0x56bb, 0xc3e3,
	0x001a, 0x54bb, 0x63bb, 0x5cbb, 0xc4e3, 0xb9e3, 0xb1e3, 0xcce3,
	0xbde3, 0x62bb, 0xd0e3, 0x5fbb, 0xcfe3, 0x001a, 0xc9e3, 0xcee3,
	0x001a, 0x001a, 0x001a, 0xd1e3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x73e7,
	0x74e7, 0x67e7, 0x66e7, 0x62e7, 0xb4bd, 0x001a, 0xacbd, 0x76e7,
	0x75e7, 0xa9df, 0x5fe7, 0x63e7, 0x5de7, 0x001a, 0x70e7, 0x61e7,
	0x001a, 0x77e7, 0x5ae7, 0x58e7, 0x64e7, 0x6ee7, 0x69e7, 0xb6bd,
	0x4fe7, 0x001a, 0x6de7, 0x001a, 0x001a, 0x001a, 0xb7bd, 0xbddf,
	0x5be7, 0x52e7, 0x55e7, 0x7be7, 0x5ce7, 0x53e7, 0x51e7, 0x4ee7,
	0x001a, 0xb0bd, 0x65e7, 0xafbd, 0xb3bd, 0x60e7, 0x68e7, 0xa9bd,
	0x78e7, 0x7ce7, 0xabbd, 0x001a, 0x57e7, 0x6be7, 0x6fe7, 0x54e7,
	0x79e7, 0xb2bd, 0x001a, 0xb1bd, 0x4ce7, 0xb5bd, 0x72e7, 0x56e7,
	0x6ae7, 0x50e7, 0x5ee7, 0x59e7, 0xadbd, 0xaebd, 0x6ce7, 0x7de7,
	0x7ae7, 0x71e7, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x4de7, 0x001a, 0xaabd, 0x49eb, 0x001a,
	0x40eb, 0x43eb, 0x001a, 0xbbbf, 0x45eb, 0xf9ea, 0x41eb, 0x47eb,
	0xb8bf, 0xbcbf, 0xb6bf, 0x001a, 0x001a, 0xfbea, 0x4ceb, 0x001a,
	0x001a, 0x46eb, 0x001a, 0xfcea, 0x55eb, 0x4feb, 0xf8ea, 0x46ee,
	0xfeea, 0xb7bf, 0x001a, 0x4aeb, 0x001a, 0x54eb, 0xbfbf, 0x001a,
	0x51eb, 0xfdea, 0x44eb, 0x48eb, 0x42eb, 0x56eb, 0x53eb, 0x50eb,
	0xb9bf, 0xbabf, 0xbebf, 0xfaea, 0x57eb, 0xbdbf, 0x4deb, 0x001a,
	0x001a, 0x4beb, 0x001a, 0x001a, 0x001a, 0x4eeb, 0x53ee, 0x40ee,
	0x45ee, 0x52ee, 0x44ee, 0xfbed, 0x41ee, 0x001a, 0xa2c1, 0x001a,
	0xf4ed, 0x4dee, 0x4fee, 0xf3ed, 0xa1c1, 0x51ee, 0x49ee, 0xa8c1,
	0x50ee, 0x42ee, 0xaac1, 0xf9ed, 0x52eb, 0x4aee, 0x47ee, 0xf5ed,
	0x55ee, 0xa4c1, 0x001a, 0x001a, 0xa5c1, 0xf7ed, 0x48ee, 0x001a,
	0x54ee, 0x4bee, 0xfded, 0xa7c1, 0xa3c1, 0x4cee, 0xfeed, 0x56ee,
	0xf8ed, 0x43ee, 0x4eee, 0xfaed, 0xfced, 0x001a, 0xcbc2, 0xf6ed,
	0xa9c1, 0xc4c2, 0x7ec1, 0x001a, 0x001a, 0x001a, 0x001a, 0xa6c1,
	0xc8c2, 0xb3f0, 0x001a, 0xa9f0, 0xa4f0, 0xaaf0, 0xb4f0, 0xb8f0,
	0xb7f0, 0xcac2, 0xc9c2, 0x001a, 0x001a, 0xabf0, 0xb9f0, 0xaef0,
	0xa6f0, 0x001a, 0xa8f0, 0xa7f0, 0xadf0, 0xb2f0, 0xa5f0, 0xacf0,
	0xb1f0, 0xc7c2, 0x001a, 0xaff0, 0x001a, 0xc5c2, 0xb0f0, 0xc3c2,
	0xc6c2, 0xd5f2, 0xb5f0, 0x001a, 0x001a, 0xc2c3, 0x001a, 0xcdf2,
	0xd1f2, 0xc9f2, 0xccf2, 0x001a, 0xd4f2, 0xc0c3, 0xd9f2, 0xd2f2,
	0x001a, 0xcaf2, 0xdaf2, 0xd3f2, 0xc3c3, 0xc4c3, 0xd7f2, 0x001a,
	0xcbf2, 0xbfc3, 0xc1c3, 0xc6f2, 0xcef2, 0xc8f2, 0x001a, 0xd8f2,
	0xd6f2, 0xc7f2, 0xcff2, 0x001a, 0x001a, 0x001a, 0xbef4, 0xc5c3,
	0xd0f2, 0xa7c4, 0xa9c4, 0xa6c4, 0x001a, 0xc3f4, 0xbbf4, 0xb9f4,
	0xbdf4, 0xbaf4, 0x001a, 0x001a, 0xbff4, 0xc1f4, 0xaac4, 0xacc4,
	0x001a, 0xc0f4, 0xadc4, 0xabc4, 0xc2f4, 0x001a, 0x001a, 0x001a,
	0x001a, 0xa8c4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf4c4,
	0xf1f5, 0xf7f5, 0xf6c4, 0xbcf4, 0xf6f5, 0x001a, 0xfdf5, 0xf4f5,
	0xfbf5, 0xfaf5, 0xb8f4, 0xf5f5, 0xb6f0, 0xfef5, 0xf3f5, 0xf8f5,
	0x001a, 0xfcf5, 0xf2f5, 0x001a, 0x4af7, 0xf5c4, 0xf9f5, 0x001a,
	0x001a, 0xf4f7, 0x4bf7, 0x49f7, 0x47f7, 0x48f7, 0x4cf7, 0x001a,
	0xd9c5, 0xf2f7, 0xf0f7, 0xf5f7, 0xf3f7, 0x001a, 0xf6f7, 0xdac5,
	0xf1f7, 0x001a, 0x001a, 0xbcf8, 0x001a, 0x001a, 0x45f9, 0x46f9,
	0x47f9, 0x001a, 0x001a, 0xc7f9, 0xbdf9, 0x4fca, 0xeaaa, 0x001a,
	0x68ad, 0x001a, 0xb8d3, 0xb7d3, 0x40b0, 0x42b3, 0x7cd7, 0x001a,
	0x001a, 0x7bd7, 0x001a, 0xeab5, 0xb8b8, 0x001a, 0xb7b8, 0xb9b8,
	0x001a, 0xd4e3, 0x7ee7, 0x58eb, 0x5aeb, 0x59eb, 0x001a, 0xabc1,
	0x57ee, 0xbaf0, 0xa5f9, 0xe4a6, 0x001a, 0xc9cd, 0xcacd, 0xc8cd,
	0xc7cd, 0xebaa, 0x001a, 0xa9d0, 0xa7d0, 0x001a, 0x001a, 0xa6d0,
	0x001a, 0x69ad, 0x6bad, 0x6aad, 0xa8d0, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc4d3, 0xc1d3, 0xbfd3,
	0x001a, 0x001a, 0x41b0, 0xc2d3, 0x46b0, 0xbcd3, 0xcbd3, 0x001a,
	0xcdd3, 0xbdd3, 0x001a, 0x43b0, 0xced3, 0xc9d3, 0xbbd3, 0xc0d3,
	0xcad3, 0xc6d3, 0xc3d3, 0x001a, 0x48b0, 0xccd3, 0xbed3, 0x001a,
	0x001a, 0xc7d3, 0xb9d3, 0x47b0, 0x44b0, 0xc5d3, 0x001a, 0xc8d3,
	0xbad3, 0x45b0, 0x42b0, 0x001a, 0x001a, 0x001a, 0x001a, 0x4cb3,
	0xa5d7, 0x4bb3, 0x001a, 0xa8d7, 0xabd7, 0x48b3, 0x46b3, 0x7ed7,
	0xa9d7, 0xa7d7, 0xa4d7, 0xacd7, 0xadd7, 0xafd7, 0xb0d7, 0x7dd7,
	0x45b3, 0xa2d7, 0xa1d7, 0xaed7, 0x47b3, 0xa3d7, 0x49b3, 0x44b3,
	0xa6d7, 0x4db3, 0x001a, 0x4ab3, 0xaad7, 0x001a, 0x001a, 0x001a,
	0xf1b5, 0xbfdb, 0x001a, 0xb4db, 0xeeb5, 0x001a, 0xe7df, 0xbddb,
	0xb1db, 0xecb5, 0xb6db, 0xefb5, 0xbadb, 0xb8db, 0xf2b5, 0xebb5,
	0x001a, 0x001a, 0xb2db, 0xb5db, 0xf0b5, 0x001a, 0xb3db, 0x001a,
	0xbedb, 0xbcdb, 0xb7db, 0xb9db, 0xbbdb, 0xedb5, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xe8df, 0xeedf, 0xe4df,
	0xeadf, 0xbab8, 0xe6df, 0xc0b8, 0x001a, 0x001a, 0xbfb8, 0x001a,
	0xbeb8, 0xeddf, 0xc1b8, 0xc2b8, 0xe3df, 0xf0df, 0xc3b8, 0xbdb8,
	0xbcb8, 0xecdf, 0xc4b8, 0xe2df, 0xe5df, 0xefdf, 0xebdf, 0x001a,
	0x001a, 0xf4e3, 0xe9e3, 0xbbb8, 0x001a, 0x001a, 0x001a, 0x001a,
	0x6abb, 0xdde3, 0xf2e3, 0xdee3, 0x65bb, 0x001a, 0xdbe3, 0x001a,
	0xe4e3, 0xdce3, 0x67bb, 0xd6e3, 0xf1e3, 0x68bb, 0xeee3, 0xefe3,
	0xd7e3, 0x6dbb, 0xe6e3, 0x001a, 0xe0e3, 0xe7e3, 0xdae3, 0x001a,
	0xf3e3, 0xebe3, 0xe5e3, 0xd5e3, 0x69bb, 0xece3, 0x001a, 0x6cbb,
	0xf0e3, 0x001a, 0xeae3, 0x66bb, 0xe8e3, 0x001a, 0xe2e3, 0x64bb,
	0xd9e3, 0xe1e3, 0xede3, 0xdfe3, 0x001a, 0x001a, 0xe3e3, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xc1bd, 0xe9df, 0xb2e7, 0xbbe7,
	0xb1e7, 0xade7, 0xaae7, 0xc2bd, 0xa8e7, 0x6bbb, 0xa1e7, 0xc0bd,
	0xa7e7, 0xbfbd, 0xace7, 0xa9e7, 0xb9e7, 0xb4e7, 0xaee7, 0xb3e7,
	0xbbbd, 0xabe7, 0xbee7, 0xa2e7, 0xa3e7, 0xbae7, 0xbcbd, 0xbfe7,
	0xbebd, 0xc0e7, 0xb0e7, 0xd8e3, 0xb6e7, 0xafe7, 0xb8e7, 0xb5e7,
	0x001a, 0x001a, 0x001a, 0xa6e7, 0xb9bd, 0xbde7, 0xbabd, 0xa4e7,
	0xbdbd, 0x64eb, 0xb7e7, 0xbce7, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x61eb, 0xb8bd, 0xc0bf, 0x6beb, 0x67eb, 0x001a, 0x65eb,
	0x60eb, 0x6feb, 0x001a, 0x001a, 0x001a, 0xc4bf, 0x001a, 0x5ceb,
	0x68eb, 0x69eb, 0x5feb, 0x5eeb, 0x6ceb, 0x001a, 0x62eb, 0x5deb,
	0x63eb, 0x001a, 0x6eeb, 0x5beb, 0x6deb, 0x6aeb, 0xc2bf, 0xc1bf,
	0x001a, 0x001a, 0xc3bf, 0x66eb, 0xcbf0, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x59ee, 0xb1c1, 0x5dee, 0x5aee, 0x61ee, 0x67ee,
	0x5cee, 0x001a, 0x70ee, 0xaec1, 0x6aee, 0x5fee, 0x6bee, 0x66ee,
	0x6dee, 0x5eee, 0xb3c1, 0xb2c1, 0x60ee, 0x6eee, 0x58ee, 0x6cee,
	0xacc1, 0x001a, 0x64ee, 0x63ee, 0x68ee, 0x5bee, 0xb0c1, 0x001a,
	0xb4c1, 0x62ee, 0x69ee, 0xb5c1, 0x65ee, 0x001a, 0x001a, 0x001a,
	0x001a, 0xadc1, 0xafc1, 0xc7f0, 0xc5f0, 0x001a, 0x001a, 0xccf0,
	0xc9f0, 0xcdf0, 0x001a, 0xbef0, 0xc6f0, 0xd1f0, 0x6fee, 0xc2f0,
	0xcfc2, 0xa5e7, 0xbdf0, 0xcaf0, 0xc4f0, 0xc1f0, 0xbcf0, 0xbbf0,
	0xd0f0, 0x001a, 0xc0f0, 0xbff0, 0xcdc2, 0xc8f0, 0x001a, 0xccc2,
	0x001a, 0x001a, 0xcec2, 0xc3f0, 0xcff0, 0x001a, 0xdef2, 0xdff2,
	0x001a, 0xc9c3, 0xdcf2, 0xc6c3, 0xe4f2, 0x001a, 0xcac3, 0xe6f2,
	0xdbf2, 0xcef0, 0xe8f2, 0xddf2, 0x001a, 0xc7c3, 0xe3f2, 0x001a,
	0xe5f2, 0xe0f2, 0xe7f2, 0xe2f2, 0xe1f2, 0xc8c3, 0x001a, 0x001a,
	0xc5f4, 0xc6f4, 0x001a, 0xc8f4, 0xaec4, 0xafc4, 0xc9f4, 0xc7f4,
	0x001a, 0xc4f4, 0x001a, 0x42f6, 0x45f6, 0x41f6, 0x001a, 0xfac4,
	0x43f6, 0xf9c4, 0xf8c4, 0xf7c4, 0x44f6, 0x51f7, 0x4ff7, 0x001a,
	0x4ef7, 0x40f6, 0x50f7, 0x46f6, 0x4df7, 0x001a, 0xf9f7, 0xd7f7,
	0xf7f7, 0xdbc5, 0xf8f7, 0xfaf7, 0x001a, 0xbff8, 0xfac5, 0xbef8,
	0xbdf8, 0xfbc5, 0x001a, 0x5ac6, 0x6ef9, 0xa7f9, 0xa6f9, 0xa8f9,
	0xe5a6, 0xaad0, 0x001a, 0xcfd3, 0xd0d3, 0x001a, 0x001a, 0x001a,
	0xc0db, 0x001a, 0x47f6, 0xc0f8, 0xe6a6, 0x6cad, 0xabd0, 0x001a,
	0x001a, 0x001a, 0xb1d7, 0x4eb3, 0x001a, 0xc2db, 0xc1db, 0xf3b5,
	0x001a, 0xc5b8, 0xc1e7, 0xc3bd, 0x001a, 0xc4bd, 0x001a, 0x001a,
	0x001a, 0xc5bf, 0xfcc5, 0xe7a6, 0x001a, 0x001a, 0x001a, 0xacd0,
	0xedaa, 0xaed0, 0xadd0, 0x6dad, 0x001a, 0xd1d3, 0x001a, 0xd8d3,
	0x49b0, 0xd6d3, 0xd4d3, 0x001a, 0xdbd3, 0xd2d3, 0xd3d3, 0x4ab0,
	0x001a, 0x4eb0, 0x001a, 0x001a, 0xdcd3, 0x4db0, 0xdad3, 0xd7d3,
	0xd5d3, 0x4bb0, 0x4cb0, 0xd9d3, 0x001a, 0x001a, 0x001a, 0x001a,
	0x50b3, 0xb2d7, 0x001a, 0x55b3, 0xc2d7, 0x54b3, 0xc4d7, 0x001a,
	0x001a, 0xb8d7, 0x52b3, 0xc3d7, 0x001a, 0xb3d7, 0x53b3, 0xbfd7,
	0xbbd7, 0xbdd7, 0xb7d7, 0xbed7, 0x001a, 0x001a, 0x4fb3, 0xbad7,
	0x001a, 0xb9d7, 0xb5d7, 0x001a, 0xc0d7, 0x001a, 0x001a, 0xbcd7,
	0xb4d7, 0x001a, 0xb6d7, 0x51b3, 0xc1d7, 0x001a, 0x001a, 0x001a,
	0x001a, 0xf6b5, 0xcddb, 0x001a, 0x001a, 0x001a, 0xc9db, 0xcbdb,
	0xc6db, 0xc5db, 0xc3db, 0x001a, 0xcadb, 0xccdb, 0xc8db, 0x001a,
	0xc7db, 0xf4b5, 0xf5b5, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xcfdb, 0xcdb8, 0xf2df, 0xf8df, 0xf3df, 0xf4df, 0xd8f9,
	0xf9df, 0x001a, 0xcfb8, 0x001a, 0xc7b8, 0xceb8, 0xf1df, 0xc4db,
	0xcab8, 0xc8b8, 0xf7df, 0xf6df, 0xc9b8, 0xcbb8, 0xf5df, 0xc6b8,
	0x001a, 0xccb8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf6e3,
	0x74bb, 0x001a, 0x001a, 0x42e4, 0x41e4, 0x001a, 0xfbe3, 0x76bb,
	0x40e4, 0xf7e3, 0xf8e3, 0x6ebb, 0x70bb, 0x001a, 0xfde3, 0xf5e3,
	0x72bb, 0x71bb, 0xf9e3, 0xfee3, 0xfce3, 0x73bb, 0xfae3, 0x001a,
	0x001a, 0xcedb, 0x6fbb, 0x001a, 0x001a, 0xc2e7, 0xc9e7, 0xc6bd,
	0x001a, 0xcde7, 0xcabd, 0xc5e7, 0xc3e7, 0x001a, 0xcce7, 0x001a,
	0xc5bd, 0xcbe7, 0xc7bd, 0xc8bd, 0xc4e7, 0xc9bd, 0xcae7, 0xc6e7,
	0xc7e7, 0xc8e7, 0x75bb, 0x001a, 0x001a, 0x001a, 0x70eb, 0x7ceb,
	0x001a, 0xcabf, 0x77eb, 0x79eb, 0x001a, 0xc8bf, 0x71eb, 0x75eb,
	0x001a, 0x78eb, 0xc6bf, 0xc9bf, 0x7beb, 0x73eb, 0x74eb, 0x7aeb,
	0x72eb, 0x76eb, 0xc7bf, 0x72ee, 0x001a, 0x71ee, 0xb7c1, 0x77ee,
	0xb9c1, 0x001a, 0x001a, 0xb6c1, 0x73ee, 0xbac1, 0x74ee, 0x001a,
	0x001a, 0x75ee, 0x78ee, 0x001a, 0xb8c1, 0x001a, 0xd6f0, 0x001a,
	0x001a, 0xd9f0, 0x001a, 0xd3f0, 0xd5f0, 0x001a, 0x001a, 0xd4f0,
	0xd7f0, 0xd8f0, 0x76ee, 0xd2f0, 0x001a, 0x001a, 0xcdc3, 0xecf2,
	0xeff2, 0xf1f2, 0xeaf2, 0xebf2, 0xeef2, 0xf0f2, 0xcec3, 0xccc3,
	0xcbc3, 0xedf2, 0xe9f2, 0xcaf4, 0xb0c4, 0x001a, 0xcbf4, 0x001a,
	0x001a, 0x49f6, 0xfbc4, 0x4bf6, 0xfcc4, 0x48f6, 0x4af6, 0xa8c5,
	0x001a, 0x52f7, 0xa7c5, 0xfdf7, 0xfcf7, 0x001a, 0xfbf7, 0x001a,
	0x001a, 0x48f9, 0x49f9, 0x4bf9, 0x4af9, 0x001a, 0x50ca, 0xe8a6,
	0x001a, 0x6ead, 0xc5d7, 0xf7b5, 0x001a, 0xfadf, 0xd0c2, 0x001a,
	0xf2f2, 0x001a, 0x001a, 0xa3a8, 0x001a, 0x001a, 0x001a, 0x57b3,
	0x001a, 0x001a, 0x001a, 0x56b3, 0x001a, 0xd0db, 0xf8b5, 0xd2db,
	0xd1db, 0x001a, 0x001a, 0xfbdf, 0xd0b8, 0x43e4, 0x46e4, 0x45e4,
	0x001a, 0x44e4, 0xcee7, 0xd0e7, 0xcfe7, 0x001a, 0xccbf, 0x001a,
	0x001a, 0x001a, 0xcbbf, 0x001a, 0xbbc1, 0x79ee, 0x7bee, 0x7aee,
	0x001a, 0x001a, 0xd1c2, 0x001a, 0x001a, 0x001a, 0xf4f2, 0xf3f2,
	0x001a, 0xccf4, 0xb1c4, 0x001a, 0x001a, 0xfdc4, 0x54f7, 0x53f7,
	0x5bc6, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xa4a8, 0xafd0, 0x6fad, 0xc8d7, 0xc6d7, 0x001a,
	0x001a, 0xc7d7, 0xd4db, 0xd5db, 0x43e0, 0xd3db, 0x001a, 0xfcdf,
	0x41e0, 0x40e0, 0x42e0, 0xd1b8, 0xfedf, 0xfddf, 0x44e0, 0x001a,
	0x49e4, 0x47e4, 0x001a, 0x48e4, 0xd3e7, 0xd1e7, 0x001a, 0x001a,
	0xd2e7, 0x7deb, 0x7cee, 0x7dee, 0xd2c2, 0x001a, 0xf5f2, 0xcdf4,
	0xb2c4, 0x001a, 0x4cf6, 0x55f7, 0xa9c5, 0x001a, 0xfef7, 0x4cf9,
	0xa5a8, 0x001a, 0x71ad, 0x72ad, 0xb0d0, 0x001a, 0x001a, 0xb1d0,
	0x70ad, 0x001a, 0x54b0, 0x001a, 0x52b0, 0x001a, 0x51b0, 0x58b0,
	0x50b0, 0x59b0, 0xddd3, 0x56b0, 0x001a, 0x53b0, 0x57b0, 0x55b0,
	0x4fb0, 0x001a, 0x001a, 0x5fb3, 0x001a, 0x59b3, 0xccd7, 0x5eb3,
	0x001a, 0x001a, 0x60b3, 0x5ab3, 0x001a, 0x5bb3, 0x001a, 0xcad7,
	0x001a, 0x001a, 0x58b3, 0x001a, 0xcbd7, 0x5db3, 0x001a, 0x001a,
	0xc9d7, 0x5cb3, 0x001a, 0x001a, 0x44b6, 0x001a, 0x46b6, 0x001a,
	0x001a, 0xd8db, 0x45b6, 0xf9b5, 0xfdb5, 0x001a, 0xe4b8, 0x49e0,
	0xdadb, 0xfeb5, 0x001a, 0x001a, 0xdddb, 0xdedb, 0x43b6, 0x001a,
	0xe0db, 0x001a, 0xe2db, 0x001a, 0xe3db, 0xd7db, 0xd6db, 0xe4db,
	0x42b6, 0xe1db, 0xdfdb, 0x001a, 0x40b6, 0xfbb5, 0x47b6, 0xdbdb,
	0xdcdb, 0xd9db, 0x001a, 0x41b6, 0x001a, 0x001a, 0xfcb5, 0x001a,
	0xfab5, 0x48e0, 0xdfb8, 0xdab8, 0x001a, 0x001a, 0xd5b8, 0x001a,
	0xe5b8, 0xd6b8, 0x001a, 0xd2b8, 0xe1b8, 0xdeb8, 0xe0b8, 0x001a,
	0xd7b8, 0xdcb8, 0xd3b8, 0xd4b8, 0x50e0, 0x4de0, 0x45e0, 0x4ae0,
	0x001a, 0xe2b8, 0x51e0, 0xe3b8, 0xd9b8, 0x001a, 0x001a, 0x47e0,
	0x001a, 0x4fe0, 0x4be0, 0x4ee0, 0x4ce0, 0xddb8, 0x46e0, 0xd8b8,
	0x001a, 0x001a, 0x001a, 0x4ce4, 0x78bb, 0x7bbb, 0x001a, 0x4ee4,
	0x001a, 0xa5bb, 0x4de4, 0x7dbb, 0x001a, 0xcfbd, 0x4fe4, 0x001a,
	0xa4bb, 0x4be4, 0xa6bb, 0x001a, 0x001a, 0x001a, 0x79bb, 0x001a,
	0xdbb8, 0x7cbb, 0x001a, 0x7abb, 0x7ebb, 0xa2bb, 0x77bb, 0xa7bb,
	0xa3bb, 0x001a, 0xa1bb, 0x4ae4, 0x001a, 0x001a, 0x001a, 0x001a,
	0xd6bd, 0x001a, 0xd2bd, 0x001a, 0x001a, 0x001a, 0xd9bd, 0x001a,
	0xd6e7, 0xdabd, 0xe2e7, 0xdbe7, 0xcbbd, 0xe3e7, 0xdde7, 0xd5bd,
	0xdee7, 0x001a, 0xd4bd, 0xe1e7, 0xcebd, 0xdfe7, 0xd5e7, 0xcdbd,
	0xaaeb, 0xd3bd, 0x001a, 0xd0bd, 0x001a, 0xd8bd, 0x001a, 0xd4e7,
	0x001a, 0xd8e7, 0xccbd, 0xd7e7, 0xd9e7, 0xdae7, 0xd7bd, 0xdce7,
	0xe0e7, 0xe4e7, 0x001a, 0xdbbd, 0xd2bf, 0xa5eb, 0xabeb, 0xa8eb,
	0x7eeb, 0xaceb, 0xa1eb, 0x001a, 0xa7eb, 0x001a, 0xcdbf, 0xd3bf,
	0xadeb, 0x001a, 0x001a, 0xcfbf, 0x001a, 0xd9bf, 0xd4bf, 0xafeb,
	0xa9eb, 0xd0bf, 0xa2eb, 0xdabf, 0xa3eb, 0xa4eb, 0xdbbf, 0xd8bf,
	0xd1bd, 0x001a, 0xcebf, 0xb0eb, 0xdcbf, 0x001a, 0xd5bf, 0xaeeb,
	0xd1bf, 0xd6bf, 0xd7bf, 0x001a, 0xc3c1, 0xa4ee, 0xadee, 0xaaee,
	0xacee, 0x001a, 0xc0c1, 0xa5ee, 0x001a, 0xabee, 0xbcc1, 0xa7ee,
	0xc4c1, 0xa3ee, 0xa8ee, 0xafee, 0xa6eb, 0xa9ee, 0xa2ee, 0xbdc1,
	0xa1ee, 0xbec1, 0xb0ee, 0xbfc1, 0xaeee, 0xc2c1, 0x7eee, 0x001a,
	0xc1c1, 0x001a, 0xa6ee, 0xdcf0, 0xeaf0, 0xe5f0, 0xe7f0, 0xdbf0,
	0xd3c2, 0x001a, 0xdaf0, 0xd6c2, 0xd5c2, 0x001a, 0xe9f0, 0xe1f0,
	0xdef0, 0xe4f0, 0x001a, 0xddf0, 0x001a, 0xdff0, 0xe8f0, 0xe6f0,
	0x001a, 0xd4c2, 0xedf0, 0xebf0, 0xe2f0, 0xecf0, 0xe3f0, 0x001a,
	0xf9f2, 0xcfc3, 0x41f3, 0x001a, 0x001a, 0x4ff6, 0xd6c3, 0xe0f0,
	0xf7f2, 0xd2c3, 0xf8f2, 0xfdf2, 0x001a, 0x001a, 0xd4c3, 0xd5c3,
	0xf6f2, 0x40f3, 0x42f3, 0xfaf2, 0xfcf2, 0xfef2, 0xfbf2, 0x43f3,
	0xd1c3, 0xd7c3, 0xd3c3, 0x001a, 0xd0c3, 0xd0f4, 0x001a, 0xb7c4,
	0xcef4, 0x001a, 0x001a, 0xd2f4, 0x001a, 0xd3f4, 0xb5c4, 0xd4f4,
	0xd1f4, 0x001a, 0xcff4, 0xb8c4, 0xb4c4, 0xd5f4, 0x001a, 0xb6c4,
	0xb3c4, 0x001a, 0x001a, 0x001a, 0xfec4, 0x001a, 0x001a, 0x40c5,
	0x4ef6, 0x4df6, 0x50f6, 0x51f6, 0x001a, 0x41c5, 0x56f7, 0x5bf7,
	0xaac5, 0x001a, 0x58f7, 0x001a, 0x57f7, 0x5af7, 0x59f7, 0x001a,
	0x43f8, 0x001a, 0xdcc5, 0x42f8, 0x40f8, 0x001a, 0x41f8, 0x001a,
	0x001a, 0x001a, 0xfec5, 0xfdc5, 0xc1f8, 0xc2f8, 0x40c6, 0x001a,
	0x4df9, 0x4ef9, 0x67c6, 0x001a, 0x6dc6, 0x001a, 0xa9f9, 0xc8f9,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa6a8,
	0x001a, 0xcdd7, 0x001a, 0xced7, 0x52e0, 0x50e4, 0xe5e7, 0xc6c1,
	0x001a, 0xc5c1, 0xeef0, 0x44f3, 0x001a, 0x44f8, 0xa7a8, 0xded3,
	0x5ab0, 0x61b3, 0x54e0, 0x53e0, 0xdcbd, 0xe6e7, 0xddbd, 0xb1ee,
	0xd7c2, 0x001a, 0x001a, 0x001a, 0x76c6, 0xa8a8, 0xcbcd, 0xdfd3,
	0x001a, 0x001a, 0x62b3, 0x001a, 0xcfd7, 0xd0d7, 0x001a, 0xe5db,
	0x001a, 0x48b6, 0xe6b8, 0x001a, 0x56e0, 0x55e0, 0x57e0, 0x001a,
	0x51e4, 0x52e4, 0xa8bb, 0xddbf, 0xdebd, 0xdebf, 0x001a, 0xb5ee,
	0xb2ee, 0xb4ee, 0xb3ee, 0xc7c1, 0x001a, 0xeff0, 0x46f3, 0x45f3,
	0xa4cb, 0x5cb0, 0x5bb0, 0xe0d3, 0x001a, 0xd1d7, 0x001a, 0x001a,
	0xe7db, 0xe6db, 0x49b6, 0x001a, 0x59e0, 0x5ae0, 0x58e0, 0x001a,
	0x001a, 0xe8b8, 0xe7b8, 0x001a, 0xaabb, 0xa9bb, 0x001a, 0xe7e7,
	0xb3eb, 0xb1eb, 0xb2eb, 0xdfbf, 0xb7ee, 0xb6ee, 0x001a, 0xf2f0,
	0xf1f0, 0xf0f0, 0x47f3, 0x001a, 0xaaf9, 0xa9a8, 0x73ad, 0x001a,
	0x74ad, 0x5db0, 0x5eb0, 0xe2d3, 0xe1d3, 0xd2d7, 0x001a, 0x68b3,
	0x66b3, 0x63b3, 0x67b3, 0x65b3, 0x64b3, 0x001a, 0x001a, 0x4ab6,
	0xeadb, 0x001a, 0xedb8, 0x4cb6, 0x51b6, 0xecdb, 0x53b6, 0x52b6,
	0x55b6, 0xebdb, 0xe8db, 0x4fb6, 0x4bb6, 0x4db6, 0xe9db, 0x54b6,
	0x50b6, 0x4eb6, 0xefb8, 0xeeb8, 0xecb8, 0xf0b8, 0x001a, 0xeab8,
	0xebb8, 0x001a, 0xe9b8, 0x001a, 0x5be0, 0x001a, 0x001a, 0x54e4,
	0x001a, 0xacbb, 0xadbb, 0xabbb, 0x001a, 0x53e4, 0x001a, 0x55e4,
	0x001a, 0xeae7, 0xece7, 0x001a, 0xe7bd, 0xede7, 0xe0bd, 0xe9e7,
	0xdfbd, 0xe9bd, 0xe5bd, 0xe6bd, 0xe2bd, 0xe8e7, 0xe1bd, 0xeee7,
	0xebe7, 0x001a, 0xe8bd, 0x001a, 0xe3bd, 0xe4bd, 0xb5eb, 0x001a,
	0xb7eb, 0xb6eb, 0x001a, 0xb8eb, 0xe0bf, 0xb4eb, 0x001a, 0x001a,
	0xcbc1, 0xb8ee, 0xc8c1, 0xccc1, 0xcac1, 0xc9c1, 0xf3f0, 0x001a,
	0xf6f0, 0x001a, 0xf5f0, 0x001a, 0xf4f0, 0xd8c2, 0x48f3, 0x49f3,
	0xd8c3, 0x4af3, 0xd9c3, 0x001a, 0x001a, 0xbac4, 0x001a, 0xb9c4,
	0x52f6, 0x001a, 0x001a, 0x42c5, 0x53f6, 0x5cf7, 0xabc5, 0xacc5,
	0x001a, 0x45f8, 0x001a, 0x42c6, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xaaa8, 0x001a, 0x6ab3, 0x69b3,
	0x5ce0, 0x5de0, 0x001a, 0xaebb, 0xb9eb, 0xeabd, 0xbaeb, 0xb9ee,
	0xaba8, 0x001a, 0xb2d0, 0x76ad, 0x75ad, 0x001a, 0xe3d3, 0x5fb0,
	0xe4d3, 0xd5d7, 0x001a, 0xd4d7, 0x001a, 0xd3d7, 0x001a, 0x001a,
	0xeedb, 0x58b6, 0x001a, 0x001a, 0xeddb, 0x57b6, 0x001a, 0x001a,
	0x001a, 0xefdb, 0x56b6, 0x001a, 0x5fe0, 0x62e0, 0x60e0, 0x61e0,
	0x65e0, 0x5ee0, 0x66e0, 0x63e0, 0x64e0, 0xb0bb, 0x56e4, 0x001a,
	0x001a, 0xafbb, 0x001a, 0xf2e7, 0xf0e7, 0x001a, 0x001a, 0xebbd,
	0xefe7, 0xf1e7, 0x001a, 0xecbd, 0x001a, 0xbbeb, 0x001a, 0xbceb,
	0xcdc1, 0x001a, 0x4cf3, 0x4ef3, 0x4bf3, 0x4df3, 0xd6f4, 0x54f6,
	0x001a, 0x001a, 0x6ff9, 0xaca8, 0x77ad, 0xe5d3, 0xe7d3, 0xe6d3,
	0x001a, 0xd8d7, 0x6cb3, 0x001a, 0xd6d7, 0x001a, 0x6bb3, 0xd9d7,
	0x001a, 0xdad7, 0xd7d7, 0x001a, 0x001a, 0xfbdb, 0x60b6, 0xf3db,
	0xf9db, 0x001a, 0x001a, 0x5bb6, 0x5eb6, 0xf2db, 0x59b6, 0xf6db,
	0x6ce0, 0x5db6, 0x001a, 0xf1db, 0x001a, 0xf7db, 0xf4db, 0xfadb,
	0xf0db, 0xf8db, 0x5cb6, 0x5fb6, 0xf5db, 0x5ab6, 0x001a, 0xf2b8,
	0x68e0, 0xf1b8, 0x6fe0, 0x6ee0, 0xf8b8, 0x001a, 0xf9b8, 0x70e0,
	0xf3b8, 0x6de0, 0xf7b8, 0x72e0, 0x69e0, 0x001a, 0x6be0, 0xf4b8,
	0x67e0, 0x6ae0, 0x71e0, 0xf5b8, 0x73e0, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf6b8, 0x001a, 0xb1bb, 0x5be4, 0x61e4, 0x59e4,
	0x62e4, 0x001a, 0x58e4, 0x5de4, 0x63e4, 0x60e4, 0x5fe4, 0x5ee4,
	0x001a, 0x57e4, 0x5ce4, 0x001a, 0x001a, 0x5ae4, 0x001a, 0xf1bd,
	0xeebd, 0xfbe7, 0x41e8, 0x43e8, 0x40e8, 0xf8e7, 0xfae7, 0x45e8,
	0x42e8, 0xfce7, 0x46e8, 0xf9e7, 0x44e8, 0xefbd, 0xf5bd, 0xf3bd,
	0xf3e7, 0xf4bd, 0xf0bd, 0xf4e7, 0xf6e7, 0xf5e7, 0xfde7, 0xfee7,
	0x001a, 0xf2bd, 0x001a, 0xedbd, 0x001a, 0x001a, 0xf7e7, 0x001a,
	0xc6eb, 0xe2bf, 0x001a, 0xbdeb, 0xe3bf, 0xe6bf, 0xc2eb, 0x001a,
	0xbfeb, 0xe5bf, 0x001a, 0x001a, 0xc3eb, 0xc4eb, 0xbeeb, 0xc7eb,
	0xc0eb, 0xc5eb, 0xe4bf, 0x001a, 0xe1bf, 0xc1eb, 0x001a, 0xbfee,
	0xd0c1, 0xcec1, 0xd1c1, 0xcfc1, 0xbeee, 0xbbee, 0xbaee, 0x001a,
	0xbdee, 0x001a, 0x001a, 0xbcee, 0x45f1, 0xdec2, 0xfbf0, 0xfaf0,
	0x001a, 0xd9c2, 0x41f1, 0x40f1, 0xf7f0, 0x43f1, 0xfcf0, 0xddc2,
	0xf9f0, 0x42f1, 0xf8f0, 0xdac2, 0xdcc2, 0xfdf0, 0xdbc2, 0xfef0,
	0x001a, 0x44f1, 0x52f3, 0x001a, 0xdec3, 0x4ff3, 0x001a, 0x53f3,
	0x001a, 0x001a, 0xdbc3, 0x51f3, 0xe0c3, 0x001a, 0xddc3, 0x001a,
	0x50f3, 0x001a, 0xdfc3, 0x54f3, 0xdac3, 0x001a, 0x001a, 0x001a,
	0x001a, 0xbcc4, 0xbec4, 0x001a, 0xd9f4, 0xbdc4, 0xd7f4, 0xdcc3,
	0xd8f4, 0xbbc4, 0x43c5, 0x45c5, 0x56f6, 0x44c5, 0x55f6, 0x001a,
	0x61f7, 0xadc5, 0x60f7, 0xaec5, 0x5ef7, 0x5df7, 0x62f7, 0x63f7,
	0x46f8, 0x001a, 0x5ff7, 0x001a, 0x001a, 0xc6f8, 0xc3f8, 0xc4f8,
	0xc5f8, 0x5cc6, 0x001a, 0x51f9, 0x50f9, 0x4ff9, 0x70f9, 0x001a,
	0xbef9, 0xabf9, 0x6ec6, 0xada8, 0x60b0, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xfab8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xf6bd, 0x001a, 0x001a, 0xc8eb, 0x001a, 0x001a,
	0xdfc2, 0x001a, 0x55f3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xacf9, 0xaea8, 0xeeaa, 0x79ad, 0x78ad, 0x001a, 0x63b0,
	0x001a, 0xe8d3, 0x61b0, 0xe9d3, 0x62b0, 0x001a, 0x001a, 0xdfd7,
	0xdbd7, 0x001a, 0x001a, 0x6db3, 0xded7, 0xddd7, 0xdcd7, 0x6eb3,
	0xe0d7, 0xe1d7, 0x001a, 0x001a, 0x001a, 0x43dc, 0x41dc, 0x45dc,
	0x46dc, 0x4cdc, 0x001a, 0x48dc, 0x4adc, 0x001a, 0x42dc, 0xfcdb,
	0x001a, 0x49dc, 0x001a, 0x001a, 0x4bdc, 0x44dc, 0x47dc, 0xfddb,
	0x62b6, 0x40dc, 0xfedb, 0x61b6, 0x63b6, 0x001a, 0xfdb8, 0x75e0,
	0x77e0, 0x76e0, 0x7be0, 0xfbb8, 0x001a, 0x78e0, 0x74e0, 0x79e0,
	0x7ae0, 0xfcb8, 0xfeb8, 0x7ce0, 0x001a, 0x67e4, 0x66e4, 0x001a,
	0x64e4, 0x65e4, 0xb3bb, 0xb5bb, 0xb2bb, 0xb4bb, 0x4de8, 0x4ee8,
	0x49e8, 0x001a, 0x4ae8, 0xf8bd, 0xfdbd, 0xf7bd, 0xfebd, 0xf9bd,
	0x4be8, 0x001a, 0x001a, 0x4ce8, 0x48e8, 0x40be, 0xfbbd, 0x001a,
	0x001a, 0xfabd, 0xfcbd, 0x001a, 0x47e8, 0x001a, 0xcaeb, 0xe8bf,
	0x001a, 0x001a, 0xcceb, 0xeabf, 0xcfeb, 0xcbeb, 0xc9eb, 0xceeb,
	0xe9bf, 0xcdeb, 0x001a, 0xe7bf, 0x001a, 0x001a, 0xd3c1, 0xd6c1,
	0xc1ee, 0x001a, 0xd4c1, 0xc0ee, 0xd2c1, 0xd5c1, 0x46f1, 0x47f1,
	0x48f1, 0xe0c2, 0x001a, 0x49f1, 0x001a, 0xe1c2, 0xe2c3, 0x58f3,
	0x59f3, 0x57f3, 0x56f3, 0x5af3, 0xe1c3, 0xddf4, 0xdbf4, 0xdcf4,
	0xdef4, 0xdaf4, 0xdff4, 0x58f6, 0x001a, 0x59f6, 0x57f6, 0x46c5,
	0x64f7, 0xafc5, 0x65f7, 0x48f8, 0x47f8, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xafa8, 0x64b6, 0x001a, 0x001a, 0x40b9,
	0x001a, 0x001a, 0x001a, 0xb6bb, 0x001a, 0x001a, 0xecbf, 0x001a,
	0xebbf, 0x001a, 0x001a, 0x001a, 0x001a, 0xe3c3, 0x7cc4, 0x47c5,
	0xb0a8, 0x64b0, 0x41b9, 0x001a, 0x5bf3, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xa6cb,
	0x001a, 0x001a, 0xb1a8, 0x001a, 0xb4a8, 0xb3a8, 0xb2a8, 0x001a,
	0x001a, 0xa5cb, 0x001a, 0xcdcd, 0x001a, 0xcfcd, 0xefaa, 0x001a,
	0x001a, 0xf1aa, 0xcccd, 0xcecd, 0xf0aa, 0xd1cd, 0xd0cd, 0xd2cd,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb6d0, 0xb4d0, 0x7cad, 0xb3d0, 0xa3ad, 0x7ead, 0x7bad, 0x001a,
	0xa4ad, 0x001a, 0x7dad, 0xa2ad, 0x001a, 0xa1ad, 0xb5d0, 0x001a,
	0x7aad, 0x001a, 0x001a, 0x001a, 0x6ab0, 0xebd3, 0xf1d3, 0x67b0,
	0x6eb0, 0x001a, 0x69b0, 0xeed3, 0xf0d3, 0x6cb0, 0xead3, 0xedd3,
	0x68b0, 0x65b0, 0xecd3, 0x6bb0, 0xefd3, 0x6db0, 0x66b0, 0x001a,
	0x001a, 0x001a, 0x001a, 0xe3d7, 0xe6d7, 0x70b3, 0x001a, 0x7ab3,
	0x76b3, 0xe4d7, 0x001a, 0x001a, 0x7eb3, 0x77b3, 0x7cb3, 0x72b3,
	0x001a, 0x6fb3, 0x71b3, 0x7db3, 0xe5d7, 0x75b3, 0x78b3, 0x74b3,
	0x79b3, 0xe7d7, 0x7bb3, 0x73b3, 0xe2d7, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4ddc, 0x65b6, 0x4fdc,
	0x001a, 0x67b6, 0x69b6, 0x001a, 0x4edc, 0x66b6, 0x6ab6, 0x001a,
	0x68b6, 0x001a, 0x001a, 0x001a, 0x47b9, 0xa3e0, 0x4fb9, 0x7ee0,
	0x001a, 0x50b9, 0x45b9, 0x001a, 0xa1e0, 0x001a, 0x001a, 0x4ab9,
	0x001a, 0xa2e0, 0x43b9, 0x42b9, 0x001a, 0x4db9, 0x4cb9, 0x4bb9,
	0x49b9, 0x4eb9, 0x7de0, 0x44b9, 0x46b9, 0x48b9, 0x001a, 0x001a,
	0xb8bb, 0xbbbb, 0x001a, 0xbfbb, 0xb9bb, 0xbebb, 0xbcbb, 0x001a,
	0xb7bb, 0x001a, 0xbdbb, 0xbabb, 0x001a, 0x001a, 0x001a, 0x52e8,
	0x43be, 0x41be, 0x001a, 0x53e8, 0x001a, 0x44be, 0x42be, 0x51e8,
	0x50e8, 0x001a, 0xf0bf, 0x4fe8, 0xeebf, 0xedbf, 0xd0eb, 0x45be,
	0xefbf, 0xd1eb, 0xf2bf, 0xd2eb, 0xf1bf, 0xd8c1, 0xc3ee, 0xd7c1,
	0xdcc1, 0xdac1, 0xdbc1, 0xe3c2, 0xd9c1, 0xc2ee, 0xd3eb, 0xe2c2,
	0xe4c2, 0x001a, 0xe4c3, 0xe5c3, 0x001a, 0xe0f4, 0x001a, 0xdec5,
	0xddc5, 0xb6a8, 0x001a, 0x001a, 0x55ca, 0x6fb0, 0x001a, 0x52ca,
	0x53ca, 0x51ca, 0x001a, 0x54ca, 0x001a, 0x001a, 0xaacb, 0xa7cb,
	0xaccb, 0xa8cb, 0xb7a8, 0xbaa8, 0x001a, 0xa9cb, 0xb9a8, 0xabcb,
	0x001a, 0x001a, 0xb8a8, 0x001a, 0x001a, 0x001a, 0x001a, 0xd5cd,
	0xd7cd, 0xf4aa, 0xd3cd, 0xd6cd, 0xd4cd, 0xf2aa, 0xf5aa, 0x001a,
	0xf3aa, 0x001a, 0x001a, 0x001a, 0x001a, 0xb8d0, 0xbcd0, 0xb9d0,
	0x001a, 0xa7ad, 0x001a, 0xa8ad, 0x001a, 0xbbd0, 0x001a, 0xbdd0,
	0xbfd0, 0x001a, 0xa5ad, 0xbed0, 0x001a, 0x001a, 0xa6ad, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xeed7, 0xbad0, 0xf2d3, 0xfbd3,
	0xf9d3, 0xf4d3, 0xf5d3, 0xfad3, 0xfcd3, 0x71b0, 0x001a, 0xf7d3,
	0xf3d3, 0x70b0, 0x72b0, 0xf6d3, 0xfdd3, 0xf8d3, 0x001a, 0x001a,
	0xa1b3, 0xf1d7, 0xe9d7, 0xefd7, 0xf0d7, 0xa2b3, 0x001a, 0xe8d7,
	0xead7, 0xb7d0, 0xecd7, 0xedd7, 0xebd7, 0x6cb6, 0x001a, 0x001a,
	0x001a, 0x56dc, 0xd4eb, 0x57dc, 0x54dc, 0xa3b3, 0x6eb6, 0x53dc,
	0x59dc, 0x58dc, 0x6bb6, 0x5cdc, 0x52dc, 0x5bdc, 0x50dc, 0x5adc,
	0x55dc, 0x6db6, 0x001a, 0xaae0, 0x001a, 0xa5e0, 0xabe0, 0xa6e0,
	0xa4e0, 0xa7e0, 0x51b9, 0x001a, 0xa9e0, 0x001a, 0xa8e0, 0x52b9,
	0xc1bb, 0xc0bb, 0x6ee4, 0x71e4, 0x69e4, 0x6de4, 0xc2bb, 0x6ce4,
	0x6ae4, 0x70e4, 0x6be4, 0x68e4, 0x6fe4, 0x001a, 0x59e8, 0x48be,
	0x4af1, 0x56e8, 0x57e8, 0x55e8, 0x51dc, 0x47be, 0x5ae8, 0x54e8,
	0x46be, 0x49be, 0x58e8, 0xd5eb, 0xf3bf, 0xd6eb, 0xd7eb, 0x001a,
	0xc4ee, 0xddc1, 0x4bf1, 0x4cf1, 0x001a, 0x001a, 0x4df1, 0x5df3,
	0x5cf3, 0xe2f4, 0x001a, 0xe1f4, 0x5bf6, 0x5cf6, 0x5af6, 0x66f7,
	0xb0c5, 0xbba8, 0xaaad, 0xa9ad, 0x75b0, 0x74b0, 0x40d4, 0x41d4,
	0xfed3, 0x001a, 0x73b0, 0xf5d7, 0x001a, 0xf6d7, 0xf2d7, 0xa4b3,
	0xf3d7, 0x001a, 0xf4d7, 0x001a, 0x001a, 0x001a, 0x001a, 0x5fdc,
	0x61dc, 0x5ddc, 0x60dc, 0x6fb6, 0x5edc, 0x70b6, 0x001a, 0x001a,
	0x73dd, 0x55b9, 0x54b9, 0x001a, 0x53b9, 0x001a, 0xace0, 0xade0,
	0x001a, 0x001a, 0x73e4, 0x75e4, 0xc6bb, 0xc3bb, 0x001a, 0xc5bb,
	0xc4bb, 0x74e4, 0x72e4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x61e8, 0x5ee8, 0x5fe8, 0x4dbe, 0x60e8, 0x5be8, 0x5ce8, 0x4abe,
	0x001a, 0x4bbe, 0x5de8, 0x4cbe, 0x001a, 0xdbeb, 0x001a, 0xdceb,
	0xd9eb, 0xdaeb, 0xf4bf, 0xd8eb, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc8ee, 0xc5ee, 0xc7ee, 0xe0c1, 0xcbee, 0xdfc1, 0xc9ee,
	0xccee, 0xcaee, 0xc6ee, 0xdec1, 0x001a, 0x4ff1, 0x001a, 0x50f1,
	0x4ef1, 0x001a, 0x52f1, 0xe5c2, 0xe6c2, 0x5ff3, 0xe7c3, 0x51f1,
	0x5ef3, 0xe6c3, 0xe5f4, 0xe6f4, 0xbfc4, 0xe4f4, 0x001a, 0xe3f4,
	0x001a, 0x5df6, 0x48c5, 0x001a, 0x49f8, 0xc8f8, 0xc7f8, 0x001a,
	0x43c6, 0x5dc6, 0xc9f8, 0x71f9, 0x001a, 0x6fc6, 0xbca8, 0xf6aa,
	0x001a, 0x56b9, 0x001a, 0xc0c4, 0xbda8, 0xabad, 0xa5b3, 0x71b6,
	0xe7c2, 0xf7aa, 0x001a, 0xc1d0, 0xc0d0, 0x42d4, 0x001a, 0x78b0,
	0x76b0, 0x7ab0, 0x44d4, 0x001a, 0x79b0, 0x77b0, 0x001a, 0x001a,
	0x001a, 0x001a, 0x43d4, 0xa8b3, 0xfcd7, 0x001a, 0xa7b3, 0xa9b3,
	0x42d8, 0xabb3, 0xfed7, 0x40d8, 0xf7d7, 0xaab3, 0x43d8, 0x001a,
	0x001a, 0xf9d7, 0x001a, 0xfad7, 0xf8d7, 0xa6b3, 0x001a, 0x41d8,
	0xfbd7, 0xfdd7, 0x001a, 0x001a, 0x001a, 0x6ddc, 0x001a, 0x6cdc,
	0x6adc, 0x62dc, 0x71dc, 0x65dc, 0x6fdc, 0x76dc, 0x6edc, 0x79b6,
	0x001a, 0x75b6, 0x63dc, 0x001a, 0x69dc, 0x77b6, 0x001a, 0x68dc,
	0x78b6, 0x7ab6, 0x6bdc, 0x001a, 0x72b6, 0x73b6, 0x77dc, 0x75dc,
	0x001a, 0x74dc, 0x66dc, 0x001a, 0x72dc, 0x001a, 0x76b6, 0x001a,
	0x001a, 0x001a, 0x001a, 0x74b6, 0x73dc, 0x64dc, 0x67dc, 0x70dc,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbae4, 0xb7e0, 0x001a,
	0xb0e0, 0xc3e0, 0xcce0, 0xb3e0, 0x61b9, 0x001a, 0xc0e0, 0x57b9,
	0x59b9, 0x65b9, 0xb1e0, 0x001a, 0x001a, 0x5ab9, 0x5cb9, 0x66b9,
	0x5bb9, 0x001a, 0x001a, 0x001a, 0x001a, 0x64b9, 0xb9e0, 0x001a,
	0xaee0, 0x62b9, 0xb8e0, 0x5eb9, 0xcae0, 0x63b9, 0xc8e0, 0xbce0,
	0xc6e0, 0x60b9, 0xafe0, 0xc9e0, 0xc4e0, 0x001a, 0xcbe0, 0x58b9,
	0x001a, 0x001a, 0x67b9, 0x5db9, 0x001a, 0x001a, 0xb5e0, 0x001a,
	0xbde0, 0xc1e0, 0x001a, 0xc5e0, 0x5fb9, 0xb4e0, 0xb2e0, 0xbee0,
	0x001a, 0x001a, 0x001a, 0x001a, 0xbbe0, 0xbae0, 0x001a, 0xbfe0,
	0xc2e0, 0x001a, 0xc7e0, 0x001a, 0x001a, 0x001a, 0x78e4, 0x001a,
	0xc7bb, 0xa4e4, 0x7ae4, 0xccbb, 0xd0bb, 0xade4, 0xb5e4, 0xa6e4,
	0xc8bb, 0x001a, 0xaae4, 0xb6e0, 0x001a, 0xc9bb, 0xb1e4, 0xb6e4,
	0xaee4, 0x001a, 0xb0e4, 0xb9e4, 0xb2e4, 0x7ee4, 0xa9e4, 0x001a,
	0x001a, 0xd1bb, 0x001a, 0xcdbb, 0x7ce4, 0xabe4, 0xcbbb, 0xa5e4,
	0xcabb, 0xb3e4, 0xa2e4, 0x79e4, 0xcebb, 0xb8e4, 0x001a, 0x001a,
	0x7be4, 0xafe4, 0xace4, 0xa7e4, 0x77e4, 0x76e4, 0xa1e4, 0xb4e4,
	0xcfbb, 0xb7e4, 0x7de4, 0xa3e4, 0x52be, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x5abe, 0x55be, 0xa4e8, 0xa1e8, 0x67e8, 0x50be,
	0x001a, 0xd7f9, 0x001a, 0x4fbe, 0x56be, 0x001a, 0x001a, 0x001a,
	0x65e8, 0x54be, 0x71e8, 0x63e8, 0x64e8, 0x4ebe, 0xa3e8, 0x58be,
	0x74e8, 0x79e8, 0x73e8, 0xeeeb, 0x6fe8, 0x77e8, 0x75e8, 0x68e8,
	0x62e8, 0x7de8, 0x57be, 0x7ee8, 0x001a, 0x78e8, 0x001a, 0x6de8,
	0x6be8, 0x66e8, 0x001a, 0x001a, 0x001a, 0x6ee8, 0x7be8, 0x6ae8,
	0x7ae8, 0xa2e8, 0x001a, 0x001a, 0x53be, 0x001a, 0x76e8, 0x7ce8,
	0x72e8, 0x6ce8, 0x51be, 0x001a, 0x001a, 0x001a, 0xa8e4, 0x70e8,
	0x59be, 0x69e8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf4eb,
	0xf7bf, 0xf3eb, 0xf0eb, 0x44ec, 0xfbbf, 0x001a, 0x41ec, 0xf8eb,
	0x43ec, 0xe9eb, 0xf6eb, 0x001a, 0xfdbf, 0x001a, 0xe1eb, 0x001a,
	0xdfeb, 0x42ec, 0x001a, 0x40ec, 0xfeeb, 0xedeb, 0xeceb, 0xe2eb,
	0x40c0, 0x001a, 0xe8eb, 0xf2eb, 0xfdeb, 0x43c0, 0x45ec, 0x001a,
	0xe8c1, 0x45c0, 0xfebf, 0xe6eb, 0x001a, 0xefeb, 0xdeeb, 0xe0eb,
	0xf5bf, 0x42c0, 0xfabf, 0xe7eb, 0xf7eb, 0xf1eb, 0x41c0, 0xddeb,
	0xe3c1, 0xf9eb, 0xfceb, 0xfcbf, 0x001a, 0xebeb, 0x44c0, 0xf9bf,
	0x001a, 0x001a, 0x001a, 0xf8bf, 0xf5eb, 0xfbeb, 0xf6bf, 0x001a,
	0xe4eb, 0xfaeb, 0x001a, 0x001a, 0xe5eb, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xeaeb, 0xd2ee,
	0x001a, 0xd7ee, 0xe5c1, 0xe7c1, 0xddee, 0xe1c1, 0xecee, 0xe3ee,
	0xd8ee, 0xd9ee, 0xe2ee, 0x001a, 0xeec1, 0xe1ee, 0xd1ee, 0xe0ee,
	0xd4ee, 0xedee, 0xedc1, 0xebc1, 0xd5ee, 0x001a, 0xe8ee, 0x001a,
	0xdaee, 0xe7ee, 0x001a, 0xe9ee, 0xd0ee, 0xe6c1, 0x001a, 0xeaee,
	0x001a, 0x001a, 0xdeee, 0x001a, 0xeac1, 0xdbee, 0x001a, 0x001a,
	0xecc1, 0xe4ee, 0x001a, 0x001a, 0x001a, 0xe4c1, 0xd6ee, 0xe5ee,
	0x001a, 0xdfee, 0xe3eb, 0xe6ee, 0xd3ee, 0x001a, 0xe9c1, 0x001a,
	0xebee, 0x001a, 0xe2c1, 0xceee, 0x001a, 0x001a, 0x001a, 0x001a,
	0x60f1, 0x59f1, 0xe9c2, 0x001a, 0x54f1, 0x63f1, 0x5bf1, 0xdcee,
	0x001a, 0x65f1, 0x55f1, 0x001a, 0xe8c2, 0x5ff1, 0xeac2, 0xf2c2,
	0xf0c2, 0x61f1, 0xf1c2, 0x57f1, 0x001a, 0x58f1, 0x5df1, 0x62f1,
	0x001a, 0xcdee, 0xebc2, 0x6af1, 0x67f1, 0x6bf1, 0x5ef1, 0x5af1,
	0x68f1, 0x6af3, 0x5cf1, 0x001a, 0xeec2, 0x001a, 0xedc2, 0xcfee,
	0xefc2, 0x64f1, 0x66f1, 0xecc2, 0x69f1, 0x53f1, 0x001a, 0x56f1,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x73f3, 0x001a, 0x63f3, 0xebc3, 0x71f3, 0x001a, 0x001a, 0x61f3,
	0xecc3, 0x001a, 0x6cf3, 0x001a, 0x68f3, 0xf1c3, 0x72f3, 0x62f3,
	0x65f3, 0xe9c3, 0x74f3, 0x001a, 0x6df3, 0x70f3, 0xefc3, 0xf4c3,
	0xf2c3, 0x69f3, 0x64f3, 0x001a, 0xedc3, 0xeec3, 0x60f3, 0xeac3,
	0x001a, 0xe8c3, 0xf0c3, 0x6ff3, 0xf3c3, 0x001a, 0x6bf3, 0x75f3,
	0xf5c3, 0x001a, 0x001a, 0x001a, 0x67f3, 0x001a, 0x6ef3, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf3f4, 0x42f5, 0xf5f4,
	0xfcf4, 0x66f3, 0xfaf4, 0xe9f4, 0x40f5, 0xc3c4, 0xedf4, 0xfef4,
	0xf4f4, 0x001a, 0x001a, 0xc2c4, 0x001a, 0x001a, 0x44f5, 0xf6f4,
	0x001a, 0xfbf4, 0xfdf4, 0xe7f4, 0x41f5, 0xf2f4, 0xf7f4, 0xebf4,
	0xeff4, 0x43f5, 0xf9f4, 0xe8f4, 0xecf4, 0xeef4, 0xf8f4, 0x001a,
	0xc1c4, 0xf1f4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xeaf4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xf0f4, 0x61f6, 0x66f6, 0x4fc5, 0x68f6, 0x001a, 0x49c5, 0x001a,
	0x64f6, 0x6af6, 0x4ec5, 0x4ac5, 0x001a, 0x4bc5, 0x60f6, 0x67f6,
	0x4dc5, 0x65f6, 0x4cc5, 0x5ff6, 0x63f6, 0x62f6, 0x001a, 0x5ef6,
	0x69f6, 0x001a, 0x001a, 0x001a, 0xb1c5, 0x6df7, 0x70f7, 0x6cf7,
	0x6ef7, 0x6ff7, 0x69f7, 0x6af7, 0x67f7, 0x001a, 0x001a, 0x6bf7,
	0x68f7, 0xb2c5, 0xb3c5, 0x001a, 0x001a, 0x4bf8, 0x001a, 0x4df8,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x4cf8, 0x4ef8, 0x001a,
	0xe0c5, 0x001a, 0x4af8, 0xdfc5, 0xe1c5, 0x001a, 0x001a, 0x001a,
	0xcbf8, 0xccf8, 0x44c6, 0xcaf8, 0x001a, 0x53f9, 0x52f9, 0x54f9,
	0x5fc6, 0x55f9, 0x5ec6, 0x56f9, 0x72f9, 0x75f9, 0x74f9, 0x68c6,
	0x73f9, 0x001a, 0x001a, 0x001a, 0x72c6, 0x70c6, 0x71c6, 0x77c6,
	0xc0f9, 0xc1f9, 0xbff9, 0xc9f9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xf8aa,
	0x001a, 0x001a, 0x44d8, 0x78dc, 0xa5e8, 0x76f3, 0x001a, 0x001a,
	0xf9aa, 0x001a, 0xacad, 0x7bb0, 0x001a, 0x001a, 0x45d8, 0x001a,
	0x46d8, 0xacb3, 0x001a, 0x7db6, 0x7adc, 0x79dc, 0xa3b6, 0x7cb6,
	0x7bdc, 0x7eb6, 0xa2b6, 0xa1b6, 0x7bb6, 0x001a, 0x001a, 0x001a,
	0x68b9, 0x001a, 0x001a, 0xd0e0, 0xcee0, 0x001a, 0xcfe0, 0xcde0,
	0x001a, 0xd2bb, 0x001a, 0xd5bb, 0xd7bb, 0xd6bb, 0x001a, 0x001a,
	0xd3bb, 0xd4bb, 0x001a, 0xa7e8, 0xa6e8, 0x5bbe, 0xa8e8, 0x001a,
	0xa9e8, 0x5cbe, 0x001a, 0x001a, 0x001a, 0x4dec, 0x4bec, 0xf3ee,
	0x001a, 0x49ec, 0x4aec, 0x46c0, 0x46ec, 0x4eec, 0x48ec, 0x4cec,
	0xefee, 0x001a, 0x001a, 0xf1ee, 0x001a, 0xf2ee, 0xf3c1, 0xeeee,
	0xf2c1, 0xf0ee, 0xefc1, 0xf0c1, 0xf1c1, 0x47ec, 0x001a, 0x001a,
	0xf5c2, 0x6ef1, 0x6cf1, 0x6df1, 0xf3c2, 0xf6c2, 0xf4c2, 0x001a,
	0x001a, 0x001a, 0x77f3, 0x78f3, 0xf6c3, 0x001a, 0x45f5, 0x47f5,
	0x46f5, 0xc4c4, 0x50c5, 0x6df6, 0x6cf6, 0x6bf6, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0xfaaa, 0x001a, 0xaac9, 0x001a,
	0x58ca, 0xe9a6, 0x56ca, 0x59ca, 0x57ca, 0x001a, 0x001a, 0x001a,
	0xaecb, 0x001a, 0xc1a8, 0x001a, 0xc2a8, 0xb0cb, 0xbfa8, 0xafcb,
	0xadcb, 0xc0a8, 0xbea8, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0xd8cd, 0xdbcd, 0xfdaa, 0xdacd, 0xd9cd, 0x001a, 0xfcaa,
	0xfbaa, 0x001a, 0x40ab, 0xdccd, 0xfeaa, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xc6d0, 0xaead, 0xafad, 0xb0ad, 0xc7d0, 0xc3d0,
	0xadad, 0xc4d0, 0x001a, 0xc5d0, 0xc2d0, 0x001a, 0x001a, 0x001a,
	0xa4b0, 0x001a, 0x001a, 0xa1b0, 0x45d4, 0xa2b0, 0xa5b0, 0x46d4,
	0x001a, 0x7eb0, 0x7cb0, 0x7db0, 0xa3b0, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0xadb3, 0x49d8, 0xb5b3, 0x48d8, 0x001a, 0x4bd8,
	0xb1b3, 0x4ad8, 0xabb6, 0xafb3, 0xb2b3, 0xaeb3, 0xb3b3, 0xb4b3,
	0xb0b3, 0x001a, 0x001a, 0x001a, 0x47d8, 0xa7b6, 0x7ddc, 0x001a,
	0xa3dc, 0x001a, 0x001a, 0xa2dc, 0xacb6, 0xa8b6, 0xa9b6, 0x7cdc,
	0x7edc, 0xa1dc, 0xa4b6, 0xa6b6, 0x001a, 0xaab6, 0xa5b6, 0x001a,
	0x001a, 0xd3e0, 0xd1e0, 0xd2e0, 0x6ab9, 0x6bb9, 0x001a, 0xd4e0,
	0x69b9, 0xd8bb, 0x001a, 0xdabb, 0xd9bb, 0x001a, 0xbbe4, 0x001a,
	0x001a, 0xbce4, 0xabe8, 0x001a, 0xaae8, 0x001a, 0x001a, 0x47c0,
	0x48c0, 0x4fec, 0x49c0, 0x001a, 0xf6ee, 0x001a, 0xf4ee, 0x001a,
	0xf5ee, 0xf4c1, 0x001a, 0x6ff1, 0xf7c3, 0x001a, 0x001a, 0x001a,
	0xf5c1, 0x41ab, 0x001a, 0xa6b0, 0x47d4, 0x001a, 0x001a, 0x4cd8,
	0xb6b3, 0xadb6, 0xa4dc, 0xa6dc, 0xafb6, 0xaeb6, 0xb0b6, 0xb1b6,
	0xa5dc, 0x6eb9, 0x6fb9, 0x6db9, 0xdbbb, 0x6cb9, 0xd5e0, 0x001a,
	0x001a, 0x001a, 0xdcbb, 0xace8, 0x50ec, 0x4ac0, 0xf6c1, 0x70f1,
	0x74f1, 0xf9c2, 0x71f1, 0xfac2, 0xf8c2, 0x75f1, 0xfbc2, 0x73f1,
	0x001a, 0x79f3, 0xf7c2, 0xf8c3, 0x001a, 0xcdf8, 0x001a, 0x001a,
	0x42ab, 0xb8b3, 0xb7b3, 0x001a, 0x001a, 0x001a, 0x001a, 0xb2b6,
	0xa8dc, 0xa7dc, 0xb3b6, 0x001a, 0x001a, 0xd9e0, 0x73b9, 0x70b9,
	0xd8e0, 0x72b9, 0xd6e0, 0x71b9, 0x001a, 0xd7e0, 0x001a, 0xbde4,
	0xddbb, 0x001a, 0xafe8, 0x001a, 0x5dbe, 0xade8, 0x5ebe, 0x5fbe,
	0xaee8, 0x60be, 0x001a, 0x51ec, 0x001a, 0x4ec0, 0x4bc0, 0x50c0,
	0x53ec, 0x4cc0, 0x52ec, 0x4fc0, 0x001a, 0x001a, 0x4dc0, 0x001a,
	0xf9ee, 0xfbee, 0x001a, 0x001a, 0xf7c1, 0xfaee, 0xf8c1, 0xf8ee,
	0xf7ee, 0x001a, 0x77f1, 0x76f1, 0xfcc2, 0x78f1, 0x7ef3, 0xfac3,
	0x7df3, 0x7af3, 0xf9c3, 0x7bf3, 0x7cf3, 0x001a, 0x48f5, 0x49f5,
	0xc5c4, 0x001a, 0x53c5, 0x001a, 0x001a, 0x6ef6, 0x001a, 0x001a,
	0x51c5, 0x52c5, 0x6ff6, 0x001a, 0x001a, 0xb4c5, 0xb5c5, 0x71f7,
	0x001a, 0x001a, 0x45c6, 0xcff8, 0x47c6, 0x001a, 0xcef8, 0xd0f8,
	0x46c6, 0x57f9, 0x001a, 0xadf9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x43ab, 0x001a, 0x001a, 0x001a, 0x74b9, 0x001a,
	0xbee4, 0x001a, 0xb0e8, 0x51c0, 0x52c0, 0x001a, 0x44ab, 0x001a,
	0x61be, 0xfbc3, 0xb1ad, 0x001a, 0x001a, 0x001a, 0x53c0, 0x001a,
	0xe2c5, 0xb2ad, 0x4dd8, 0x001a, 0xa9dc, 0x001a, 0xabdc, 0x001a,
	0xaadc, 0x001a, 0xdde0, 0xdae0, 0x75b9, 0x001a, 0x76b9, 0xdbe0,
	0xdce0, 0x001a, 0xc0e4, 0xc5e4, 0xdebb, 0xbfe4, 0xc1e4, 0xc8e4,
	0xc3e4, 0xc7e4, 0xc4e4, 0xc2e4, 0xc6e4, 0xdfbb, 0x001a, 0x001a,
	0xb3e8, 0x001a, 0xb1e8, 0x63be, 0x001a, 0x62be, 0xb2e8, 0x64be,
	0x001a, 0x001a, 0x001a, 0x001a, 0x56ec, 0x001a, 0x001a, 0x55ec,
	0x54c0, 0x54ec, 0xfcee, 0x001a, 0xfeee, 0x41ef, 0x40ef, 0x001a,
	0xf9c1, 0xfdee, 0xa1f1, 0xfdc2, 0x7df1, 0xa2f1, 0xfec2, 0x001a,
	0x7bf1, 0x001a, 0x7ef1, 0x7cf1, 0x79f1, 0x40c3, 0x7af1, 0x001a,
	0x001a, 0x001a, 0x001a, 0xa1f3, 0x001a, 0x001a, 0xa3f3, 0xa2f3,
	0x001a, 0x4af5, 0x001a, 0x4bf5, 0x001a, 0x001a, 0x001a, 0x70f6,
	0x001a, 0xb7c5, 0x001a, 0xb6c5, 0x4ff8, 0x50f8, 0x48c6, 0xd1f8,
	0x001a, 0x69c6, 0x001a, 0xb3ad, 0xb4b6, 0xcae4, 0xc9e4, 0xb5e8,
	0xb4e8, 0x001a, 0x001a, 0xfac1, 0x43ef, 0x42ef, 0xa5f1, 0xa3f1,
	0xa6f1, 0xa4f1, 0x001a, 0x001a, 0xfcc3, 0xa4f3, 0xa5f3, 0xa6f3,
	0x001a, 0x71f6, 0x001a, 0x72f7, 0x001a, 0xd2f8, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xb4ad, 0x001a, 0x001a,
	0x57ec, 0x44ef, 0x001a, 0xb5ad, 0x001a, 0x001a, 0xe0bb, 0x001a,
	0x58ec, 0x41c3, 0xa7f1, 0xfdc3, 0x001a, 0x4cf5, 0x4df5, 0x54c5,
	0x51f8, 0xb6ad, 0xbbb3, 0xbcb3, 0x4ed8, 0xb5b6, 0xb6b6, 0xacdc,
	0xb7b6, 0x001a, 0x7ab9, 0x001a, 0x7cb9, 0xdfe0, 0xe0e0, 0xdee0,
	0x77b9, 0x78b9, 0x7bb9, 0x79b9, 0x001a, 0x001a, 0xcbe4, 0xe1bb,
	0xe2bb, 0x001a, 0x001a, 0xbce8, 0x67be, 0xb7e8, 0xb6e8, 0x001a,
	0xbbe8, 0x65be, 0x001a, 0x001a, 0x5bc0, 0x001a, 0xb8e8, 0xbde8,
	0xbae8, 0xb9e8, 0x001a, 0x66be, 0x001a, 0x59c0, 0x001a, 0x5aec,
	0x55c0, 0x001a, 0x5bec, 0x001a, 0x001a, 0x59ec, 0x001a, 0x58c0,
	0x56c0, 0x5ac0, 0x001a, 0x57c0, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x45ef, 0x001a, 0x4aef, 0x46ef, 0x49ef, 0xfbc1, 0x001a,
	0xd4ed, 0x48ef, 0x47ef, 0x001a, 0x44c3, 0x42c3, 0x45c3, 0x43c3,
	0xa8f1, 0xa9f1, 0xaaf1, 0x46c3, 0x001a, 0x001a, 0x001a, 0xaaf3,
	0x40c4, 0xa8f3, 0x001a, 0x41c4, 0xa7f3, 0xa9f3, 0xfec3, 0x51f5,
	0x4ef5, 0x001a, 0x4ff5, 0x50f5, 0x72f6, 0x56c5, 0x001a, 0x55c5,
	0x001a, 0x74f7, 0x73f7, 0xb8c5, 0x001a, 0x001a, 0x001a, 0xe3c5,
	0x49c6, 0x60c6, 0x58f9, 0xaef9, 0xaff9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xb7ad, 0xaddc, 0x001a, 0x001a, 0xe1e0, 0xcce4, 0xcde4, 0xe3bb,
	0x001a, 0xe4bb, 0xbee8, 0x68be, 0x001a, 0x001a, 0xfcc1, 0x001a,
	0xabf1, 0x001a, 0x47c3, 0xadf3, 0x42c4, 0xacf3, 0xaef3, 0xabf3,
	0x75f6, 0x52f5, 0x53f5, 0x001a, 0xc6c4, 0x001a, 0x74f6, 0x001a,
	0x001a, 0x73f6, 0x001a, 0x75f7, 0xb0f9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xb8ad, 0x001a, 0x001a, 0x001a, 0xb9ad,
	0x001a, 0x001a, 0xa7b0, 0x48d4, 0x001a, 0x4fd8, 0x001a, 0xb8b6,
	0x001a, 0xbbb6, 0xb9b6, 0xaedc, 0x001a, 0xbdb6, 0x001a, 0xbab6,
	0x001a, 0x001a, 0xbcb6, 0x001a, 0x7eb9, 0x001a, 0xe2e0, 0x001a,
	0x001a, 0xe3e0, 0xc0e8, 0x001a, 0x7db9, 0xa1b9, 0xa2b9, 0x001a,
	0xcfe4, 0x001a, 0xcee4, 0xe5bb, 0x001a, 0xe6bb, 0x001a, 0xd0e4,
	0xbfe8, 0xe8bb, 0x69be, 0x001a, 0xe7bb, 0x001a, 0x001a, 0x001a,
	0x5cc0, 0xc1e8, 0x6bbe, 0x6abe, 0xc2e8, 0xc5e8, 0xc3e8, 0xc4e8,
	0x6cbe, 0x001a, 0x61c0, 0x5fc0, 0x001a, 0x001a, 0x5ec0, 0x5dec,
	0x001a, 0x60c0, 0x001a, 0x001a, 0x5cec, 0x4bef, 0x001a, 0x5eec,
	0x5dc0, 0x5fec, 0x4eef, 0x4cef, 0x4def, 0x52ef, 0x4bc3, 0x51ef,
	0x54ef, 0x53ef, 0x50ef, 0x4fef, 0x001a, 0xfdc1, 0x001a, 0x001a,
	0x001a, 0x001a, 0xaef1, 0x001a, 0xadf1, 0x4ac3, 0x48c3, 0x49c3,
	0x001a, 0xacf1, 0x001a, 0xb1f3, 0x001a, 0x43c4, 0x001a, 0xb0f3,
	0xaff3, 0x44c4, 0x001a, 0x58f5, 0x57f5, 0x001a, 0x55f5, 0x001a,
	0x54f5, 0xc8c4, 0xc7c4, 0x59f5, 0x76f7, 0xb9c5, 0x77f6, 0x57c5,
	0x76f6, 0x56f5, 0x001a, 0x77f7, 0xe4c5, 0x001a, 0x61c6, 0x59f9,
	0x001a, 0xb1f9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbaad, 0x50d8,
	0x55ef, 0xbbad, 0x001a, 0x001a, 0xd2e4, 0xd1e4, 0x60ec, 0x001a,
	0x001a, 0x57ef, 0x001a, 0x56ef, 0x001a, 0x4cc3, 0xb2f3, 0xb3f3,
	0xc9c4, 0x001a, 0x001a, 0xb2f9, 0xa8b0, 0xbfb6, 0xbeb6, 0xe4e0,
	0xe6e0, 0xa4b9, 0xe5e0, 0xa3b9, 0xa5b9, 0xe7e0, 0x001a, 0x001a,
	0x001a, 0xd4e4, 0xd6e4, 0xd5e4, 0x001a, 0xd8e4, 0x001a, 0x001a,
	0x001a, 0xe9bb, 0xd7e4, 0xd3e4, 0x001a, 0x001a, 0x001a, 0xd9e4,
	0x001a, 0xcce8, 0x001a, 0xcfe8, 0xd1e8, 0xc7e8, 0xcbe8, 0xc8e8,
	0x6ebe, 0x71be, 0x73be, 0xc9e8, 0xcae8, 0x72be, 0xcde8, 0xd0e8,
	0xcee8, 0x74be, 0x001a, 0x70be, 0xc6e8, 0x6dbe, 0x001a, 0x6fbe,
	0x001a, 0x001a, 0x63c0, 0x66ec, 0x64ec, 0x63ec, 0x001a, 0x69ec,
	0x001a, 0x68ec, 0x67ec, 0x001a, 0x62ec, 0x62c0, 0x61ec, 0x001a,
	0x65ec, 0x64c0, 0x001a, 0x001a, 0x5aef, 0x001a, 0x5eef, 0x5bef,
	0x5def, 0x5cef, 0x59ef, 0x5fef, 0x62ef, 0x60ef, 0x61ef, 0x40c2,
	0x001a, 0xfec1, 0x58ef, 0x63ef, 0xb3f1, 0xb6f1, 0xb8f1, 0xb7f1,
	0x001a, 0xb1f1, 0xb5f1, 0xb0f1, 0x001a, 0xb2f1, 0x4dc3, 0xaff1,
	0x001a, 0xb4f1, 0x001a, 0x001a, 0xc0f3, 0xb5f3, 0x45c4, 0x001a,
	0x001a, 0x46c4, 0xb4f3, 0xb9f3, 0xbff3, 0xb7f3, 0xbef3, 0x001a,
	0xbbf3, 0x001a, 0xbaf3, 0xbdf3, 0xb8f3, 0xb6f3, 0x001a, 0xbcf3,
	0x001a, 0x60f5, 0x5ef5, 0xcac4, 0x5df5, 0x63f5, 0x61f5, 0x001a,
	0xcbc4, 0x5cf5, 0x5af5, 0x001a, 0x5bf5, 0xcdc4, 0x5ff5, 0xccc4,
	0x62f5, 0x78f6, 0x7ef6, 0x001a, 0x001a, 0x79f6, 0x5bc5, 0xa1f6,
	0x5ac5, 0x7df6, 0x7cf6, 0x59c5, 0x7bf6, 0x58c5, 0x7af6, 0x001a,
	0x7df7, 0xa1f7, 0x7ef7, 0x001a, 0x7bf7, 0xbbc5, 0x78f7, 0x7cf7,
	0xa3f7, 0x001a, 0xa2f7, 0x79f7, 0x7af7, 0xbac5, 0x52f8, 0xe7c5,
	0x001a, 0x53f8, 0xe5c5, 0xe6c5, 0x001a, 0x001a, 0xd3f8, 0x4ac6,
	0x76f9, 0x001a, 0x6ac6, 0x001a, 0xb3f9, 0x6bc6, 0xb4f9, 0xb5f9,
	0xc3f9, 0xc2f9, 0x7ac6, 0xcdf9, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xa9b0, 0x001a, 0x001a, 0xe9e0, 0x001a, 0xe8e0, 0x001a, 0xeabb,
	0xebbb, 0xdae4, 0x001a, 0xd2e8, 0x6cec, 0x001a, 0x001a, 0x75be,
	0x65c0, 0x6aec, 0x001a, 0x6dec, 0x66c0, 0x001a, 0x64ef, 0x6bec,
	0xb9f1, 0x4ec3, 0xc1f3, 0x001a, 0x001a, 0x001a, 0x66f5, 0x64f5,
	0x001a, 0x001a, 0x65f5, 0x001a, 0x001a, 0xa2f6, 0x001a, 0x5cc5,
	0xa4f7, 0xeac5, 0xbcc5, 0xe8c5, 0xe9c5, 0xd4f8, 0x62c6, 0x001a,
	0xaab0, 0x001a, 0x001a, 0x001a, 0xbaf1, 0x001a, 0x001a, 0x49d4,
	0x001a, 0xa6b9, 0x001a, 0xdbe4, 0x001a, 0x001a, 0xecbb, 0xdce4,
	0x001a, 0x001a, 0x001a, 0xd4e8, 0xd3e8, 0x68c0, 0x76be, 0x77be,
	0x001a, 0xd7e8, 0xd6e8, 0xd5e8, 0x001a, 0x001a, 0x6eec, 0x71ec,
	0x001a, 0x70ec, 0x6fec, 0x67c0, 0x68ef, 0x66ef, 0x65ef, 0x001a,
	0x001a, 0x67ef, 0x001a, 0x4fc3, 0xbcf1, 0xbdf1, 0x50c3, 0x001a,
	0xbbf1, 0x001a, 0xc3f3, 0xc2f3, 0xc5f3, 0x47c4, 0xc4f3, 0x001a,
	0x67f5, 0x69f5, 0x68f5, 0x001a, 0x001a, 0xa3f6, 0xa6f6, 0xa4f6,
	0xa5f6, 0xa5f7, 0xbdc5, 0x001a, 0x001a, 0x001a, 0x54f8, 0x55f8,
	0x56f8, 0x001a, 0x4bc6, 0x63c6, 0xb6f9, 0xabb0, 0x001a, 0x78be,
	0x69c0, 0xbef1, 0x001a, 0xa6f7, 0x001a, 0x001a, 0xc4f9, 0x4ad4,
	0x001a, 0x7bc6, 0xacb0, 0x72ec, 0x001a, 0xbff1, 0x001a, 0xc6f3,
	0x001a, 0x001a, 0xa7f6, 0xa7f7, 0xadb0, 0x001a, 0xdde4, 0xdee4,
	0x001a, 0xedbb, 0xeebb, 0xd9e8, 0x7abe, 0x79be, 0xd8e8, 0x001a,
	0x69ef, 0x001a, 0xc0f1, 0xc2f1, 0xc1f1, 0x53c3, 0x52c3, 0x51c3,
	0x001a, 0x5ec5, 0xa8f6, 0x001a, 0x5dc5, 0xa9f7, 0xa8f7, 0x001a,
	0x4cc6, 0xd5f8, 0xbdb3, 0xeae0, 0x001a, 0x001a, 0x001a, 0xe1e4,
	0xdfe4, 0xe0e4, 0x001a, 0x001a, 0xe2e8, 0x001a, 0xdde8, 0xdae8,
	0xe1e8, 0x001a, 0x001a, 0x001a, 0xe3e8, 0x001a, 0x001a, 0x7cbe,
	0xe0e8, 0xdce8, 0x001a, 0x001a, 0xdbe8, 0xdfe8, 0xdee8, 0x7bbe,
	0x001a, 0x001a, 0x7dec, 0x78ec, 0x76ec, 0xa1ec, 0x77ec, 0x001a,
	0x73ec, 0x001a, 0x79ec, 0x001a, 0x001a, 0x74ec, 0x72ef, 0x75ec,
	0xa2ec, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x7cec, 0x6ac0, 0x7bec, 0x7aec, 0x001a, 0x7eec, 0x001a, 0x001a,
	0x001a, 0x001a, 0x6aef, 0x6def, 0x001a, 0x001a, 0x6cef, 0x001a,
	0x74ef, 0x6fef, 0x73ef, 0x001a, 0x71ef, 0x70ef, 0x6eef, 0x001a,
	0x6bef, 0x001a, 0x43c2, 0x42c2, 0x001a, 0x44c2, 0x41c2, 0x75ef,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xc8f1, 0xcbf1, 0x001a,
	0xc9f1, 0xcdf1, 0x001a, 0x001a, 0x001a, 0xcef1, 0x001a, 0xc6f1,
	0x58c3, 0xc7f1, 0x001a, 0xc5f1, 0xccf1, 0x001a, 0xc4f1, 0xc3f1,
	0x57c3, 0x55c3, 0x54c3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0xcaf1, 0xcff3, 0xd5f3, 0x4ac4, 0xd0f3,
	0x001a, 0xd3f3, 0xd7f3, 0x4bc4, 0xd2f3, 0x001a, 0xcaf3, 0x001a,
	0xc9f3, 0xd6f3, 0xcdf3, 0x001a, 0xcbf3, 0xd4f3, 0xccf3, 0x49c4,
	0x48c4, 0x001a, 0xc7f3, 0xc8f3, 0xd1f3, 0x001a, 0x001a, 0x001a,
	0xcef3, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6cf5,
	0x6ff5, 0x001a, 0x001a, 0x001a, 0x001a, 0x56c3, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x6df5, 0x73f5, 0x71f5,
	0x6bf5, 0x76f5, 0x001a, 0x6af5, 0x001a, 0xcfc4, 0x72f5, 0x001a,
	0x001a, 0x001a, 0x6ef5, 0xcec4, 0x75f5, 0x001a, 0x001a, 0x74f5,
	0x001a, 0x001a, 0x001a, 0x001a, 0xabf6, 0xaaf6, 0x001a, 0x001a,
	0x001a, 0xb1f6, 0x001a, 0xadf6, 0xb0f6, 0x60c5, 0x001a, 0x001a,
	0xaef6, 0xaff6, 0x001a, 0xa9f6, 0xacf6, 0x5fc5, 0x001a, 0x001a,
	0x001a, 0xbfc5, 0xb4f7, 0xaff7, 0xb3f7, 0x001a, 0xb6f7, 0xb2f7,
	0x001a, 0xaef7, 0x001a, 0xc1c5, 0xb1f7, 0xb5f7, 0xc0c5, 0xacf7,
	0x70f5, 0xb0f7, 0x001a, 0x001a, 0xadf7, 0x001a, 0xaaf7, 0x001a,
	0xabf7, 0xbec5, 0x5af8, 0x5cf8, 0x5ff8, 0x5bf8, 0x60f8, 0x001a,
	0x59f8, 0x001a, 0x57f8, 0x001a, 0xebc5, 0x5df8, 0xedc5, 0xecc5,
	0x58f8, 0x5ef8, 0x001a, 0x001a, 0x001a, 0x001a, 0xdaf8, 0x4dc6,
	0xdbf8, 0x001a, 0xd9f8, 0xd6f8, 0x001a, 0x001a, 0xd8f8, 0xd7f8,
	0x5af9, 0x001a, 0x001a, 0x001a, 0x001a, 0x5cf9, 0x5bf9, 0x001a,
	0x001a, 0x79f9, 0x001a, 0x78f9, 0x77f9, 0x7af9, 0x001a, 0x73c6,
	0x74c6, 0xcaf9, 0xcef9, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbeb3, 0xafdc, 0xede0,
	0x001a, 0xa7b9, 0xebe0, 0x001a, 0x001a, 0xece0, 0x001a, 0x001a,
	0x001a, 0xe2e4, 0xe3e4, 0xf1bb, 0xefbb, 0xe4e4, 0xf0bb, 0xe8e8,
	0x001a, 0xebe8, 0xe5e8, 0xece8, 0xe4e8, 0xe6e8, 0x001a, 0xe7e8,
	0xeae8, 0x001a, 0x001a, 0xa1be, 0xefe8, 0xeee8, 0x7dbe, 0xe9e8,
	0xede8, 0x7ebe, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0xacec, 0x001a, 0x6fc0, 0x001a, 0xa7ec, 0x6bc0, 0x001a, 0xa4ec,
	0xaaec, 0xadec, 0x001a, 0x70c0, 0x001a, 0xa9ec, 0xa6ec, 0xaeec,
	0xa5ec, 0x001a, 0xabec, 0x6cc0, 0x001a, 0xa3ec, 0x6dc0, 0x001a,
	0x6ec0, 0xa8ec, 0x001a, 0x001a, 0x001a, 0xa9ef, 0x7aef, 0x7bef,
	0x7eef, 0x7cef, 0x001a, 0x76ef, 0x001a, 0x001a, 0x79ef, 0xa5ef,
	0x7def, 0x001a, 0x001a, 0x45c2, 0x001a, 0xa7ef, 0xa4ef, 0x46c2,
	0xa6ef, 0x77ef, 0xa2ef, 0xa3ef, 0x001a, 0xa1ef, 0x001a, 0x001a,
	0x001a, 0x001a, 0xd2f1, 0xd4f1, 0xd7f1, 0x001a, 0x001a, 0xd1f1,
	0x001a, 0x59c3, 0xd9f1, 0xd0f1, 0xdaf1, 0x001a, 0xd6f1, 0xd8f1,
	0xdcf1, 0xd5f1, 0xddf1, 0xd3f1, 0xcff1, 0x5ac3, 0x001a, 0xdbf1,
	0x5bc3, 0x4dc4, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x78ef,
	0xf1f3, 0xe8f3, 0x4fc4, 0xe4f3, 0x50c4, 0x001a, 0x001a, 0xedf3,
	0xe7f3, 0xddf3, 0x4ec4, 0xeaf3, 0xe5f3, 0xe6f3, 0x001a, 0xd8f3,
	0xdff3, 0xeef3, 0x001a, 0xebf3, 0x001a, 0xe3f3, 0x001a, 0xeff3,
	0xdef3, 0xd9f3, 0xecf3, 0x001a, 0xdbf3, 0xe9f3, 0xe0f3, 0xf0f3,
	0xdcf3, 0x4cc4, 0xdaf3, 0xe1f3, 0xe2f3, 0x001a, 0x001a, 0x001a,
	0x7df5, 0x001a, 0x7bf5, 0x001a, 0xa2f5, 0x001a, 0xaef5, 0xa5f5,
	0x7cf5, 0x78f5, 0xa7f5, 0x7ef5, 0xa3f5, 0x7af5, 0xaaf5, 0x77f5,
	0xa1f5, 0xa6f5, 0xa8f5, 0xabf5, 0x79f5, 0x001a, 0xaff5, 0xb0f5,
	0xa9f5, 0xadf5, 0xa4f5, 0x001a, 0xc1f6, 0xc4f6, 0x001a, 0x61c5,
	0x001a, 0xc3f6, 0xc8f6, 0xc6f6, 0x62c5, 0xbdf6, 0xb3f6, 0xb2f6,
	0x64c5, 0xbff6, 0xc0f6, 0xbcf6, 0xb4f6, 0x001a, 0xb9f6, 0xacf5,
	0x001a, 0xb5f6, 0x63c5, 0xbbf6, 0x001a, 0xbaf6, 0x001a, 0xb6f6,
	0xc2f6, 0x001a, 0xb7f6, 0xbbf7, 0xc5f6, 0xc7f6, 0xbef6, 0xb8f6,
	0xbcf7, 0xbef7, 0xb8f7, 0xc2c5, 0x001a, 0xc5f7, 0xc3f7, 0xc3c5,
	0xc2f7, 0xc1f7, 0xbaf7, 0xb7f7, 0xbdf7, 0xc6f7, 0xb9f7, 0xbff7,
	0x001a, 0x69f8, 0x6ef8, 0x64f8, 0x67f8, 0xeec5, 0x6bf8, 0x001a,
	0x72f8, 0xc0f7, 0x001a, 0x65f8, 0x6ff8, 0x73f8, 0x6af8, 0x63f8,
	0x6df8, 0x001a, 0x6cf8, 0x71f8, 0x70f8, 0xc4f7, 0x68f8, 0x62f8,
	0x66f8, 0x4ec6, 0x4fc6, 0x61f8, 0x001a, 0xe6f8, 0xddf8, 0xe5f8,
	0xe2f8, 0xe3f8, 0xdcf8, 0xdff8, 0xe7f8, 0xe1f8, 0xe0f8, 0xdef8,
	0x001a, 0xe4f8, 0x001a, 0x5df9, 0x001a, 0x5ef9, 0x001a, 0x60f9,
	0x5ff9, 0x62f9, 0x61f9, 0x7cf9, 0x7bf9, 0xb7f9, 0x001a, 0xb8f9,
	0x001a, 0xc5f9, 0x78c6, 0x7cc6, 0x001a, 0xcff9, 0x7dc6, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0xbfb3, 0x001a, 0x001a,
	0x001a, 0xd0c4, 0xc9f6, 0x001a, 0x50c6, 0x51c6, 0x001a, 0xc0b3,
	0xeee0, 0x001a, 0xa8b9, 0xf0e8, 0x001a, 0x001a, 0xb0ec, 0xb1ec,
	0xafec, 0xabef, 0xaaef, 0x47c2, 0xdff1, 0xacef, 0xdef1, 0x001a,
	0x001a, 0xf3f3, 0x51c4, 0x53c4, 0xf2f3, 0x001a, 0x001a, 0x52c4,
	0x001a, 0xb1f5, 0xb3f5, 0xb2f5, 0xcaf6, 0x65c5, 0x001a, 0xefc5,
	0xe8f8, 0x63f9, 0x001a, 0x001a, 0xd2f9, 0xc1b3, 0x001a, 0xe5e4,
	0x001a, 0xa2be, 0x001a, 0x001a, 0x001a, 0xb3ec, 0xb2ec, 0x001a,
	0xadef, 0x001a, 0x001a, 0x001a, 0x54c4, 0xd1c4, 0xc7f7, 0xcbf9,
	0x001a, 0x001a, 0x001a, 0xc2b3, 0xf2bb, 0x001a, 0xa3be, 0x001a,
	0xf4f3, 0x001a, 0x74f8, 0xc0b6, 0x001a, 0x001a, 0x001a, 0x001a,
	0xaeef, 0x001a, 0x001a, 0x001a, 0x64c6, 0xc1b6, 0xa4be, 0x48c2,
	0x75f8, 0xc2b6, 0x001a, 0xf1e8, 0x72c0, 0xb4ec, 0xb5ec, 0x001a,
	0x71c0, 0x001a, 0xafef, 0x4cc2, 0x4ac2, 0x4bc2, 0x49c2, 0xe0f1,
	0x5cc3, 0x001a, 0x001a, 0x001a, 0xb5f5, 0xb4f5, 0xb7f5, 0xb6f5,
	0xd2c4, 0x001a, 0x001a, 0xcbf6, 0x001a, 0xcdf6, 0xccf6, 0x66c5,
	0xc8f7, 0x001a, 0x76f8, 0x77f8, 0xf0c5, 0x64f9, 0x7df9, 0x75c6,
	0x001a, 0xb0dc, 0xb6ec, 0xb0ef, 0xf5f3, 0xefe0, 0x001a, 0xb1ef,
	0xe2f1, 0xe1f1, 0x001a, 0x001a, 0x001a, 0x001a, 0x78f8, 0x52c6,
	0x001a, 0x65f9, 0x7ef9, 0x001a, 0x001a, 0x001a, 0xa9b9, 0xf2e8,
	0xf3e8, 0x001a, 0xb7ec, 0xaab9, 0x001a, 0x5dc3, 0xe3f1, 0x001a,
	0xcff6, 0x67c5, 0xd0f6, 0xcef6, 0x79f8, 0x001a, 0xe9f8, 0x001a,
	0xabb9, 0x001a, 0xb4ef, 0xb3ef, 0xb2ef, 0xe4f1, 0x001a, 0x001a,
	0xe8f1, 0xe7f1, 0xe6f1, 0xe5f1, 0x5ec3, 0xf6f3, 0xb9f5, 0xd3c4,
	0xb8f5, 0xd1f6, 0xcbf7, 0xcaf7, 0xc4c5, 0xc9f7, 0x7cf8, 0x7bf8,
	0x7af8, 0x001a, 0x001a, 0xf3bb, 0x001a, 0xb8ec, 0x4dc2, 0x001a,
	0xf7f3, 0xf8f3, 0xccf7, 0x7df8, 0x001a, 0x001a, 0xeaf8, 0x66f9,
	0xb9f9, 0xd4f9, 0xf4bb, 0x4ec2, 0xe9f1, 0xf9f3, 0xd2f6, 0x7ef8,
	0x001a, 0x001a, 0xa6be, 0x001a, 0xb5ef, 0xeaf1, 0xfaf3, 0xfbf3,
	0xfcf3, 0xbef5, 0x001a, 0xbaf5, 0x68c5, 0xbdf5, 0xbcf5, 0xd4c4,
	0xbbf5, 0xd6c4, 0x001a, 0xd5c4, 0xd4f6, 0xd3f6, 0x69c5, 0x6ac5,
	0x001a, 0x001a, 0xc6c5, 0xcdf7, 0xc5c5, 0x001a, 0xa3f8, 0xa4f8,
	0xa2f8, 0xa1f8, 0x54c6, 0x001a, 0xebf8, 0xecf8, 0xedf8, 0x53c6,
	0x67f9, 0x6af9, 0x69f9, 0x68f9, 0x001a, 0x001a, 0xd3f9, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x73c0, 0x001a, 0x001a,
	0x65c3, 0xbff5, 0xd5f6, 0x001a, 0xc7c5, 0xcef7, 0x001a, 0x001a,
	0xd5f9, 0x001a, 0x001a, 0x001a, 0x74c0, 0x001a, 0x001a, 0x001a,
	0xb6ef, 0x001a, 0xcff7, 0x001a, 0xa1f9, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_950_unicode_to_byte_stream_base_0xfa00[ 64 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x4ac9, 0xfcdd, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};

const uint16_t libuna_codepage_windows_950_unicode_to_byte_stream_base_0xfe00[ 512 ] = {
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x4aa1, 0x57a1, 0x001a, 0x59a1, 0x5ba1, 0x5fa1, 0x60a1, 0x63a1,
	0x64a1, 0x67a1, 0x68a1, 0x6ba1, 0x6ca1, 0x6fa1, 0x70a1, 0x73a1,
	0x74a1, 0x77a1, 0x78a1, 0x7ba1, 0x7ca1, 0x001a, 0x001a, 0x001a,
	0x001a, 0xc6a1, 0xc7a1, 0xcaa1, 0xcba1, 0xc8a1, 0xc9a1, 0x5ca1,
	0x4da1, 0x4ea1, 0x4fa1, 0x001a, 0x51a1, 0x52a1, 0x53a1, 0x54a1,
	0x001a, 0x7da1, 0x7ea1, 0xa1a1, 0xa2a1, 0xa3a1, 0xa4a1, 0xcca1,
	0xcda1, 0xcea1, 0xdea1, 0xdfa1, 0xe0a1, 0xe1a1, 0xe2a1, 0x001a,
	0x42a2, 0x4ca2, 0x4da2, 0x4ea2, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x49a1, 0x001a, 0xada1, 0x43a2, 0x48a2, 0xaea1, 0x001a,
	0x5da1, 0x5ea1, 0xafa1, 0xcfa1, 0x41a1, 0xd0a1, 0x44a1, 0xfea1,
	0xafa2, 0xb0a2, 0xb1a2, 0xb2a2, 0xb3a2, 0xb4a2, 0xb5a2, 0xb6a2,
	0xb7a2, 0xb8a2, 0x47a1, 0x46a1, 0xd5a1, 0xd7a1, 0xd6a1, 0x48a1,
	0x49a2, 0xcfa2, 0xd0a2, 0xd1a2, 0xd2a2, 0xd3a2, 0xd4a2, 0xd5a2,
	0xd6a2, 0xd7a2, 0xd8a2, 0xd9a2, 0xdaa2, 0xdba2, 0xdca2, 0xdda2,
	0xdea2, 0xdfa2, 0xe0a2, 0xe1a2, 0xe2a2, 0xe3a2, 0xe4a2, 0xe5a2,
	0xe6a2, 0xe7a2, 0xe8a2, 0x001a, 0x40a2, 0x001a, 0x001a, 0xc4a1,
	0x001a, 0xe9a2, 0xeaa2, 0xeba2, 0xeca2, 0xeda2, 0xeea2, 0xefa2,
	0xf0a2, 0xf1a2, 0xf2a2, 0xf3a2, 0xf4a2, 0xf5a2, 0xf6a2, 0xf7a2,
	0xf8a2, 0xf9a2, 0xfaa2, 0xfba2, 0xfca2, 0xfda2, 0xfea2, 0x40a3,
	0x41a3, 0x42a3, 0x43a3, 0x61a1, 0x55a1, 0x62a1, 0xe3a1, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x46a2, 0x47a2, 0x001a, 0xc3a1, 0x001a, 0x44a2, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a,
	0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a, 0x001a
};


/* Determines the size of a byte stream character from an Unicode character
 * Adds the size to the byte stream character size value
 * Returns 1 if successful or -1 on error
 */
LIBUNA_INLINE \
int libuna_codepage_windows_950_unicode_character_size_to_byte_stream(
     libuna_unicode_character_t unicode_character,
     size_t *byte_stream_character_size,
     libcerror_error_t **error )
{
	static char *function      = "libuna_codepage_windows_950_unicode_character_size_to_byte_stream";
	uint16_t byte_stream_value = 0x001a;

	if( byte_stream_character_size == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream character size.",
		 function );

		return( -1 );
	}
	if( unicode_character < 0x80 )
	{
		byte_stream_value = (uint16_t) unicode_character;
	}
	else if( ( unicode_character >= 0x0080 )
	      && ( unicode_character < 0x0100 ) )
	{
		unicode_character -= 0x0080;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x0080[ unicode_character ];
	}
	else if( ( unicode_character >= 0x02c0 )
	      && ( unicode_character < 0x0400 ) )
	{
		unicode_character -= 0x02c0;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x02c0[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2000 )
	      && ( unicode_character < 0x2300 ) )
	{
		unicode_character -= 0x2000;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x2000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2500 )
	      && ( unicode_character < 0x2680 ) )
	{
		unicode_character -= 0x2500;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x2500[ unicode_character ];
	}
	else if( ( unicode_character >= 0x3000 )
	      && ( unicode_character < 0x3400 ) )
	{
		unicode_character -= 0x3000;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x3000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x4e00 )
	      && ( unicode_character < 0x9fc0 ) )
	{
		unicode_character -= 0x4e00;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x4e00[ unicode_character ];
	}
	else if( ( unicode_character >= 0xfa00 )
	      && ( unicode_character < 0xfa40 ) )
	{
		unicode_character -= 0xfa00;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0xfa00[ unicode_character ];
	}
	else if( ( unicode_character >= 0xfe00 )
	      && ( unicode_character < 0x10000 ) )
	{
		unicode_character -= 0xfe00;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0xfe00[ unicode_character ];
	}
	byte_stream_value >>= 8;

	if( byte_stream_value != 0 )
	{
		*byte_stream_character_size += 2;
	}
	else
	{
		*byte_stream_character_size += 1;
	}
	return( 1 );
}

/* Copies a Unicode character from a byte stream
 * Returns 1 if successful or -1 on error
 */
LIBUNA_INLINE \
int libuna_codepage_windows_950_copy_from_byte_stream(
     libuna_unicode_character_t *unicode_character,
     const uint8_t *byte_stream,
     size_t byte_stream_size,
     size_t *byte_stream_index,
     libcerror_error_t **error )
{
	static char *function         = "libuna_codepage_windows_950_copy_from_byte_stream";
	uint8_t additional_character  = 0;
	uint8_t byte_stream_character = 0;

	if( unicode_character == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid Unicode character.",
		 function );

		return( -1 );
	}
	if( byte_stream == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream.",
		 function );

		return( -1 );
	}
	if( byte_stream_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid byte stream size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( byte_stream_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream index.",
		 function );

		return( -1 );
	}
	if( *byte_stream_index >= byte_stream_size )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: byte stream too small.",
		 function );

		return( -1 );
	}
	byte_stream_character = byte_stream[ *byte_stream_index ];

	if( byte_stream_character < 0x80 )
	{
		*unicode_character = byte_stream_character;
	}
	else if( ( *byte_stream_index + 1 ) <= byte_stream_size )
	{
		*byte_stream_index += 1;

		additional_character = byte_stream[ *byte_stream_index ];

		if( ( byte_stream_character >= 0xa1 )
		 && ( byte_stream_character <= 0xa2 ) )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x7f ) )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0xa1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa140[ additional_character ];
						break;

					case 0xa2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa240[ additional_character ];
						break;
				}
			}
			else if( ( additional_character >= 0xa1 )
			      && ( additional_character < 0xff ) )
			{
				additional_character -= 0xa1;

				switch( byte_stream_character )
				{
					case 0xa1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa1a1[ additional_character ];
						break;

					case 0xa2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa2a1[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xa3 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x7f ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa340[ additional_character ];
			}
			else if( ( additional_character >= 0xa1 )
			      && ( additional_character < 0xc0 ) )
			{
				additional_character -= 0xa1;

				*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa3a1[ additional_character ];
			}
			else if( additional_character == 0xe1 )
			{
				*unicode_character = 0x20ac;
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0xa4 )
		      && ( byte_stream_character <= 0xc5 ) )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x7f ) )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0xa4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa440[ additional_character ];
						break;

					case 0xa5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa540[ additional_character ];
						break;

					case 0xa6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa640[ additional_character ];
						break;

					case 0xa7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa740[ additional_character ];
						break;

					case 0xa8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa840[ additional_character ];
						break;

					case 0xa9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa940[ additional_character ];
						break;

					case 0xaa:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaa40[ additional_character ];
						break;

					case 0xab:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xab40[ additional_character ];
						break;

					case 0xac:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xac40[ additional_character ];
						break;

					case 0xad:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xad40[ additional_character ];
						break;

					case 0xae:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xae40[ additional_character ];
						break;

					case 0xaf:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaf40[ additional_character ];
						break;

					case 0xb0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb040[ additional_character ];
						break;

					case 0xb1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb140[ additional_character ];
						break;

					case 0xb2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb240[ additional_character ];
						break;

					case 0xb3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb340[ additional_character ];
						break;

					case 0xb4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb440[ additional_character ];
						break;

					case 0xb5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb540[ additional_character ];
						break;

					case 0xb6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb640[ additional_character ];
						break;

					case 0xb7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb740[ additional_character ];
						break;

					case 0xb8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb840[ additional_character ];
						break;

					case 0xb9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb940[ additional_character ];
						break;

					case 0xba:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xba40[ additional_character ];
						break;

					case 0xbb:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbb40[ additional_character ];
						break;

					case 0xbc:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbc40[ additional_character ];
						break;

					case 0xbd:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbd40[ additional_character ];
						break;

					case 0xbe:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbe40[ additional_character ];
						break;

					case 0xbf:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbf40[ additional_character ];
						break;

					case 0xc0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc040[ additional_character ];
						break;

					case 0xc1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc140[ additional_character ];
						break;

					case 0xc2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc240[ additional_character ];
						break;

					case 0xc3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc340[ additional_character ];
						break;

					case 0xc4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc440[ additional_character ];
						break;

					case 0xc5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc540[ additional_character ];
						break;
				}
			}
			else if( ( additional_character >= 0xa1 )
			      && ( additional_character < 0xff ) )
			{
				additional_character -= 0xa1;

				switch( byte_stream_character )
				{
					case 0xa4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa4a1[ additional_character ];
						break;

					case 0xa5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa5a1[ additional_character ];
						break;

					case 0xa6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa6a1[ additional_character ];
						break;

					case 0xa7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa7a1[ additional_character ];
						break;

					case 0xa8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa8a1[ additional_character ];
						break;

					case 0xa9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xa9a1[ additional_character ];
						break;

					case 0xaa:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaaa1[ additional_character ];
						break;

					case 0xab:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaba1[ additional_character ];
						break;

					case 0xac:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaca1[ additional_character ];
						break;

					case 0xad:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xada1[ additional_character ];
						break;

					case 0xae:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xaea1[ additional_character ];
						break;

					case 0xaf:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xafa1[ additional_character ];
						break;

					case 0xb0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb0a1[ additional_character ];
						break;

					case 0xb1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb1a1[ additional_character ];
						break;

					case 0xb2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb2a1[ additional_character ];
						break;

					case 0xb3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb3a1[ additional_character ];
						break;

					case 0xb4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb4a1[ additional_character ];
						break;

					case 0xb5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb5a1[ additional_character ];
						break;

					case 0xb6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb6a1[ additional_character ];
						break;

					case 0xb7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb7a1[ additional_character ];
						break;

					case 0xb8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb8a1[ additional_character ];
						break;

					case 0xb9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xb9a1[ additional_character ];
						break;

					case 0xba:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbaa1[ additional_character ];
						break;

					case 0xbb:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbba1[ additional_character ];
						break;

					case 0xbc:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbca1[ additional_character ];
						break;

					case 0xbd:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbda1[ additional_character ];
						break;

					case 0xbe:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbea1[ additional_character ];
						break;

					case 0xbf:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xbfa1[ additional_character ];
						break;

					case 0xc0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc0a1[ additional_character ];
						break;

					case 0xc1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc1a1[ additional_character ];
						break;

					case 0xc2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc2a1[ additional_character ];
						break;

					case 0xc3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc3a1[ additional_character ];
						break;

					case 0xc4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc4a1[ additional_character ];
						break;

					case 0xc5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc5a1[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( byte_stream_character == 0xc6 )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x7f ) )
			{
				additional_character -= 0x40;

				*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc640[ additional_character ];
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else if( ( byte_stream_character >= 0xc9 )
		      && ( byte_stream_character <= 0xf9 ) )
		{
			if( ( additional_character >= 0x40 )
			 && ( additional_character < 0x7f ) )
			{
				additional_character -= 0x40;

				switch( byte_stream_character )
				{
					case 0xc9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc940[ additional_character ];
						break;

					case 0xca:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xca40[ additional_character ];
						break;

					case 0xcb:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcb40[ additional_character ];
						break;

					case 0xcc:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcc40[ additional_character ];
						break;

					case 0xcd:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcd40[ additional_character ];
						break;

					case 0xce:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xce40[ additional_character ];
						break;

					case 0xcf:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcf40[ additional_character ];
						break;

					case 0xd0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd040[ additional_character ];
						break;

					case 0xd1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd140[ additional_character ];
						break;

					case 0xd2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd240[ additional_character ];
						break;

					case 0xd3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd340[ additional_character ];
						break;

					case 0xd4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd440[ additional_character ];
						break;

					case 0xd5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd540[ additional_character ];
						break;

					case 0xd6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd640[ additional_character ];
						break;

					case 0xd7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd740[ additional_character ];
						break;

					case 0xd8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd840[ additional_character ];
						break;

					case 0xd9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd940[ additional_character ];
						break;

					case 0xda:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xda40[ additional_character ];
						break;

					case 0xdb:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdb40[ additional_character ];
						break;

					case 0xdc:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdc40[ additional_character ];
						break;

					case 0xdd:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdd40[ additional_character ];
						break;

					case 0xde:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xde40[ additional_character ];
						break;

					case 0xdf:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdf40[ additional_character ];
						break;

					case 0xe0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe040[ additional_character ];
						break;

					case 0xe1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe140[ additional_character ];
						break;

					case 0xe2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe240[ additional_character ];
						break;

					case 0xe3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe340[ additional_character ];
						break;

					case 0xe4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe440[ additional_character ];
						break;

					case 0xe5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe540[ additional_character ];
						break;

					case 0xe6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe640[ additional_character ];
						break;

					case 0xe7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe740[ additional_character ];
						break;

					case 0xe8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe840[ additional_character ];
						break;

					case 0xe9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe940[ additional_character ];
						break;

					case 0xea:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xea40[ additional_character ];
						break;

					case 0xeb:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeb40[ additional_character ];
						break;

					case 0xec:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xec40[ additional_character ];
						break;

					case 0xed:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xed40[ additional_character ];
						break;

					case 0xee:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xee40[ additional_character ];
						break;

					case 0xef:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xef40[ additional_character ];
						break;

					case 0xf0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf040[ additional_character ];
						break;

					case 0xf1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf140[ additional_character ];
						break;

					case 0xf2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf240[ additional_character ];
						break;

					case 0xf3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf340[ additional_character ];
						break;

					case 0xf4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf440[ additional_character ];
						break;

					case 0xf5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf540[ additional_character ];
						break;

					case 0xf6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf640[ additional_character ];
						break;

					case 0xf7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf740[ additional_character ];
						break;

					case 0xf8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf840[ additional_character ];
						break;

					case 0xf9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf940[ additional_character ];
						break;
				}
			}
			else if( ( additional_character >= 0xa1 )
			      && ( additional_character < 0xff ) )
			{
				additional_character -= 0xa1;

				switch( byte_stream_character )
				{
					case 0xc9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xc9a1[ additional_character ];
						break;

					case 0xca:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcaa1[ additional_character ];
						break;

					case 0xcb:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcba1[ additional_character ];
						break;

					case 0xcc:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcca1[ additional_character ];
						break;

					case 0xcd:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcda1[ additional_character ];
						break;

					case 0xce:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcea1[ additional_character ];
						break;

					case 0xcf:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xcfa1[ additional_character ];
						break;

					case 0xd0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd0a1[ additional_character ];
						break;

					case 0xd1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd1a1[ additional_character ];
						break;

					case 0xd2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd2a1[ additional_character ];
						break;

					case 0xd3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd3a1[ additional_character ];
						break;

					case 0xd4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd4a1[ additional_character ];
						break;

					case 0xd5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd5a1[ additional_character ];
						break;

					case 0xd6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd6a1[ additional_character ];
						break;

					case 0xd7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd7a1[ additional_character ];
						break;

					case 0xd8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd8a1[ additional_character ];
						break;

					case 0xd9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xd9a1[ additional_character ];
						break;

					case 0xda:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdaa1[ additional_character ];
						break;

					case 0xdb:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdba1[ additional_character ];
						break;

					case 0xdc:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdca1[ additional_character ];
						break;

					case 0xdd:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdda1[ additional_character ];
						break;

					case 0xde:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdea1[ additional_character ];
						break;

					case 0xdf:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xdfa1[ additional_character ];
						break;

					case 0xe0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe0a1[ additional_character ];
						break;

					case 0xe1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe1a1[ additional_character ];
						break;

					case 0xe2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe2a1[ additional_character ];
						break;

					case 0xe3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe3a1[ additional_character ];
						break;

					case 0xe4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe4a1[ additional_character ];
						break;

					case 0xe5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe5a1[ additional_character ];
						break;

					case 0xe6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe6a1[ additional_character ];
						break;

					case 0xe7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe7a1[ additional_character ];
						break;

					case 0xe8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe8a1[ additional_character ];
						break;

					case 0xe9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xe9a1[ additional_character ];
						break;

					case 0xea:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeaa1[ additional_character ];
						break;

					case 0xeb:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeba1[ additional_character ];
						break;

					case 0xec:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeca1[ additional_character ];
						break;

					case 0xed:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeda1[ additional_character ];
						break;

					case 0xee:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xeea1[ additional_character ];
						break;

					case 0xef:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xefa1[ additional_character ];
						break;

					case 0xf0:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf0a1[ additional_character ];
						break;

					case 0xf1:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf1a1[ additional_character ];
						break;

					case 0xf2:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf2a1[ additional_character ];
						break;

					case 0xf3:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf3a1[ additional_character ];
						break;

					case 0xf4:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf4a1[ additional_character ];
						break;

					case 0xf5:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf5a1[ additional_character ];
						break;

					case 0xf6:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf6a1[ additional_character ];
						break;

					case 0xf7:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf7a1[ additional_character ];
						break;

					case 0xf8:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf8a1[ additional_character ];
						break;

					case 0xf9:
						*unicode_character = libuna_codepage_windows_950_byte_stream_to_unicode_base_0xf9a1[ additional_character ];
						break;
				}
			}
			else
			{
				*unicode_character = 0xfffd;
			}
		}
		else
		{
			*unicode_character = 0xfffd;
		}
	}
	else
	{
		*unicode_character = 0xfffd;
	}
	*byte_stream_index += 1;

	return( 1 );
}

/* Copies a Unicode character to a byte stream
 * Returns 1 if successful or -1 on error
 */
LIBUNA_INLINE \
int libuna_codepage_windows_950_copy_to_byte_stream(
     libuna_unicode_character_t unicode_character,
     uint8_t *byte_stream,
     size_t byte_stream_size,
     size_t *byte_stream_index,
     libcerror_error_t **error )
{
	static char *function      = "libuna_codepage_windows_950_copy_to_byte_stream";
	uint16_t byte_stream_value = 0x001a;

	if( byte_stream == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream.",
		 function );

		return( -1 );
	}
	if( byte_stream_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid byte stream size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( byte_stream_index == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid byte stream index.",
		 function );

		return( -1 );
	}
	if( *byte_stream_index >= byte_stream_size )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: byte stream too small.",
		 function );

		return( -1 );
	}
	if( unicode_character < 0x80 )
	{
		byte_stream_value = (uint16_t) unicode_character;
	}
	else if( ( unicode_character >= 0x0080 )
	      && ( unicode_character < 0x0100 ) )
	{
		unicode_character -= 0x0080;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x0080[ unicode_character ];
	}
	else if( ( unicode_character >= 0x02c0 )
	      && ( unicode_character < 0x0400 ) )
	{
		unicode_character -= 0x02c0;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x02c0[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2000 )
	      && ( unicode_character < 0x2300 ) )
	{
		unicode_character -= 0x2000;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x2000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x2500 )
	      && ( unicode_character < 0x2680 ) )
	{
		unicode_character -= 0x2500;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x2500[ unicode_character ];
	}
	else if( ( unicode_character >= 0x3000 )
	      && ( unicode_character < 0x3400 ) )
	{
		unicode_character -= 0x3000;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x3000[ unicode_character ];
	}
	else if( ( unicode_character >= 0x4e00 )
	      && ( unicode_character < 0x9fc0 ) )
	{
		unicode_character -= 0x4e00;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0x4e00[ unicode_character ];
	}
	else if( ( unicode_character >= 0xfa00 )
	      && ( unicode_character < 0xfa40 ) )
	{
		unicode_character -= 0xfa00;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0xfa00[ unicode_character ];
	}
	else if( ( unicode_character >= 0xfe00 )
	      && ( unicode_character < 0x10000 ) )
	{
		unicode_character -= 0xfe00;
		byte_stream_value  = libuna_codepage_windows_950_unicode_to_byte_stream_base_0xfe00[ unicode_character ];
	}
	byte_stream[ *byte_stream_index ] = (uint8_t) ( byte_stream_value & 0x00ff );

	byte_stream_value >>= 8;

	if( byte_stream_value != 0 )
	{
		*byte_stream_index += 1;

		byte_stream[ *byte_stream_index ] = (uint8_t) ( byte_stream_value & 0x00ff );
	}
	*byte_stream_index += 1;

	return( 1 );
}

