/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "igenericfilter.h"


#include "icommoneventobservers.h"
#include "ierror.h"
#include "iviewmodule.h"
#include "iviewsubject.h"

#include <vtkDataSetAttributes.h>
#include <vtkDemandDrivenPipeline.h>
#include <vtkFloatArray.h>
#include <vtkImageData.h>
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>


//
//  Generic Source class
//
template<class Source, class OutputType>
iGenericSource<Source,OutputType>::iGenericSource(iViewSubject *vo, bool usesLimits) : iViewSubjectComponent(vo), mUsesLimits(usesLimits)
{
	mExecuteMethod = 0;
	
	if(vo->IsCreatingMainPipeline())
	{
		this->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetAbortRenderEventObserver());
	}
	else
	{
		this->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetSlaveAbortRenderEventObserver());
	}
}


template<class Source, class OutputType>
iGenericSource<Source,OutputType>::~iGenericSource()
{
}


template<class Source, class OutputType>
void iGenericSource<Source,OutputType>::ExecuteData(vtkDataObject * )
{
	OutputType *output = this->GetOutput();
	IERROR_ASSERT(output);

	mExecuteMethod = 0;
	this->ProduceOutput();

	if(this->IsOptimizedForMemory()) output->Squeeze();
}


template<class Source, class OutputType>
void iGenericSource<Source,OutputType>::SaveRequest(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo)
{
	mSavedRequest = request;
	mSavedInputVector = inInfo;
	mSavedOutputVector = outInfo;
	mReturnCode = 1;
}


template<class Source, class OutputType>
int iGenericSource<Source,OutputType>::RequestData(vtkInformation* request, vtkInformationVector** inInfo, vtkInformationVector* outInfo)
{
	OutputType *output = this->GetOutput();
	IERROR_ASSERT(output);

	mExecuteMethod = 1;
	this->SaveRequest(request,inInfo,outInfo);
	this->ProduceOutput();

	if(this->IsOptimizedForMemory()) output->Squeeze();

	return mReturnCode;
}


template<class Source, class OutputType>
void iGenericSource<Source,OutputType>::ExecuteParent()
{
	switch(mExecuteMethod)
	{
	case 0:
		{
			this->Source::ExecuteData(this->GetOutput());
			break;
		}
	case 1:
		{
            mReturnCode = this->Source::RequestData(mSavedRequest,mSavedInputVector,mSavedOutputVector);
			break;
		}
	default:
		{
			IERROR_LOW("Invalid ExecuteMethod.");
		}
	}
}


template<class Source, class OutputType>
void iGenericSource<Source,OutputType>::SyncWithData(const iDataSyncRequest &)
{
	if(mUsesLimits) this->Modified();
}


template<class Source, class OutputType>
int iGenericSource<Source,OutputType>::GetNumberOfOutputs()
{
	return this->Source::GetNumberOfOutputPorts();
}


template<class Source, class OutputType>
float iGenericSource<Source,OutputType>::GetMemorySize()
{
	int j;
	float s = 0.0f;

	for(j=0; j<this->GetNumberOfOutputs(); j++)
	{
		s += this->GetOutput(j)->GetActualMemorySize();
	}

	return s;
}


//
//  Generic Filter class
//
template<class Filter, class InputType, class OutputType>
iGenericFilter<Filter,InputType,OutputType>::iGenericFilter(iViewSubject *vo, int numInputs, bool ownsOutput, bool usesLimits) : iViewSubjectComponent(vo), mOwnsOutput(ownsOutput), mUsesLimits(usesLimits)
{
	mExecuteMethod = 0;

	if(vo->IsCreatingMainPipeline())
	{
		this->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetAbortRenderEventObserver());
	}
	else
	{
		this->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetSlaveAbortRenderEventObserver());
	}

	if(numInputs > 0)  //  not all filters default to 1 input, some have 0 inputs
	{
		this->SetNumberOfInputConnections(0,numInputs);
	}
}


template<class Filter, class InputType, class OutputType>
iGenericFilter<Filter,InputType,OutputType>::~iGenericFilter()
{
}


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::ExecuteData(vtkDataObject * )
{
	InputType *input = InputType::SafeDownCast(this->GetInput());
	OutputType *output = this->GetOutput();

	IERROR_ASSERT(output);
	if(input == 0)
	{
		output->Initialize();
		return;
	}

	mExecuteMethod = 0;
#ifdef I_DEBUG
	iConsole::PrintDebugMessage(iString("Executing filter ")+Filter::GetClassName()+" by method 0.");
#endif
	this->ProduceOutput();

	if(this->IsOptimizedForMemory()) output->Squeeze();

	this->VerifyResults();
}

	
template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::SaveRequest(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo)
{
	mSavedRequest = request;
	mSavedInputVector = inInfo;
	mSavedOutputVector = outInfo;
	mReturnCode = 1;
}


template<class Filter, class InputType, class OutputType>
int iGenericFilter<Filter,InputType,OutputType>::ProcessRequest(vtkInformation *request, vtkInformationVector **inInfo, vtkInformationVector *outInfo)
{
	//
	//  This replaces ExecuteInformation call
	//
	if(request->Has(vtkDemandDrivenPipeline::REQUEST_INFORMATION()))
	{
		InputType *input = InputType::SafeDownCast(this->GetInput());
		OutputType *output = this->GetOutput();

		IERROR_ASSERT(output);
		if(input != 0)
		{
			this->SaveRequest(request,inInfo,outInfo);
			this->InitExecution();
		}

		if(this->IsA("vtkSource"))
		{
			//
			//  PipelineInformation may contain wrong values of Origin and Spacing. These values get written into the Origin and Spacing fields
			//  of vtkImageData in vtkSource::ProcessRequest. We need to fix this bug by putting correct values into PipelineInformation.
			//
			int i;
			vtkInformation *info;
			for(i=0; i<inInfo[0]->GetNumberOfInformationObjects(); i++)
			{
				info = inInfo[0]->GetInformationObject(i);
				if(info != 0)
				{
					vtkImageData *data = vtkImageData::SafeDownCast(info->Get(vtkDataObject::DATA_OBJECT()));
					if(data != 0)
					{
						double v[3];
						vtkInformation *info = data->GetPipelineInformation();
						if(info != 0)
						{
							if(info->Has(vtkDataObject::ORIGIN()))
							{
								data->GetOrigin(v);
								info->Set(vtkDataObject::ORIGIN(),v,3);
							}
							if(info->Has(vtkDataObject::SPACING()))
							{
								data->GetSpacing(v);
								info->Set(vtkDataObject::SPACING(),v,3);
							}
						}
					}
				}
			}
		}
	}

	return this->Filter::ProcessRequest(request,inInfo,outInfo);
}


template<class Filter, class InputType, class OutputType>
int iGenericFilter<Filter,InputType,OutputType>::RequestData(vtkInformation* request, vtkInformationVector** inInfo, vtkInformationVector* outInfo)
{
	InputType *input = InputType::SafeDownCast(this->GetInput());
	OutputType *output = this->GetOutput();

	IERROR_ASSERT(output);
	if(input == 0)
	{
		output->Initialize();
		return 1;
	}

	mExecuteMethod = 1;
#ifdef I_DEBUG
	iConsole::PrintDebugMessage(iString("Executing filter ")+Filter::GetClassName()+" by method 1.");
#endif
	this->SaveRequest(request,inInfo,outInfo);
	this->ProduceOutput();

	if(this->IsOptimizedForMemory()) output->Squeeze();

	this->VerifyResults();

	return mReturnCode;
}


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::ExecuteParent()
{
	switch(mExecuteMethod)
	{
	case 0:
		{
			this->Filter::ExecuteData(this->GetOutput());
			break;
		}
	case 1:
		{
			mReturnCode = this->Filter::RequestData(mSavedRequest,mSavedInputVector,mSavedOutputVector);
			break;
		}
	default:
		{
			IERROR_LOW("Invalid ExecuteMethod.");
		}
	}
}


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::SyncWithData(const iDataSyncRequest &)
{
	if(mUsesLimits) this->Modified();
}


template<class Filter, class InputType, class OutputType>
int iGenericFilter<Filter,InputType,OutputType>::GetNumberOfOutputs()
{
	return this->Filter::GetNumberOfOutputPorts();
}


template<class Filter, class InputType, class OutputType>
float iGenericFilter<Filter,InputType,OutputType>::GetMemorySize()
{
	int j;
	float s = 0.0f;

	if(mOwnsOutput)
	{
		for(j=0; j<this->GetNumberOfOutputs(); j++)
		{
			s += this->GetOutput(j)->GetActualMemorySize();
		}
	}

	return s;
}


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::InitExecution()
{
	//
	//  Default implementation is to do nothing
	//
}


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::VerifyResults()
{
	//
	//  Default implementation is to do nothing
	//
}


template<class Filter, class InputType, class OutputType>
bool iGenericFilter<Filter,InputType,OutputType>::ScalarsInit(vtkDataSetAttributes *data, vtkIdType nSizeOut, int nDimOut)
{
	if(data==0 || data->GetScalars()==0)
	{
		wScalarArrIn = wScalarArrOut = 0;
		wScalarPtrIn = wScalarPtrOut = 0;
		wScalarDimIn = wScalarDimOut = 0;
		return true;
	}

	wScalarArrIn = vtkFloatArray::SafeDownCast(data->GetScalars()); 
	if(wScalarArrIn == 0)
    {
		vtkErrorMacro("Input scalars are not float.");
		return false;
    }
	wScalarDimIn = wScalarArrIn->GetNumberOfComponents();
	wScalarPtrIn = wScalarArrIn->GetPointer(0);

	if(nSizeOut < 1) nSizeOut = wScalarArrIn->GetNumberOfTuples();
	if(nDimOut < 1) nDimOut = wScalarDimIn;

	wScalarArrOut = vtkFloatArray::New();
	if(wScalarArrOut == 0)
    {
		vtkErrorMacro("Not enough memory to create output scalars.");
		return false;
    }
	wScalarArrOut->SetNumberOfComponents(nDimOut);
	wScalarArrOut->SetNumberOfTuples(nSizeOut);

	wScalarDimOut = nDimOut;
	wScalarPtrOut = wScalarArrOut->GetPointer(0);
	if(wScalarPtrOut == 0)
    {
		vtkErrorMacro("Not enough memory to create output scalars.");
		return false;
    }
	return true;
}


template<class Filter, class InputType, class OutputType>
void iGenericFilter<Filter,InputType,OutputType>::ScalarsDone(vtkDataSetAttributes *data)
{
	if(data!=0 && wScalarArrOut!=0)
	{
		data->SetScalars(wScalarArrOut);
		wScalarArrOut->Delete();
	}
}


template<class Filter>
void iGenericPolyDataToPolyDataFilter<Filter>::VerifyResults()
{
	vtkPolyData *input = vtkPolyData::SafeDownCast(this->GetInput());
	vtkPolyData *output = this->GetOutput();

	if(input!=0 && output!=0 && input->GetPoints()!=0 && output->GetPoints()!=0 && (input->GetPoints()->GetDataType()!=output->GetPoints()->GetDataType()))
	{
#ifdef I_DEBUG
		int it = input->GetPoints()->GetDataType();
		int ot = output->GetPoints()->GetDataType();
#endif
		vtkErrorMacro("Incompatible point types of input and output");
	}
}

