/* $LastChangedDate: 2023-04-15 00:54:24 +0200 (Sat, 15 Apr 2023) $ */
/*
 fim.h : Fim main header file

 (c) 2007-2023 Michele Martone

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/
#ifndef FIM_FIM_H
#define FIM_FIM_H

/*
 * config.h is generated by autoconf and defines some symbols used by Fim
 * */
#ifdef HAVE_CONFIG_H
/*
 * when we get rid of string.h we will use <config.h>, too
#include <config.h>
*/
#include "../config.h"
#endif /* HAVE_CONFIG_H */

#define FIM_USE_OLDCXX ( __cplusplus<201103L ) /* */
#define FIM_USE_CXX11 ( __cplusplus>=201103L ) /* */
#define FIM_USE_CXX14 ( __cplusplus> 201402L ) /* */
#define FIM_USE_CXX17 ( __cplusplus> 201707L ) /* */

#ifdef FIM_WITH_DEBUG 
#else
#define NDEBUG 1	/* turns off assert() functionality */
#endif

/*
 *	This is the main fim program header file.
 *	Established 20061225 ( yes, i know what do you think about me now )
 * */
#include <cstdlib>	/* <cstdlib> standard C library definitions			*/
#include <cstdio>	/* <cstdlib> standard C buffered I/O 				*/

#include <cstring>	/* <string.h> standard C string manipulation functions		*/
#include <iostream>	/* C++ I/O stream manipulation ( cout,cin, ... )		*/
#include <algorithm>	/* STL (Standard Template Library) algorithm library		*/
#include <map>		/* STL (Standard Template Library) associative array template	*/
#include <stack>	/* STL (Standard Template Library) stack structure template	*/

#include <vector>	/* STL (Standard Template Library) vector structure template	*/
#include <queue>	/* STL (Standard Template Library) queue adaptor		*/
#include <utility>	/* STL (Standard Template Library) ?				*/
#include <list> 	/* STL (Standard Template Library) list structure template	*/
#include <set> 		/* STL (Standard Template Library) set structure template 	*/
#include <cassert>	/* <assert.h> C assertions ( IEEE Std 1003.1-2001 aka Posix ) 	*/

#define FIM_USE_CXX_REGEX FIM_USE_CXX11 && HAVE_REGEX

#if FIM_USE_CXX_REGEX
# include <regex>		/*	C++11 STL regular expressions: regex_search regex_replace */
#else /* FIM_USE_CXX_REGEX */
#if HAVE_REGEX_H
#ifndef USE_GNU_REGEX
# include <regex.h>		/*	the Posix (GNU implementation,not functionality) readline library	*/
#else
# include "regex.h"		/*	the GNU (implementation and functionality) readline library	*/
#endif /* USE_GNU_REGEX */
#endif /* HAVE_REGEX_H */
#endif /* FIM_USE_CXX_REGEX */

#ifdef HAVE_CLIMITS
 #include <climits>
 /* From:
  * ISO C++ 14882: 18.2.2  Implementation properties: C library 
  * we get some more stuff than limits.h. */
#else /* HAVE_CLIMITS */
 #ifdef HAVE_LIMITS_H
 /* According to IEEE Std 1003.1-2001, 
  * this should define _POSIX_PATH_MAX
  * */
  #include <limits.h>
 #endif /* HAVE_LIMITS_H */
#endif /* HAVE_LIMITS_H */
#ifndef _POSIX_PATH_MAX
 /* I don't know in what case could arise this situation 
  * (no limits.h nor climits : i guess a badly configured system),
  * but this would be the fix :*/
 #define _POSIX_PATH_MAX 4096
#endif /* _POSIX_PATH_MAX */

#include <sys/types.h>		/* */
#if HAVE_SYS_STAT_H
#include <sys/stat.h>		/* stat */
#endif /* HAVE_SYS_STAT_H */
#include <unistd.h>		/* */
#ifdef FIM_WITH_PTHREADS
#include <pthread.h>		/* */
#endif /* FIM_WITH_PTHREADS */

#if 0
#ifdef HAVE_UNIX_H
# include <unix.h>
# ifndef _POSIX_PATH_MAX
#  include <posix1_lim.h>
#  ifndef _POSIX_PATH_MAX
/*  a strict limit to pathname length */
#   define _POSIX_PATH_MAX 4096
#  endif
# endif
#else
#endif
#endif
#define FIM_PATH_MAX _POSIX_PATH_MAX

#if HAVE_TERMIOS_H
#include <termios.h>	/* general terminal interface (Posix)			*/
#endif /* HAVE_TERMIOS_H */
#include <fcntl.h>	/* file descriptor manipulation interface (Posix)	*/
#include <ctime>	/* time related functionality (Posix)			*/
#include "common.h"	/* misc FIM stuff					*/
/* #define FIM_USE_GPM 0 */
#ifdef FIM_USE_GPM
#include <gpm.h>	/* mouse events						*/
#endif /* FIM_USE_GPM */
/*#include <unistd.h>*/ /* standard Posix symbolic constants and types		*/
/*#include <sys/types.h>*/
#include <cmath>
 
#define FIM_LINE_CERR std::cerr << "fatal error" << __FILE__ << ":" << __LINE__ << "\n";
#define FIM_LINE_COUT std::cout << "in " <<__func__ << " # " << __FILE__ << ":" << __LINE__ << "\n";
#define FIM_INT_PCNT(P,L)        (static_cast<fim_int>(    ceilf(static_cast<float>((P)*(L))/100.0)))
#define FIM_FLT_PCNT(P,L)        (static_cast<fim_float_t>(ceilf(static_cast<float>((P)*(L))/100.0)))
#define FIM_INT_PCNT_OF_100(P,L)  static_cast<fim_int>(        FIM_MIN(FIM_INT_PCNT(P,L),L) )
#define FIM_FLT_PCNT_OF_100(P,L) FIM_MIN(FIM_FLT_PCNT(P,L),L)
#define FIM_INT_PCNT_SAFE(P,L)   (FIM_INT_PCNT(FIM_MIN(100,FIM_MAX(P,0)),(L)))
#define FIM_INT_DET_PCNT(P,T)    (((static_cast<double>(P))/(static_cast<double>(T)))*100.0) 
#define FIM_DELIMIT_TO_100(X) FIM_MIN(FIM_MAX(X,0),100)
#define FIM_INT_DET_PX(P,T,X)    (((static_cast<double>(P))/(static_cast<double>(T)))*(1.0*X)) 
#define FIM_DELIMIT_TO_X(Y,X) FIM_MIN(FIM_MAX(Y,0),X)
#define FIM_XOR(X,Y) ((X)^(Y))

#include "fim_types.h"
#include "fim_limits.h"
#include "fim_wrappers.h"
#include "Arg.h"

#define FIM_WANT_REMEMBER_LAST_FILE_LOADER 1	/* TODO: shall use this feature to set a i:file_loader attribute; FIXME: move this down */
#define FIM_WANT_BENCHMARKS	1	/* FIXME: move this down */
#define FIM_ALLOW_LOADER_VERBOSITY 1
#define FIM_WANT_FAT_BROWSER 1
#define FIM_WANT_BDI 1
#define FIM_WANT_TII 1
#define FIM_WANT_MIPMAPS 1
#define FIM_WANT_GOTOLAST 1
#define FIM_WANT_LASTGOTODIRECTION (FIM_WANT_GOTOLAST && 1)
#define FIM_WANT_PIC_CMTS 1
#define FIM_WANT_PIC_LISTUNMARK 1
#define FIM_OSW_LOAD_IMG_DSC_FILE_VID_COMMENT_OI_VAL 2
#define FIM_WANT_PIC_LVDN 1 /* pictures load variables description namespace (FIXME: experimental) */
#define FIM_WANT_PIC_LBFL 1 /* limit browser file list (see FIM_FLT_LIMIT) */
#define FIM_WANT_PIC_RCMT 1 /* remember last comment with special "#!fim:=" syntax */
#define FIM_WANT_PIC_CCMT 1 && FIM_WANT_PIC_LVDN /* contextual last comment with special "#!fim:^=" and "#!fim:$=" syntax */
#define FIM_WANT_DESC_VEXP ( FIM_WANT_PIC_LVDN && 1 ) /* description variable expansion */
#define FIM_WANT_VAR_GOTO 1 && FIM_WANT_PIC_CCMT /* goto based on variables values */
#define FIM_WANT_CMDLINE_KEYPRESS 1 /* commandline option to execute certain keypresses at startup time */
#define FIM_WANT_FLIST_STAT ( HAVE_SYS_STAT_H && 1 ) /* stat() info in the file list */
#define FIM_WANT_SORT_BY_STAT_INFO FIM_WANT_FLIST_STAT /* using stat() info for sorting (FIXME: shall cache this info at first sort or on -R) */
#define FIM_WANT_MAGIC_FIMDESC 1
#if FIM_USE_CXX11
#define FIM_WANT_CROP 1 /* experimental, unfinished */
#else /* FIM_USE_CXX11 */
#define FIM_WANT_CROP 0 /* experimental, unfinished */
#endif /* FIM_USE_CXX11 */
#define FIM_WANT_MOUSE_CROP FIM_WANT_CROP /* experimental, unfinished */
#define FIM_EXPERIMENTAL_IMG_NMSPC 1
#define FIM_EXPERIMENTAL_VAR_EXPANDOS 1
#define FIM_FONT_MAGNIFY_FACTOR FIM_WANT_FONT_MAGNIFY_FACTOR /* Framebuffer font magnifying factor, >=1 (EXPERIMENTAL). Useful with very high resolutions and small fonts. */
#define FIM_FONT_MAGNIFY_FACTOR_MIN 1  /* Framebuffer font magnifying factor, minimal value. */
#define FIM_FONT_MAGNIFY_FACTOR_MAX 16 /* Framebuffer font magnifying factor, maximal value. */
#define FIM_FONT_MAGNIFY_FACTOR_DEFAULT ( FIM_WANT_FONT_MAGNIFY_FACTOR > 0 ? FIM_WANT_FONT_MAGNIFY_FACTOR : ( FIM_WANT_FONT_MAGNIFY_FACTOR < 0 ? -FIM_WANT_FONT_MAGNIFY_FACTOR : 1 ) ) /* Framebuffer font magnifying factor, default value. */
#if defined(__MINGW32__ ) || defined(__MINGW64__)
#define FIM_USING_MINGW 1
#else
#define FIM_USING_MINGW 0
#endif
#define FIM_WANT_BACKGROUND_LOAD ( FIM_USE_CXX11 && !FIM_USING_MINGW ) /* FIXME: this is experimental */
#define FIM_WANT_PIC_CMTS_RELOAD ( FIM_USE_CXX11 && FIM_WANT_PIC_CMTS )
#define FIM_WANT_IMAGE_LOAD_TIME 1
#define FIM_RECURSIVE_HIDDEN_DIRS_SKIP_CHECK 1
#define FIM_WANT_PROGRESS_RECURSIVE_LOADING 0 /* print a line when loading recursively directories in background */
#define FIM_WANT_BIND_ALL 0 /* bind all keys to a specific command; FIXME: activate this */
#define FIM_WANT_RESIZE_HUGE_AFTER_LOAD 1 /* resize huge images at load time */
#define FIM_WANT_GOTO_DIR 1
#define FIM_WANT_FBI_INNER_DIAGNOSTICS false
#define FIM_WANT_FBI_FBDEV_DIAGNOSTICS false
#define FIM_IMG_NAKED_PTRS (!FIM_USE_CXX11) /* from C++11 onwards, use smart pointers over Image* */
#define FIM_WANT_OBSOLETE 0 /* for functionality to be removed or overhauled  */

/* shadow list of high-res images to jump on demand */
#ifndef FIM_EXPERIMENTAL_SHADOW_DIRS
#if HAVE_FILESYSTEM
#if FIM_USE_CXX17
#define FIM_EXPERIMENTAL_SHADOW_DIRS 2 /* 1: at load time  2: at view time */
#endif /* FIM_USE_CXX17 */
#endif /* HAVE_FILESYSTEM */
#endif /* FIM_EXPERIMENTAL_SHADOW_DIRS */
#ifndef FIM_EXPERIMENTAL_SHADOW_DIRS
#if FIM_USE_CXX11
#if FIM_READ_DIRS
#define FIM_EXPERIMENTAL_SHADOW_DIRS 1 /* 1: at load time  2: at view time */
#endif /* FIM_READ_DIRS */
#endif /* FIM_USE_CXX11 */
#endif /* FIM_EXPERIMENTAL_SHADOW_DIRS */

#define FIM_EXPERIMENTAL_FONT_CMD HAVE_DIRENT_H /*  */
#define FIM_WANT_HELP_ARGS 1 /*  */
#define FIM_WANT_VARIABLE_RESOLUTION_MOUSE_SCROLL 1 /* the larger the image, the more scroll steps  */
#if FIM_USE_CXX11
#define FIM_NOEXCEPT noexcept
#define FIM_CONSTEXPR constexpr
#define FIM_OVERRIDE override
#define FIM_FINAL final
#define FIM_NORETURN [[noreturn]]
#else /* FIM_USE_CXX11 */
#define FIM_NOEXCEPT
#define FIM_CONSTEXPR const
#define FIM_OVERRIDE 
#define FIM_FINAL 
#define FIM_NORETURN 
#endif /* FIM_USE_CXX11 */
#if FIM_USE_CXX14
#define FIM_DEPRECATED(MSG) [[deprecated(MSG)]]
#else /* FIM_USE_CXX14 */
#define FIM_DEPRECATED(MSG) 
#endif /* FIM_USE_CXX14 */
#if FIM_USE_CXX17
#define FIM_FALLTHROUGH [[fallthrough]];
#else /* FIM_USE_CXX17 */
#define FIM_FALLTHROUGH 
#endif /* FIM_USE_CXX17 */

#ifdef FIM_AUTOCMDS
#define FIM_AUTOCMD_EXEC autocmd_exec
#define FIM_AUTOCMD_EXEC_PRE(AC,CF) { const fim_fn_t _c = CF; autocmd_exec(AC,_c);
#define FIM_AUTOCMD_EXEC_POST(AC)   autocmd_exec(AC,_c); }
#else /* FIM_AUTOCMDS */
#define FIM_AUTOCMD_EXEC
#endif /* FIM_AUTOCMDS */

	fim::string fim_help_opt(const char*qs, const char dl);
namespace fim
{
	#if FIM_USE_CXX11
	using fim_fn_t = string; /* file name */
	using fim_ds_t = string; /* file description */
	using sym_keys_t = std::map<fim::string,fim_key_t >;	//symbol->code
	using fim_cxr = fim::string; // command execution result
	#else /* FIM_USE_CXX11 */
	typedef string fim_fn_t; /* file name */
	typedef string fim_ds_t; /* file description */
	typedef std::map<fim::string,fim_key_t > sym_keys_t;	//symbol->code
	typedef fim::string fim_cxr; // command execution result
	#endif /* FIM_USE_CXX11 */

	void status(const fim_char_t *desc, const fim_char_t *info);

class ViewportState
{
#if FIM_USE_CXX11
	public:
	fim_off_t	steps_{},hsteps_{},vsteps_{},top_{},left_{},panned_{} ;	/* viewport state variables */
#else /* FIM_USE_CXX11 */
	public:
	fim_off_t	steps_,hsteps_,vsteps_,top_,left_,panned_ ;	/* viewport state variables */
	ViewportState(void)	:steps_(0) ,hsteps_(0) ,vsteps_(0) ,top_(0) ,left_(0) ,panned_(0x0) {}
#endif /* FIM_USE_CXX11 */
};

	class Browser;
	class FontServer;
	class FramebufferDevice;
	class CommandConsole;
	class Command;
	class Image;
	class Var;
	class MiniConsole;
#ifdef FIM_WINDOWS
	class FimWindow;
#endif /* FIM_WINDOWS */
	class Viewport;
	class fim_stream;
}

namespace rl
{
	void initialize_readline (fim_bool_t with_no_display_device);
}

/* using GCC builtins (__GNUC__ should be defined by gcc) */

#ifndef FIM_IS_SLOWER_THAN_FBI
	#define FIM_UNLIKELY(expr) __builtin_expect(!!(expr),0)
	#define FIM_LIKELY(expr)   __builtin_expect(!!(expr),1)

#if FIM_USE_CXX17 && defined(__cpp_aligned_new)
	#define FIM_ALIGNED __attribute__((aligned (64)))
#else /* FIM_USE_CXX17 */
	#define FIM_ALIGNED
#endif /* FIM_USE_CXX17 */
#else /* FIM_IS_SLOWER_THAN_FBI */
	#define FIM_UNLIKELY(expr)  (expr)
	#define FIM_LIKELY(expr)   (expr)
	#define FIM_ALIGNED
#endif /* FIM_IS_SLOWER_THAN_FBI */

//#define FIM_FBI_PRINTF( ... ) fprintf(stderr, __VA_ARGS__ )
/* " warning: anonymous variadic macros were introduced in C99" (here and elsewhere) */
#define FIM_WANT_FBI_PRINTF 0
#define FIM_FBI_PRINTF( ... ) {}
#define FIM_VERB_PRINTF printf
#define FIM_VERB_COUT std::cout

namespace fim{
enum fim_image_source_t FIM_ENUM_BASE { FIM_E_FILE=-11, FIM_E_STDIN=-22};	/* these */
enum fim_xflags_t /*FIM_ENUM_BASE*/ { FIM_X_NULL=0,FIM_X_HISTORY=1,FIM_X_QUIET=2,FIM_X_NOAUTOCMD=4};	/* TODO: may introduce 'sandbox' like flags, here (for instance, for no-system/pipe-interacting executions) */
typedef std::pair<fim::string,fim_image_source_t > 	   cache_key_t;	//the current cache key
enum FimDocRefMode FIM_ENUM_BASE { Txt, Man, DefRefMode=Txt};
}

#define FIM_STDIN_IMAGE_NAME "<STDIN>"
/* should belong to a namespace different from the file name space, and possibly figuring alphabetically as the first one */

/*
 * Fim Symbols
 * */
#define FIM_SYM_CONSOLE_KEY	':'
#define FIM_SYM_CONSOLE_KEY_STR	":"
#define FIM_SYM_FW_SEARCH_KEY	'/'
#define FIM_SYM_FW_SEARCH_KEY_STR	"/"
#define FIM_SYM_BW_SEARCH_KEY	'?'
#define FIM_SYM_BW_SEARCH_KEY_STR	"?"
#define FIM_SYM_NULL_KEY	0x0 /* null key value for fim_key_t */
#define FIM_SYM_NAMESPACE_SEP	':'
#define FIM_SYM_DEVOPTS_SEP	'='
#define FIM_SYM_DEVOPTS_SEP_STR	"="
#define FIM_SYM_NULL_NAMESPACE_CHAR	'\0'
#define FIM_SYM_NAMESPACE_BROWSER_CHAR	'b'
#define FIM_SYM_NAMESPACE_IMAGE_CHAR	'i'
#define FIM_SYM_NAMESPACE_IMAGE_ALL_STR	"i:*"
#define FIM_SYM_NAMESPACE_GLOBAL_CHAR	'g'
#define FIM_SYM_NAMESPACE_WINDOW_CHAR	'w'
#define FIM_SYM_NAMESPACE_VIEWPORT_CHAR	'v'
/* #define FIM_SYM_NAMESPACE_PREFIXES	"'i:', 'b:', 'w:', 'v:', 'g:'" */
/* #define FIM_SYM_NAMESPACE_PREFIXES_DSC	"current image, browser, window, viewport, global" */
/* #define FIM_SYM_NAMESPACE_REGEX	"^[givbw]:" */
#define FIM_SYM_NAMESPACE_PREFIXES	"'i:'" /* removed g: because is not mandatory */
#define FIM_SYM_NAMESPACE_PREFIXES_DSC	"current image, global"
#define FIM_SYM_NAMESPACE_REGEX	"^[gi]:"
#define FIM_SYM_PROMPT_CHAR	':'
#define FIM_SYM_PROMPT_SLASH	'/'
#define FIM_SYM_PROMPT_NUL	'\0'
#define FIM_SYM_CHAR_NUL	'\0'
#define FIM_SYM_CHAR_ENDL	'\n'
#define FIM_SYM_ESC		0x1B
#define FIM_SYM_ENTER		0x0D
#define FIM_SYM_FLIPCHAR	'F'
#define FIM_SYM_MIRRCHAR	'M'
#define FIM_SYM_TYPE_FLOAT	'f'
#define FIM_SYM_TYPE_INT	'i'
#define FIM_SYM_STRING_CONCAT	'.'
#define FIM_SYM_DOT_CHAR	'.'
#define FIM_SYM_SEMICOLON	';'
#define FIM_SYM_SEMICOLON_STRING	";"
#define FIM_SYM_ENDL	"\n"
#define FIM_SYM_PIC_CMT_CHAR    '#'
#define FIM_SYM_UNKNOWN_CHAR    ' ' /* replacement for non-printable characters */
#define FIM_SYM_UNKNOWN_STRING  " " /* *FIM_SYM_UNKNOWN_STRING == FIM_SYM_UNKNOWN_CHAR */
#define FIM_SYM_SORT_SZ	'z'
#define FIM_SYM_SORT_MD	'm'
#define FIM_SYM_SORT_BN	'b'
#define FIM_SYM_SORT_BV	'v'
#define FIM_SYM_SORT_FN 'f'
#define FIM_SYM_CMD_SLSL FIM_VID_RE_SEARCH_OPTS "=\"f\""

/*
 * External programs used by fim.
 */
#define FIM_EPR_FIG2DEV		"fig2dev"
#define FIM_EPR_XCFTOPNM	"xcftopnm"
#define FIM_EPR_DIA		"dia"
#define FIM_EPR_INKSCAPE	"inkscape"
#define FIM_EPR_CONVERT		"convert"
#define FIM_EPR_ZCAT		"zcat"

/*
 * Some Fim (internal) error codes.
 */
#define FIM_ERR_NO_ERROR	0
#define FIM_ERR_GENERIC	-1
#define FIM_ERR_UNSUPPORTED	-2
#define FIM_ERR_BUFFER_FULL	-1024 // FIXME: -2 seems in use
#define FIM_ERR_UNSUPPORTED_DEVICE	-4
#define FIM_ERR_BAD_PARAMS	-8
#define FIM_ERR_NO_SUCH_OPTION	-16
#define FIM_ERR_OOPS		42

#define FIM_ERR_TO_PERR(E)	(((unsigned char)(E)))

/*
 * Some Fim (program) error codes.
 */
#define FIM_PERR_NO_ERROR	FIM_ERR_TO_PERR(FIM_ERR_NO_ERROR)
#define FIM_PERR_GENERIC	FIM_ERR_TO_PERR(FIM_ERR_GENERIC)
#define FIM_PERR_UNSUPPORTED	FIM_ERR_TO_PERR(FIM_ERR_UNSUPPORTED)
#define FIM_PERR_NO_SUCH_OPTION	FIM_ERR_TO_PERR(FIM_ERR_NO_SUCH_OPTION)
//#define FIM_PERR_BUFFER_FULL	FIM_ERR_TO_PERR(FIM_ERR_BUFFER_FULL)
#define FIM_PERR_UNSUPPORTED_DEVICE	FIM_ERR_TO_PERR(FIM_ERR_UNSUPPORTED_DEVICE)
#define FIM_PERR_BAD_PARAMS	 FIM_ERR_TO_PERR(FIM_ERR_BAD_PARAMS)
#define FIM_PERR_OOPS	FIM_ERR_TO_PERR(FIM_ERR_OOPS)

/*
 * Some Fim error messages.
 */
#define FIM_EMSG_NO_SCRIPTING	"sorry, no scripting available!\n"
#define FIM_EMSG_NO_READ_STDIN_IMAGE	"sorry, the reading of images from stdin was disabled at compile time\n"
#define FIM_EMSG_CACHING_STDIN	"problems caching standard input image!\n"
#define FIM_EMSG_OUT_OF_MEM	"out of memory\n"
#define FIM_EMSG_UNFINISHED	"sorry, feature incomplete!\n"
#define FIM_EMSG_ITE		"internal type error\n"
#define FIM_EMSG_ICE		"internal consistency error\n"
#define FIM_EMSG_DSMO		"Warning: Asking to ignore already specified stdin reading option!\n"

/* Command related error messages */
#define FIM_EMSG_NOMARKUNMARK	"sorry, mark/unmark functionality was opted out."

/*
 * Some environment variables used by Fim.
 */
#define FIM_ENV_DISPLAY "DISPLAY"
#define FIM_ENV_SSH "SSH_TTY"
#define FIM_ENV_FRAMEBUFFER "FRAMEBUFFER"
#define FIM_ENV_FBGAMMA "FBGAMMA"
#define FIM_ENV_FBFONT "FBFONT"

/*
 * Fim Option (long) Switches
 * */
#define FIM_OSW_OUTPUT_DEVICE	"output-device"
#define FIM_OSW_BINARY	"binary"
#define FIM_OSW_TEXT	"as-text"
#define FIM_OSW_EXECUTE_COMMANDS	"execute-commands"
#define FIM_OSW_EXECUTE_COMMANDS_EARLY	"execute-commands-early"
#define FIM_OSW_EXECUTE_SCRIPT	"execute-script"
#define FIM_OSW_FINAL_COMMANDS	"final-commands"
#define FIM_OSW_SCRIPT_FROM_STDIN	"script-from-stdin"
#define FIM_OSW_READ_FROM_STDIN	"read-from-stdin"
#define FIM_OSW_IMAGE_FROM_STDIN	"image-from-stdin"
#define FIM_OSW_DUMP_SCRIPTOUT "write-scriptout"
#define FIM_OSW_CHARS_PRESS              "chars-press"
#define FIM_OSW_KEYSYM_PRESS             "keysym-press"
#define FIM_OSW_LOAD_IMG_DSC_FILE	"load-image-descriptions-file"
#define FIM_OSW_IMG_DSC_FILE_SEPC	"image-descriptions-file-separator"
#define FIM_OSW_SORT_FSIZE		"sort-fsize"
#define FIM_OSW_SORT_MTIME		"sort-mtime"
#define FIM_OSW_BGREC			"background-recursive"
#define FIM_OSW_RECURSIVE		"recursive"
#define FIM_OSW_NOEXTPIPLOA		"no-pipe-load"
#define FIM_OSW_NO_STAT_PUSH		"no-stat-push"
#define FIM_OSW_NO_ETC_FIMRC		"no-etc-fimrc-file"
#define FIM_OSW_ETC_FIMRC		"etc-fimrc"

/*
 * Fim Constants
 * */
#define FIM_CNS_FIM	 "FIM - Fbi IMproved"
#define FIM_CNS_LCY	 "2023" /* latest copyright year */
//#define FIM_CNS_FIM_TXT	 "FIM.TXT"
#define FIM_CNS_BUGS_FILE	 "BUGS"
#define FIM_CNS_TERM_VAR	 "TERM"
#define FIM_CNS_HOME_VAR	 "HOME"
#define FIM_CNS_HIST_FILENAME	 ".fim_history"
#define FIM_CNS_HIST_COMPLETE_FILENAME	 "~/.fim_history"
//#ifdef FIM_PATHS_IN_MAN
#define FIM_CNS_SYS_RC_FILEPATH	 "@sysconfdir@/fimrc"
#define FIM_CNS_DOC_PATH	 "@docdir@"
//#else /*  FIM_PATHS_IN_MAN */
//#define FIM_CNS_SYS_RC_FILEPATH	 "/etc/fimrc"
//#define FIM_CNS_DOC_PATH	 "/usr/share/doc/fim"
//#endif /* FIM_PATHS_IN_MAN */
#define FIM_CNS_USR_RC_FILEPATH	 ".fimrc"
#define FIM_CNS_USR_RC_COMPLETE_FILEPATH	 "~/.fimrc"
#define FIM_CNS_EXAMPLE_FILENAME	 "file.jpg"
#define FIM_CNS_SCALEFACTOR	 1.322f
#define FIM_CNS_SCALEFACTOR_ONE 1.0f
#define FIM_CNS_SCALE_DEFAULT FIM_CNS_SCALEFACTOR_ONE
#define FIM_CNS_ANGLE_ONE 1.0f
#define FIM_CNS_ANGLE_ZERO 0.0f
#define FIM_CNS_ANGLE_DEFAULT FIM_CNS_ANGLE_ZERO
#define FIM_CNS_GAMMA_DEFAULT 1.0
#define FIM_CNS_GAMMA_DEFAULT_STR FIM_XSTRINGIFY(FIM_CNS_GAMMA_DEFAULT)
#define FIM_CNS_SCALEFACTOR_MULTIPLIER 1.1f
#define FIM_CNS_SCALEFACTOR_DELTA 0.1f
#define FIM_CNS_SCALEFACTOR_ZERO 0.0f
//#define FIM_CNS_STEPS_DEFAULT	 	50
#define FIM_CNS_SCROLL_DEFAULT	 	"90%"
#define FIM_CNS_STEPS_DEFAULT_N	 	50
#define FIM_CNS_STEPS_DEFAULT	 	"20%"
#define FIM_CNS_STEPS_DEFAULT_P		true	/* FIXME */
#define FIM_CNS_STEPS_MIN	 1
#define FIM_CNS_WGROW_STEPS_DEFAULT	 1
#define FIM_CNS_WENLARGE_STEPS_DEFAULT	 10
#define FIM_CNS_SCALEDELTA	 0.01f
#define FIM_CNS_EMPTY_STRING	""
#define FIM_CNS_NEWLINE		"\n"
//#define FIM_CNS_NEWLINE		std::endl
#define FIM_CNS_DEFAULT_IFNAME	FIM_CNS_EMPTY_STRING
#define FIM_CNS_SLASH_CHAR	'/' /* directory delimiter */
#define FIM_CNS_SLASH_STRING	"/" /* directory delimiter */
#define FIM_CNS_QU_MA_STRING	"?" /* see FIM_SYM_BW_SEARCH_KEY_STR */
#define FIM_CNS_DIRSEP_STRING	FIM_CNS_SLASH_STRING
#define FIM_CNS_DIRSEP_CHAR	FIM_CNS_SLASH_CHAR
#define FIM_CNS_FP_ZERO		0.0
#define FIM_CNS_EMPTY_FP_VAL	FIM_CNS_FP_ZERO
#define FIM_CNS_EMPTY_INT_VAL	0
#define FIM_CNS_ERR_QUIT	0
#define FIM_CNS_EMPTY_RESULT	FIM_CNS_EMPTY_STRING
#ifdef SVN_REVISION
#define FIM_CNS_FIM_APPTITLE FIM_CNS_FIM ", v." PACKAGE_VERSION " (r." SVN_REVISION ")"
#else /* SVN_REVISION */
#define FIM_CNS_FIM_APPTITLE FIM_CNS_FIM ", v." PACKAGE_VERSION ""
#endif /* SVN_REVISION */
#ifdef SVN_REVISION_NUMBER
#define FIM_REVISION_NUMBER (SVN_REVISION_NUMBER)
#else /* SVN_REVISION_NUMBER */
#define FIM_REVISION_NUMBER (-1)
#endif /* SVN_REVISION_NUMBER */
#define FIM_CNS_MAGIC_DESC	"#!fim:desc"
#define FIM_CNS_EX_KSY_STRING	"{keysym}"
#define FIM_CNS_EX_CMD_STRING	"{command}"
#define FIM_CNS_EX_FN_STRING	"{filename}"
#define FIM_CNS_EX_FNS_STRING	"{filename(s)}"
#define FIM_CNS_EX_KC_STRING	"{keycode}"
#define FIM_CNS_EX_ID_STRING	"{identifier}"
#define FIM_CNS_EX_VAR_STRING	"{var}"
#define FIM_CNS_EX_IDS_STRING	"{identifier|identifier2...}]"
#define FIM_CNS_EX_EXP_STRING	"{expression}"
#define FIM_CNS_EX_PAT_STRING	"{pattern}"
#define FIM_CNS_EX_CMDS_STRING	"{commands}"
#define FIM_CNS_EX_EVT_STRING	"{event}"
#define FIM_CNS_EX_ARGS_STRING	"{args}"
#define FIM_CNS_EX_DSC_STRING	"{description}"
#define FIM_CNS_EX_PATH_STRING	"{path}"
#define FIM_CNS_EX_SYSC_STRING	"{syscmd}"
#define FIM_CNS_EX_FCT_STRING	"{factor}"
#define FIM_CNS_EX_RE_STRING	"{regexp}"
#define FIM_CNS_EX_RES_STRING	"{regexp(s)}"
#define FIM_CNS_EX_NUM_STRING	"{number}"
#define FIM_CNS_EX_SCALE_STRING	"{scale}"
#define FIM_CNS_SHELL	"/bin/sh"
#define FIM_CNS_DSFF_SL	"JPEG, PNG, GIF, TIFF, PPM, PGM, PBM" 
#define FIM_CNS_DSFF_SN	"BMP, PCX" 
#if FIM_USE_CXX11
/* this default FIM_CNS_PUSHDIR_RE might be built based on build options */
#define FIM_CNS_PUSHDIR_RE	R"***(\.(JPG|PNG|GIF|BMP|TIFF|TIF|JPEG|JFIF|PPM|PGM|PBM|PCX|WEBP)$)***" 
#else /* FIM_USE_CXX11 */
#define FIM_CNS_PUSHDIR_RE	    "\\.(JPG|PNG|GIF|BMP|TIFF|TIF|JPEG|JFIF|PPM|PGM|PBM|PCX|WEBP)$"
#endif /* FIM_USE_CXX11 */
#define FIM_CNS_ARCHIVE_RE	"\\.(RAR|ZIP|TAR|TAR.GZ|TGZ|TAR.BZ2|TBZ|TBZ2|CBR|CBZ|LHA|7Z|XAR|ISO)$" /* there might be more: CAB.. */
#define FIM_CNS_PIPEABLE_PATH_RE	"^[/A-Za-z0-9_.][/A-Za-z0-9_.-]*$"
#define FIM_CNS_BPP_INVALID	0
#define FIM_CNS_K 1024
#define FIM_CNS_M (FIM_CNS_K*FIM_CNS_K)
#define FIM_CNS_LAST FIM_MAX_INT
#define FIM_CNS_FIRST 0
#define FIM_CNS_VICSZ FIM_CNS_K * 64 /* viewport info cache size */
#define FIM_CNS_CSU FIM_CNS_K  /* cache size unit */ 
#define FIM_CNS_CLEARTERM "\x1B\x4D" /* clear term char */
#define FIM_CNS_WHITE 0xFFFFFFFF	/* Temporarily here. Will better need an enum or another portable solution. */
#define FIM_CNS_BLACK 0x00000000	/* Temporarily here. Will better need an enum or another portable solution. */
#define FIM_CNS_DBG_CMDS_MAX "ackCi" /* for FIM_VID_DBG_COMMANDS */
#define FIM_CNS_DBG_CMDS_MID "ackC"  /* for FIM_VID_DBG_COMMANDS */
#define FIM_CNS_DBG_CMDS_PFX "#"    /* for FIM_VID_DBG_COMMANDS */

#if FIM_USE_CXX14
template<typename T, typename S>
inline auto FIM_MIN(T x, S y)
{
	return ((x)<(y)?(x):(y));
}
template<typename T, typename S>
inline auto FIM_MAX(T x, S y)
{
	return ((x)>(y)?(x):(y));
}
#else /* FIM_USE_CXX14 */
#define FIM_MAX(x,y)        ((x)>(y)?(x):(y))
#define FIM_MIN(x,y)        ((x)<(y)?(x):(y))
#endif /* FIM_USE_CXX14 */

#define FIM_MOD(X,C)        ((((X)%(C))+(C))%(C))
#define FIM_INT_FRAC(Q,D) (((Q)+((D)-1))/(D))
#define FIM_INT_SCALE_FRAC(Q,D) ((Q)/(D)) /* division of integer values by fim_scale_t */

#define FIM_INTERNAL_LANGUAGE_SHORTCUT_SHORT_HELP \
".nf\n"\
":" "       enter command line mode\n"\
":" FIM_CNS_EX_NUM_STRING "       jump to " FIM_CNS_EX_NUM_STRING "^th image in the list\n"\
":^	        jump to first image in the list\n"\
":$	        jump to last image in the list\n"\
":*" FIM_CNS_EX_FCT_STRING "      scale the image by " FIM_CNS_EX_FCT_STRING "\n"\
":" FIM_CNS_EX_SCALE_STRING "%       scale the image to the desired " FIM_CNS_EX_SCALE_STRING "\n"\
":+" FIM_CNS_EX_SCALE_STRING "%       scale the image up to the desired percentage " FIM_CNS_EX_SCALE_STRING " (relatively to the original)\n"\
":-" FIM_CNS_EX_SCALE_STRING "%       scale the image down to the desired percentage " FIM_CNS_EX_SCALE_STRING " (relatively to the original)\n"\
"\n"\
"/" FIM_CNS_EX_RE_STRING "		 entering the pattern " FIM_CNS_EX_RE_STRING " (with /) makes fim jump to the next image whose filename matches " FIM_CNS_EX_RE_STRING "\n"\
"/*.png$		 entering this pattern (with /) makes fim jump to the next image whose filename ends with 'png'\n"\
"/png		 a shortcut for /.*png.*\n"\
"\n"\
"2,4 " FIM_FLT_STDOUT " '{}'	print three filenames to standard output.\n"\
"\n"\
"\n"\
"!" FIM_CNS_EX_SYSC_STRING "		executes the " FIM_CNS_EX_SYSC_STRING " quoted string as an argument to the  \"" FIM_FLT_SYSTEM "\" fim command.\n"\
""

#ifdef FIM_WITH_LIBCACA
#define FIM_USE_ASCII_ART_DEFAULT FIM_DDN_INN_CACA
#else /* FIM_WITH_LIBCACA */
#ifdef FIM_WITH_AALIB
#define FIM_USE_ASCII_ART_DEFAULT FIM_DDN_INN_AA
#else /* FIM_WITH_AALIB */
#undef FIM_USE_ASCII_ART_DEFAULT /*undefined */
#endif /* FIM_WITH_AALIB */
#endif /* FIM_WITH_LIBCACA */


#define FIM_MAX_MIPMAPS 32 /* pretty large ;-) */
#define FIM_HYPOTHENUSE_OF(C1,C2) sqrt( (C1)*(C1) + (C2)*(C2) )
#define FIM_HYPOTHENUSE_OF_INT(C1,C2) ceil( FIM_HYPOTHENUSE_OF(((double)(C1)),((double)(C2))) + 1.0 )

enum fim_mmo_t { FIM_MMO_NORMAL=0, FIM_MMO_FASTER=1 };
class fim_mipmap_t FIM_FINAL
{
	public:
	size_t mmoffs[FIM_MAX_MIPMAPS]; /* mipmap offset */
	size_t mmsize[FIM_MAX_MIPMAPS]; /* mipmap size */
	fim_int mmw[FIM_MAX_MIPMAPS]; /* mipmap width */
	fim_int mmh[FIM_MAX_MIPMAPS]; /* mipmap height */
	int nmm; /* number of mipmaps */
	size_t mmb; /* mipmap bytes (total) */
	fim_byte_t* mdp; /* mipmap data pointer */
	enum fim_mmo_t mmo;

	fim_mipmap_t(const fim_mipmap_t&mm){reset();}
	fim_mipmap_t(void){reset();}
	size_t byte_size(void)const{return mmb+sizeof(*this);}
	void dealloc(void){if(mdp)fim_free(mdp);reset();}
	~fim_mipmap_t(void){this->dealloc();}
	bool ok(void)const{return mmb > 0;}
	private:
	void reset(void){nmm=0;mmb=0;mdp=FIM_NULL;}
};

#include "fim_string.h"
#include "Command.h"
#include "Benchmarkable.h"
#include "FontServer.h"
#include "FbiStuff.h"
#include "Var.h"
#include "Namespace.h"
#include "Image.h"
#include "Cache.h"
#include "Viewport.h"
#include "FimWindow.h"
#include "Browser.h"
#include "DebugConsole.h"
#include "DummyDisplayDevice.h"
#ifdef FIM_WITH_LIBIMLIB2
#include "Imlib2Device.h"
#endif /* FIM_WITH_LIBIMLIB2 */
#ifdef FIM_WITH_LIBSDL
#include "SDLDevice.h"
#endif /* FIM_WITH_LIBSDL */
#ifdef FIM_WITH_LIBCACA
#include "CACADevice.h"
#endif /* FIM_WITH_LIBCACA */
#ifdef FIM_WITH_AALIB
#include "AADevice.h"
#endif /* FIM_WITH_AALIB */
#include "FramebufferDevice.h"
#include "CommandConsole.h"
#include "fim_stream.h"

namespace fim
{
	extern fim_stream cout;
	extern fim_stream cerr;
	extern fim_char_t g_sc;
}

#define FIM_I2BI(VID) VID "_" /* identifier to backup identifier */

/*
 * Fim language variable identifiers and help comment strings (notice their format).
 * Please note that the prefix FIM_VID is reserved to variable identifiers exclusively. 
 * */
//#define FIM_VID_NEWLINE 			"_newline"	/* "" */
//#define FIM_VID_TAB 				"_tab"	/* "" */
#define FIM_VID_ARCHIVE_FILES			"_archive_files"	/* "[in,g:] Regular expression matching filenames to be treated as (multipage) archives. If empty, \"" FIM_CNS_ARCHIVE_RE "\" will be used. Within each archive, only filenames matching the regular expression in the " FIM_VID_PUSHDIR_RE " variable will be considered for opening." */
#define FIM_VID_RANDOM 				"random"	/* "[out] a pseudorandom number." */
#define FIM_VID_BINARY_DISPLAY 			"_display_as_binary"	/* "[in,g:] will force loading of the specified files as pixelmaps (no image decoding will be performed); if 1, using one bit per pixel;  if 24, using 24 bits per pixel; otherwise will load and decode the files as usual." */
#define FIM_VID_TEXT_DISPLAY 			"_display_as_rendered_text"	/* "[in,g:] will force loading of the specified files as text files (no image decoding will be performed); if 1; otherwise will load and decode the files as usual." */
#define FIM_VID_CACHE_CONTROL 			"_cache_control"	/* "[in,g:] string for cache control. if it starts with 'm' mipmaps will be cached; if it starts with 'M' then not. otherwise defaults will apply." */
#define  FIM_VID_CROP_ONCE 		"_crop_once"		/* "[in,g:] Unfinished." */
#define FIM_VID_CACHE_STATUS 			"_cache_status"		/* "[out,g:] string with current information on cache status." */
#define FIM_VID_DISPLAY_CONSOLE 		"_display_console"	/* "[in,g:] if 1, will display the output console." */
#define FIM_VID_DEVICE_DRIVER 			"_device_string"	/* "[out,g:] the current display device string." */
#define FIM_VID_DISPLAY_STATUS			"_display_status"	/* "[in,g:] if 1, will display the status bar." */
#define FIM_VID_DISPLAY_STATUS_FMT		"_display_status_fmt"	/* "[in,g:] custom info format string, displayed in the lower left corner of the status bar; if unset: full pathname; otherwise a custom format string specified just as " FIM_VID_INFO_FMT_STR"." */
#define FIM_VID_PUSH_PUSHES_DIRS		"_push_pushes_dirs"	/* "[in,g:] if 1, the push command will also accept and push directories (using pushdir). if 2, will also push hidden files/directories, that is, ones whose names begin with a dot (.). " */
#define FIM_VID_SANITY_CHECK			"_do_sanity_check"	/* "[in,experimental,g:] if 1, will execute a sanity check on startup." */
#define FIM_VID_LAST_SYSTEM_OUTPUT		"_last_system_output"	/* "[out,experimental,g:] the standard output of the last call to the system command." */
#define FIM_VID_LAST_CMD_OUTPUT			"_last_cmd_output"	/* "[out,experimental,g:] the last command output." */
#define FIM_VID_LOAD_DEFAULT_ETC_FIMRC 		"_load_default_etc_fimrc"	/* "[in,g:] if 1 at startup, will load the system wide initialization file." */
#define FIM_VID_LOAD_HIDDEN_DIRS 		"_load_hidden_dirs"	/* "[in,g:] if not 1, when pushing directories/files, those whose name begins with a dot (.) will be skipped." */
#define FIM_VID_DEFAULT_ETC_FIMRC 		"_sys_rc_file"		/* "[in,g:] string with the global configuration file name." */
#if FIM_WANT_IMAGE_LOAD_TIME
#define FIM_VID_IMAGE_LOAD_TIME			"_file_load_time"	/* "[out,i:] time taken to load the file and decode the image, in seconds." */
#endif /* FIM_WANT_IMAGE_LOAD_TIME */
#define FIM_VID_FILE_LOADER 		"_file_loader"		/* "[in,i:,g:] if not empty, this string will force a file loader (among the ones listed in the -V switch output); [out] i:" FIM_VID_FILE_LOADER " stores the loader of the current image." */
#define FIM_VID_FILE_BUFFERED_FROM 		"_buffered_in_tmpfile"		/* "[out,i:] if an image has been temporarily converted and decoded from a temporary file, its name is here." */
#define FIM_VID_RETRY_LOADER_PROBE 		"_retry_loader_probe"		/* "[in,g:] if 1 and user specified a file loader and this fails, will probe for a different loader." */
#define FIM_VID_NO_RC_FILE			"_no_rc_file"		/* "[in,g:] if 1, the ~/.fimrc file will not be loaded at startup." */
#define FIM_VID_NO_EXTERNAL_LOADERS		"_no_external_loader_programs"		/* "[in,g:] if 1, no external loading programs will be tried for piping in an unsupported type image file." */
#define FIM_VID_SCRIPTOUT_FILE			"_fim_scriptout_file"	/* "[in,g:] the name of the file to write to when recording sessions." */
#define FIM_VID_PUSHDIR_RE			"_pushdir_re"	/* "[in] regular expression to match against when pushing files from a directory or an archive. By default this is \"" FIM_CNS_PUSHDIR_RE "\"." */
#define FIM_VID_STATUS_LINE 			"_status_line"		/* "[in,g:] if 1, will display the status bar." */
#define FIM_VID_WANT_PREFETCH 			"_want_prefetch"	/* "[in,g:] if 1, will prefetch further files just after display of the first file; if 2 (and configured with --enable-cxx11) will load in the background." */
#define FIM_VID_LOADING_IN_BACKGROUND		"_loading_in_background"	/* "[out,g:] 1 if program has been invoked with --" FIM_OSW_BGREC " and still loading in background. " */
#define FIM_VID_RESIZE_HUGE_ON_LOAD		"_downscale_huge_at_load"	/* "[in,g:] if 1, will downscale automatically huge images at load time." */
#define FIM_VID_WANT_SLEEPS 			"_want_sleep_seconds"	/* "[in,g:] number of seconds of sleep during slideshow mode." */
#define FIM_VID_WANT_EXIF_ORIENTATION		"_want_exif_orientation"	/* "[in,g:] if 1, will reorient images using information from EXIF metadata (and stored in in " FIM_VID_EXIF_ORIENTATION ", " FIM_VID_EXIF_MIRRORED ", " FIM_VID_EXIF_FLIPPED" )." */
#define FIM_VID_EXIF_ORIENTATION 		"__exif_orientation"	/* "[out,i:] orientation information in the same format of " FIM_VID_ORIENTATION ", read from the orientation EXIF tags (i:EXIF_Orientation)." */
#define FIM_VID_EXIF_MIRRORED 			"__exif_mirrored"	/* "[out,i:] mirroring information, read from the EXIF tags of a given image." */
#define FIM_VID_EXIF_FLIPPED 			"__exif_flipped"	/* "[out,i:] flipping information, read from the EXIF tags of a given image." */
#define FIM_VID_AUTOTOP				"_autotop"		/* "[in,g:] if 1, will align to the top freshly loaded images." */
#define FIM_VID_SCALE_STYLE			"_scale_style"		/* "[in,g:] if non empty, this string will be fed to the " FIM_VID_SCALE " command; see its documentation for possible values." */
#define FIM_VID_FILEINDEX			"_fileindex"		/* "[out,g:] the current image numeric index." */
#define FIM_VID_LOOP_ONCE			"_loop_only_once"	/* "[internal,g:] if 1 and doing a --slideshow, do it once." */
#define FIM_VID_LASTFILEINDEX			"_lastfileindex"	/* "[out,g:] the last visited image numeric index. Useful for jumping back and forth easily between two images with 'goto _lastfileindex'." */
#define FIM_VID_LASTGOTODIRECTION			"_lastgotodirection"	/* "[out,g:] the last file goto direction (either string '+1' or string '-1')." */
#define FIM_VID_FILELISTLEN			"_filelistlen"		/* "[out,g:] current image list length (number of visible images)." */
#define FIM_VID_INFO_FMT_STR			"_info_fmt_str"		/* "[in,g:] custom info format string, displayed in the lower right corner of the status bar; may contain ordinary text and special 'expando' sequences. These are: %p for current scale, in percentage; %w for width; %h for height; %i for image index in list; %k for the value of i:_comment (comment description) variable in square brackets; %l for current image list length; %L for flip/mirror/orientation information; %P for page information; %F for file size; %M for screen image memory size; %m for memory used by mipmap; %C for memory used by cache; %T for total memory used (approximation); %R for total max memory used (as detected by getrusage()); %n for the current file path name; %N for the current file path name basename; ; %c for centering information; %v for the fim program/version identifier string; %% for an ordinary %. A sequence like %?VAR?EXP? expands to EXP if i:VAR is set; EXP will be copied verbatim except for contained sequences of the form %:VAR:, which will be expanded to the value of variable i:VAR; this is meant to be used like in e.g. '%?EXIF_DateTimeOriginal?[%:EXIF_DateTimeOriginal:]?', where the EXIF-set variable EXIF_DateTimeOriginal (make sure you have libexif for this) will be used only if present." */
#define FIM_VID_FILENAME			"_filename"		/* "[out,i:] the current file name string." */
#define FIM_VID_FIM_DEFAULT_CONFIG_FILE_CONTENTS "_fim_default_config_file_contents"/* "[out,g:] the contents of the default (hardcoded) configuration file (executed after the minimal hardcoded config)." */
#define FIM_VID_FIM_DEFAULT_GRAMMAR_FILE_CONTENTS "_fim_default_grammar_file_contents" /* "[out,g:] the contents of the default (hardcoded) grammar file." */
#define FIM_VID_FRESH				"fresh"			/* "[in,out,i:,experimental] 1 if the image was loaded, before all autocommands (autocmd) execution." */
#define FIM_VID_PAGE				"page"			/* "[out,experimental,g:] the current page." */
#define FIM_VID_PAGES				"pages"			/* "[out,experimental,i:] the current number of pages of an image." */
#define FIM_VID_OVERRIDE_DISPLAY		"_inhibit_display"	/* "[internal,g:] if 1, will inhibit display." */
#define FIM_VID_MAX_ITERATED_COMMANDS		"_max_iterated_commands"	/* "[g:] the iteration limit for N in \"N[commandname]\" iterated command invocations." */
#define FIM_VID_WANT_CAPTION_STATUS		"_want_wm_caption_status"	/* "[in,g:] this works only if supported by the display device (currently only SDL). if set to a number that is not 0, will show the status (or command) line in the window manager caption; if set to a non-empty string, will interpret it just as a file info format string (see " FIM_VID_INFO_FMT_STR "); if empty, will show the program version." */
#define FIM_VID_WANT_MOUSE_CTRL		"_want_wm_mouse_ctrl"	/* "[in,g:] if at least 9 chars long, enable mouse click/movement behaviour when in SDL mode; the 9 chars will correspond to a 3x3 screen clickable grid and the equivalent command keys; clicking middle or right button will toggle on-screen usage info." */
#define FIM_VID_MAGNIFY_FACTOR			"_magnify_factor"	/* "[in,g:] the image scale multiplier used when magnifying images size." */
#define FIM_VID_PWD				"_pwd"			/* "[out,g:] the current working directory; will be updated at startup and whenever the working directory changes." */
#define FIM_VID_REDUCE_FACTOR			"_reduce_factor"		/* "[in,g:] the image scale multiplier used when reducing images size." */
#define FIM_VID_SCALE_FACTOR_MULTIPLIER		"_scale_factor_multiplier"	/* "[in,g:] value used for scaling up/down the scaling factors." */
#define FIM_VID_SCALE_FACTOR_DELTA		"_scale_factor_delta"		/* "[in,g:] value used for incrementing/decrementing the scaling factors." */
#define FIM_VID_COMMENT 				"_comment"				/* "[i:,out] the image comment, extracted from the image file (if any)." */
#define FIM_VID_COMMENT_OI 				"_caption_over_image"				/* "[in,g:] if set not to 0, will display a custom comment string specified according to the value of_caption_over_image_fmt; if larger than 1, with black background; if 3, image will be drawn possibly below. It will take at most half of the screen." */
#define FIM_VID_COMMENT_OI_FMT 				"_caption_over_image_fmt"		/* "[in,g:] custom info format string, displayed in a caption over the image; if unset: i:_comment; otherwise a custom format string specified just as " FIM_VID_INFO_FMT_STR"." */
#define FIM_VID_EXIFTOOL_COMMENT 				"_exiftool_comment"				/* "[out,g:] comment extracted via the exiftool interface; see " FIM_VID_EXIFTOOL "." */
#define FIM_VID_STEPS 				"_steps"				/* "[in,g:] the default steps, in pixels, when panning images." */
#define FIM_VID_PREAD 				"_pread_cmd"	/* "[in,g:] a user-specified shell command emitting an image on stdout, in a format readable by the convert utility. If the current filename matches \"" FIM_CNS_PIPEABLE_PATH_RE "\", it will be substituted to any occurrence of '{}'." */
#define FIM_VID_VERSION				"_fim_version"	/* "[out,g:] fim version number; may be used for keeping compatibility of fim scripts across evolving versions."  */
#define FIM_VID_FBFONT                           "_fbfont"       /* "[out,g:] The current console font file string. If the internal hardcoded font has been used, then its value is \"" FIM_DEFAULT_HARDCODEDFONT_STRING "\"." */
#define FIM_VID_FBFMF                           "_fbfont_magnify_factor"       /* "[in,g:] The rendered text will use a font magnified by this (integer) factor. Maximal value is \"" FIM_XSTRINGIFY(FIM_FONT_MAGNIFY_FACTOR_MAX) "\". (Only enabled if configured with --with-font-magnifying-factor=FACTOR, with FACTOR<1)." */
#define FIM_VID_FBAFS                           "_fbfont_as_screen_fraction"       /* "[in,g:] The rendered text will be scaled at least to this (integer) fraction of the screen. Disable font autoscaling with -1. (Only enabled if configured with --with-font-magnifying-factor=FACTOR, with FACTOR<1)." */
#define FIM_VID_FB_VERBOSITY			"_fbfont_verbosity"       /* "[in,g:] if > 0, verbose font loading" */
#define FIM_VID_HSTEPS 				"_hsteps"				/* "[in,g:] the default steps, in pixels, when panning images horizontally (overrides steps)." */
#define FIM_VID_VSTEPS 				"_vsteps"				/* "[in,g:] the default steps, in pixels, when panning images vertically (overrides steps)." */
#define FIM_VID_CONSOLE_ROWS 			"_rows"			/* "[in,g:] if >0, will set the number of displayed text lines in the console." */
#define FIM_VID_CONSOLE_LINE_WIDTH 		"_lwidth"		/* "[in,g:] if>0, will force the output console text width." */
#define FIM_VID_CONSOLE_LINE_OFFSET 		"_console_offset"	/* "[in,out,g:] position of the text beginning in the output console, expressed in lines." */
#define FIM_VID_CONSOLE_BUFFER_LINES		"_console_lines"		/* "[out,g:] the number of buffered output console text lines." */
#define FIM_VID_CONSOLE_BUFFER_TOTAL		"_console_buffer_total"		/* "[out,g:] amount of memory allocated for the output console buffer." */
#define FIM_VID_CONSOLE_BUFFER_FREE		"_console_buffer_free"		/* "[out,g:] amount of unused memory in the output console buffer." */
#define FIM_VID_CONSOLE_BUFFER_USED		"_console_buffer_used"		/* "[out,g:] amount of used memory in the output console buffer." */
#define FIM_VID_VERBOSE_KEYS			"_verbose_keys"			/* "[in,g:] if 1, after each interactive mode key hit, the console will display the hit key raw keycode." */
#define FIM_VID_CMD_EXPANSION			"_command_expansion"			/* "[in,g:] if 1, will enable autocompletion (on execution) of alias and command strings." */
#define FIM_VID_VERBOSE_ERRORS			"_verbose_errors"			/* "[in,g:] if 1, will display on stdout internal errors, while parsing commands." */
#define FIM_VID_VERBOSITY			"_verbosity"			/* "[in,experimental,g:] program verbosity." */
#define FIM_VID_CONSOLE_KEY			"_console_key"		/* "[in,g:] the key binding (an integer variable) for spawning the command line; will have precedence over any other binding." */
#define FIM_VID_IGNORECASE			"_ignorecase"		/* "[in,g:] if 1, will allow for case insensitive regexp-based match in autocommands (autocmd)." */
#define FIM_VID_RE_SEARCH_OPTS			"_re_search_opts"	/* "[in,g:] affects regexp-based searches; if an empty string, defaults will apply; if contains 'i' ('I'), case insensitive (sensitive) searches will occur; if contains 'b', will match on basename, if contains 'f' on full pathname." */
#define FIM_VID_SAVE_FIM_HISTORY		"_save_fim_history"	/* "[in,g:] if 1 on exit, will save the " FIM_CNS_HIST_COMPLETE_FILENAME " file on exit." */
#define FIM_VID_LOAD_FIM_HISTORY		"_load_fim_history"	/* "[in,g:] if 1 on startup, will load the " FIM_CNS_HIST_COMPLETE_FILENAME " file on startup." */
#define FIM_VID_TERM				"_TERM"			/* "[out,g:] the environment TERM variable." */
#define FIM_VID_NO_DEFAULT_CONFIGURATION	"_no_default_configuration"	/* "[in,g:] if 0, a default, hardcoded configuration will be executed at startup, after the minimal hardcoded one." */
#define FIM_VID_DISPLAY_STATUS_BAR		"_display_status_bar"		/* "[in,g:] if 1, will display the status bar." */
#define FIM_VID_DISPLAY_BUSY			"_display_busy"			/* "[in,g:] if 1, will display a message on the status bar when processing." */
#define FIM_VID_WANT_MIPMAPS			"_use_mipmaps"			/* "[in,g:] if >0, will use mipmaps to speed up downscaling of images (this has a memory overhead equivalent to one image copy); mipmaps will not be cached. If 2, will use every fourth source pixel instead of averaging (good for photos, not for graphs)." */
#define FIM_VID_EXIFTOOL			"_use_exiftool"			/* "[in,g:] if >0 and supported, exiftool will be used to get additional information. if 1, this will be appened to _comment; if 2, will go to " FIM_VID_EXIFTOOL_COMMENT "." */
#define FIM_VID_SCALE				"scale"				/* "[in,i:] the scale of the current image." */
#define FIM_VID_ASCALE				"ascale"			/* "[in,out,i:] the asymmetric scaling of the current image." */
#define FIM_VID_ANGLE				"angle"				/* "[in,out,i:] a floating point number specifying the rotation angle, in degrees." */
#define FIM_VID_ORIENTATION			"_orientation"		/* "[internal,i:] Orthogonal clockwise rotation (orientation) is controlled by: 'i:" FIM_VID_ORIENTATION "', 'g:" FIM_VID_ORIENTATION "' and applied on a per-image basis. In particular, the values of the three variables are summed up and the sum is interpreted as the image orientation.  If the sum is 0, no rotation will apply; if it is 1, a single ( 90') rotation will apply; if it is 2, a double (180') rotation will apply; if it is 3, a triple (270') rotation will apply.  If the sum is not one of 0,1,2,3, the value of the sum modulo 4 is considered.  Therefore, \":i:" FIM_VID_ORIENTATION "=1\" and \":i:" FIM_VID_ORIENTATION "=5\" are equivalent: they rotate the image one time by 90'." */
#define FIM_VID_WIDTH				"width"			/* "[out,i:] the current image original width." */
#define FIM_VID_HEIGHT				"height"		/* "[out,i:] the current image original height." */
#define FIM_VID_SWIDTH				"swidth"		/* "[out,i:] the current image scaled width." */
#define FIM_VID_SHEIGHT				"sheight"		/* "[out,i:] the current image scaled height." */
#define FIM_VID_AUTOFLIP			"_autoflip"		/* "[in,g:] if 1, will flip images by default." */
#define FIM_VID_AUTONEGATE			"_autonegate"		/* "[in,g:] if 1, will negate images by default." */
#define FIM_VID_AUTODESATURATE			"_autodesaturate"	/* "[in,g:] if 1, will desaturate images by default." */
#if FIM_WANT_REMEMBER_LAST_FILE_LOADER
#define FIM_VID_LAST_FILE_LOADER		"_last_file_loader"	/* "[out,g:] after each image load, " FIM_VID_LAST_FILE_LOADER " will be set to the last file loader used." */
#define FIM_VID_ALL_FILE_LOADERS		"_all_file_loaders"	/* "[out,g:] space-separated list of hardcoded file loaders usable with " FIM_VID_FILE_LOADER "." */
#endif /* FIM_WANT_REMEMBER_LAST_FILE_LOADER */
#define FIM_VID_FLIPPED				"flipped"		/* "[out,i:] 1, if the image is flipped." */
#define FIM_VID_NEGATED				"negated"		/* "[out,i:] 1, if the image is negated." */
#define FIM_VID_DESATURATED			"desaturated"		/* "[out,i:] 1, if the image is desaturated." */
#define FIM_VID_FIM_BPP				"_fim_bpp"		/* "[out,g:] the bits per pixel count." */
#define FIM_VID_AUTOMIRROR			"_automirror"		/* "[in,g:] if 1, will mirror images by default." */
#define FIM_VID_MIRRORED			"mirrored"		/* "[out,i:] 1, if the image is mirrored." */
#define FIM_VID_WANT_AUTOCENTER			"_want_autocenter"	/* "[in,g:] if 1, the image will be displayed centered." */
#define FIM_VID_MAX_CACHED_IMAGES		"_max_cached_images"	/* "[in,g:] the maximum number of images after which evictions will be forced. Setting this to 0 (no limits) is ok provided _max_cached_memory is set meaningfully." */
#define FIM_VID_MIN_CACHED_IMAGES		"_min_cached_images"	/* "[in,g:] the minimum number of images to keep from eviction; if less than four can lead to inefficiencies: e.g. when jumping between two images, each time an erase and a prefetch of neighboring images would trigger. default value is 4." */
#define FIM_VID_MAX_CACHED_MEMORY		"_max_cached_memory"	/* "[in,g:] the maximum amount of memory (in KiB) at which images will be continued being added to the cache. Setting this to 0 (no limit) will lead to a crash (there is no protection currently)." */
#define FIM_VID_CACHED_IMAGES			"_cached_images"	/* "[out,g:] the number of images currently cached." */
#define FIM_VID_SCREEN_WIDTH			"_screen_width"		/* "[out,g:] the screen width." */
#define FIM_VID_SCREEN_HEIGHT			"_screen_height"		/* "[out] the screen height." */
#define FIM_VID_SKIP_SCROLL			"_scroll_skip_page_fraction"		/* "[int,g:] if >1, fraction of page which will be skipped in when scrolling (e.g. 'scrollforward'); if 1, auto chosen; if <1, disabled." */
#define FIM_VID_DBG_AUTOCMD_TRACE_STACK		"_autocmd_trace_stack"	/* "[in,g:] dump to stdout autocommands (autocmd) stack trace during their execution (for debugging purposes)." */
#define FIM_VID_DBG_COMMANDS			"_debug_commands"	/* "[in,g:] debugging option string for printing out . if containing 'a', print out autocmd info; if containing 'c', print out each command; if containing 'k', print out each pressed key; if containing 'j', print interpreter internal steps; if containing 'B', clear screen and print background loading files; if containing 'C', print cache activity." */
#define FIM_VID_OPEN_OFFSET_L			"_open_offset"		/* "[in,optional,g:,i:] offset (specified in bytes) used when opening a file; [out] i:" FIM_VID_OPEN_OFFSET_L " will be assigned to images opened at a nonzero offset." */
#define FIM_VID_OPEN_OFFSET_U		"_open_offset_retry"	/* "[in,optional,g:] number of adjacent bytes to probe in opening the file." */
#define FIM_VID_SEEK_MAGIC			"_seek_magic"		/* "[optional,g:] will seek a 'magic' signature in the file after opening it, and will try decoding it starting within the range of that signature (use like this: fim -C '" FIM_VID_SEEK_MAGIC "=MAGIC_STRING;push filename')." */
#define FIM_VID_PREFERRED_RENDERING_WIDTH	"_preferred_rendering_width"		/* "[in,optional,g:] if >0, bit based rendering will use this value for a default document width (instead of a default value)." */
#define FIM_VID_PREFERRED_RENDERING_DPI	"_preferred_rendering_dpi"		/* "[in,optional,g:] if >0, pdf, ps, djvu rendering will use this value for a default document dpi (instead of a default value)." */
#define FIM_VID_PRELOAD_CHECKS			"_push_checks"	/* "[in,experimental,g:] if 1 (default), will check with stat() existence of input files before push'ing them (set this to 0 to speed up loading very long file lists; in these cases a trailing slash (/) will have to be used to tell fim a pathname is a directory). This only works after initialization (thus, after command line files have been push'ed); use --" FIM_OSW_NO_STAT_PUSH " if you wish to set this to 0 at command line files specification." */

/*
 * Fim Keyboard Keys Descriptions
 * */
#define FIM_KBD_TAB			"Tab"
#define FIM_KBD_ENTER			"Enter"
#define FIM_KBD_PAUSE			"Pause"
#define FIM_KBD_MENU			"Menu"
#define FIM_KBD_BACKSPACE		"BackSpace"
#define FIM_KBD_BACKSPACE_		"Backspace"
#define FIM_KBD_SPACE			" "
#define FIM_KBD_DEL			"Del"
#define FIM_KBD_INS			"Ins"
#define FIM_KBD_HOME			"Home"
#define FIM_KBD_END			"End"
#define FIM_KBD_ANY			"Any"
#define FIM_KBD_ESC			"Esc"
#define FIM_KBD_LEFT			"Left"
#define FIM_KBD_RIGHT			"Right"
#define FIM_KBD_UP			"Up"
#define FIM_KBD_DOWN			"Down"
#define FIM_KBD_PAGEUP			"PageUp"
#define FIM_KBD_PAGEDOWN			"PageDown"
#define FIM_KBD_COLON			":"
#define FIM_KBD_SEMICOLON			";"
#define FIM_KBD_MOUSE_LEFT			"MouseLeft" /* still unused */
#define FIM_KBD_MOUSE_RIGHT			"MouseRight" /* still unused */
#define FIM_KBD_PLUS			"+"
#define FIM_KBD_MINUS			"-"
#define FIM_KBD_SLASH			"/"
#define FIM_KBD_ASTERISK			"*"
#define FIM_KBD_GT			">"
#define FIM_KBD_LT			"<"
#define FIM_KBD_UNDERSCORE		"_"
#define FIM_KBD_F1			"F1"
#define FIM_KBD_F2			"F2"
#define FIM_KBD_F3			"F3"
#define FIM_KBD_F4			"F4"
#define FIM_KBD_F5			"F5"
#define FIM_KBD_F6			"F6"
#define FIM_KBD_F7			"F7"
#define FIM_KBD_F8			"F8"
#define FIM_KBD_F9			"F9"
#define FIM_KBD_F10			"F10"
#define FIM_KBD_F11			"F11"
#define FIM_KBD_F12			"F12"

/*
 * Fim Keyboard Keys Encodings
 * */
#define FIM_KKE_ENTER			10
#define FIM_KKE_ESC			27
#define FIM_KKE_BACKSPACE		127
#define FIM_KKE_LEFT			4479771
#define FIM_KKE_RIGHT			4414235
#define FIM_KKE_UP			4283163
#define FIM_KKE_DOWN			4348699
#define FIM_KKE_PAGE_UP			2117425947
#define FIM_KKE_PAGE_DOWN		2117491483
#define FIM_KKE_INSERT			2117229339
#define FIM_KKE_HOME			2117163803
#define FIM_KKE_END			2117360411
#define FIM_KKE_DEL			2117294875
#define FIM_KKE_F1			282
#define FIM_KKE_F2			283
#define FIM_KKE_F3			284
#define FIM_KKE_F4			285
#define FIM_KKE_F5			286
#define FIM_KKE_F6			287
#define FIM_KKE_F7			288
#define FIM_KKE_F8			289
#define FIM_KKE_F9			290
#define FIM_KKE_F10			291
#define FIM_KKE_F11			292
#define FIM_KKE_F12			293

/*
 * Fim Display Driver Names
 * Note: When adding a new driver, you shall update FIM_DDN_VARS as well. 
 * p.s.: ICI = If Configured In
 * */
#define FIM_DDN_INN_FB	 "fb"
#define FIM_DDN_VAR_FB	 "fb"
#ifndef FIM_WITH_NO_FRAMEBUFFER
#define FIM_DDN_ICI_FB	 "fb|"
#else /* FIM_WITH_NO_FRAMEBUFFER */
#define FIM_DDN_ICI_FB	 ""
#endif /* FIM_WITH_NO_FRAMEBUFFER */

#define FIM_DDN_INN_SDL	 "sdl"
#define FIM_DDN_VAR_SDL	 "sdl"
#ifdef FIM_WITH_LIBSDL
#define FIM_DDN_ICI_SDL	 "sdl|"
#else /* FIM_WITH_LIBSDL */
#define FIM_DDN_ICI_SDL	 ""
#endif /* FIM_WITH_LIBSDL */

#define FIM_DDN_INN_IL2	 "imlib2"
#define FIM_DDN_VAR_IL2	 "imlib2"
#ifdef FIM_WITH_LIBIMLIB2
#define FIM_DDN_ICI_IL2	 "imlib2|"
#else /* FIM_WITH_LIBIMLIB2 */
#define FIM_DDN_ICI_IL2	 ""
#endif /* FIM_WITH_LIBIMLIB2 */

#define FIM_DDN_INN_AA	 "aa"
#define FIM_DDN_VAR_AA	 "aa"
#ifdef FIM_WITH_AALIB
#define FIM_DDN_ICI_AA	 "aa|"
#else /* FIM_WITH_AALIB */
#define FIM_DDN_ICI_AA	 ""
#endif /* FIM_WITH_AALIB */

#define FIM_DDN_INN_CACA	 "ca"
#define FIM_DDN_VAR_CACA	 "ca"
#ifdef FIM_WITH_LIBCACA
#define FIM_DDN_ICI_CACA	 "ca|"
#else /* FIM_WITH_LIBCACA */
#define FIM_DDN_ICI_CACA	 ""
#endif /* FIM_WITH_LIBCACA */

#define FIM_DDN_INN_DUMB	 "dumb"
//#define FIM_DDN_VAR_DUMB	 "dummy"
#define FIM_DDN_VAR_DUMB	 "dumb"

#define FIM_DDN_VARS_NB	 FIM_DDN_INN_FB"|" FIM_DDN_INN_SDL "|" FIM_DDN_INN_CACA "|" FIM_DDN_INN_AA  "|" FIM_DDN_INN_IL2 "|" FIM_DDN_INN_DUMB
#define FIM_DDN_VARS	 "[" FIM_DDN_VARS_NB "]" 
#define FIM_DDN_VARS_IN "[" FIM_DDN_ICI_FB FIM_DDN_ICI_SDL FIM_DDN_ICI_CACA FIM_DDN_ICI_AA FIM_DDN_ICI_IL2 FIM_DDN_INN_DUMB "]" 

#define FIM_HELP_EXTRA_OPTS "s|d|l|m"

/*
 * Fim Autocommands
 * FIXME: need autodocumentation for these.
 * */
//#define FIM_ACM_POSTSCREENRESIZE	"PostScreenResize"	/* "" */
//#define FIM_ACM_PRESCREENRESIZE		"PreScreenResize"	/* "" */
#define FIM_ACM_PRESCALE	"PreScale"		/* "" */
#define FIM_ACM_POSTSCALE	"PostScale"		/* "" */
#define FIM_ACM_PREPAN		"PrePan"		/* "" */
#define FIM_ACM_POSTPAN		"PostPan"		/* "" */
#define FIM_ACM_PREREDISPLAY	"PreRedisplay"		/* "" */
#define FIM_ACM_POSTREDISPLAY	"PostRedisplay"		/* "" */
#define FIM_ACM_PREDISPLAY	"PreDisplay"		/* "" */
#define FIM_ACM_POSTDISPLAY	"PostDisplay"		/* "" */
#define FIM_ACM_PREPREFETCH	"PrePrefetch"		/* "" */
#define FIM_ACM_POSTPREFETCH	"PostPrefetch"		/* "" */
#define FIM_ACM_PRERELOAD	"PreReload"		/* "" */
#define FIM_ACM_POSTRELOAD	"PostReload"		/* "" */
#define FIM_ACM_PRELOAD		"PreLoad"		/* "" */
#define FIM_ACM_POSTLOAD	"PostLoad"		/* "" */
#define FIM_ACM_PREGOTO		"PreGoto"		/* "" */
#define FIM_ACM_POSTGOTO	"PostGoto"		/* "" */
#define FIM_ACM_PRECONF		"PreConfigLoading"	/* "before loading any configuration file" */
#define FIM_ACM_POSTCONF	"PostConfigLoading"	/* "after  loading any configuration file" */
#define FIM_ACM_PREHFIMRC	"PreHardcodedConfigLoading"	/* "before loading hardcoded configuration" */
#define FIM_ACM_POSTHFIMRC	"PostHardcodedConfigLoading"	/* "after  loading hardcoded configuration" */
#define FIM_ACM_PREUFIMRC	"PreUserConfigLoading"		/* "before loading user configuration file" */
#define FIM_ACM_POSTUFIMRC	"PostUserConfigLoading"		/* "after  loading user configuration file" */
#define FIM_ACM_PREGFIMRC	"PreGlobalConfigLoading"	/* "before loading global configuration file" */
#define FIM_ACM_POSTGFIMRC	"PostGlobalConfigLoading"	/* "after  loading global configuration file" */
#define FIM_ACM_PREINTERACTIVECOMMAND	"PreInteractiveCommand"		/* "" */
#define FIM_ACM_POSTINTERACTIVECOMMAND	"PostInteractiveCommand"	/* "" */
#define FIM_ACM_PREEXECUTIONCYCLE	"PreExecutionCycle"		/* "" */
#define FIM_ACM_POSTEXECUTIONCYCLE	"PostExecutionCycle"		/* "" */
#define FIM_ACM_PREEXECUTIONCYCLEARGS	"PreExecutionCycleArgs"		/* "" */
#define FIM_ACM_PREWINDOW	"PreWindow"	/* "before a display reinit (after  PreRedisplay)" */
#define FIM_ACM_POSTWINDOW	"PostWindow"	/* "after  a display reinit (before PreRedisplay)" */
//#define FIM_ACM_PREROTATE	"PreRotate"	/* "" */
//#define FIM_ACM_POSTROTATE	"PostRotate"	/* "" */

/*
 * Fim Language Tokens
 * */
#define FIM_FLT_ALIAS			"alias"	/* not in vim */
#define FIM_FLT_ALIGN			"align"	/* not in vim */
#define FIM_FLT_AUTOCMD			"autocmd" /* in vim */
#define FIM_FLT_AUTOCMD_DEL		"autocmd_del"	/* not in vim */
#define FIM_FLT_BIND			"bind" /* not in vim */
#define FIM_FLT_BASENAME		"basename" /* not in vim */
#define FIM_FLT_CD			"cd" /* in vim */
#define FIM_FLT_CLEAR			"clear" /* not in vim */
#define FIM_FLT_COLOR	              	"color" /* not in vim */
#define FIM_FLT_COMMANDS		"commands" /* not in vim */
#define FIM_FLT_CROP			"crop" /* not in vim */
#define FIM_FLT_DISPLAY			"display" /* in vim, with another meaning */
#define FIM_FLT_DUMP_KEY_CODES		"dump_key_codes" /* not in vim */
#define FIM_FLT_DESC			"desc" /* not in vim */
#define FIM_FLT_ECHO			"echo" /* in vim */
#define FIM_FLT_ELSE			"else" /* in vim */
#define FIM_FLT_EXEC			"exec" /* not in vim */
#define FIM_FLT_EVAL			"eval" /* not in vim */
#define FIM_FLT_FILE			"file" /* in vim */
#define FIM_FLT_FONT			"font" /* not in vim */
#define FIM_FLT_GETENV			"getenv" /* not in vim */
#define FIM_FLT_GOTO			"goto" /* in vim */
#define FIM_FLT_HELP			"help" /* in vim */
#define FIM_FLT_IF			"if" /* in vim */
#define FIM_FLT_INFO			"info" /* not in vim */
#define FIM_FLT_LOAD			"load" /* not in vim */
#define FIM_FLT_LIMIT			"limit" /* not in vim */
#define FIM_FLT_LIST			"list" /* not in vim */
#define FIM_FLT_PAN			"pan" /* not in vim */
#define FIM_FLT_POPEN			"popen" /* not in vim */
#define FIM_FLT_PREAD			"pread" /* not in vim */
#define FIM_FLT_PREFETCH		"prefetch" /* in vim */
#define FIM_FLT_PWD			"pwd" /* in vim */
#define FIM_FLT_REDISPLAY		"redisplay" /* not in vim */
#define FIM_FLT_RELOAD			"reload" /* not in vim */
#define FIM_FLT_ROTATE			"rotate" /* not in vim */
#define FIM_FLT_SCALE			"scale" /* not in vim */
#define FIM_FLT_SCROLL			"scroll" /* not in vim */
#define FIM_FLT_SET			"set" /* in vim */
#define FIM_FLT_SET_INTERACTIVE_MODE	"set_interactive_mode" /* not in vim */
#define FIM_FLT_SET_CONSOLE_MODE	"set_commandline_mode" /* not in vim */
#define FIM_FLT_STATUS			"status" /* not in vim */
#define FIM_FLT_STDERR			"stderr" /* not in vim */
#define FIM_FLT_STDOUT			"stdout" /* not in vim */
#define FIM_FLT_QUIT			"quit" /* in vim */
#define FIM_FLT_RECORDING		"recording" /* not in vim */
#define FIM_FLT_SYSTEM			"system" /* not in vim */
#define FIM_FLT_SLEEP			"sleep" /* in vim */
#define FIM_FLT_USLEEP			"usleep" /* not in vim */
#define FIM_FLT_UNALIAS			"unalias" /* not in vim */
#define FIM_FLT_UNBIND			"unbind" /* not in vim */
#define FIM_FLT_VARIABLES		"variables" /* not in vim */
#define FIM_FLT_WHILE			"while" /* in vim */
#define FIM_FLT_WINDOW			"window" /* not in vim */

/* composite commands or hardcoded aliases */
#define FIM_FLA_NEXT_FILE		"next_file" /* not in vim */ // WAS: FIM_FLT_NEXT_PIC
#define FIM_FLA_PREV_FILE		"prev_file" /* not in vim */ // WAS: FIM_FLT_PREC_PIC
#define FIM_FLA_NEXT_PAGE		"next_page" /* not in vim */
#define FIM_FLA_PREV_PAGE		"prev_page" /* not in vim */
#define FIM_FLA_NEXT			"next" /* in vim */
#define FIM_FLA_PREV			"prev" /* in vim */
#define FIM_FLA_MAGNIFY			"magnify" /* not in vim */
#define FIM_FLA_REDUCE			"reduce" /* not in vim */
#define FIM_FLC_NEXT			"goto '+1'" /* in vim */
#define FIM_FLC_PREV			"goto '-1'" /* in vim */
#define FIM_FLC_MIRROR			"mirror" /* not in vim */
#define FIM_FLC_FLIP			"flip" /* not in vim */
#define FIM_FLC_PAN_UP			"pan 'up'" /* not in vim */
#define FIM_FLC_PAN_DOWN		"pan 'down'" /* not in vim */
#define FIM_FLC_SCROLL_DOWN		"pan 'down+'" /* not in vim */
#define FIM_FLC_SCROLL_UP		"pan 'up-'" /* not in vim */
#define FIM_FLC_PAN_LEFT		"pan 'left'" /* not in vim */
#define FIM_FLC_PAN_RIGHT		"pan 'right'" /* not in vim */
#define FIM_FLC_NEXT_FILE		"goto '+1f'" /* not in vim */ // WAS: FIM_FLT_NEXT_PIC
#define FIM_FLC_PREV_FILE		"goto '-1f'" /* not in vim */ // WAS: FIM_FLT_PREC_PIC
#define FIM_FLC_NEXT_PAGE		"goto '+1p'" /* not in vim */
#define FIM_FLC_PREV_PAGE		"goto '-1p'" /* not in vim */
#define FIM_FLC_MAGNIFY			"scale '+'" /* not in vim */
#define FIM_FLC_REDUCE			"scale '-'" /* not in vim */

#define FIM_CMD_ACM_INFO(PREACM,POSTACM) FIM_CNS_CMDSEP "Executes autocommands for events " PREACM " and " POSTACM "."
#define FIM_CNS_CMDSEP "\n" /* see CommandConsole::get_commands_reference() */
/* Help messages for Fim internal commands. */
#define FIM_CMD_HELP_ALIAS  FIM_FLT_ALIAS " [" FIM_CNS_EX_ID_STRING " [" FIM_CNS_EX_CMDS_STRING " [" FIM_CNS_EX_DSC_STRING "]]]" FIM_CNS_CMDSEP "Without arguments, lists the current aliases." FIM_CNS_CMDSEP "With one, shows an identifier's assigned command." FIM_CNS_CMDSEP "With two, assigns to an identifier a user defined command or sequence of commands." FIM_CNS_CMDSEP "With three, also assigns a help string."
#define FIM_CMD_HELP_ALIGN FIM_FLT_ALIGN " ['bottom'|'top'|'left'|right']: if image larger than drawing area, align one side of the image to the border." FIM_CNS_CMDSEP FIM_FLT_ALIGN " 'center': align equally far from all sides." FIM_CNS_CMDSEP FIM_FLT_ALIGN " 'info': print internal alignment information."
#define FIM_CMD_HELP_AUTOCMD FIM_FLT_AUTOCMD " " FIM_CNS_EX_EVT_STRING " " FIM_CNS_EX_PAT_STRING " " FIM_CNS_EX_CMDS_STRING ": manipulate autocommands (inspired from Vim autocmd's)." FIM_CNS_CMDSEP "Without arguments, list autocommands." FIM_CNS_CMDSEP "With arguments, specifies for which type of event and which current file open, which commands to execute." FIM_CNS_CMDSEP "See the default built-in configuration files for examples."
#define FIM_CMD_HELP_AUTOCMD_DEL FIM_FLT_AUTOCMD_DEL ": specify autocommands to delete." FIM_CNS_CMDSEP "Usage: " FIM_FLT_AUTOCMD_DEL " " FIM_CNS_EX_EVT_STRING " " FIM_CNS_EX_PAT_STRING " " FIM_CNS_EX_CMDS_STRING "."
#define FIM_CMD_HELP_BASENAME FIM_FLT_BASENAME" {filename}: returns the basename of {filename} in the '" FIM_VID_LAST_CMD_OUTPUT "' variable."
#define FIM_CMD_HELP_BIND FIM_FLT_BIND" [" FIM_CNS_EX_KSY_STRING " [" FIM_CNS_EX_CMDS_STRING "]]: bind a keyboard symbol/shortcut " FIM_CNS_EX_KSY_STRING " to " FIM_CNS_EX_CMDS_STRING "." FIM_CNS_CMDSEP FIM_CNS_RAW_KEYS_MESG FIM_CNS_CMDSEP "Key binding is dynamical, so you can rebind keys even during program's execution." FIM_CNS_CMDSEP "You can get a list of valid symbols (keysyms) by invoking " FIM_FLT_DUMP_KEY_CODES " or in the man page."
#define FIM_CMD_HELP_LIST	FIM_FLT_LIST ": display the files list." FIM_CNS_CMDSEP FIM_FLT_LIST " 'random_shuffle': randomly shuffle the file list." FIM_CNS_CMDSEP FIM_FLT_LIST " 'reverse': reverse the file list." FIM_CNS_CMDSEP FIM_FLT_LIST " 'clear': clear the file list." FIM_CNS_CMDSEP FIM_FLT_LIST " 'sort': sort the file list." FIM_CNS_CMDSEP FIM_FLT_LIST " 'sort_basename': sort the file list according to base name." FIM_CNS_CMDSEP FIM_FLT_LIST " 'sort_comment': sort the file list according to the value of the " FIM_VID_COMMENT " variable." FIM_CNS_CMDSEP FIM_FLT_LIST " 'sort_var' " FIM_CNS_EX_VAR_STRING ": sort the file list according to the value of the i:" FIM_CNS_EX_VAR_STRING " variable." FIM_CNS_CMDSEP FIM_FLT_LIST " 'vars|variables': list variables in all i:* read from description file." FIM_CNS_CMDSEP FIM_FLT_LIST " 'sort_fsize': sort the file list according to file size." FIM_CNS_CMDSEP FIM_FLT_LIST " 'sort_mtime': sort the file list according to modification date. " FIM_CNS_CMDSEP FIM_FLT_LIST " 'pop': remove the current file from the files list, and step back." FIM_CNS_CMDSEP FIM_FLT_LIST " 'remove' [" FIM_CNS_EX_FNS_STRING "]: remove the current file, or the " FIM_CNS_EX_FNS_STRING ", if specified." FIM_CNS_CMDSEP FIM_FLT_LIST " 'push' " FIM_CNS_EX_FNS_STRING ": push " FIM_CNS_EX_FNS_STRING " to the back of the files list." FIM_CNS_CMDSEP FIM_FLT_LIST " 'filesnum': display the number of files in the files list." FIM_CNS_CMDSEP FIM_FLT_LIST " 'mark' [{args}]: mark image file names for stdout printing at exit, with {args} mark the ones matching according to the rules of the '" FIM_FLT_LIMIT "' command, otherwise the current file." FIM_CNS_CMDSEP FIM_FLT_LIST " 'unmark' [{args}]: unmark marked image file names, with {args} unmark the ones matching according to the rules of the '" FIM_FLT_LIMIT "' command, otherwise the current file." FIM_CNS_CMDSEP FIM_FLT_LIST " 'marked': show which files have been marked so far." FIM_CNS_CMDSEP FIM_FLT_LIST " 'dumpmarked': dump to stdout the marked files (you will want usually to 'unmarkall' afterwards)." FIM_CNS_CMDSEP FIM_FLT_LIST " 'markall': mark all the current list files." FIM_CNS_CMDSEP FIM_FLT_LIST " 'unmarkall': unmark all the marked files." FIM_CNS_CMDSEP FIM_FLT_LIST " 'pushdir' {dirname}: will push all the files in {dirname}, when matching the regular expression in variable " FIM_VID_PUSHDIR_RE " or, if empty, from constant regular expression '" FIM_CNS_PUSHDIR_RE "'." FIM_CNS_CMDSEP FIM_FLT_LIST " 'pushdirr' {dirname}: like pushdir, but will also push encountered directory entries recursively." FIM_CNS_CMDSEP FIM_FLT_LIST " 'swap': will move the current image filename to the first in the list (you'll have to invoke " FIM_FLT_RELOAD " to see the effect)." FIM_CNS_CMDSEP "Of the above commands, several will be temporarily non available for the duration of a background load (enabled by --" FIM_OSW_BGREC "), which will last until " FIM_VID_LOADING_IN_BACKGROUND " is 0."
#define FIM_CMD_HELP_CLEAR FIM_FLT_CLEAR ": clear the virtual console."
#define FIM_CMD_HELP_CROP \
	FIM_FLT_CROP ": crop image to a centered rectangle, half the width and half the height.\n" \
	FIM_FLT_CROP " {p}: crop image to the middle {p} horizontal percent and {p} vertical percent of the image.\n" \
	FIM_FLT_CROP " {w} {h}: crop image to the middle {w} horizontal percent and {h} vertical percent of the image.\n" \
	FIM_FLT_CROP " {x1} {y1} {x2} {y2}: crop image to the area between the upper left ({x1},{y1}) and lower right ({x2},{y2}) corner.\n"\
	"Units are intended as percentage (0 to 100).\n" \
	"Note: still experimental functionality."
#define FIM_CMD_HELP_COMMANDS FIM_FLT_COMMANDS ": display the existing commands."
#define FIM_CMD_HELP_COLOR \
			FIM_FLT_COLOR " [desaturate]: desaturate the displayed image colors."\
	FIM_CNS_CMDSEP FIM_FLT_COLOR " [negate]: negate the displayed image colors."\
	FIM_CNS_CMDSEP FIM_FLT_COLOR " [colorblind|c|deuteranopia|d]: simulate a form of the deuteranopia color vision deficiency (cvd)."\
	FIM_CNS_CMDSEP FIM_FLT_COLOR " [protanopia|p]: simulate a form of the protanopia cvd."\
	FIM_CNS_CMDSEP FIM_FLT_COLOR " [tritanopia|t]: simulate a form of the tritanopia cvd."\
	FIM_CNS_CMDSEP FIM_FLT_COLOR " [daltonize|D]: if following a cvd specification, will attempt correcting it."\
	FIM_CNS_CMDSEP FIM_FLT_COLOR " [identity]: populate the image with 'RGB identity' pixels."\
	FIM_CNS_CMDSEP "To get back the original you will have to reload the image." 
#define FIM_CMD_HELP_DUMP_KEY_CODES FIM_FLT_DUMP_KEY_CODES ": dump the active key codes (unescaped, for inspection purposes)."
#define FIM_CMD_HELP_ECHO FIM_FLT_ECHO " " FIM_CNS_EX_ARGS_STRING ": print the " FIM_CNS_EX_ARGS_STRING " on console."
#define FIM_CMD_HELP_ELSE FIM_CNS_IFELSE ": see " FIM_FLT_ELSE "."
#define FIM_CMD_HELP_EXEC FIM_FLT_EXEC " " FIM_CNS_EX_FNS_STRING ": execute script " FIM_CNS_EX_FNS_STRING "."
#define FIM_CMD_HELP_FONT \
		FIM_FLT_FONT " 'scan' [{dirname}]: scan {dirname} or " FIM_LINUX_CONSOLEFONTS_DIR " looking for fonts in the internal fonts list."\
	FIM_CNS_CMDSEP FIM_FLT_FONT " 'load' {filename}: load font {filename}."\
	FIM_CNS_CMDSEP FIM_FLT_FONT " 'next'/'prev': load next/previous font in the internal fonts list."\
	FIM_CNS_CMDSEP FIM_FLT_FONT " 'info': print current font filename."
#define FIM_CMD_HELP_CD			FIM_FLT_CD " " FIM_CNS_EX_PATH_STRING ": change the current directory to " FIM_CNS_EX_PATH_STRING "." FIM_CNS_CMDSEP "If " FIM_CNS_EX_PATH_STRING " is a file, use its base directory name." FIM_CNS_CMDSEP FIM_FLT_CD " '-' will change to the previous current directory (before the last ':" FIM_FLT_CD " " FIM_CNS_EX_PATH_STRING "' command)."
#define FIM_CMD_HELP_GETENV FIM_FLT_GETENV " " FIM_CNS_EX_ID_STRING ": create a variable with the same value as the '" FIM_CNS_EX_ID_STRING "' environment variable, but with an identifier prefixed by 'ENV_'. So e.g. " FIM_FLT_GETENV " 'DISPLAY' creates 'ENV_DISPLAY'. Nothing is being printed; no variable is created if " FIM_CNS_EX_ID_STRING " is empty."
#define FIM_CMD_HELP_IFELSE  FIM_CNS_IFELSE ": see '" FIM_FLT_ELSE "'."
#define FIM_CMD_HELP_INFO   FIM_FLT_INFO ": display information about the current file."
// #define FIM_CMD_HELP_LIMIT   // missing
#define FIM_CMD_HELP_LOAD   FIM_FLT_LOAD ": load the image, if not yet loaded (see also '" FIM_FLT_RELOAD "')." FIM_CMD_ACM_INFO(FIM_ACM_PRELOAD,FIM_ACM_POSTLOAD)
#define FIM_CMD_HELP_PAN    \
	FIM_FLT_PAN " {vsteps}% {hsteps}%: pan the image to {vsteps} percentage steps from the top and {hsteps} percentage steps from left. " FIM_CNS_CMDSEP \
	FIM_FLT_PAN " {vsteps} {hsteps}: pan the image to {vsteps} pixels from the top and {hsteps} pixels from left. " FIM_CNS_CMDSEP \
	FIM_FLT_PAN " {'down'|'up'|'left'|'right'|'ne'|'nw'|'se'|'sw'}[+-] [{steps}['%']]: pan the image {steps} pixels in the desired direction." FIM_CNS_CMDSEP "If the '%' specifier is present, {steps} will be treated as a percentage of current screen dimensions." FIM_CNS_CMDSEP "If {steps} is not specified, the '" FIM_VID_STEPS "' variable will be used." FIM_CNS_CMDSEP "If present, the '" FIM_VID_HSTEPS "' variable will be considered for horizontal panning." FIM_CNS_CMDSEP "A '+' or '-' sign at the end of the first argument will make jump to next or prev if border is reached." FIM_CNS_CMDSEP "If present, the '" FIM_VID_VSTEPS "' variable will be considered for vertical panning." FIM_CNS_CMDSEP "The variables may be terminated by the \'%\' specifier." FIM_CMD_ACM_INFO(FIM_ACM_PREPAN,FIM_ACM_POSTPAN)
#define FIM_CMD_HELP_POPEN FIM_FLT_POPEN " " FIM_CNS_EX_SYSC_STRING ": pipe a command, invoking popen(): spawns a shell, invoking '" FIM_CNS_EX_SYSC_STRING "' and executing as fim commands the output of '" FIM_CNS_EX_SYSC_STRING "'."
#define FIM_CMD_HELP_PREAD FIM_FLT_PREAD " " FIM_CNS_EX_ARGS_STRING ": execute " FIM_CNS_EX_ARGS_STRING " as a shell command and read the output as an image file (using '" FIM_FLT_POPEN "')."
#define FIM_CMD_HELP_PREFETCH FIM_FLT_PREFETCH ": prefetch (read into the cache) the two nearby image files (next and previous), for a faster subsequent opening." FIM_CMD_ACM_INFO(FIM_ACM_PREPREFETCH,FIM_ACM_POSTPREFETCH) FIM_CNS_CMDSEP "See also the '" FIM_VID_WANT_PREFETCH "' variable."
#define FIM_CMD_HELP_QUIT FIM_FLT_QUIT " [{number}]: terminate the program." FIM_CNS_CMDSEP "If {number} is specified, use it as the program return status." FIM_CNS_CMDSEP "Note that autocommand '" FIM_ACM_POSTINTERACTIVECOMMAND "' does not trigger after this command."
#define FIM_CMD_HELP_RECORDING FIM_FLT_RECORDING " 'start': start recording the executed commands. " FIM_CNS_CMDSEP FIM_FLT_RECORDING " 'stop': stop  recording the executed commands." FIM_CNS_CMDSEP FIM_FLT_RECORDING " 'dump': dump in the console the record buffer." FIM_CNS_CMDSEP FIM_FLT_RECORDING " 'execute': execute the record buffer." FIM_CNS_CMDSEP FIM_FLT_RECORDING " 'repeat_last': repeat the last performed action."
#define FIM_CMD_HELP_REDISPLAY  FIM_FLT_REDISPLAY ": re-display the current file contents."
#define FIM_CMD_HELP_RELOAD FIM_FLT_RELOAD" [{arg}]: load the image into memory." FIM_CNS_CMDSEP "If {arg} is present, will force reloading, bypassing the cache (see also '" FIM_FLT_LOAD "')." FIM_CMD_ACM_INFO(FIM_ACM_PRERELOAD,FIM_ACM_POSTRELOAD)
#define FIM_CMD_HELP_ROTATE FIM_FLT_ROTATE " " FIM_CNS_EX_NUM_STRING ": rotate the image the specified amount of degrees. If unspecified, by one. If you are interested in orthogonal rotations, see '" FIM_VID_ORIENTATION "' and related aliases." FIM_CMD_ACM_INFO(FIM_ACM_PRESCALE,FIM_ACM_POSTSCALE)
#define FIM_CMD_HELP_SCROLL FIM_FLT_SCROLL ": scroll down the image, going next when hitting the bottom." FIM_CNS_CMDSEP FIM_FLT_SCROLL " 'forward': scroll the image as we were reading left to right (see '" FIM_VID_SKIP_SCROLL "' variable)." FIM_CMD_ACM_INFO(FIM_ACM_PREPAN,FIM_ACM_POSTPAN)
#define FIM_CMD_HELP_SET_CONSOLE_MODE FIM_FLT_SET_CONSOLE_MODE ": set console mode."
#define FIM_CMD_HELP_SET_INTERACTIVE_MODE FIM_FLT_SET_INTERACTIVE_MODE ": set interactive mode."
#define FIM_CMD_HELP_SLEEP FIM_FLT_SLEEP " [" FIM_CNS_EX_NUM_STRING "=1]: sleep for the specified (default 1) number of seconds."
#define FIM_CMD_HELP_STATUS FIM_FLT_STATUS ": set the status line to the collation of the given arguments."
#define FIM_CMD_HELP_STDERR FIM_FLT_STDERR " " FIM_CNS_EX_ARGS_STRING ": writes to stderr its arguments " FIM_CNS_EX_ARGS_STRING "."
#define FIM_CMD_HELP_STDOUT FIM_FLT_STDOUT " " FIM_CNS_EX_ARGS_STRING ": writes to stdout its arguments " FIM_CNS_EX_ARGS_STRING "."
#define FIM_CMD_HELP_VARIABLES FIM_FLT_VARIABLES ": display the existing variables."
#define FIM_CMD_HELP_UNALIAS FIM_FLT_UNALIAS " " FIM_CNS_EX_ID_STRING " | '-a': delete the alias " FIM_CNS_EX_ID_STRING " or all aliases (use '-a', not -a)."
#define FIM_CMD_HELP_UNBIND FIM_FLT_UNBIND " " FIM_CNS_EX_KSY_STRING ": unbind the action associated to a specified " FIM_CNS_EX_KSY_STRING FIM_CNS_CMDSEP FIM_CNS_RAW_KEYS_MESG ""
#define FIM_CMD_HELP_WHILE "while(expression){action;}: A conditional cycle construct." FIM_CNS_CMDSEP "May be interrupted by hitting the '" FIM_KBD_ESC "' or the '" FIM_KBD_COLON "' key."
#define FIM_CMD_HELP_SET	FIM_FLT_SET ": returns a list of variables which are set." FIM_CNS_CMDSEP FIM_FLT_SET " " FIM_CNS_EX_ID_STRING ": returns the value of variable " FIM_CNS_EX_ID_STRING "." FIM_CNS_CMDSEP FIM_FLT_SET " " FIM_CNS_EX_ID_STRING " " FIM_CNS_EX_CMDS_STRING ": sets variable " FIM_CNS_EX_ID_STRING " to value " FIM_CNS_EX_CMDS_STRING "." 
#define FIM_CMD_HELP_PWD	FIM_FLT_PWD ": print the current directory name, and updates the '" FIM_VID_PWD "' variable."
#define FIM_CMD_HELP_EVAL	FIM_FLT_EVAL " " FIM_CNS_EX_ARGS_STRING ": evaluate " FIM_CNS_EX_ARGS_STRING " as commands, executing them."
#define FIM_CMD_HELP_SYSTEM	FIM_FLT_SYSTEM " " FIM_CNS_EX_SYSC_STRING ": get the output of executing the " FIM_CNS_EX_SYSC_STRING " system command. Uses the popen() system call. Usually popen invokes \"" FIM_CNS_SHELL " -c " FIM_CNS_EX_SYSC_STRING "\". This might not handle a multi-word command; in that case you will have to put it into a script. See 'man popen' for more."
#if FIM_DISABLE_WINDOW_SPLITTING
#define FIM_CMD_HELP_WINDOW FIM_FLT_WINDOW " " FIM_CNS_EX_ARGS_STRING ": this command is disabled." /*FIM_CNS_CMDSEP " It might be re-enabled in a future version."*/
#else /* FIM_DISABLE_WINDOW_SPLITTING */
#define FIM_CMD_HELP_WINDOW FIM_FLT_WINDOW " " FIM_CNS_EX_ARGS_STRING ": manipulates the window system windows; each value of " FIM_CNS_EX_ARGS_STRING " shall be one of ['split' | 'hsplit' | 'vsplit' | 'normalize' | 'enlarge' | 'venlarge' | 'henlarge' | 'up' | 'down' | 'left' | 'right' | 'close' | 'swap']."
#endif /* FIM_DISABLE_WINDOW_SPLITTING */
#define FIM_CMD_HELP_GOTO 	FIM_FLT_GOTO " {['+'|'-']" FIM_CNS_EX_NUM_STRING "['%']['f'|'p'|'F'|'P']} | {/" FIM_CNS_EX_RE_STRING "/} | {" FIM_SYM_BW_SEARCH_KEY_STR FIM_CNS_EX_FN_STRING "} | {'+//'} | {'+/'|'-/'}[C] | {{'+'|'-'}" FIM_CNS_EX_ID_STRING "['+']}: jump to an image." FIM_CNS_CMDSEP "If " FIM_CNS_EX_NUM_STRING " is given, and not surrounded by any specifier, will go to image at index " FIM_CNS_EX_NUM_STRING "." FIM_CNS_CMDSEP "If followed by '%', the effective index will be computed as a percentage to the current available images." FIM_CNS_CMDSEP "If prepended by '-' or '+', the jump will be relative to the current index." FIM_CNS_CMDSEP "The 'f' specifier asks for the jump to occur within the files (same for 'F', but accelerates if keep pressing)." FIM_CNS_CMDSEP "The 'p' specifier asks for the jump to occur in terms of pages, within the current file (same for 'P', but accelerates if keep pressing)." FIM_CNS_CMDSEP "If /" FIM_CNS_EX_RE_STRING "/ is given, will jump to the first image matching the given /" FIM_CNS_EX_RE_STRING "/ regular expression pattern." FIM_CNS_CMDSEP "If the argument starts with " FIM_SYM_BW_SEARCH_KEY_STR ", jump to the filename following " FIM_SYM_BW_SEARCH_KEY_STR "." FIM_CNS_CMDSEP "If given '+//', will jump to the first different image matching the last given regular expression pattern." FIM_CNS_CMDSEP "With '+/'C or '-/'C will jump to next or previous file according to C: if 's' if same directory, if 'd' if down the directory hierarchy, if 'u' if down the directory hierarchy, if 'b' if same basename, if upper case match is negative, if missing defaults to 'S' (jump to file in different dir)." FIM_CNS_CMDSEP "If " FIM_CNS_EX_IDS_STRING " is encountered after a '+' or '-' sign, jump to next or previous image having a different value for any corresponding i:" FIM_CNS_EX_ID_STRING " (a trailing '+' will require a non empty value)." FIM_CNS_CMDSEP "Match will occur on both file name and description, possibly loaded via " FIM_FLT_DESC " or --" FIM_OSW_LOAD_IMG_DSC_FILE "; see also '" FIM_VID_LASTGOTODIRECTION "' and '" FIM_VID_RE_SEARCH_OPTS "'." FIM_CNS_CMDSEP " You can specify multiple arguments to " FIM_FLT_GOTO ": those after the first triggering a jump will be ignored." FIM_CMD_ACM_INFO(FIM_ACM_PREGOTO,FIM_ACM_POSTGOTO) " Keeping pressed shall accelerate images browsing."
#define FIM_CMD_HELP_SCALE	FIM_FLT_SCALE " {['+'|'-']{value}['%']|'*'{value}|'w'|'h'|'a'|'b'|'+[+-*/]'|['<'|'>']|'shadow'}: scale the image according to a scale {value} (e.g.: 0.5,40%,'w','h','a','b')." FIM_CNS_CMDSEP "If given '*' and a value, will multiply the current scale by that value." FIM_CNS_CMDSEP "If given 'w', will scale according to the screen width." FIM_CNS_CMDSEP "If given 'h', scale to the screen height." FIM_CNS_CMDSEP "If given 'a', to the minimum of 'w' and 'h'." FIM_CNS_CMDSEP "If given 'b', like 'a', provided that the image width exceeds 'w' or 'h'." FIM_CNS_CMDSEP "If {value} is a number, will scale relatively to the original image width." FIM_CNS_CMDSEP "If the number is followed by '%', the relative scale will be treated on a percent scale." FIM_CNS_CMDSEP "If given '++'('+-'), will increment (decrement) the '" FIM_VID_MAGNIFY_FACTOR "', '" FIM_VID_REDUCE_FACTOR "' variables by '" FIM_VID_SCALE_FACTOR_DELTA "'." FIM_CNS_CMDSEP "If given '+*'('+/'), will multiply (divide) the '" FIM_VID_MAGNIFY_FACTOR "', '" FIM_VID_REDUCE_FACTOR "' variables by '" FIM_VID_SCALE_FACTOR_MULTIPLIER "'." FIM_CNS_CMDSEP "If given '<' or '>', will shrink or magnify image using nearest mipmap (cached pre-scaled version). If given 'shadow' as a parameter, search a same-named file in one of the directories specified to --load-shadow-dir." FIM_CMD_ACM_INFO(FIM_ACM_PRESCALE,FIM_ACM_POSTSCALE)
#define FIM_CMD_HELP_HELP	FIM_FLT_HELP " [" FIM_CNS_EX_ID_STRING "]: provide online help, assuming " FIM_CNS_EX_ID_STRING " is a variable, alias, or command identifier." FIM_CNS_CMDSEP "If " FIM_CNS_EX_ID_STRING " begins with '" FIM_CNS_SLASH_STRING "', the search will be on the help contents and a list of matching items will be given instead." FIM_CNS_CMDSEP "A list of commands can be obtained simply invoking '" FIM_FLT_COMMANDS "'; a list of aliases with '" FIM_FLT_ALIAS "'; a list of bindings with '" FIM_FLT_BIND "'."
#define FIM_FLT_HELP_DESC FIM_FLT_DESC " 'load' {filename} [{sepchar}]: load description file {filename}, using the optional {sepchar} character as separator." FIM_CNS_CMDSEP FIM_FLT_DESC " 'reload': load once again description files specified at the command line with --" FIM_OSW_LOAD_IMG_DSC_FILE ", with respective separators." FIM_CNS_CMDSEP FIM_FLT_DESC " ['-all'] ['-append'] ['-nooverw'] 'save' {filename} [{sepchar}]: save current list descriptions to file {filename}, using the optional {sepchar} character as separator, and if '-all' is present will save the variables, and if '-append' is present, will only append, and if '-nooverw' is present, will not overwrite existing files." FIM_CNS_CMDSEP "See documentation of --" FIM_OSW_LOAD_IMG_DSC_FILE " for the format of {filename}."
#define FIM_FLT_HELP_DISPLAY FIM_FLT_DISPLAY " ['reinit' {string}]|'resize' {w} {h}]: display the current file contents." FIM_CNS_CMDSEP "If 'reinit' switch is supplied, the '{string}' specifier will be used to reinitialize (e.g.: change resolution, window system options) the display device." FIM_CNS_CMDSEP "See documentation for the --" FIM_OSW_OUTPUT_DEVICE " command line switch for allowed values of {string}." FIM_CNS_CMDSEP "If 'resize' and no argument, will ask the window manager to size the window like the image." FIM_CNS_CMDSEP "If 'resize' and two arguments, these will be used as width and height of window, to set."

/*  */
#define FIM_CNS_SLIDESHOW_CMD FIM_FLT_RELOAD "; i:fresh=1; while(" FIM_VID_FILEINDEX " <= " FIM_VID_FILELISTLEN "-" FIM_VID_LOOP_ONCE "){sleep " FIM_VID_WANT_SLEEPS "; next;} sleep " FIM_VID_WANT_SLEEPS "; "
#define FIM_CNS_QUIET_CMD FIM_VID_DISPLAY_STATUS "=0;" FIM_VID_DISPLAY_BUSY "=0;"

/*
 * Some Fim compilation defaults
 */
#define FIM_WANT_SDL_PROOF_OF_CONCEPT_MOUSE_SUPPORT 1 /* experimental mouse support in sdl mode */
#define FIM_WANT_SCREEN_KEY_REMAPPING_PATCH 1
#define FIM_WANT_OVERLY_VERBOSE_DUMB_CONSOLE 0
#define FIM_WANT_MILDLY_VERBOSE_DUMB_CONSOLE 1
#define FIM_WANT_SINGLE_SYSTEM_INVOCATION 1
#define FIM_WANT_SDL_OPTIONS_STRING 1
#define FIM_WANT_OUTPUT_DEVICE_STRING_CASE_INSENSITIVE 1
#define FIM_WANT_HISTORY 1
#define FIM_WANT_AVOID_FP_EXCEPTIONS 1
#define FIM_WANT_CAPTION_CONTROL	1
#define FIM_WANT_READLINE_CLEAR_WITH_ESC	1
#define FIM_WANT_DOUBLE_ESC_TO_ENTER 0	/* if enabled in the console mode, would require three presses the first time; two later on; this is non consistent, so we keep it disabled until we find a fix */
#define FIM_WANT_EXPERIMENTAL_PLUGINS 1	/* unfinished */
#define FIM_WANT_STDIN_FILELOAD_AFTER_CONFIG 1
#define FIM_WANT_KEEP_FILESIZE 1
#define FIM_WANT_DISPLAY_FILESIZE (FIM_WANT_KEEP_FILESIZE && 0)
#define FIM_WANT_DISPLAY_MEMSIZE  0
#define FIM_WANT_APPROXIMATE_EXPONENTIAL_SCALING  FIM_WANT_MIPMAPS && 1
#define FIM_WANT_R_SWITCH  FIM_WITH_LIBSDL /* --resolution width:height switch (the fim way, differently from fbi's) */
#define FIM_WANT_RECURSE_FILTER_OPTION 1 /* Enable -R=... */
#define FIM_WANT_NOEXTPROPIPE 1 /* */
/* #define FIM_WANT_CUSTOM_INFO_STRING  1 */
#if defined(FIM_FRAMEBUFFER_DEVICE_H) && defined(FIM_WANT_FB_CONSOLE_SWITCH_WHILE_LOADING) /* only needed in Linux framebuffer device on computers loading files so slowly that we might want to switch console in between */
#define FIM_WANT_CONSOLE_SWITCH_WHILE_LOADING 1
#endif
#define FIM_WANT_RELOAD_ON_FILE_CHANGE 1 /* detect file change and reload */
#define FIM_WANT_NEXT_ACCEL 1 /* keeping pressed accelerates browse speed */
#define FIM_STREAM_BUFSIZE	16*4096
#define FIM_MAXLINE_BUFSIZE	1024
#define FIM_PRINTFNUM_BUFSIZE	32
#define FIM_PRINTINUM_BUFSIZE	4*sizeof(fim_int) /* for fim_int */
#define FIM_METAINFO_BUFSIZE	256
#define FIM_EXIF_BUFSIZE FIM_METAINFO_BUFSIZE
#define FIM_RL_COMPLETION_BUFSIZE	256
#define FIM_LIBERR_BUFSIZE	1024
#define FIM_LINUX_LINKFILENAME_BUFSIZE 64
#define FIM_STRING_BUFSIZE	4096
#define FIM_PIPE_BUFSIZE	1024
#define FIM_PIPE_CMD_BUFSIZE	1024
#define FIM_FILE_PROBE_BLKSIZE	512
#define FIM_CONSOLE_BLOCKSIZE	1024
#define FIM_CONSOLE_DEF_WIDTH	128
#define FIM_BITRENDERING_DEF_WIDTH	1024
#define FIM_DEFAULT_WINDOW_WIDTH	640
#define FIM_DEFAULT_WINDOW_HEIGHT	512
#define FIM_RENDERING_DPI	200
#define FIM_RENDERING_MAX_ROWS	1024
#define FIM_RENDERING_MAX_COLS	1024
#define FIM_CONSOLE_DEF_ROWS	24
#define FIM_VERBOSE_KEYS_BUFSIZE	64
#define FIM_FILE_BUF_SIZE 	(1024*256)
#define FIM_ATOX_BUFSIZE 	(32)
#define FIM_STATUSLINE_BUF_SIZE 	(2*128)
#define FIM_IMG_DESC_ITEM_MAX_LEN 	(16*FIM_STATUSLINE_BUF_SIZE)
#define FIM_FBI_PPM_LINEBUFSIZE 	(1024)
#define FIM_LINUX_CONSOLEFONTS_DIR "/usr/share/consolefonts"
#define FIM_LINUX_STDIN_FILE "/dev/stdin"
#define FIM_LINUX_STDOUT_FILE "/dev/stdout"
#define FIM_LINUX_RAND_FILE "/dev/urandom"
#define FIM_FBDEV_FILE_MAX_CHARS 16
#define FIM_DEFAULT_FB_FILE "/dev/fb0"
#define FIM_DEFAULT_AS_BINARY_BPP 24
#define FIM_DEFAULT_HARDCODEDFONT_STRING	"fim://" 
#if __cplusplus < 201703L
#define FIM_REGISTER register
#else
#define FIM_REGISTER 
#endif

/*
 * Various  Fim messages.
 */
#define FIM_MSG_CONSOLE_FIRST_LINE_BANNER "=== This is the fim output console. You can scroll it up and down. ===\n"
#define FIM_MSG_CONSOLE_LONG_LINE "\n============================================================\n"

/*
 * Some Fim internals flags.
 */
#define FIM_FLAG_MIRROR 1
#define FIM_FLAG_FLIP 2
#define FIM_FLAG_RGB2GRAY 4
#define FIM_FLAG_RGB2GRAYGRAYGRAY 8

/* FIM_XSTRINGIFY evaluates to a string with the supplied preprocessor symbol value */
#define FIM_XSTRINGIFY(X) FIM_STRINGIFY(X)
/* FIM_STRINGIFY evaluates to a string with the supplied preprocessor symbol identifier */
#define FIM_STRINGIFY(X) #X
#define FIM_MAN_Bn(X) ".B " X "\n.fi \n" /* bold newline */
#define FIM_MAN_fB(X) "\\fB" X "\\fP" /* bold inline */
#define FIM_MAN_fR(X) "\\fR\\fI" X "\\fR"
#define FIM_MAN_iB "\\:" /* invisible break */

#endif /* FIM_FIM_H */
