/****************************************************************************
 *    lib/c/Parser.y - This file is part of coala							*
 *																			*
 *    Copyright (C) 2009  Torsten Grote										*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/
%name Parser

%define LSP_NEEDED
%define ERROR_BODY = 0
%define LEX_BODY = 0
%define MEMBERS void preProcessing(Coala::CompilerOptions*); void postProcessing(Printer*); virtual ~Parser(){}\
	private: std::string to_string(int integer);

%header{
#include <stdio.h>
#include <iostream>
#include <string>
#include <vector>
#include <algorithm>

#include "../options.h"
#include "Program.h"
#include "Printer.h"

using namespace std;
using namespace C;
%}

%{
Program* program;
%}

%union {
	int number;
	string* str;
	Formula* form;
	Type* typ;
	Types* types;
	Arguments* args;
	Constraint* constr;
	Pool* var_pool;
	Interval* interval;
	LTLNode* ltl_node;
}

%token <str> IDENTIFIER
%token <str> NEG_IDENTI
%token <str> VARIABLE
%token <number> NUMBER
%token <number> ACT		// <action>
%token <number> FLU		// <fluent>
%token <number> CAUS	// <caused>
%token <number> IF		// <if>
%token <number> AFTER	// <after>
%token <number> TRUE	// <true>
%token <number> FALSE	// <false>
%token <number> DEF		// <default>
%token <number> INERT	// <inertial>
%token <number> CAUSES	// <causes>
%token <number> ALWAYS	// <always>
%token <number> MAY		// <may cause>
%token <number> NONEXE	// <nonexecutable>
%token <number> WHERE	// <where>
%token <number> OC_AT	// <occurs at>
%token <number> HO_AT	// <holds at>
%token <number> OCCURS	// <occurs>
%token <number> HOLDS	// <holds>
%token <number> NECES	// <necessarily>
%token <number> AT		// <at>
%token <number> NOT		// not
%token <number> COMMA	// ,
%token <number> SEMIC	// ;
%token <number> COLON	// :
%token <number> DOT		// .
%token <number> DDOT	// ..
%token <number> LBRAC	// (
%token <number> RBRAC	// )
%token <number> EQ		// ==
%token <number> NEQ		// !=
%token <number> LT		// <
%token <number> GT		// >
%token <number> LE		// <=
%token <number> GE		// >=
%token <number> PLUS	// +
%token <number> MINUS	// -
%token <number> TIMES	// *
%token <number> DIV		// /
%token <number> MOD		// mod
%token <number> AND		// &
%token <number> OR		// ?
%token <number> XOR		// ^
%token <number> LTL		// LTL:
%token <number> LTLNOT	// !
%token <number> LTLOR	// |
%token <number> IMPL	// ->
%token <number> EQUIV	// <->
%token <number> X		// X
%token <number> F		// F
%token <number> G		// G
%token <number> U		// U
%token <number> R		// R

// LTL Operator Precedence
%left EQUIV IMPL LTLOR AND
%left U R
%nonassoc G F X
%nonassoc LTLNOT

%type <form>		formula
%type <types>		types_list
%type <typ>			type
%type <args>		term_list
%type <form>		identifier
%type <constr>		constraint
%type <var_pool>	pool
%type <interval>	interval
%type <constr>		operation
%type <str>			operator
%type <str>			eqoperator
%type <ltl_node>	ltl_expr

%start program

%%
/***************************************************************************************************************************/

program: /* empty */ | program rule DOT;

rule: fact | law | shortcut | query;

fact: act_fact | flu_fact;

law: static_rule | dyn_rule;

shortcut: def_rule | inert_rule | caus_rule | always_rule | may_rule | nonexe_rule;

query: oc_query | ho_query | hypo_query | ltl_query;

// <action> a_1,...,a_n <where> bla.
act_fact:
  ACT formula					{ program->addActionDef($2); }
| ACT formula WHERE types_list	{ program->addActionDef($2, $4); }

// <fluent> f_1,...,f_n <where> bla.
flu_fact:
  FLU formula					{ program->addFluentDef($2); }
| FLU formula WHERE types_list	{ program->addFluentDef($2, $4); }

// <caused> f_1,...,f_n <if> g_1,...,g_m <where> bla.
static_rule:
  CAUS formula								{ program->addStaticRule($2,	NULL); }
| CAUS formula WHERE types_list				{ program->addStaticRule($2,	NULL, $4); }
| CAUS formula IF formula					{ program->addStaticRule($2,	$4); }
| CAUS formula IF formula WHERE types_list	{ program->addStaticRule($2,	$4, $6); }

// <caused> f_1,...,f_n <if> g_1,...,g_m <after> h_1,...,h_l <where> bla.
dyn_rule:
  CAUS formula AFTER formula								{ program->addDynamicRule($2, NULL, $4); }
| CAUS formula AFTER formula WHERE types_list				{ program->addDynamicRule($2, NULL, $4, $6); }
| CAUS formula IF formula AFTER formula						{ program->addDynamicRule($2, $4, $6); }
| CAUS formula IF formula AFTER formula WHERE types_list	{ program->addDynamicRule($2, $4, $6, $8); }

// <default> f_1,...,f_n <if> g_1,...,g_m <where> bla.	=	<caused> f_1 <if> f_1, g_1,...,g_m <where> bla.
def_rule:
  DEF formula								{ program->addDefaultRule($2, NULL); }
| DEF formula WHERE types_list				{ program->addDefaultRule($2, NULL, $4); }
| DEF formula IF formula					{ program->addDefaultRule($2, $4); }
| DEF formula IF formula WHERE types_list	{ program->addDefaultRule($2, $4, $6); }

// <inertial> f_1, ..., f_n <where> bla. 	=	<caused> f <if> f <after> f <where> bla.
inert_rule:
  INERT formula						{ program->addInertialRule($2); }
| INERT formula WHERE types_list	{ program->addInertialRule($2, $4); }

// a1,...,a_o <causes> f_1, ..., f_n <if> g_1, ..., g_m <where> bla.	=	<caused> f <if> <true> <after> g, a <where> bla.
caus_rule:
  formula CAUSES formula								{ program->addCausesRule($1, $3, NULL); }
| formula CAUSES formula WHERE types_list				{ program->addCausesRule($1, $3, NULL, $5); }
| formula CAUSES formula IF formula						{ program->addCausesRule($1, $3, $5); }
| formula CAUSES formula IF formula WHERE types_list	{ program->addCausesRule($1, $3, $5, $7); }

// <always> f_1, ..., f_n <where> bla.	=	<caused> <false> <if> -f_1, ..., -f_n <where> bla.
always_rule:
| ALWAYS formula					{ program->addAlwaysRule($2); }
| ALWAYS formula WHERE types_list	{ program->addAlwaysRule($2, $4); }

// a_1,...,a_n <may cause> f_1,...,f_m <if> g_1,...,g_l <where> bla. = <caused> f_1 <if> f_1 <after> g_1,...,g_l,a_1,...,a_n <where>  bla.
may_rule:
  formula MAY formula								{ program->addMayCauseRule($1, $3, NULL); }
| formula MAY formula WHERE types_list				{ program->addMayCauseRule($1, $3, NULL, $5); }
| formula MAY formula IF formula					{ program->addMayCauseRule($1, $3, $5); }
| formula MAY formula IF formula WHERE types_list	{ program->addMayCauseRule($1, $3, $5, $7); }

// <nonexecutable> a_1,..., a_n <if> f_1,..., f_m <where> bla.	=	<caused> <false> <after> a_1,...,a_n,f_1,...,f_m <where> bla.
nonexe_rule:
  NONEXE formula								{ program->addNonexecutableRule($2, NULL); }
| NONEXE formula WHERE types_list				{ program->addNonexecutableRule($2, NULL, $4); }
| NONEXE formula IF formula						{ program->addNonexecutableRule($2, $4); }
| NONEXE formula IF formula WHERE types_list	{ program->addNonexecutableRule($2, $4, $6); }

// a_1,...,a_n <occurs at> t_i <where> bla.
oc_query:
  formula OC_AT NUMBER						{ program->addQQuery("occurs", $1, to_string($3)); }
| formula OC_AT IDENTIFIER					{ program->addQQuery("occurs", $1, *$3); }
| formula OC_AT NUMBER WHERE types_list		{ program->addQQuery("occurs", $1, to_string($3), $5); }
| formula OC_AT IDENTIFIER WHERE types_list	{ program->addQQuery("occurs", $1, *$3, $5); }

// f_1,...,f_n <holds at> t_i <where> bla.
ho_query:
  formula HO_AT NUMBER						{ program->addQQuery("holds", $1, to_string($3)); }
| formula HO_AT IDENTIFIER					{ program->addQQuery("holds", $1, *$3); }
| formula HO_AT NUMBER WHERE types_list		{ program->addQQuery("holds", $1, to_string($3), $5); }
| formula HO_AT IDENTIFIER WHERE types_list	{ program->addQQuery("holds", $1, *$3, $5); }

// necessarily f_1,...,f_n after A1,...,Ak at ti <where> bla.
hypo_query:
  NECES formula AFTER formula AT NUMBER						{ program->addRQuery($2, $4, to_string($6)); }
| NECES formula AFTER formula AT NUMBER WHERE types_list 	{ program->addRQuery($2, $4, to_string($6), $8); }

// f_1,...,f_n
formula:
  identifier					{ $$ = program->addFormula('c', $1); }
| TRUE							{ $$ = program->addFormula('c', new FluentAction(new string("0"))); }
| FALSE							{ $$ = program->addFormula('c', new Formula('n', new FluentAction(new string("0")))); }
| formula COMMA identifier		{ $1->addPart($3); $$ = $1; }

identifier:
  IDENTIFIER						{ $$ = new FluentAction($1); }
| IDENTIFIER LBRAC term_list RBRAC	{ $$ = new FluentAction($1, $3); }
| NEG_IDENTI						{ $$ = new Formula('n', new FluentAction($1)); }
| NEG_IDENTI LBRAC term_list RBRAC	{ $$ = new Formula('n', new FluentAction($1, $3)); }

// X, a, 3, X;Y, 1..Z
term_list:
  VARIABLE											{ $$ = new Arguments(program->addVariable($1)); }
| IDENTIFIER										{ $$ = new Arguments(new Constant($1)); }
| NUMBER											{ $$ = new Arguments(new Constant($1)); }
| pool												{ $$ = new Arguments($1); }
| interval											{ $$ = new Arguments($1); }
| IDENTIFIER LBRAC term_list RBRAC					{ $$ = new Arguments(new Function($1, $3->getArgs())); }
| term_list COMMA VARIABLE							{ $1->addVariable(program->addVariable($3)); $$ = $1; }
| term_list COMMA IDENTIFIER						{ $1->addConstant(new Constant($3)); $$ = $1; }
| term_list COMMA NUMBER							{ $1->addConstant(new Constant($3)); $$ = $1; }
| term_list COMMA pool								{ $1->addPool($3); $$ = $1; }
| term_list COMMA interval							{ $1->addInterval($3); $$ = $1; }
| term_list COMMA IDENTIFIER LBRAC term_list RBRAC	{ $1->addFunction(new Function($3, $5->getArgs())); $$ = $1; }

// X;Y
pool:
  VARIABLE SEMIC VARIABLE	{ $$ = new Pool(program->addVariable($1), program->addVariable($3)); }
| pool SEMIC VARIABLE		{ $1->addVariable(program->addVariable($3)); $$ = $1; }

// 1..X
interval:
  NUMBER DDOT NUMBER		{ $$ = new Interval(new Constant($1), new Constant($3)); }
| NUMBER DDOT VARIABLE		{ $$ = new Interval(new Constant($1), program->addVariable($3)); }
| VARIABLE DDOT NUMBER		{ $$ = new Interval(program->addVariable($1), new Constant($3)); }
| VARIABLE DDOT VARIABLE	{ $$ = new Interval(program->addVariable($1), program->addVariable($3)); }

// type_1(X),...,type_n(Y), X!=Y
types_list:
  constraint						{ $$ = new Types($1); }
| type								{ $$ = new Types($1); }
| NUMBER COLON type					{ $3->setMin($1); $$ = new Types($3); }
| type COLON NUMBER					{ $1->setMax($3); $$ = new Types($1); }
| NUMBER COLON type COLON NUMBER	{ $3->setMin($1); $3->setMax($5); $$ = new Types($3); }
| types_list COMMA constraint						{ $1->addType($3); }
| types_list COMMA type								{ $1->addType($3); }
| types_list COMMA NUMBER COLON type				{ $5->setMin($3); $1->addType($5); }
| types_list COMMA type COLON NUMBER				{ $3->setMax($5); $1->addType($3); }
| types_list COMMA NUMBER COLON type COLON NUMBER	{ $5->setMin($3); $5->setMax($7); $1->addType($5); }

// type_xyz(X,Y,Z)
type:
  IDENTIFIER LBRAC term_list RBRAC		{ $$ = new Type($1, $3, true); }
| NOT IDENTIFIER LBRAC term_list RBRAC	{ $$ = new Type($2, $4, false); }

// X!=Y etc.
constraint:
  operation eqoperator operation	{ $1->merge($3, $2); $$ = $1; }
| NUMBER eqoperator operation		{ $3->mergeLeft(new string(to_string($1)+*$2), $2); $$ = $3; }
| operation eqoperator NUMBER		{ $1->merge(new string(*$2+to_string($3)), $2); $$ = $1; }
| IDENTIFIER eqoperator operation	{ $3->mergeLeft(new string(*$1+*$2), $2); $$ = $3; }
| operation eqoperator IDENTIFIER	{ $1->merge(new string(*$2+*$3), $2); $$ = $1; }
| VARIABLE eqoperator operation		{ $3->mergeLeft(new string(*$1+*$2), program->addVariable($1), $2); $$ = $3; }
| operation eqoperator VARIABLE		{ $1->merge(new string(*$2+*$3), program->addVariable($3), $2); $$ = $1; }
| VARIABLE eqoperator VARIABLE		{ $$ = new Constraint(program->addVariable($1), program->addVariable($3), new string(*$1+*$2+*$3), $2); }
| NUMBER eqoperator VARIABLE		{ $$ = new Constraint(program->addVariable($3), new string(to_string($1)+*$2+*$3), $2); }
| VARIABLE eqoperator NUMBER		{ $$ = new Constraint(program->addVariable($1), new string(*$1+*$2+to_string($3)), $2); }
| IDENTIFIER eqoperator VARIABLE	{ $$ = new Constraint(program->addVariable($3), new string(*$1+*$2+*$3), $2); }
| VARIABLE eqoperator IDENTIFIER	{ $$ = new Constraint(program->addVariable($1), new string(*$1+*$2+*$3), $2); }
| NUMBER eqoperator IDENTIFIER		{ $$ = new Constraint(new string(to_string($1)+*$2+*$3), $2); }
| IDENTIFIER eqoperator NUMBER		{ $$ = new Constraint(new string(*$1+*$2+to_string($3)), $2); }

// X+1+Y
operation:
  NUMBER operator VARIABLE		{ $$ = new Constraint(program->addVariable($3), new string(to_string($1)+*$2+*$3)); }
| VARIABLE operator NUMBER		{ $$ = new Constraint(program->addVariable($1), new string(*$1+*$2+to_string($3))); }
| VARIABLE operator VARIABLE	{ $$ = new Constraint(program->addVariable($1), program->addVariable($3), new string(*$1+*$2+*$3)); }
| operation operator NUMBER		{ $1->merge(new string(*$2+to_string($3))); $$ = $1; }
| operation operator VARIABLE	{ $1->merge(new string(*$2+*$3), program->addVariable($3)); $$ = $1; }

eqoperator:
  EQ 	{ $$ = new string("=="); }
| NEQ	{ $$ = new string("!="); }
| LT	{ $$ = new string("<"); }
| GT	{ $$ = new string(">"); }
| LE	{ $$ = new string("<="); }
| GE	{ $$ = new string(">="); }

operator:
  PLUS	{ $$ = new string("+"); }
| MINUS	{ $$ = new string("-"); }
| TIMES	{ $$ = new string("*"); }
| DIV	{ $$ = new string("/"); }
| MOD	{ $$ = new string(" mod "); }
| AND	{ $$ = new string("&"); }
| OR	{ $$ = new string("?"); }
| XOR	{ $$ = new string("^"); }

ltl_query: LTL ltl_expr { program->addLTLQuery($2); }

ltl_expr:
  identifier				{ $$ = program->getLTLNode(IDENTIFIER, NULL, NULL, $1); }
| LBRAC ltl_expr RBRAC		{ $$ = $2; }
| LTLNOT ltl_expr			{ $$ = program->getLTLNode(LTLNOT, NULL, $2); }
| ltl_expr AND ltl_expr		{ $$ = program->getLTLNode(AND, $1, $3); }
| ltl_expr LTLOR ltl_expr	{ $$ = program->getLTLNode(LTLOR, $1, $3); }
| ltl_expr IMPL ltl_expr	{ $$ = program->getLTLNode(IMPL, $1, $3); }
| ltl_expr EQUIV ltl_expr	{ $$ = program->getLTLNode(EQUIV, $1, $3); }
| X ltl_expr				{ $$ = program->getLTLNode(X, NULL, $2); }
| G ltl_expr				{ $$ = program->getLTLNode(G, NULL, $2); }
| F ltl_expr				{ $$ = program->getLTLNode(F, NULL, $2); }
| ltl_expr U ltl_expr		{ $$ = program->getLTLNode(U, $1, $3); }
| ltl_expr R ltl_expr		{ $$ = program->getLTLNode(R, $1, $3); }


%%

/***************************************************************************************************************************/

void Parser::preProcessing(Coala::CompilerOptions* options) {
	program = new Program(options);
	program->setLine(&yylloc.first_line);
}

void Parser::postProcessing(Printer* p) {
	program->print(p);
	delete program;
}

std::string Parser::to_string(int integer) {
	std::stringstream ss;
	std::string str;
	ss << integer;
	ss >> str;
	return str;
}
