/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "MeshDataModel.h"
#include "MeshComponent.h"


#include <vtkCellData.h>
#include <vtkPointData.h>

namespace camitk {

// -------------------- constructor --------------------
MeshDataModel::MeshDataModel(MeshComponent * meshComp) :
    QAbstractTableModel(meshComp), meshComponent(meshComp) {
    // get ready to reset table
    beginResetModel();
}

// -------------------- rowCount --------------------
int MeshDataModel::rowCount(const QModelIndex & parent) const {
    if (!meshComponent || !(meshComponent->getPointSet())) {
        return 0;
    }

    return meshComponent->getPointSet()->GetCellData()->GetNumberOfArrays() + meshComponent->getPointSet()->GetPointData()->GetNumberOfArrays();
}

// -------------------- columnCount --------------------
int MeshDataModel::columnCount(const QModelIndex & parent) const {
    return 3;
}

// -------------------- getFieldTypeOfRow --------------------
MeshDataModel::FieldType MeshDataModel::getFieldTypeOfRow(const int row, int *dataIndex) const {
    int nbPointData = meshComponent->getPointSet()->GetPointData()->GetNumberOfArrays();
    int nbCellData =  meshComponent->getPointSet()->GetCellData()->GetNumberOfArrays();

    if (row < nbPointData) {
        *dataIndex = row;
        return POINTS;
    } else if (row < (nbPointData + nbCellData)) {
        *dataIndex = row - nbPointData;
        return CELLS;
    } else {
        *dataIndex = row - (nbPointData + nbCellData);
        return MESH;
    }
}

// -------------------- data --------------------
QVariant MeshDataModel::data(const QModelIndex & index, int role) const {
    if (!meshComponent || !(meshComponent->getPointSet())) {
        return QVariant();
    }

    const int row = index.row();
    int col = index.column();

    vtkSmartPointer<vtkFieldData> data;
    int dataNum;
    QString field;
    QString type;
    FieldType fieldType = getFieldTypeOfRow(row,&dataNum);

    switch (fieldType) {
    case POINTS:
        data = meshComponent->getPointSet()->GetPointData();
        field = "points";
        break;

    case CELLS:
        data = meshComponent->getPointSet()->GetCellData();
        field = "cells";
        break;
    case MESH:
    default:
        data = meshComponent->getPointSet()->GetFieldData();
        field = "mesh";
        break;
    }

    switch (data->GetArray(dataNum)->GetNumberOfComponents()) {
    case 1:
        type = QString("scalars");
        break;

    case 3 :
        type = QString("vectors");
        break;

    case 9 :
        type = QString("tensors");
        break;

    default:
        type = QString::number(data->GetArray(dataNum)->GetNumberOfComponents());
        break;
    }

    switch (role) {
    case Qt::DisplayRole:
        switch (col) {
        case 0 :
            return QString(data->GetArray(dataNum)->GetName());
            break;

        case 1 :
            return field;
            break;

        case 2 :
            return type;
            break;

        default:
            return QVariant();
            break;
        }

        break;

    case Qt::CheckStateRole:

        if ((col == 0) && field != "mesh") {
            switch (data->GetArray(dataNum)->GetNumberOfComponents()) {
            case 1: {
                vtkSmartPointer<vtkDataSetAttributes> dataSetAttr = vtkDataSetAttributes::SafeDownCast(data);

                if (dataSetAttr && dataSetAttr->GetScalars() && dataSetAttr->GetArray(dataNum) && QString(dataSetAttr->GetScalars()->GetName()) == QString(dataSetAttr->GetArray(dataNum)->GetName())) {
                    return Qt::Checked;
                } else {
                    return Qt::Unchecked;
                }
            }
            break;

            case 3:
            case 9: {
                vtkSmartPointer<vtkProp> prop = meshComponent->getDataProp(fieldType, data->GetArray(dataNum)->GetName());

                if (prop) {
                    if (prop->GetVisibility()) {
                        return Qt::Checked;
                    } else {
                        return Qt::Unchecked;
                    }
                }
            }
            break;

            default:
                break;
            }
        }

        break;

    case Qt::DecorationRole : {
        if (col == 1) {
            if (field == "points") {
                return QIcon(":/points");
            } else if (field == "cells") {
                return QIcon(":/cell");
            } else {
                return QVariant();
            }
        } else if (col ==  2) {
            if (type == "scalars") {
                return QIcon(":/scalars");
            } else if (type == "vectors") {
                return QIcon(":/vectors");
            } else {
                return QVariant();
            }
        }

        return QVariant();
        break;
    }

    case Qt::BackgroundRole : {
        //data->GetScalars()->GetName();

        break;
    }

    default:
        return QVariant();
        break;
    }

    return QVariant();
}

// -------------------- setData --------------------
bool MeshDataModel::setData(const QModelIndex & index, const QVariant & value, int role) {

    if (role == Qt::CheckStateRole && index.column() == 0) {
        int row = index.row();
        vtkDataSetAttributes * data;
        vtkProp * prop;
        int dataNum;
        FieldType fieldType = getFieldTypeOfRow(row,&dataNum);

        switch (fieldType) {
        case POINTS:
            data = meshComponent->getPointSet()->GetPointData();
            prop = meshComponent->getDataProp(POINTS, data->GetArray(dataNum)->GetName());
            break;

        case CELLS:
            data = meshComponent->getPointSet()->GetCellData();
            prop = meshComponent->getDataProp(CELLS, data->GetArray(dataNum)->GetName());
            break;

        default:
            // nothing to do if this is a MESH data
            return false;
        }

        // Some data do not have an additional vtkProp (e.g. scalars are represented directly as point data colors)
        if (!prop) {
            if (data->GetArray(dataNum)->GetNumberOfComponents() == 1) {
                if (value == Qt::Checked) {
                    vtkSmartPointer<vtkDataSetAttributes> dataSetAttr = vtkDataSetAttributes::SafeDownCast(data);
                    // if this is a scalar value, set it to active (to display the colors on the cells/points)
                    meshComponent->setActiveData(fieldType, dataSetAttr->GetArray(dataNum)->GetName());
                } else {
                    meshComponent->setActiveData(fieldType, NULL);
                }
                // nothing to if the button is not checked: the unactive scalars are going to be refreshed in data(..) method
            } else {
                // not managed
                return false;
            }
        } else {
            if (value == Qt::Checked) {
                prop->VisibilityOn();
            } else {
                prop->VisibilityOff();
            }
        }

        meshComponent->refresh();
        return true;
    }

    return false;
}

Qt::ItemFlags MeshDataModel::flags(const QModelIndex & index) const {
    if (index.column() == 0) {
        return Qt::ItemIsSelectable | /* Qt::ItemIsEditable | */ Qt::ItemIsEnabled | Qt::ItemIsUserCheckable;
    } else {
        return Qt::ItemIsSelectable  | Qt::ItemIsEnabled ;
    }
}


QVariant MeshDataModel::headerData(int section, Qt::Orientation orientation, int role) const {
    if (role == Qt::DisplayRole) {
        if (orientation == Qt::Horizontal) {
            switch (section) {
            case 0:
                return QString("Name");
                break;

            case 1:
                return QString("Field");
                break;

            case 2:
                return QString("Type");
                break;

            default:
                return QVariant();
                break;
            }
        }
    }

    return QVariant();
}

void MeshDataModel::refresh() {
    endResetModel();
}

const QMap< int, QString > & MeshDataModel::getFieldNames() {
    static QMap<int, QString> fieldNames = initFieldNames();

    return fieldNames;
}

const QMap< int, QString > & MeshDataModel::getDataNames() {
    static QMap<int, QString> dataNames = initDataNames();

    return dataNames;
}

QMap< int, QString > MeshDataModel::initFieldNames() {
    QMap< int, QString > fieldNames;
    fieldNames[MeshDataModel::POINTS] = "points";
    fieldNames[MeshDataModel::CELLS] = "cells";
    fieldNames[MeshDataModel::CELLS] = "mesh";

    return fieldNames;
}

QMap< int, QString > MeshDataModel::initDataNames() {
    QMap< int, QString > dataNames;
    dataNames[MeshDataModel::SCALARS] = "scalars";
    dataNames[MeshDataModel::VECTORS] = "vectors";
    dataNames[MeshDataModel::TENSORS] = "tensors";
    dataNames[MeshDataModel::OTHERS] = "";

    return dataNames;
}

MeshDataFilterModel::MeshDataFilterModel(int fieldFilter, int dataFilter, QObject * parent) :
    QSortFilterProxyModel(parent),
    fieldTypeFilter(fieldFilter),
    dataTypeFilter(dataFilter) {
}

void MeshDataFilterModel::setFieldTypeFilter(int fieldFilter) {
    fieldTypeFilter = fieldFilter;
}

void MeshDataFilterModel::setDataTypeFilter(int dataFilter) {
    dataTypeFilter = dataFilter;
}

bool MeshDataFilterModel::filterAcceptsRow(int sourceRow, const QModelIndex & sourceParent) const {
    QModelIndex fieldIndex = sourceModel()->index(sourceRow, 1, sourceParent);
    QModelIndex typeIndex = sourceModel()->index(sourceRow, 2, sourceParent);

    return (MeshDataModel::getFieldNames().key(sourceModel()->data(fieldIndex).toString(), 0) & fieldTypeFilter) &&
           (MeshDataModel::getDataNames().key(sourceModel()->data(typeIndex).toString(), 0) & dataTypeFilter);
}

}
