/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core image component stuff
#include "SingleImageComponent.h"
#include "ImageComponent.h"

// -- Core stuff
#include "Slice.h"
#include "InteractiveViewer.h"
#include "Log.h"
#include "Application.h"

// -- VTK stuff
#include <vtkUnstructuredGrid.h>
#include <vtkImageClip.h>
#include <vtkImageChangeInformation.h>
#include <vtkProperty.h>

#include <cmath>

namespace camitk {
// -------------------- constructor  --------------------
SingleImageComponent::SingleImageComponent(Component *parentComponent, InterfaceBitMap::PossibleOrientation sliceOrientation, const QString & name, vtkSmartPointer<vtkWindowLevelLookupTable> lut)
    : Component(parentComponent, name, Component::SLICE) {

    this->sliceOrientation = sliceOrientation;

    this->lut = lut;

    // build the slice 3D
    initRepresentation();

    // by default, this slice is not shown in 3D
    this->viewSliceIn3D = false;
}

// -------------------- destructor  --------------------
SingleImageComponent::~SingleImageComponent() {
}

// -------------------- setSelected --------------------
void SingleImageComponent::setSelected(const bool b, const bool r) {
    dynamic_cast<ImageComponent*>(getParent())->setSelected(b, false);
}

// -------------------- singleImageSelected --------------------
void SingleImageComponent::singleImageSelected(const bool b) {
    Component::setSelected(b, false);
}

// -------------------- getViewSliceIn3D --------------------
bool SingleImageComponent::getViewSliceIn3D() const {
    return viewSliceIn3D;
}

// -------------------- setViewSliceIn3D --------------------
void SingleImageComponent::setViewSliceIn3D(bool toggle) {
    this->viewSliceIn3D = toggle;
    this->setVisibility(InteractiveViewer::get3DViewer(), viewSliceIn3D);
    InteractiveViewer::get3DViewer()->refresh();
}


// -------------------- initRepresentation --------------------
void SingleImageComponent::initRepresentation() {
    mySlice = new Slice(dynamic_cast<ImageComponent *>(getParentComponent())->getImageData(), sliceOrientation, this->lut);

    switch (sliceOrientation) {
    case InterfaceBitMap::AXIAL_ORIENTATION:
        setVisibility(InteractiveViewer::getAxialViewer(), true);
        break;
    case InterfaceBitMap::CORONAL_ORIENTATION:
        setVisibility(InteractiveViewer::getCoronalViewer(), true);
        break;
    case InterfaceBitMap::SAGITTAL_ORIENTATION:
        setVisibility(InteractiveViewer::getSagittalViewer(), true);
        break;
    case InterfaceBitMap::ARBITRARY_ORIENTATION:
        setVisibility(InteractiveViewer::getArbitraryViewer(), true);
        break;
    }
}


// ---------------------- pixelPicked  ----------------------------
void SingleImageComponent::pixelPicked(double i, double j, double k) {
    // At this point, coordinates are represented in real frame (world frame)
    double xyz[3];
    double ijk[3];
    double resliceSpacing[3];

    // Get the spacing of the reslice data.
    ((Slice*)mySlice)->getImageData()->GetSpacing( resliceSpacing );

    // Compute coordiantes in the resliced volume. Coordinates are rounded.
    ijk[0] = floor((i+0.5) / resliceSpacing[0]);
    ijk[1] = floor((j+0.5) / resliceSpacing[1]);
    ijk[2] = (int) ((Slice*)mySlice)->getSlice();

    // Compute coordinates in original volume.
    ((Slice*)mySlice)->reslicedToVolumeCoords(ijk, xyz);

    //-- tell my parent
    ((ImageComponent*)getParent())->pixelPicked(xyz[0], xyz[1], xyz[2], this);
}

}
