// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __bclist_h__
#define __bclist_h__

namespace Puma {

template <class Item, int BUCKET_SIZE = 32>
class BCList {

protected:
  struct Bucket {
    Bucket *_prev;
    Bucket *_next;
    Item *_write_pos;
    Item *_end_pos;
    Item _data[BUCKET_SIZE]; // large amount of token pointers to avoid new

    Bucket () : _prev (0), _next (0), _write_pos (_data),
      _end_pos (_data + BUCKET_SIZE) {}
    ~Bucket () { if (_next) delete _next; }
    void clear () {
      if (_next) delete _next;
      _next = 0;
      _write_pos = _data;
    }
    Item *first () { return _data; }
    Item *pos () const { return _write_pos; }
    Item *end () const { return _end_pos; }
    bool full () const { return _write_pos == _end_pos; }
    Bucket *new_bucket () {
      _next = new Bucket;
      _next->_prev = this;
      return _next;
    }
    void add (Item token) {
      *_write_pos = token;
      _write_pos++;
    }
  };

  Bucket *_first_bucket; // the first bucket is part of the container
  Bucket *_last_bucket; // pointer to the last bucket for insertion

  void check () {
    if (_first_bucket == 0) {
      _first_bucket = new Bucket ();
      _last_bucket = _first_bucket;
    }
  }

public:
  class Iterator {
    friend class BCList<Item, BUCKET_SIZE>;

    Item *_item;
    Bucket *_bucket;

    Iterator (Item *item, Bucket *bucket) :
      _item (item), _bucket (bucket) {}

  public:
    Iterator () : _item (0) {}

    // this one is needed, because somewhere in Puma 0 is passed for an
    // Iterator argument
    Iterator (int i) { _item = 0; }

    operator bool () const { return _item != 0; }

    // comparison function
    bool operator == (const Iterator& other) const {
      return _item == other._item;
    }
    bool operator != (const Iterator &other) const {
      return !(*this == other);
    }

    // ++ operators to iterate
    Iterator &operator ++ () {
      _item++;
      if (_item == _bucket->pos ()) {
        _bucket = _bucket->_next;
        _item = _bucket ? _bucket->first () : 0;
      }
      return *this;
    }
    Iterator operator ++ (int) {
      Iterator temp (*this);
      ++(*this);
      return temp;
    }
    Iterator &operator -- () {
      if (_item == _bucket->first ()) {
        _bucket = _bucket->_prev;
        if (_bucket) {
          _item = _bucket->pos ();
          _item--;
        }
        else {
          _item = 0;
        }
      }
      else {
        _item--; 
      }
      return *this;
    }
    Iterator operator -- (int) {
      Iterator temp (*this);
      --(*this);
      return temp;
    }

    // access functions
    Item &operator * () const { return *_item; }
    Item *operator -> () const { return _item; }
  };

  // constructor
  BCList () : _first_bucket (0), _last_bucket (0) {}

  // destructor
  ~BCList () {
    if (_first_bucket != 0)
      delete _first_bucket;
  }

  // start and end of the list
  Iterator begin () {
    check ();
    return Iterator (_first_bucket->first (), _first_bucket);
  }

  Iterator end () const {
    return Iterator ();
  }

  // add an element to the list
  Iterator add (Item item) {
    check ();
    // if the last bucket is full extend the bucket chain
    if (_last_bucket->full ())
      _last_bucket = _last_bucket->new_bucket ();
    // the result is the position of the inserted item
    Item *result = _last_bucket->pos ();
    // add the item; the bucket is not full!
    _last_bucket->add (item);
    // return an iterator for the new item
    return Iterator (result, _last_bucket);
  }

  void reset () {
    if (_first_bucket != 0) {
      _first_bucket->clear ();
      _last_bucket = _first_bucket;
    }
  }
};

} // namespace Puma

#endif /* __bclist_h__ */
