/************************************************************************************
  Copyright (C) 2014 MariaDB Corporation Ab

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not see <http://www.gnu.org/licenses>
  or write to the Free Software Foundation, Inc.,
  51 Franklin St., Fifth Floor, Boston, MA 02110, USA

  Author: Georg Richter

 *************************************************************************************/
#include "ma_schannel.h"
#include <assert.h>

#define SC_IO_BUFFER_SIZE 0x4000
#define MAX_SSL_ERR_LEN 100

#define SCHANNEL_PAYLOAD(A) (A).cbMaximumMessage - (A).cbHeader - (A).cbTrailer

/* {{{ void ma_schannel_set_sec_error */
void ma_schannel_set_sec_error(MARIADB_PVIO *pvio, DWORD ErrorNo)
{
  MYSQL *mysql= pvio->mysql;
  switch(ErrorNo) {
  case SEC_E_UNTRUSTED_ROOT:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Untrusted root certificate");
    break;
  case SEC_E_BUFFER_TOO_SMALL:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Buffer too small");
    break;
  case SEC_E_CRYPTO_SYSTEM_INVALID:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Cipher is not supported");
    break;
  case SEC_E_INSUFFICIENT_MEMORY:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Out of memory");
    break;
  case SEC_E_OUT_OF_SEQUENCE:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Invalid message sequence");
    break;
  case SEC_E_DECRYPT_FAILURE:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "An error occured during decrypting data");
    break;
  case SEC_I_INCOMPLETE_CREDENTIALS:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Incomplete credentials");
    break;
  case SEC_E_ENCRYPT_FAILURE:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "An error occured during encrypting data");
    break;
  case SEC_I_CONTEXT_EXPIRED:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Context expired: ");
  case SEC_E_OK:
    break;
  default:
    pvio->set_error(mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Unknown SSL error (%d)", ErrorNo);
  }
}
/* }}} */

/* {{{ void ma_schnnel_set_win_error */
void ma_schannel_set_win_error(MARIADB_PVIO *pvio)
{
  ulong ssl_errno= GetLastError();
  char *ssl_error_reason= NULL;

  if (!ssl_errno)
  {
    pvio->set_error(pvio->mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Unknown SSL error");
    return;
  }
  /* todo: obtain error messge */
  FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
                NULL, ssl_errno, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                (LPTSTR) &ssl_error_reason, 0, NULL );
  pvio->set_error(pvio->mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, ssl_error_reason);

  if (ssl_error_reason)
    LocalFree(ssl_error_reason);
  return;
}
/* }}} */

/* {{{ LPBYTE ma_schannel_load_pem(const char *PemFileName, DWORD *buffer_len) */
/*
   Load a pem or clr file and convert it to a binary DER object

   SYNOPSIS
     ma_schannel_load_pem()
     PemFileName           name of the pem file (in)
     buffer_len            length of the converted DER binary

   DESCRIPTION
     Loads a X509 file (ca, certification, key or clr) into memory and converts
     it to a DER binary object. This object can be decoded and loaded into
     a schannel crypto context.
     If the function failed, error can be retrieved by GetLastError()
     The returned binary object must be freed by caller.

   RETURN VALUE
     NULL                  if the conversion failed or file was not found
     LPBYTE *              a pointer to a binary der object
                           buffer_len will contain the length of binary der object
*/
static LPBYTE ma_schannel_load_pem(MARIADB_PVIO *pvio, const char *PemFileName, DWORD *buffer_len)
{
  HANDLE hfile;
  char   *buffer= NULL;
  DWORD dwBytesRead= 0;
  LPBYTE der_buffer= NULL;
  DWORD der_buffer_length;

  if (buffer_len == NULL)
    return NULL;

  
  if ((hfile= CreateFile(PemFileName, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 
                          FILE_ATTRIBUTE_NORMAL, NULL )) == INVALID_HANDLE_VALUE)
  {
    ma_schannel_set_win_error(pvio);
    return NULL;
  }

  if (!(*buffer_len = GetFileSize(hfile, NULL)))
  {
     pvio->set_error(pvio->mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Invalid pem format");
     goto end;
  }

  if (!(buffer= LocalAlloc(0, *buffer_len + 1)))
  {
    pvio->set_error(pvio->mysql, CR_OUT_OF_MEMORY, SQLSTATE_UNKNOWN, NULL);
    goto end;
  }

  if (!ReadFile(hfile, buffer, *buffer_len, &dwBytesRead, NULL))
  {
    ma_schannel_set_win_error(pvio);
    goto end;
  }

  CloseHandle(hfile);

  /* calculate the length of DER binary */
  if (!CryptStringToBinaryA(buffer, *buffer_len, CRYPT_STRING_BASE64HEADER,
                            NULL, &der_buffer_length, NULL, NULL))
  {
    ma_schannel_set_win_error(pvio);
    goto end;
  }
  /* allocate DER binary buffer */
  if (!(der_buffer= (LPBYTE)LocalAlloc(0, der_buffer_length)))
  {
    pvio->set_error(pvio->mysql, CR_OUT_OF_MEMORY, SQLSTATE_UNKNOWN, NULL);
    goto end;
  }
  /* convert to DER binary */
  if (!CryptStringToBinaryA(buffer, *buffer_len, CRYPT_STRING_BASE64HEADER,
                            der_buffer, &der_buffer_length, NULL, NULL))
  {
    ma_schannel_set_win_error(pvio);
    goto end;
  }

  *buffer_len= der_buffer_length;
  LocalFree(buffer);
  
  return der_buffer;

end:
  if (hfile != INVALID_HANDLE_VALUE)
    CloseHandle(hfile);
  if (buffer)
    LocalFree(buffer);
  if (der_buffer)
    LocalFree(der_buffer);
  *buffer_len= 0;
  return NULL;
}
/* }}} */

/* {{{ CERT_CONTEXT *ma_schannel_create_cert_context(MARIADB_PVIO *pvio, const char *pem_file) */
/*
  Create a certification context from ca or cert file

  SYNOPSIS
    ma_schannel_create_cert_context()
    pvio                    pvio object
    pem_file               name of certificate or ca file

  DESCRIPTION
    Loads a PEM file (certificate authority or certificate) creates a certification
    context and loads the binary representation into context.
    The returned context must be freed by caller.
    If the function failed, error can be retrieved by GetLastError().

  RETURNS
    NULL                   If loading of the file or creating context failed
    CERT_CONTEXT *         A pointer to a certification context structure
*/
CERT_CONTEXT *ma_schannel_create_cert_context(MARIADB_PVIO *pvio, const char *pem_file)
{
  DWORD der_buffer_length;
  LPBYTE der_buffer= NULL;

  CERT_CONTEXT *ctx= NULL;

  /* create DER binary object from ca/certification file */
  if (!(der_buffer= ma_schannel_load_pem(pvio, pem_file, (DWORD *)&der_buffer_length)))
    goto end;
  if (!(ctx= (CERT_CONTEXT *)CertCreateCertificateContext(X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
                                    der_buffer, der_buffer_length)))
    ma_schannel_set_win_error(pvio);

end:
  if (der_buffer)
    LocalFree(der_buffer);
  return ctx;
}
/* }}} */

/* {{{ PCCRL_CONTEXT ma_schannel_create_crl_context(MARIADB_PVIO *pvio, const char *pem_file) */
/*
  Create a crl context from crlfile

  SYNOPSIS
    ma_schannel_create_crl_context()
    pem_file               name of certificate or ca file

  DESCRIPTION
    Loads a certification revocation list file, creates a certification
    context and loads the binary representation into crl context.
    The returned context must be freed by caller.
    If the function failed, error can be retrieved by GetLastError().

  RETURNS
    NULL                   If loading of the file or creating context failed
    PCCRL_CONTEXT          A pointer to a certification context structure
*/
PCCRL_CONTEXT ma_schannel_create_crl_context(MARIADB_PVIO *pvio, const char *pem_file)
{
  DWORD der_buffer_length;
  LPBYTE der_buffer= NULL;

  PCCRL_CONTEXT ctx= NULL;

  /* load ca pem file into memory */
  if (!(der_buffer= ma_schannel_load_pem(pvio, pem_file, (DWORD *)&der_buffer_length)))
    goto end;
  if (!(ctx= CertCreateCRLContext(X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
                                    der_buffer, der_buffer_length)))
    ma_schannel_set_win_error(pvio);
end:
  if (der_buffer)
    LocalFree(der_buffer);
  return ctx;
}
/* }}} */

/* {{{ my_bool ma_schannel_load_private_key(MARIADB_PVIO *pvio, CERT_CONTEXT *ctx, char *key_file) */
/*
  Load privte key into context

  SYNOPSIS
    ma_schannel_load_private_key()
    ctx                    pointer to a certification context
    pem_file               name of certificate or ca file

  DESCRIPTION
    Loads a certification revocation list file, creates a certification
    context and loads the binary representation into crl context.
    The returned context must be freed by caller.
    If the function failed, error can be retrieved by GetLastError().

  RETURNS
    NULL                   If loading of the file or creating context failed
    PCCRL_CONTEXT          A pointer to a certification context structure
*/

my_bool ma_schannel_load_private_key(MARIADB_PVIO *pvio, CERT_CONTEXT *ctx, char *key_file)
{
   DWORD der_buffer_len= 0;
   LPBYTE der_buffer= NULL;
   DWORD priv_key_len= 0;
   LPBYTE priv_key= NULL;
   HCRYPTPROV  crypt_prov= 0;
   HCRYPTKEY  crypt_key= 0;
   CERT_KEY_CONTEXT kpi;
   my_bool rc= 0;

   /* load private key into der binary object */
   if (!(der_buffer= ma_schannel_load_pem(pvio, key_file, &der_buffer_len)))
     return 0;

   /* determine required buffer size for decoded private key */
   if (!CryptDecodeObjectEx(X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
                            PKCS_RSA_PRIVATE_KEY,
                            der_buffer, der_buffer_len,
                            0, NULL,
                            NULL, &priv_key_len))
   {
     ma_schannel_set_win_error(pvio);
     goto end;
   }

   /* allocate buffer for decoded private key */
   if (!(priv_key= LocalAlloc(0, priv_key_len)))
   {
     pvio->set_error(pvio->mysql, CR_OUT_OF_MEMORY, SQLSTATE_UNKNOWN, NULL);
     goto end;
   }

   /* decode */
   if (!CryptDecodeObjectEx(X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
                            PKCS_RSA_PRIVATE_KEY,
                            der_buffer, der_buffer_len,
                            0, NULL,
                            priv_key, &priv_key_len))
   {
     ma_schannel_set_win_error(pvio);
     goto end;
   }

   /* Acquire context:
      If pvio_schannel context doesn't exist, create a new one */
   if (!CryptAcquireContext(&crypt_prov, "pvio_schannel", MS_ENHANCED_PROV, PROV_RSA_FULL, 0))
   if (!CryptAcquireContext(&crypt_prov, "pvio_schannel", MS_ENHANCED_PROV, PROV_RSA_FULL, CRYPT_NEWKEYSET))
   {
     ma_schannel_set_win_error(pvio);
     goto end;
   }
   /* ... and import the private key */
   if (!CryptImportKey(crypt_prov, priv_key, priv_key_len, 0, 0, (HCRYPTKEY *)&crypt_key))
   {
     ma_schannel_set_win_error(pvio);
     goto end;
   }

   SecureZeroMemory(&kpi, sizeof(kpi));
   kpi.hCryptProv= crypt_prov;
   kpi.dwKeySpec = AT_KEYEXCHANGE;
   kpi.cbSize= sizeof(kpi);

   /* assign private key to certificate context */
   if (CertSetCertificateContextProperty(ctx, CERT_KEY_CONTEXT_PROP_ID, 0, &kpi))
     rc= 1;
   else
     ma_schannel_set_win_error(pvio);

end:
  if (der_buffer)
    LocalFree(der_buffer);
  if (priv_key)
  {
    if (crypt_key)
      CryptDestroyKey(crypt_key);
    LocalFree(priv_key);
  if (!rc)
    if (crypt_prov)
      CryptReleaseContext(crypt_prov, 0);
  }
  return rc;
}
/* }}} */

/* {{{ SECURITY_STATUS ma_schannel_handshake_loop(MARIADB_PVIO *pvio, my_bool InitialRead, SecBuffer *pExtraData) */
/*
  perform handshake loop

  SYNOPSIS
    ma_schannel_handshake_loop()
    pvio            Pointer to an Communication/IO structure
    InitialRead    TRUE if it's the very first read
    ExtraData      Pointer to an SecBuffer which contains extra data (sent by application)

    
*/

SECURITY_STATUS ma_schannel_handshake_loop(MARIADB_PVIO *pvio, my_bool InitialRead, SecBuffer *pExtraData)
{
  SecBufferDesc   OutBuffer, InBuffer;
  SecBuffer       InBuffers[2], OutBuffers[1];
  DWORD           dwSSPIFlags, dwSSPIOutFlags, cbData, cbIoBuffer;
  TimeStamp       tsExpiry;
  SECURITY_STATUS rc;
  PUCHAR          IoBuffer;
  BOOL            fDoRead;
  MARIADB_SSL     *cssl= pvio->cssl;
  SC_CTX          *sctx= (SC_CTX *)cssl->ssl;


  dwSSPIFlags = ISC_REQ_SEQUENCE_DETECT |
                ISC_REQ_REPLAY_DETECT |
                ISC_REQ_CONFIDENTIALITY |
                ISC_RET_EXTENDED_ERROR |
                ISC_REQ_ALLOCATE_MEMORY | 
                ISC_REQ_STREAM;


  /* Allocate data buffer */
  if (!(IoBuffer = LocalAlloc(LMEM_FIXED, SC_IO_BUFFER_SIZE)))
    return SEC_E_INSUFFICIENT_MEMORY;

  cbIoBuffer = 0;
  fDoRead = InitialRead;

  /* handshake loop: We will leave a handshake is finished
     or an error occurs */

  rc = SEC_I_CONTINUE_NEEDED;

  while (rc == SEC_I_CONTINUE_NEEDED ||
         rc == SEC_E_INCOMPLETE_MESSAGE ||
         rc == SEC_I_INCOMPLETE_CREDENTIALS )
  {
    /* Read data */
    if (rc == SEC_E_INCOMPLETE_MESSAGE ||
        !cbIoBuffer)
    {
      if(fDoRead)
      {
        cbData = (DWORD)pvio->methods->read(pvio, IoBuffer + cbIoBuffer, (size_t)(SC_IO_BUFFER_SIZE - cbIoBuffer));
        if (cbData == SOCKET_ERROR || cbData == 0)
        {
          rc = SEC_E_INTERNAL_ERROR;
          break;
        }
        cbIoBuffer += cbData;
      }
      else
        fDoRead = TRUE;
    }

    /* input buffers
       First buffer stores data received from server. leftover data
       will be stored in second buffer with BufferType SECBUFFER_EXTRA */

    InBuffers[0].pvBuffer   = IoBuffer;
    InBuffers[0].cbBuffer   = cbIoBuffer;
    InBuffers[0].BufferType = SECBUFFER_TOKEN;

    InBuffers[1].pvBuffer   = NULL;
    InBuffers[1].cbBuffer   = 0;
    InBuffers[1].BufferType = SECBUFFER_EMPTY;

    InBuffer.cBuffers       = 2;
    InBuffer.pBuffers       = InBuffers;
    InBuffer.ulVersion      = SECBUFFER_VERSION;


    /* output buffer */
    OutBuffers[0].pvBuffer  = NULL;
    OutBuffers[0].BufferType= SECBUFFER_TOKEN;
    OutBuffers[0].cbBuffer  = 0;

    OutBuffer.cBuffers      = 1;
    OutBuffer.pBuffers      = OutBuffers;
    OutBuffer.ulVersion     = SECBUFFER_VERSION;


    rc = InitializeSecurityContextA(&sctx->CredHdl,
                                    &sctx->ctxt,
                                    NULL,
                                    dwSSPIFlags,
                                    0,
                                    SECURITY_NATIVE_DREP,
                                    &InBuffer,
                                    0,
                                    NULL,
                                    &OutBuffer,
                                    &dwSSPIOutFlags,
                                    &tsExpiry );


    if (rc == SEC_E_OK  ||
        rc == SEC_I_CONTINUE_NEEDED ||
        FAILED(rc) && (dwSSPIOutFlags & ISC_RET_EXTENDED_ERROR))
    {
      if(OutBuffers[0].cbBuffer && OutBuffers[0].pvBuffer)
      {
        cbData= (DWORD)pvio->methods->write(pvio, (uchar *)OutBuffers[0].pvBuffer, (size_t)OutBuffers[0].cbBuffer);
        if(cbData == SOCKET_ERROR || cbData == 0)
        {
          FreeContextBuffer(OutBuffers[0].pvBuffer);
          DeleteSecurityContext(&sctx->ctxt);
          return SEC_E_INTERNAL_ERROR;
        }

        /* Free output context buffer */
        FreeContextBuffer(OutBuffers[0].pvBuffer);
        OutBuffers[0].pvBuffer = NULL;
      }
    }

    /* check if we need to read more data */
    switch (rc) {
    case SEC_E_INCOMPLETE_MESSAGE:
      /* we didn't receive all data, so just continue loop */
      continue;
      break;
    case SEC_E_OK:
      /* handshake completed, but we need to check if extra
         data was sent (which contains encrypted application data) */
      if (InBuffers[1].BufferType == SECBUFFER_EXTRA)
      {
        if (!(pExtraData->pvBuffer= LocalAlloc(0, InBuffers[1].cbBuffer)))
          return SEC_E_INSUFFICIENT_MEMORY;
        
        MoveMemory(pExtraData->pvBuffer, IoBuffer + (cbIoBuffer - InBuffers[1].cbBuffer), InBuffers[1].cbBuffer );
        pExtraData->BufferType = SECBUFFER_TOKEN;
        pExtraData->cbBuffer   = InBuffers[1].cbBuffer;
      }
      else
      {
        pExtraData->BufferType= SECBUFFER_EMPTY;
        pExtraData->pvBuffer= NULL;
        pExtraData->cbBuffer= 0;
      }
    break;
 
    case SEC_I_INCOMPLETE_CREDENTIALS:
      /* Provided credentials didn't contain a valid client certificate.
         We will try to connect anonymously, using current credentials */
      fDoRead= FALSE;
      rc= SEC_I_CONTINUE_NEEDED;
      continue;
      break;
    default:
      if (FAILED(rc))
      {
        ma_schannel_set_sec_error(pvio, rc);
        goto loopend;
      }
      break;
    }

    if ( InBuffers[1].BufferType == SECBUFFER_EXTRA )
    {
      MoveMemory( IoBuffer, IoBuffer + (cbIoBuffer - InBuffers[1].cbBuffer), InBuffers[1].cbBuffer );
      cbIoBuffer = InBuffers[1].cbBuffer;
    }

    cbIoBuffer = 0;
  }
loopend:
  if (FAILED(rc)) 
    DeleteSecurityContext(&sctx->ctxt);
  LocalFree(IoBuffer);

  return rc;
}
/* }}} */

/* {{{ SECURITY_STATUS ma_schannel_client_handshake(MARIADB_SSL *cssl) */
/*
   performs client side handshake 

   SYNOPSIS
     ma_schannel_client_handshake()
     cssl             Pointer to a MARIADB_SSL structure

   DESCRIPTION
     initiates a client/server handshake. This function can be used
     by clients only

   RETURN
     SEC_E_OK         on success
*/

SECURITY_STATUS ma_schannel_client_handshake(MARIADB_SSL *cssl)
{
  MARIADB_PVIO *pvio;
  SECURITY_STATUS sRet;
  DWORD OutFlags;
  DWORD r;
  SC_CTX *sctx;
  SecBuffer ExtraData;
  DWORD SFlags= ISC_REQ_SEQUENCE_DETECT | ISC_REQ_REPLAY_DETECT |
                ISC_REQ_CONFIDENTIALITY | ISC_RET_EXTENDED_ERROR | 
                ISC_REQ_USE_SUPPLIED_CREDS |
                ISC_REQ_ALLOCATE_MEMORY | ISC_REQ_STREAM;
  
  SecBufferDesc	BufferOut;
  SecBuffer  BuffersOut[1];

  if (!cssl || !cssl->pvio)
    return 1;

  pvio= cssl->pvio;
  sctx= (SC_CTX *)cssl->ssl;

  /* Initialie securifty context */
  BuffersOut[0].BufferType= SECBUFFER_TOKEN;
  BuffersOut[0].cbBuffer= 0;
  BuffersOut[0].pvBuffer= NULL;


  BufferOut.cBuffers= 1;
  BufferOut.pBuffers= BuffersOut;
  BufferOut.ulVersion= SECBUFFER_VERSION;

  sRet = InitializeSecurityContext(&sctx->CredHdl,
                                    NULL,
                                    pvio->mysql->host,
                                    SFlags,
                                    0,
                                    SECURITY_NATIVE_DREP,
                                    NULL,
                                    0,
                                    &sctx->ctxt,
                                    &BufferOut,
                                    &OutFlags,
                                    NULL);

  if(sRet != SEC_I_CONTINUE_NEEDED)
  {
    ma_schannel_set_sec_error(pvio, sRet);
    return sRet;
  }

  /* send client hello packaet */
  if(BuffersOut[0].cbBuffer != 0 && BuffersOut[0].pvBuffer != NULL)
  {  
    r= (DWORD)pvio->methods->write(pvio, (uchar *)BuffersOut[0].pvBuffer, (size_t)BuffersOut[0].cbBuffer);
    if (r <= 0)
    {
      sRet= SEC_E_INTERNAL_ERROR;
      goto end;
    }
  }
  sRet= ma_schannel_handshake_loop(pvio, TRUE, &ExtraData);

  /* allocate IO-Buffer for write operations: After handshake
  was successfull, we are able now to calculate payload */
  if ((sRet = QueryContextAttributes(&sctx->ctxt, SECPKG_ATTR_STREAM_SIZES, &sctx->Sizes )))
    goto end;

  sctx->IoBufferSize= SCHANNEL_PAYLOAD(sctx->Sizes);
  if (!(sctx->IoBuffer= (PUCHAR)LocalAlloc(0, sctx->IoBufferSize)))
  {
    sRet= SEC_E_INSUFFICIENT_MEMORY;
    goto end;
  }
    
  return sRet;
end:
  LocalFree(sctx->IoBuffer);
  sctx->IoBufferSize= 0;
  FreeContextBuffer(BuffersOut[0].pvBuffer);
  DeleteSecurityContext(&sctx->ctxt);
  return sRet;
}
/* }}} */

/* {{{ SECURITY_STATUS ma_schannel_read_decrypt(MARIADB_PVIO *pvio, PCredHandle phCreds, CtxtHandle * phContext,
                                                DWORD DecryptLength, uchar *ReadBuffer, DWORD ReadBufferSize) */
/*
  Reads encrypted data from a SSL stream and decrypts it.

  SYNOPSIS
    ma_schannel_read
    pvio              pointer to Communication IO structure
    phContext        a context handle
    DecryptLength    size of decrypted buffer
    ReadBuffer       Buffer for decrypted data
    ReadBufferSize   size of ReadBuffer


  DESCRIPTION
    Reads decrypted data from a SSL stream and encrypts it.

  RETURN
    SEC_E_OK         on success
    SEC_E_*          if an error occured
*/  

SECURITY_STATUS ma_schannel_read_decrypt(MARIADB_PVIO *pvio,
                                         PCredHandle phCreds,
                                         CtxtHandle * phContext,
                                         DWORD *DecryptLength,
                                         uchar *ReadBuffer,
                                         DWORD ReadBufferSize)
{
  DWORD dwBytesRead= 0;
  DWORD dwOffset= 0;
  SC_CTX *sctx;
  SECURITY_STATUS sRet= 0;
  SecBufferDesc Msg;
  SecBuffer Buffers[4],
            *pData, *pExtra;
  int i;

  if (!pvio || !pvio->methods || !pvio->methods->read || !pvio->cssl || !DecryptLength)
    return SEC_E_INTERNAL_ERROR;

  sctx= (SC_CTX *)pvio->cssl->ssl;
  *DecryptLength= 0;

  while (1)
  {
    if (!dwBytesRead || sRet == SEC_E_INCOMPLETE_MESSAGE)
    {
      dwBytesRead= (DWORD)pvio->methods->read(pvio, sctx->IoBuffer + dwOffset, (size_t)(sctx->IoBufferSize - dwOffset));
      if (dwBytesRead == 0)
      {
        /* server closed connection */
        // todo: error 
        return SEC_E_INVALID_HANDLE;
      }
      if (dwBytesRead < 0)
      {
        /* socket error */
        // todo: error
        return SEC_E_INVALID_HANDLE;
      }
      dwOffset+= dwBytesRead;
    }
    ZeroMemory(Buffers, sizeof(SecBuffer) * 4);
    Buffers[0].pvBuffer= sctx->IoBuffer;
    Buffers[0].cbBuffer= dwOffset;

    Buffers[0].BufferType= SECBUFFER_DATA; 
    Buffers[1].BufferType=
    Buffers[2].BufferType=
    Buffers[3].BufferType= SECBUFFER_EMPTY;

    Msg.ulVersion= SECBUFFER_VERSION;    // Version number
    Msg.cBuffers= 4;
    Msg.pBuffers= Buffers;

    sRet = DecryptMessage(phContext, &Msg, 0, NULL);

    /* Check for possible errors: we continue in case context has 
       expired or renogitiation is required */
    if (sRet != SEC_E_OK && sRet != SEC_I_CONTEXT_EXPIRED &&
        sRet != SEC_I_RENEGOTIATE && sRet != SEC_E_INCOMPLETE_MESSAGE)
    {
      ma_schannel_set_sec_error(pvio, sRet);
      return sRet;
    }

    pData= pExtra= NULL;
    for (i=0; i < 4; i++)
    {
      if (!pData && Buffers[i].BufferType == SECBUFFER_DATA)
        pData= &Buffers[i];
      if (!pExtra && Buffers[i].BufferType == SECBUFFER_EXTRA)
        pExtra= &Buffers[i];
      if (pData && pExtra)
        break;
    }
    
    if (pData && pData->cbBuffer)
    {
      memcpy(ReadBuffer + *DecryptLength, pData->pvBuffer, pData->cbBuffer);
      *DecryptLength+= pData->cbBuffer;
      return sRet;
    }

    if (pExtra)
    {
      MoveMemory(sctx->IoBuffer, pExtra->pvBuffer, pExtra->cbBuffer);
      dwOffset= pExtra->cbBuffer;
    }
    else
      dwOffset= 0;
  }    
}
/* }}} */

my_bool ma_schannel_verify_certs(SC_CTX *sctx, DWORD dwCertFlags)
{
  SECURITY_STATUS sRet;
  DWORD flags;
  MARIADB_PVIO *pvio= sctx->mysql->net.pvio;
  PCCERT_CONTEXT pServerCert= NULL;

  if ((sRet= QueryContextAttributes(&sctx->ctxt, SECPKG_ATTR_REMOTE_CERT_CONTEXT, (PVOID)&pServerCert)) != SEC_E_OK)
  {
    ma_schannel_set_sec_error(pvio, sRet);
    return 0;
  }

  flags= CERT_STORE_SIGNATURE_FLAG |
         CERT_STORE_TIME_VALIDITY_FLAG;


    
  if (sctx->client_ca_ctx)
  {
 	  if (!(sRet= CertVerifySubjectCertificateContext(pServerCert,
                                                    sctx->client_ca_ctx,
                                                    &flags)))
    {
      ma_schannel_set_win_error(pvio);
      return 0;
    }

    if (flags)
    {
      if ((flags & CERT_STORE_SIGNATURE_FLAG) != 0)
        pvio->set_error(sctx->mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Certificate signature check failed");
      else if ((flags & CERT_STORE_REVOCATION_FLAG) != 0)
        pvio->set_error(sctx->mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "certificate was revoked");
      else if ((flags & CERT_STORE_TIME_VALIDITY_FLAG) != 0)
        pvio->set_error(sctx->mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "certificate has expired");
      else
        pvio->set_error(sctx->mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "Unknown error during certificate validation");
      return 0;
    }
  }

  /* Check if none of the certificates in the certificate chain have been revoked. */
  if (sctx->client_crl_ctx)
  {
    PCRL_INFO Info[1];

    Info[0]= sctx->client_crl_ctx->pCrlInfo;
    if (!(CertVerifyCRLRevocation(X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
                                  pServerCert->pCertInfo,
                                  1, Info))                               )
    {
      pvio->set_error(pvio->mysql, CR_SSL_CONNECTION_ERROR, SQLSTATE_UNKNOWN, "CRL Revocation failed");
      return 0;
    }
  }
  return 1;
}


/* {{{ size_t ma_schannel_write_encrypt(MARIADB_PVIO *pvio, PCredHandle phCreds, CtxtHandle * phContext) */
/*
  Decrypts data and write to SSL stream
  SYNOPSIS
    ma_schannel_write_decrypt
    pvio              pointer to Communication IO structure
    phContext        a context handle
    DecryptLength    size of decrypted buffer
    ReadBuffer       Buffer for decrypted data
    ReadBufferSize   size of ReadBuffer

  DESCRIPTION
    Write encrypted data to SSL stream.

  RETURN
    SEC_E_OK         on success
    SEC_E_*          if an error occured
*/ 
size_t ma_schannel_write_encrypt(MARIADB_PVIO *pvio,
                                 uchar *WriteBuffer,
                                 size_t WriteBufferSize)
{
  SECURITY_STATUS scRet;
  SecBufferDesc Message;
  SecBuffer Buffers[4];
  DWORD cbMessage;
  PBYTE pbMessage;
  SC_CTX *sctx= (SC_CTX *)pvio->cssl->ssl;
  size_t payload;

  payload= MIN(WriteBufferSize, sctx->IoBufferSize);

  memcpy(&sctx->IoBuffer[sctx->Sizes.cbHeader], WriteBuffer, payload);
  pbMessage = sctx->IoBuffer + sctx->Sizes.cbHeader; 
  cbMessage = (DWORD)payload;
  
  Buffers[0].pvBuffer     = sctx->IoBuffer;
  Buffers[0].cbBuffer     = sctx->Sizes.cbHeader;
  Buffers[0].BufferType   = SECBUFFER_STREAM_HEADER;    // Type of the buffer

  Buffers[1].pvBuffer     = &sctx->IoBuffer[sctx->Sizes.cbHeader];
  Buffers[1].cbBuffer     = (DWORD)payload;
  Buffers[1].BufferType   = SECBUFFER_DATA;

  Buffers[2].pvBuffer     = &sctx->IoBuffer[sctx->Sizes.cbHeader] + payload;
  Buffers[2].cbBuffer     = sctx->Sizes.cbTrailer;
  Buffers[2].BufferType   = SECBUFFER_STREAM_TRAILER;

  Buffers[3].pvBuffer     = SECBUFFER_EMPTY;                    // Pointer to buffer 4
  Buffers[3].cbBuffer     = SECBUFFER_EMPTY;                    // length of buffer 4
  Buffers[3].BufferType   = SECBUFFER_EMPTY;                    // Type of the buffer 4


  Message.ulVersion       = SECBUFFER_VERSION;
  Message.cBuffers        = 4;
  Message.pBuffers        = Buffers;
  if ((scRet = EncryptMessage(&sctx->ctxt, 0, &Message, 0))!= SEC_E_OK)
    return -1;
  
  if (pvio->methods->write(pvio, sctx->IoBuffer, Buffers[0].cbBuffer + Buffers[1].cbBuffer + Buffers[2].cbBuffer))
    return payload; 
  return 0;
}
/* }}} */

extern char *ssl_protocol_version[5];

/* {{{ ma_ssl_get_protocol_version(MARIADB_SSL *cssl, struct st_ssl_version *version) */
my_bool ma_ssl_get_protocol_version(MARIADB_SSL *cssl, struct st_ssl_version *version)
{
  SC_CTX *sctx;
  SecPkgContext_ConnectionInfo ConnectionInfo;

  if (!cssl->ssl)
    return 1;

  sctx= (SC_CTX *)cssl->ssl;

  if (QueryContextAttributes(&sctx->ctxt, SECPKG_ATTR_CONNECTION_INFO, &ConnectionInfo) != SEC_E_OK)
    return 1;

  switch(ConnectionInfo.dwProtocol)
  {
  case SP_PROT_SSL3_CLIENT:
    version->iversion= 1;
    break;
  case SP_PROT_TLS1_CLIENT:
    version->iversion= 2;
    break;
  case SP_PROT_TLS1_1_CLIENT:
    version->iversion= 3;
    break;
  case SP_PROT_TLS1_2_CLIENT:
    version->iversion= 4;
    break;
  default:
    version->iversion= 0;
    break;
  }
  version->cversion= ssl_protocol_version[version->iversion];
  return 0;
}
/* }}} */
